/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cleanroomsml.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides configuration information for the dockerized container where the model algorithm is stored.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ContainerConfig implements SdkPojo, Serializable, ToCopyableBuilder<ContainerConfig.Builder, ContainerConfig> {
    private static final SdkField<String> IMAGE_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageUri").getter(getter(ContainerConfig::imageUri)).setter(setter(Builder::imageUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageUri").build()).build();

    private static final SdkField<List<String>> ENTRYPOINT_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("entrypoint")
            .getter(getter(ContainerConfig::entrypoint))
            .setter(setter(Builder::entrypoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entrypoint").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ARGUMENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("arguments")
            .getter(getter(ContainerConfig::arguments))
            .setter(setter(Builder::arguments))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arguments").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<MetricDefinition>> METRIC_DEFINITIONS_FIELD = SdkField
            .<List<MetricDefinition>> builder(MarshallingType.LIST)
            .memberName("metricDefinitions")
            .getter(getter(ContainerConfig::metricDefinitions))
            .setter(setter(Builder::metricDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metricDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_URI_FIELD,
            ENTRYPOINT_FIELD, ARGUMENTS_FIELD, METRIC_DEFINITIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String imageUri;

    private final List<String> entrypoint;

    private final List<String> arguments;

    private final List<MetricDefinition> metricDefinitions;

    private ContainerConfig(BuilderImpl builder) {
        this.imageUri = builder.imageUri;
        this.entrypoint = builder.entrypoint;
        this.arguments = builder.arguments;
        this.metricDefinitions = builder.metricDefinitions;
    }

    /**
     * <p>
     * The registry path of the docker image that contains the algorithm. Clean Rooms ML currently only supports the
     * <code>registry/repository[:tag]</code> image path format. For more information about using images in Clean Rooms
     * ML, see the <a href=
     * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AlgorithmSpecification.html#sagemaker-Type-AlgorithmSpecification-TrainingImage"
     * >Sagemaker API reference</a>.
     * </p>
     * 
     * @return The registry path of the docker image that contains the algorithm. Clean Rooms ML currently only supports
     *         the <code>registry/repository[:tag]</code> image path format. For more information about using images in
     *         Clean Rooms ML, see the <a href=
     *         "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AlgorithmSpecification.html#sagemaker-Type-AlgorithmSpecification-TrainingImage"
     *         >Sagemaker API reference</a>.
     */
    public final String imageUri() {
        return imageUri;
    }

    /**
     * For responses, this returns true if the service returned a value for the Entrypoint property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEntrypoint() {
        return entrypoint != null && !(entrypoint instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The entrypoint script for a Docker container used to run a training job. This script takes precedence over the
     * default train processing instructions. See How Amazon SageMaker Runs Your Training Image for additional
     * information. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
     * Sagemaker runs your training image</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEntrypoint} method.
     * </p>
     * 
     * @return The entrypoint script for a Docker container used to run a training job. This script takes precedence
     *         over the default train processing instructions. See How Amazon SageMaker Runs Your Training Image for
     *         additional information. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
     *         Sagemaker runs your training image</a>.
     */
    public final List<String> entrypoint() {
        return entrypoint;
    }

    /**
     * For responses, this returns true if the service returned a value for the Arguments property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasArguments() {
        return arguments != null && !(arguments instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The arguments for a container used to run a training job. See How Amazon SageMaker Runs Your Training Image for
     * additional information. For more information, see <a
     * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
     * Sagemaker runs your training image</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArguments} method.
     * </p>
     * 
     * @return The arguments for a container used to run a training job. See How Amazon SageMaker Runs Your Training
     *         Image for additional information. For more information, see <a
     *         href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
     *         Sagemaker runs your training image</a>.
     */
    public final List<String> arguments() {
        return arguments;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricDefinitions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricDefinitions() {
        return metricDefinitions != null && !(metricDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of metric definition objects. Each object specifies the metric name and regular expressions used to parse
     * algorithm logs. Amazon Web Services Clean Rooms ML publishes each metric to all members' Amazon CloudWatch using
     * IAM role configured in <a>PutMLConfiguration</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricDefinitions} method.
     * </p>
     * 
     * @return A list of metric definition objects. Each object specifies the metric name and regular expressions used
     *         to parse algorithm logs. Amazon Web Services Clean Rooms ML publishes each metric to all members' Amazon
     *         CloudWatch using IAM role configured in <a>PutMLConfiguration</a>.
     */
    public final List<MetricDefinition> metricDefinitions() {
        return metricDefinitions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(imageUri());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntrypoint() ? entrypoint() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasArguments() ? arguments() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricDefinitions() ? metricDefinitions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ContainerConfig)) {
            return false;
        }
        ContainerConfig other = (ContainerConfig) obj;
        return Objects.equals(imageUri(), other.imageUri()) && hasEntrypoint() == other.hasEntrypoint()
                && Objects.equals(entrypoint(), other.entrypoint()) && hasArguments() == other.hasArguments()
                && Objects.equals(arguments(), other.arguments()) && hasMetricDefinitions() == other.hasMetricDefinitions()
                && Objects.equals(metricDefinitions(), other.metricDefinitions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ContainerConfig").add("ImageUri", imageUri())
                .add("Entrypoint", hasEntrypoint() ? entrypoint() : null).add("Arguments", hasArguments() ? arguments() : null)
                .add("MetricDefinitions", hasMetricDefinitions() ? metricDefinitions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "imageUri":
            return Optional.ofNullable(clazz.cast(imageUri()));
        case "entrypoint":
            return Optional.ofNullable(clazz.cast(entrypoint()));
        case "arguments":
            return Optional.ofNullable(clazz.cast(arguments()));
        case "metricDefinitions":
            return Optional.ofNullable(clazz.cast(metricDefinitions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("imageUri", IMAGE_URI_FIELD);
        map.put("entrypoint", ENTRYPOINT_FIELD);
        map.put("arguments", ARGUMENTS_FIELD);
        map.put("metricDefinitions", METRIC_DEFINITIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ContainerConfig, T> g) {
        return obj -> g.apply((ContainerConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ContainerConfig> {
        /**
         * <p>
         * The registry path of the docker image that contains the algorithm. Clean Rooms ML currently only supports the
         * <code>registry/repository[:tag]</code> image path format. For more information about using images in Clean
         * Rooms ML, see the <a href=
         * "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AlgorithmSpecification.html#sagemaker-Type-AlgorithmSpecification-TrainingImage"
         * >Sagemaker API reference</a>.
         * </p>
         * 
         * @param imageUri
         *        The registry path of the docker image that contains the algorithm. Clean Rooms ML currently only
         *        supports the <code>registry/repository[:tag]</code> image path format. For more information about
         *        using images in Clean Rooms ML, see the <a href=
         *        "https://docs.aws.amazon.com/sagemaker/latest/APIReference/API_AlgorithmSpecification.html#sagemaker-Type-AlgorithmSpecification-TrainingImage"
         *        >Sagemaker API reference</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUri(String imageUri);

        /**
         * <p>
         * The entrypoint script for a Docker container used to run a training job. This script takes precedence over
         * the default train processing instructions. See How Amazon SageMaker Runs Your Training Image for additional
         * information. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Sagemaker runs your training image</a>.
         * </p>
         * 
         * @param entrypoint
         *        The entrypoint script for a Docker container used to run a training job. This script takes precedence
         *        over the default train processing instructions. See How Amazon SageMaker Runs Your Training Image for
         *        additional information. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Sagemaker runs your training image</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entrypoint(Collection<String> entrypoint);

        /**
         * <p>
         * The entrypoint script for a Docker container used to run a training job. This script takes precedence over
         * the default train processing instructions. See How Amazon SageMaker Runs Your Training Image for additional
         * information. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Sagemaker runs your training image</a>.
         * </p>
         * 
         * @param entrypoint
         *        The entrypoint script for a Docker container used to run a training job. This script takes precedence
         *        over the default train processing instructions. See How Amazon SageMaker Runs Your Training Image for
         *        additional information. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Sagemaker runs your training image</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entrypoint(String... entrypoint);

        /**
         * <p>
         * The arguments for a container used to run a training job. See How Amazon SageMaker Runs Your Training Image
         * for additional information. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Sagemaker runs your training image</a>.
         * </p>
         * 
         * @param arguments
         *        The arguments for a container used to run a training job. See How Amazon SageMaker Runs Your Training
         *        Image for additional information. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Sagemaker runs your training image</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arguments(Collection<String> arguments);

        /**
         * <p>
         * The arguments for a container used to run a training job. See How Amazon SageMaker Runs Your Training Image
         * for additional information. For more information, see <a
         * href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html">How
         * Sagemaker runs your training image</a>.
         * </p>
         * 
         * @param arguments
         *        The arguments for a container used to run a training job. See How Amazon SageMaker Runs Your Training
         *        Image for additional information. For more information, see <a
         *        href="https://docs.aws.amazon.com/sagemaker/latest/dg/your-algorithms-training-algo-dockerfile.html"
         *        >How Sagemaker runs your training image</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arguments(String... arguments);

        /**
         * <p>
         * A list of metric definition objects. Each object specifies the metric name and regular expressions used to
         * parse algorithm logs. Amazon Web Services Clean Rooms ML publishes each metric to all members' Amazon
         * CloudWatch using IAM role configured in <a>PutMLConfiguration</a>.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of metric definition objects. Each object specifies the metric name and regular expressions
         *        used to parse algorithm logs. Amazon Web Services Clean Rooms ML publishes each metric to all members'
         *        Amazon CloudWatch using IAM role configured in <a>PutMLConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions);

        /**
         * <p>
         * A list of metric definition objects. Each object specifies the metric name and regular expressions used to
         * parse algorithm logs. Amazon Web Services Clean Rooms ML publishes each metric to all members' Amazon
         * CloudWatch using IAM role configured in <a>PutMLConfiguration</a>.
         * </p>
         * 
         * @param metricDefinitions
         *        A list of metric definition objects. Each object specifies the metric name and regular expressions
         *        used to parse algorithm logs. Amazon Web Services Clean Rooms ML publishes each metric to all members'
         *        Amazon CloudWatch using IAM role configured in <a>PutMLConfiguration</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDefinitions(MetricDefinition... metricDefinitions);

        /**
         * <p>
         * A list of metric definition objects. Each object specifies the metric name and regular expressions used to
         * parse algorithm logs. Amazon Web Services Clean Rooms ML publishes each metric to all members' Amazon
         * CloudWatch using IAM role configured in <a>PutMLConfiguration</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cleanroomsml.model.MetricDefinition.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.cleanroomsml.model.MetricDefinition#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cleanroomsml.model.MetricDefinition.Builder#build()} is called
         * immediately and its result is passed to {@link #metricDefinitions(List<MetricDefinition>)}.
         * 
         * @param metricDefinitions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cleanroomsml.model.MetricDefinition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricDefinitions(java.util.Collection<MetricDefinition>)
         */
        Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions);
    }

    static final class BuilderImpl implements Builder {
        private String imageUri;

        private List<String> entrypoint = DefaultSdkAutoConstructList.getInstance();

        private List<String> arguments = DefaultSdkAutoConstructList.getInstance();

        private List<MetricDefinition> metricDefinitions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ContainerConfig model) {
            imageUri(model.imageUri);
            entrypoint(model.entrypoint);
            arguments(model.arguments);
            metricDefinitions(model.metricDefinitions);
        }

        public final String getImageUri() {
            return imageUri;
        }

        public final void setImageUri(String imageUri) {
            this.imageUri = imageUri;
        }

        @Override
        public final Builder imageUri(String imageUri) {
            this.imageUri = imageUri;
            return this;
        }

        public final Collection<String> getEntrypoint() {
            if (entrypoint instanceof SdkAutoConstructList) {
                return null;
            }
            return entrypoint;
        }

        public final void setEntrypoint(Collection<String> entrypoint) {
            this.entrypoint = ContainerEntrypointCopier.copy(entrypoint);
        }

        @Override
        public final Builder entrypoint(Collection<String> entrypoint) {
            this.entrypoint = ContainerEntrypointCopier.copy(entrypoint);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entrypoint(String... entrypoint) {
            entrypoint(Arrays.asList(entrypoint));
            return this;
        }

        public final Collection<String> getArguments() {
            if (arguments instanceof SdkAutoConstructList) {
                return null;
            }
            return arguments;
        }

        public final void setArguments(Collection<String> arguments) {
            this.arguments = ContainerArgumentsCopier.copy(arguments);
        }

        @Override
        public final Builder arguments(Collection<String> arguments) {
            this.arguments = ContainerArgumentsCopier.copy(arguments);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder arguments(String... arguments) {
            arguments(Arrays.asList(arguments));
            return this;
        }

        public final List<MetricDefinition.Builder> getMetricDefinitions() {
            List<MetricDefinition.Builder> result = MetricDefinitionListCopier.copyToBuilder(this.metricDefinitions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricDefinitions(Collection<MetricDefinition.BuilderImpl> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copyFromBuilder(metricDefinitions);
        }

        @Override
        public final Builder metricDefinitions(Collection<MetricDefinition> metricDefinitions) {
            this.metricDefinitions = MetricDefinitionListCopier.copy(metricDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(MetricDefinition... metricDefinitions) {
            metricDefinitions(Arrays.asList(metricDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricDefinitions(Consumer<MetricDefinition.Builder>... metricDefinitions) {
            metricDefinitions(Stream.of(metricDefinitions).map(c -> MetricDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ContainerConfig build() {
            return new ContainerConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
