/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloud9;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.cloud9.internal.Cloud9ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.cloud9.model.BadRequestException;
import software.amazon.awssdk.services.cloud9.model.Cloud9Exception;
import software.amazon.awssdk.services.cloud9.model.ConcurrentAccessException;
import software.amazon.awssdk.services.cloud9.model.ConflictException;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentEc2Request;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentEc2Response;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.CreateEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.cloud9.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentMembershipsResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentStatusRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentStatusResponse;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentsRequest;
import software.amazon.awssdk.services.cloud9.model.DescribeEnvironmentsResponse;
import software.amazon.awssdk.services.cloud9.model.ForbiddenException;
import software.amazon.awssdk.services.cloud9.model.InternalServerErrorException;
import software.amazon.awssdk.services.cloud9.model.LimitExceededException;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.cloud9.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloud9.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloud9.model.NotFoundException;
import software.amazon.awssdk.services.cloud9.model.TagResourceRequest;
import software.amazon.awssdk.services.cloud9.model.TagResourceResponse;
import software.amazon.awssdk.services.cloud9.model.TooManyRequestsException;
import software.amazon.awssdk.services.cloud9.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloud9.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentMembershipRequest;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentMembershipResponse;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.cloud9.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.cloud9.transform.CreateEnvironmentEc2RequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.CreateEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DeleteEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentMembershipsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentStatusRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.DescribeEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UpdateEnvironmentMembershipRequestMarshaller;
import software.amazon.awssdk.services.cloud9.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link Cloud9AsyncClient}.
 *
 * @see Cloud9AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloud9AsyncClient implements Cloud9AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloud9AsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloud9AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Cloud9 development environment, launches an Amazon Elastic Compute Cloud (Amazon EC2) instance, and
     * then connects from the instance to the environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param createEnvironmentEc2Request
     * @return A Java Future containing the result of the CreateEnvironmentEC2 operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.CreateEnvironmentEC2
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/CreateEnvironmentEC2" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentEc2Response> createEnvironmentEC2(
            CreateEnvironmentEc2Request createEnvironmentEc2Request) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentEc2Request,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentEc2Request
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironmentEC2");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentEc2Response> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentEc2Response::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentEc2Response> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentEc2Request, CreateEnvironmentEc2Response>()
                            .withOperationName("CreateEnvironmentEC2").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEnvironmentEc2RequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEnvironmentEc2Request));
            CompletableFuture<CreateEnvironmentEc2Response> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds an environment member to an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param createEnvironmentMembershipRequest
     * @return A Java Future containing the result of the CreateEnvironmentMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.CreateEnvironmentMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/CreateEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentMembershipResponse> createEnvironmentMembership(
            CreateEnvironmentMembershipRequest createEnvironmentMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironmentMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentMembershipRequest, CreateEnvironmentMembershipResponse>()
                            .withOperationName("CreateEnvironmentMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEnvironmentMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEnvironmentMembershipRequest));
            CompletableFuture<CreateEnvironmentMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Cloud9 development environment. If an Amazon EC2 instance is connected to the environment, also
     * terminates the instance.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DeleteEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                            .withOperationName("DeleteEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEnvironmentRequest));
            CompletableFuture<DeleteEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an environment member from a development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param deleteEnvironmentMembershipRequest
     * @return A Java Future containing the result of the DeleteEnvironmentMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DeleteEnvironmentMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DeleteEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentMembershipResponse> deleteEnvironmentMembership(
            DeleteEnvironmentMembershipRequest deleteEnvironmentMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironmentMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentMembershipRequest, DeleteEnvironmentMembershipResponse>()
                            .withOperationName("DeleteEnvironmentMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEnvironmentMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEnvironmentMembershipRequest));
            CompletableFuture<DeleteEnvironmentMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about environment members for an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param describeEnvironmentMembershipsRequest
     * @return A Java Future containing the result of the DescribeEnvironmentMemberships operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DescribeEnvironmentMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEnvironmentMembershipsResponse> describeEnvironmentMemberships(
            DescribeEnvironmentMembershipsRequest describeEnvironmentMembershipsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEnvironmentMembershipsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeEnvironmentMembershipsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEnvironmentMemberships");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEnvironmentMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEnvironmentMembershipsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEnvironmentMembershipsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentMembershipsRequest, DescribeEnvironmentMembershipsResponse>()
                            .withOperationName("DescribeEnvironmentMemberships").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEnvironmentMembershipsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEnvironmentMembershipsRequest));
            CompletableFuture<DescribeEnvironmentMembershipsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets status information for an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param describeEnvironmentStatusRequest
     * @return A Java Future containing the result of the DescribeEnvironmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DescribeEnvironmentStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEnvironmentStatusResponse> describeEnvironmentStatus(
            DescribeEnvironmentStatusRequest describeEnvironmentStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEnvironmentStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEnvironmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEnvironmentStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEnvironmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEnvironmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEnvironmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentStatusRequest, DescribeEnvironmentStatusResponse>()
                            .withOperationName("DescribeEnvironmentStatus").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEnvironmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEnvironmentStatusRequest));
            CompletableFuture<DescribeEnvironmentStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about Cloud9 development environments.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param describeEnvironmentsRequest
     * @return A Java Future containing the result of the DescribeEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.DescribeEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/DescribeEnvironments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEnvironmentsResponse> describeEnvironments(
            DescribeEnvironmentsRequest describeEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEnvironmentsRequest, DescribeEnvironmentsResponse>()
                            .withOperationName("DescribeEnvironments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEnvironmentsRequest));
            CompletableFuture<DescribeEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of Cloud9 development environment identifiers.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important> <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                            .withOperationName("ListEnvironments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEnvironmentsRequest));
            CompletableFuture<ListEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of the tags associated with an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important> <important>
     * <p>
     * Tags that you add to an Cloud9 environment by using this method will NOT be automatically propagated to
     * underlying resources.
     * </p>
     * </important>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConcurrentAccessException A concurrent access issue occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConcurrentAccessException A concurrent access issue occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the settings of an existing Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param updateEnvironmentRequest
     * @return A Java Future containing the result of the UpdateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UpdateEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentResponse> updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                            .withOperationName("UpdateEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEnvironmentRequest));
            CompletableFuture<UpdateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Changes the settings of an existing environment member for an Cloud9 development environment.
     * </p>
     * <important>
     * <p>
     * Cloud9 is no longer available to new customers. Existing customers of Cloud9 can continue to use the service as
     * normal. <a
     * href="http://aws.amazon.com/blogs/devops/how-to-migrate-from-aws-cloud9-to-aws-ide-toolkits-or-aws-cloudshell/"
     * >Learn more"</a>
     * </p>
     * </important>
     *
     * @param updateEnvironmentMembershipRequest
     * @return A Java Future containing the result of the UpdateEnvironmentMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>BadRequestException The target request is invalid.</li>
     *         <li>ConflictException A conflict occurred.</li>
     *         <li>NotFoundException The target resource cannot be found.</li>
     *         <li>ForbiddenException An access permissions issue occurred.</li>
     *         <li>TooManyRequestsException Too many service requests were made over the given time period.</li>
     *         <li>LimitExceededException A service limit was exceeded.</li>
     *         <li>InternalServerErrorException An internal server error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>Cloud9Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample Cloud9AsyncClient.UpdateEnvironmentMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloud9-2017-09-23/UpdateEnvironmentMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentMembershipResponse> updateEnvironmentMembership(
            UpdateEnvironmentMembershipRequest updateEnvironmentMembershipRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnvironmentMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cloud9");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironmentMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnvironmentMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentMembershipRequest, UpdateEnvironmentMembershipResponse>()
                            .withOperationName("UpdateEnvironmentMembership").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEnvironmentMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEnvironmentMembershipRequest));
            CompletableFuture<UpdateEnvironmentMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final Cloud9ServiceClientConfiguration serviceClientConfiguration() {
        return new Cloud9ServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Cloud9Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentAccessException")
                                .exceptionBuilderSupplier(ConcurrentAccessException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        Cloud9ServiceClientConfigurationBuilder serviceConfigBuilder = new Cloud9ServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
