/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudcontrol.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the current status of a resource operation request. For more information, see <a
 * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html">Managing
 * resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProgressEvent implements SdkPojo, Serializable, ToCopyableBuilder<ProgressEvent.Builder, ProgressEvent> {
    private static final SdkField<String> TYPE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TypeName").getter(getter(ProgressEvent::typeName)).setter(setter(Builder::typeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TypeName").build()).build();

    private static final SdkField<String> IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Identifier").getter(getter(ProgressEvent::identifier)).setter(setter(Builder::identifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Identifier").build()).build();

    private static final SdkField<String> REQUEST_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequestToken").getter(getter(ProgressEvent::requestToken)).setter(setter(Builder::requestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestToken").build()).build();

    private static final SdkField<String> OPERATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Operation").getter(getter(ProgressEvent::operationAsString)).setter(setter(Builder::operation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operation").build()).build();

    private static final SdkField<String> OPERATION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperationStatus").getter(getter(ProgressEvent::operationStatusAsString))
            .setter(setter(Builder::operationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationStatus").build()).build();

    private static final SdkField<Instant> EVENT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EventTime").getter(getter(ProgressEvent::eventTime)).setter(setter(Builder::eventTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventTime").build()).build();

    private static final SdkField<String> RESOURCE_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceModel").getter(getter(ProgressEvent::resourceModel)).setter(setter(Builder::resourceModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceModel").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(ProgressEvent::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorCode").getter(getter(ProgressEvent::errorCodeAsString)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorCode").build()).build();

    private static final SdkField<Instant> RETRY_AFTER_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("RetryAfter").getter(getter(ProgressEvent::retryAfter)).setter(setter(Builder::retryAfter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetryAfter").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_NAME_FIELD,
            IDENTIFIER_FIELD, REQUEST_TOKEN_FIELD, OPERATION_FIELD, OPERATION_STATUS_FIELD, EVENT_TIME_FIELD,
            RESOURCE_MODEL_FIELD, STATUS_MESSAGE_FIELD, ERROR_CODE_FIELD, RETRY_AFTER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String typeName;

    private final String identifier;

    private final String requestToken;

    private final String operation;

    private final String operationStatus;

    private final Instant eventTime;

    private final String resourceModel;

    private final String statusMessage;

    private final String errorCode;

    private final Instant retryAfter;

    private ProgressEvent(BuilderImpl builder) {
        this.typeName = builder.typeName;
        this.identifier = builder.identifier;
        this.requestToken = builder.requestToken;
        this.operation = builder.operation;
        this.operationStatus = builder.operationStatus;
        this.eventTime = builder.eventTime;
        this.resourceModel = builder.resourceModel;
        this.statusMessage = builder.statusMessage;
        this.errorCode = builder.errorCode;
        this.retryAfter = builder.retryAfter;
    }

    /**
     * <p>
     * The name of the resource type used in the operation.
     * </p>
     * 
     * @return The name of the resource type used in the operation.
     */
    public final String typeName() {
        return typeName;
    }

    /**
     * <p>
     * The primary identifier for the resource.
     * </p>
     * <note>
     * <p>
     * In some cases, the resource identifier may be available before the resource operation has reached a status of
     * <code>SUCCESS</code>.
     * </p>
     * </note>
     * 
     * @return The primary identifier for the resource.</p> <note>
     *         <p>
     *         In some cases, the resource identifier may be available before the resource operation has reached a
     *         status of <code>SUCCESS</code>.
     *         </p>
     */
    public final String identifier() {
        return identifier;
    }

    /**
     * <p>
     * The unique token representing this resource operation request.
     * </p>
     * <p>
     * Use the <code>RequestToken</code> with <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html"
     * >GetResourceRequestStatus</a> to return the current status of a resource operation request.
     * </p>
     * 
     * @return The unique token representing this resource operation request.</p>
     *         <p>
     *         Use the <code>RequestToken</code> with <a
     *         href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html"
     *         >GetResourceRequestStatus</a> to return the current status of a resource operation request.
     */
    public final String requestToken() {
        return requestToken;
    }

    /**
     * <p>
     * The resource operation type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link Operation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The resource operation type.
     * @see Operation
     */
    public final Operation operation() {
        return Operation.fromValue(operation);
    }

    /**
     * <p>
     * The resource operation type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operation} will
     * return {@link Operation#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #operationAsString}.
     * </p>
     * 
     * @return The resource operation type.
     * @see Operation
     */
    public final String operationAsString() {
        return operation;
    }

    /**
     * <p>
     * The current status of the resource operation request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code>: The resource operation has not yet started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code>: The resource operation is currently in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code>: The resource operation has successfully completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for more
     * information.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operationStatus}
     * will return {@link OperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operationStatusAsString}.
     * </p>
     * 
     * @return The current status of the resource operation request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code>: The resource operation has not yet started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code>: The resource operation is currently in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code>: The resource operation has successfully completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for
     *         more information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
     *         </p>
     *         </li>
     * @see OperationStatus
     */
    public final OperationStatus operationStatus() {
        return OperationStatus.fromValue(operationStatus);
    }

    /**
     * <p>
     * The current status of the resource operation request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code>: The resource operation has not yet started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code>: The resource operation is currently in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCESS</code>: The resource operation has successfully completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for more
     * information.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #operationStatus}
     * will return {@link OperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #operationStatusAsString}.
     * </p>
     * 
     * @return The current status of the resource operation request.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code>: The resource operation has not yet started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code>: The resource operation is currently in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCESS</code>: The resource operation has successfully completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for
     *         more information.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
     *         </p>
     *         </li>
     * @see OperationStatus
     */
    public final String operationStatusAsString() {
        return operationStatus;
    }

    /**
     * <p>
     * When the resource operation request was initiated.
     * </p>
     * 
     * @return When the resource operation request was initiated.
     */
    public final Instant eventTime() {
        return eventTime;
    }

    /**
     * <p>
     * A JSON string containing the resource model, consisting of each resource property and its current value.
     * </p>
     * 
     * @return A JSON string containing the resource model, consisting of each resource property and its current value.
     */
    public final String resourceModel() {
        return resourceModel;
    }

    /**
     * <p>
     * Any message explaining the current status.
     * </p>
     * 
     * @return Any message explaining the current status.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * For requests with a status of <code>FAILED</code>, the associated error code.
     * </p>
     * <p>
     * For error code definitions, see <a
     * href="https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
     * >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
     * Development</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link HandlerErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return For requests with a status of <code>FAILED</code>, the associated error code.</p>
     *         <p>
     *         For error code definitions, see <a href=
     *         "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
     *         >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
     *         Development</i>.
     * @see HandlerErrorCode
     */
    public final HandlerErrorCode errorCode() {
        return HandlerErrorCode.fromValue(errorCode);
    }

    /**
     * <p>
     * For requests with a status of <code>FAILED</code>, the associated error code.
     * </p>
     * <p>
     * For error code definitions, see <a
     * href="https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
     * >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
     * Development</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link HandlerErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return For requests with a status of <code>FAILED</code>, the associated error code.</p>
     *         <p>
     *         For error code definitions, see <a href=
     *         "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
     *         >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
     *         Development</i>.
     * @see HandlerErrorCode
     */
    public final String errorCodeAsString() {
        return errorCode;
    }

    /**
     * <p>
     * When to next request the status of this resource operation request.
     * </p>
     * 
     * @return When to next request the status of this resource operation request.
     */
    public final Instant retryAfter() {
        return retryAfter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeName());
        hashCode = 31 * hashCode + Objects.hashCode(identifier());
        hashCode = 31 * hashCode + Objects.hashCode(requestToken());
        hashCode = 31 * hashCode + Objects.hashCode(operationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(operationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventTime());
        hashCode = 31 * hashCode + Objects.hashCode(resourceModel());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(errorCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(retryAfter());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProgressEvent)) {
            return false;
        }
        ProgressEvent other = (ProgressEvent) obj;
        return Objects.equals(typeName(), other.typeName()) && Objects.equals(identifier(), other.identifier())
                && Objects.equals(requestToken(), other.requestToken())
                && Objects.equals(operationAsString(), other.operationAsString())
                && Objects.equals(operationStatusAsString(), other.operationStatusAsString())
                && Objects.equals(eventTime(), other.eventTime()) && Objects.equals(resourceModel(), other.resourceModel())
                && Objects.equals(statusMessage(), other.statusMessage())
                && Objects.equals(errorCodeAsString(), other.errorCodeAsString())
                && Objects.equals(retryAfter(), other.retryAfter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProgressEvent").add("TypeName", typeName()).add("Identifier", identifier())
                .add("RequestToken", requestToken()).add("Operation", operationAsString())
                .add("OperationStatus", operationStatusAsString()).add("EventTime", eventTime())
                .add("ResourceModel", resourceModel() == null ? null : "*** Sensitive Data Redacted ***")
                .add("StatusMessage", statusMessage()).add("ErrorCode", errorCodeAsString()).add("RetryAfter", retryAfter())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TypeName":
            return Optional.ofNullable(clazz.cast(typeName()));
        case "Identifier":
            return Optional.ofNullable(clazz.cast(identifier()));
        case "RequestToken":
            return Optional.ofNullable(clazz.cast(requestToken()));
        case "Operation":
            return Optional.ofNullable(clazz.cast(operationAsString()));
        case "OperationStatus":
            return Optional.ofNullable(clazz.cast(operationStatusAsString()));
        case "EventTime":
            return Optional.ofNullable(clazz.cast(eventTime()));
        case "ResourceModel":
            return Optional.ofNullable(clazz.cast(resourceModel()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCodeAsString()));
        case "RetryAfter":
            return Optional.ofNullable(clazz.cast(retryAfter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProgressEvent, T> g) {
        return obj -> g.apply((ProgressEvent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProgressEvent> {
        /**
         * <p>
         * The name of the resource type used in the operation.
         * </p>
         * 
         * @param typeName
         *        The name of the resource type used in the operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder typeName(String typeName);

        /**
         * <p>
         * The primary identifier for the resource.
         * </p>
         * <note>
         * <p>
         * In some cases, the resource identifier may be available before the resource operation has reached a status of
         * <code>SUCCESS</code>.
         * </p>
         * </note>
         * 
         * @param identifier
         *        The primary identifier for the resource.</p> <note>
         *        <p>
         *        In some cases, the resource identifier may be available before the resource operation has reached a
         *        status of <code>SUCCESS</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifier(String identifier);

        /**
         * <p>
         * The unique token representing this resource operation request.
         * </p>
         * <p>
         * Use the <code>RequestToken</code> with <a
         * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html"
         * >GetResourceRequestStatus</a> to return the current status of a resource operation request.
         * </p>
         * 
         * @param requestToken
         *        The unique token representing this resource operation request.</p>
         *        <p>
         *        Use the <code>RequestToken</code> with <a
         *        href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html"
         *        >GetResourceRequestStatus</a> to return the current status of a resource operation request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestToken(String requestToken);

        /**
         * <p>
         * The resource operation type.
         * </p>
         * 
         * @param operation
         *        The resource operation type.
         * @see Operation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Operation
         */
        Builder operation(String operation);

        /**
         * <p>
         * The resource operation type.
         * </p>
         * 
         * @param operation
         *        The resource operation type.
         * @see Operation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Operation
         */
        Builder operation(Operation operation);

        /**
         * <p>
         * The current status of the resource operation request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code>: The resource operation has not yet started.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code>: The resource operation is currently in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code>: The resource operation has successfully completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for more
         * information.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operationStatus
         *        The current status of the resource operation request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code>: The resource operation has not yet started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code>: The resource operation is currently in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code>: The resource operation has successfully completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for
         *        more information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
         *        </p>
         *        </li>
         * @see OperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationStatus
         */
        Builder operationStatus(String operationStatus);

        /**
         * <p>
         * The current status of the resource operation request.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code>: The resource operation has not yet started.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code>: The resource operation is currently in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCESS</code>: The resource operation has successfully completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for more
         * information.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
         * </p>
         * </li>
         * </ul>
         * 
         * @param operationStatus
         *        The current status of the resource operation request.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code>: The resource operation has not yet started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code>: The resource operation is currently in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCESS</code>: The resource operation has successfully completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The resource operation has failed. Refer to the error code and status message for
         *        more information.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCEL_IN_PROGRESS</code>: The resource operation is in the process of being canceled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCEL_COMPLETE</code>: The resource operation has been canceled.
         *        </p>
         *        </li>
         * @see OperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationStatus
         */
        Builder operationStatus(OperationStatus operationStatus);

        /**
         * <p>
         * When the resource operation request was initiated.
         * </p>
         * 
         * @param eventTime
         *        When the resource operation request was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTime(Instant eventTime);

        /**
         * <p>
         * A JSON string containing the resource model, consisting of each resource property and its current value.
         * </p>
         * 
         * @param resourceModel
         *        A JSON string containing the resource model, consisting of each resource property and its current
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceModel(String resourceModel);

        /**
         * <p>
         * Any message explaining the current status.
         * </p>
         * 
         * @param statusMessage
         *        Any message explaining the current status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * For requests with a status of <code>FAILED</code>, the associated error code.
         * </p>
         * <p>
         * For error code definitions, see <a href=
         * "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
         * >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
         * Development</i>.
         * </p>
         * 
         * @param errorCode
         *        For requests with a status of <code>FAILED</code>, the associated error code.</p>
         *        <p>
         *        For error code definitions, see <a href=
         *        "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
         *        >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
         *        Development</i>.
         * @see HandlerErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HandlerErrorCode
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * For requests with a status of <code>FAILED</code>, the associated error code.
         * </p>
         * <p>
         * For error code definitions, see <a href=
         * "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
         * >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
         * Development</i>.
         * </p>
         * 
         * @param errorCode
         *        For requests with a status of <code>FAILED</code>, the associated error code.</p>
         *        <p>
         *        For error code definitions, see <a href=
         *        "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-test-contract-errors.html"
         *        >Handler error codes</a> in the <i>CloudFormation Command Line Interface User Guide for Extension
         *        Development</i>.
         * @see HandlerErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HandlerErrorCode
         */
        Builder errorCode(HandlerErrorCode errorCode);

        /**
         * <p>
         * When to next request the status of this resource operation request.
         * </p>
         * 
         * @param retryAfter
         *        When to next request the status of this resource operation request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryAfter(Instant retryAfter);
    }

    static final class BuilderImpl implements Builder {
        private String typeName;

        private String identifier;

        private String requestToken;

        private String operation;

        private String operationStatus;

        private Instant eventTime;

        private String resourceModel;

        private String statusMessage;

        private String errorCode;

        private Instant retryAfter;

        private BuilderImpl() {
        }

        private BuilderImpl(ProgressEvent model) {
            typeName(model.typeName);
            identifier(model.identifier);
            requestToken(model.requestToken);
            operation(model.operation);
            operationStatus(model.operationStatus);
            eventTime(model.eventTime);
            resourceModel(model.resourceModel);
            statusMessage(model.statusMessage);
            errorCode(model.errorCode);
            retryAfter(model.retryAfter);
        }

        public final String getTypeName() {
            return typeName;
        }

        public final void setTypeName(String typeName) {
            this.typeName = typeName;
        }

        @Override
        @Transient
        public final Builder typeName(String typeName) {
            this.typeName = typeName;
            return this;
        }

        public final String getIdentifier() {
            return identifier;
        }

        public final void setIdentifier(String identifier) {
            this.identifier = identifier;
        }

        @Override
        @Transient
        public final Builder identifier(String identifier) {
            this.identifier = identifier;
            return this;
        }

        public final String getRequestToken() {
            return requestToken;
        }

        public final void setRequestToken(String requestToken) {
            this.requestToken = requestToken;
        }

        @Override
        @Transient
        public final Builder requestToken(String requestToken) {
            this.requestToken = requestToken;
            return this;
        }

        public final String getOperation() {
            return operation;
        }

        public final void setOperation(String operation) {
            this.operation = operation;
        }

        @Override
        @Transient
        public final Builder operation(String operation) {
            this.operation = operation;
            return this;
        }

        @Override
        @Transient
        public final Builder operation(Operation operation) {
            this.operation(operation == null ? null : operation.toString());
            return this;
        }

        public final String getOperationStatus() {
            return operationStatus;
        }

        public final void setOperationStatus(String operationStatus) {
            this.operationStatus = operationStatus;
        }

        @Override
        @Transient
        public final Builder operationStatus(String operationStatus) {
            this.operationStatus = operationStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder operationStatus(OperationStatus operationStatus) {
            this.operationStatus(operationStatus == null ? null : operationStatus.toString());
            return this;
        }

        public final Instant getEventTime() {
            return eventTime;
        }

        public final void setEventTime(Instant eventTime) {
            this.eventTime = eventTime;
        }

        @Override
        @Transient
        public final Builder eventTime(Instant eventTime) {
            this.eventTime = eventTime;
            return this;
        }

        public final String getResourceModel() {
            return resourceModel;
        }

        public final void setResourceModel(String resourceModel) {
            this.resourceModel = resourceModel;
        }

        @Override
        @Transient
        public final Builder resourceModel(String resourceModel) {
            this.resourceModel = resourceModel;
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        @Transient
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        @Transient
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        @Override
        @Transient
        public final Builder errorCode(HandlerErrorCode errorCode) {
            this.errorCode(errorCode == null ? null : errorCode.toString());
            return this;
        }

        public final Instant getRetryAfter() {
            return retryAfter;
        }

        public final void setRetryAfter(Instant retryAfter) {
            this.retryAfter = retryAfter;
        }

        @Override
        @Transient
        public final Builder retryAfter(Instant retryAfter) {
            this.retryAfter = retryAfter;
            return this;
        }

        @Override
        public ProgressEvent build() {
            return new ProgressEvent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
