/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudcontrol;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.cloudcontrol.model.CancelResourceRequestRequest;
import software.amazon.awssdk.services.cloudcontrol.model.CancelResourceRequestResponse;
import software.amazon.awssdk.services.cloudcontrol.model.CreateResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.CreateResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.model.DeleteResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.DeleteResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequestStatusRequest;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequestStatusResponse;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsResponse;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourcesResponse;
import software.amazon.awssdk.services.cloudcontrol.model.UpdateResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.UpdateResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.paginators.ListResourceRequestsPublisher;
import software.amazon.awssdk.services.cloudcontrol.paginators.ListResourcesPublisher;
import software.amazon.awssdk.services.cloudcontrol.waiters.CloudControlAsyncWaiter;

/**
 * Service client for accessing CloudControlApi asynchronously. This can be created using the static {@link #builder()}
 * method.The asynchronous client performs non-blocking I/O when configured with any {@link SdkAsyncHttpClient}
 * supported in the SDK. However, full non-blocking is not guaranteed as the async client may perform blocking calls in
 * some cases such as credentials retrieval and endpoint discovery as part of the async API call.
 *
 * <p>
 * For more information about Amazon Web Services Cloud Control API, see the <a
 * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/what-is-cloudcontrolapi.html">Amazon Web Services
 * Cloud Control API User Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface CloudControlAsyncClient extends AwsClient {
    String SERVICE_NAME = "cloudcontrolapi";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "cloudcontrolapi";

    /**
     * <p>
     * Cancels the specified resource operation request. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-cancel"
     * >Canceling resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * Only resource operations requests with a status of <code>PENDING</code> or <code>IN_PROGRESS</code> can be
     * canceled.
     * </p>
     *
     * @param cancelResourceRequestRequest
     * @return A Java Future containing the result of the CancelResourceRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException The resource is currently being modified by another operation.</li>
     *         <li>RequestTokenNotFoundException A resource operation with the specified request token can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.CancelResourceRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/CancelResourceRequest"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CancelResourceRequestResponse> cancelResourceRequest(
            CancelResourceRequestRequest cancelResourceRequestRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Cancels the specified resource operation request. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-cancel"
     * >Canceling resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * Only resource operations requests with a status of <code>PENDING</code> or <code>IN_PROGRESS</code> can be
     * canceled.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CancelResourceRequestRequest.Builder} avoiding the
     * need to create one manually via {@link CancelResourceRequestRequest#builder()}
     * </p>
     *
     * @param cancelResourceRequestRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.CancelResourceRequestRequest.Builder} to create
     *        a request.
     * @return A Java Future containing the result of the CancelResourceRequest operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException The resource is currently being modified by another operation.</li>
     *         <li>RequestTokenNotFoundException A resource operation with the specified request token can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.CancelResourceRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/CancelResourceRequest"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CancelResourceRequestResponse> cancelResourceRequest(
            Consumer<CancelResourceRequestRequest.Builder> cancelResourceRequestRequest) {
        return cancelResourceRequest(CancelResourceRequestRequest.builder().applyMutation(cancelResourceRequestRequest).build());
    }

    /**
     * <p>
     * Creates the specified resource. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-create.html">Creating a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource creation request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> type returned
     * by <code>CreateResource</code>.
     * </p>
     *
     * @param createResourceRequest
     * @return A Java Future containing the result of the CreateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>ConcurrentOperationException Another resource operation is currently being performed on this
     *         resource.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ClientTokenConflictException The specified client token has already been used in another resource
     *         request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.CreateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/CreateResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateResourceResponse> createResource(CreateResourceRequest createResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates the specified resource. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-create.html">Creating a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource creation request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> type returned
     * by <code>CreateResource</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateResourceRequest.Builder} avoiding the need to
     * create one manually via {@link CreateResourceRequest#builder()}
     * </p>
     *
     * @param createResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.CreateResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the CreateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>ConcurrentOperationException Another resource operation is currently being performed on this
     *         resource.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ClientTokenConflictException The specified client token has already been used in another resource
     *         request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.CreateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/CreateResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<CreateResourceResponse> createResource(Consumer<CreateResourceRequest.Builder> createResourceRequest) {
        return createResource(CreateResourceRequest.builder().applyMutation(createResourceRequest).build());
    }

    /**
     * <p>
     * Deletes the specified resource. For details, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-delete.html">Deleting a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource deletion request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> returned by
     * <code>DeleteResource</code>.
     * </p>
     *
     * @param deleteResourceRequest
     * @return A Java Future containing the result of the DeleteResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>ConcurrentOperationException Another resource operation is currently being performed on this
     *         resource.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ClientTokenConflictException The specified client token has already been used in another resource
     *         request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.DeleteResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/DeleteResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteResourceResponse> deleteResource(DeleteResourceRequest deleteResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified resource. For details, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-delete.html">Deleting a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource deletion request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> returned by
     * <code>DeleteResource</code>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteResourceRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteResourceRequest#builder()}
     * </p>
     *
     * @param deleteResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.DeleteResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the DeleteResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>ConcurrentOperationException Another resource operation is currently being performed on this
     *         resource.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ClientTokenConflictException The specified client token has already been used in another resource
     *         request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.DeleteResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/DeleteResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<DeleteResourceResponse> deleteResource(Consumer<DeleteResourceRequest.Builder> deleteResourceRequest) {
        return deleteResource(DeleteResourceRequest.builder().applyMutation(deleteResourceRequest).build());
    }

    /**
     * <p>
     * Returns information about the current state of the specified resource. For details, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-read.html">Reading a
     * resource's current state</a>.
     * </p>
     * <p>
     * You can use this action to return information about an existing resource in your account and Amazon Web Services
     * Region, whether those resources were provisioned using Cloud Control API.
     * </p>
     *
     * @param getResourceRequest
     * @return A Java Future containing the result of the GetResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.GetResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/GetResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetResourceResponse> getResource(GetResourceRequest getResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the current state of the specified resource. For details, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-read.html">Reading a
     * resource's current state</a>.
     * </p>
     * <p>
     * You can use this action to return information about an existing resource in your account and Amazon Web Services
     * Region, whether those resources were provisioned using Cloud Control API.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetResourceRequest.Builder} avoiding the need to
     * create one manually via {@link GetResourceRequest#builder()}
     * </p>
     *
     * @param getResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequest.Builder} to create a request.
     * @return A Java Future containing the result of the GetResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.GetResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/GetResource" target="_top">AWS API
     *      Documentation</a>
     */
    default CompletableFuture<GetResourceResponse> getResource(Consumer<GetResourceRequest.Builder> getResourceRequest) {
        return getResource(GetResourceRequest.builder().applyMutation(getResourceRequest).build());
    }

    /**
     * <p>
     * Returns the current status of a resource operation request. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-track"
     * >Tracking the progress of resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User
     * Guide</i>.
     * </p>
     *
     * @param getResourceRequestStatusRequest
     * @return A Java Future containing the result of the GetResourceRequestStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RequestTokenNotFoundException A resource operation with the specified request token can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.GetResourceRequestStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/GetResourceRequestStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetResourceRequestStatusResponse> getResourceRequestStatus(
            GetResourceRequestStatusRequest getResourceRequestStatusRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the current status of a resource operation request. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-track"
     * >Tracking the progress of resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User
     * Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetResourceRequestStatusRequest.Builder} avoiding
     * the need to create one manually via {@link GetResourceRequestStatusRequest#builder()}
     * </p>
     *
     * @param getResourceRequestStatusRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequestStatusRequest.Builder} to
     *        create a request.
     * @return A Java Future containing the result of the GetResourceRequestStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RequestTokenNotFoundException A resource operation with the specified request token can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.GetResourceRequestStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/GetResourceRequestStatus"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetResourceRequestStatusResponse> getResourceRequestStatus(
            Consumer<GetResourceRequestStatusRequest.Builder> getResourceRequestStatusRequest) {
        return getResourceRequestStatus(GetResourceRequestStatusRequest.builder().applyMutation(getResourceRequestStatusRequest)
                .build());
    }

    /**
     * <p>
     * Returns existing resource operation requests. This includes requests of all status types. For more information,
     * see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-list"
     * >Listing active resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <note>
     * <p>
     * Resource operation requests expire after 7 days.
     * </p>
     * </note>
     *
     * @param listResourceRequestsRequest
     * @return A Java Future containing the result of the ListResourceRequests operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResourceRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResourceRequests"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListResourceRequestsResponse> listResourceRequests(
            ListResourceRequestsRequest listResourceRequestsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns existing resource operation requests. This includes requests of all status types. For more information,
     * see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-list"
     * >Listing active resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <note>
     * <p>
     * Resource operation requests expire after 7 days.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListResourceRequestsRequest.Builder} avoiding the
     * need to create one manually via {@link ListResourceRequestsRequest#builder()}
     * </p>
     *
     * @param listResourceRequestsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListResourceRequests operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResourceRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResourceRequests"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListResourceRequestsResponse> listResourceRequests(
            Consumer<ListResourceRequestsRequest.Builder> listResourceRequestsRequest) {
        return listResourceRequests(ListResourceRequestsRequest.builder().applyMutation(listResourceRequestsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listResourceRequests(software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourceRequestsPublisher publisher = client.listResourceRequestsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourceRequestsPublisher publisher = client.listResourceRequestsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResourceRequests(software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResourceRequestsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResourceRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResourceRequests"
     *      target="_top">AWS API Documentation</a>
     */
    default ListResourceRequestsPublisher listResourceRequestsPaginator(ListResourceRequestsRequest listResourceRequestsRequest) {
        return new ListResourceRequestsPublisher(this, listResourceRequestsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listResourceRequests(software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourceRequestsPublisher publisher = client.listResourceRequestsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourceRequestsPublisher publisher = client.listResourceRequestsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResourceRequests(software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest)}
     * operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListResourceRequestsRequest.Builder} avoiding the
     * need to create one manually via {@link ListResourceRequestsRequest#builder()}
     * </p>
     *
     * @param listResourceRequestsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResourceRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResourceRequests"
     *      target="_top">AWS API Documentation</a>
     */
    default ListResourceRequestsPublisher listResourceRequestsPaginator(
            Consumer<ListResourceRequestsRequest.Builder> listResourceRequestsRequest) {
        return listResourceRequestsPaginator(ListResourceRequestsRequest.builder().applyMutation(listResourceRequestsRequest)
                .build());
    }

    /**
     * <p>
     * Returns information about the specified resources. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-list.html">Discovering
     * resources</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * You can use this action to return information about existing resources in your account and Amazon Web Services
     * Region, whether those resources were provisioned using Cloud Control API.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns information about the specified resources. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-list.html">Discovering
     * resources</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * You can use this action to return information about existing resources in your account and Amazon Web Services
     * Region, whether those resources were provisioned using Cloud Control API.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListResourcesRequest.Builder} avoiding the need to
     * create one manually via {@link ListResourcesRequest#builder()}
     * </p>
     *
     * @param listResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<ListResourcesResponse> listResources(Consumer<ListResourcesRequest.Builder> listResourcesRequest) {
        return listResources(ListResourcesRequest.builder().applyMutation(listResourcesRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listResources(software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudcontrol.model.ListResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudcontrol.model.ListResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResources(software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest)} operation.</b>
     * </p>
     *
     * @param listResourcesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    default ListResourcesPublisher listResourcesPaginator(ListResourcesRequest listResourcesRequest) {
        return new ListResourcesPublisher(this, listResourcesRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listResources(software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudcontrol.paginators.ListResourcesPublisher publisher = client.listResourcesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudcontrol.model.ListResourcesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudcontrol.model.ListResourcesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResources(software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListResourcesRequest.Builder} avoiding the need to
     * create one manually via {@link ListResourcesRequest#builder()}
     * </p>
     *
     * @param listResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest.Builder} to create a
     *        request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    default ListResourcesPublisher listResourcesPaginator(Consumer<ListResourcesRequest.Builder> listResourcesRequest) {
        return listResourcesPaginator(ListResourcesRequest.builder().applyMutation(listResourcesRequest).build());
    }

    /**
     * <p>
     * Updates the specified property values in the resource.
     * </p>
     * <p>
     * You specify your resource property updates as a list of patch operations contained in a JSON patch document that
     * adheres to the <a href="https://datatracker.ietf.org/doc/html/rfc6902"> <i>RFC 6902 - JavaScript Object Notation
     * (JSON) Patch</i> </a> standard.
     * </p>
     * <p>
     * For details on how Cloud Control API performs resource update operations, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-update.html">Updating a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource update request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> returned by
     * <code>UpdateResource</code>.
     * </p>
     * <p>
     * For more information about the properties of a specific resource, refer to the related topic for the resource in
     * the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">
     * Resource and property types reference</a> in the <i>CloudFormation Users Guide</i>.
     * </p>
     *
     * @param updateResourceRequest
     * @return A Java Future containing the result of the UpdateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>ConcurrentOperationException Another resource operation is currently being performed on this
     *         resource.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ClientTokenConflictException The specified client token has already been used in another resource
     *         request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.UpdateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/UpdateResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateResourceResponse> updateResource(UpdateResourceRequest updateResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the specified property values in the resource.
     * </p>
     * <p>
     * You specify your resource property updates as a list of patch operations contained in a JSON patch document that
     * adheres to the <a href="https://datatracker.ietf.org/doc/html/rfc6902"> <i>RFC 6902 - JavaScript Object Notation
     * (JSON) Patch</i> </a> standard.
     * </p>
     * <p>
     * For details on how Cloud Control API performs resource update operations, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-update.html">Updating a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource update request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> returned by
     * <code>UpdateResource</code>.
     * </p>
     * <p>
     * For more information about the properties of a specific resource, refer to the related topic for the resource in
     * the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">
     * Resource and property types reference</a> in the <i>CloudFormation Users Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateResourceRequest#builder()}
     * </p>
     *
     * @param updateResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.cloudcontrol.model.UpdateResourceRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the UpdateResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AlreadyExistsException The resource with the name requested already exists.</li>
     *         <li>HandlerInternalFailureException The resource handler has returned that an unexpected error occurred
     *         within the resource handler.</li>
     *         <li>GeneralServiceException The resource handler has returned that the downstream service generated an
     *         error that doesn't map to any other handler error code.</li>
     *         <li>NotUpdatableException One or more properties included in this resource operation are defined as
     *         create-only, and therefore can't be updated.</li>
     *         <li>TypeNotFoundException The specified extension doesn't exist in the CloudFormation registry.</li>
     *         <li>ConcurrentOperationException Another resource operation is currently being performed on this
     *         resource.</li>
     *         <li>InvalidRequestException The resource handler has returned that invalid input from the user has
     *         generated a generic exception.</li>
     *         <li>PrivateTypeException Cloud Control API hasn't received a valid response from the resource handler,
     *         due to a configuration error. This includes issues such as the resource handler returning an invalid
     *         response, or timing out.</li>
     *         <li>ResourceNotFoundException A resource with the specified identifier can't be found.</li>
     *         <li>NetworkFailureException The resource handler has returned that the request couldn't be completed due
     *         to networking issues, such as a failure to receive a response from the server.</li>
     *         <li>UnsupportedActionException The specified resource doesn't support this resource operation.</li>
     *         <li>NotStabilizedException The resource handler has returned that the downstream resource failed to
     *         complete all of its ready-state checks.</li>
     *         <li>ServiceInternalErrorException The resource handler has returned that the downstream service returned
     *         an internal error, typically with a <code>5XX HTTP</code> status code.</li>
     *         <li>HandlerFailureException The resource handler has failed without a returning a more specific error
     *         code. This can include timeouts.</li>
     *         <li>ServiceLimitExceededException The resource handler has returned that a non-transient resource limit
     *         was reached on the service side.</li>
     *         <li>InvalidCredentialsException The resource handler has returned that the credentials provided by the
     *         user are invalid.</li>
     *         <li>ResourceConflictException The resource is temporarily unavailable to be acted upon. For example, if
     *         the resource is currently undergoing an operation and can't be acted upon until that operation is
     *         finished.</li>
     *         <li>ClientTokenConflictException The specified client token has already been used in another resource
     *         request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudControlException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudControlAsyncClient.UpdateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/UpdateResource" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<UpdateResourceResponse> updateResource(Consumer<UpdateResourceRequest.Builder> updateResourceRequest) {
        return updateResource(UpdateResourceRequest.builder().applyMutation(updateResourceRequest).build());
    }

    /**
     * Create an instance of {@link CloudControlAsyncWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link CloudControlAsyncWaiter}
     */
    default CloudControlAsyncWaiter waiter() {
        throw new UnsupportedOperationException();
    }

    @Override
    default CloudControlServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link CloudControlAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static CloudControlAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link CloudControlAsyncClient}.
     */
    static CloudControlAsyncClientBuilder builder() {
        return new DefaultCloudControlAsyncClientBuilder();
    }
}
