/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.clouddirectory.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchReadRequest extends CloudDirectoryRequest implements
        ToCopyableBuilder<BatchReadRequest.Builder, BatchReadRequest> {
    private static final SdkField<String> DIRECTORY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DirectoryArn").getter(getter(BatchReadRequest::directoryArn)).setter(setter(Builder::directoryArn))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-data-partition").build())
            .build();

    private static final SdkField<List<BatchReadOperation>> OPERATIONS_FIELD = SdkField
            .<List<BatchReadOperation>> builder(MarshallingType.LIST)
            .memberName("Operations")
            .getter(getter(BatchReadRequest::operations))
            .setter(setter(Builder::operations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Operations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<BatchReadOperation> builder(MarshallingType.SDK_POJO)
                                            .constructor(BatchReadOperation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONSISTENCY_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConsistencyLevel").getter(getter(BatchReadRequest::consistencyLevelAsString))
            .setter(setter(Builder::consistencyLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-consistency-level").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DIRECTORY_ARN_FIELD,
            OPERATIONS_FIELD, CONSISTENCY_LEVEL_FIELD));

    private final String directoryArn;

    private final List<BatchReadOperation> operations;

    private final String consistencyLevel;

    private BatchReadRequest(BuilderImpl builder) {
        super(builder);
        this.directoryArn = builder.directoryArn;
        this.operations = builder.operations;
        this.consistencyLevel = builder.consistencyLevel;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>. For more information, see
     * <a>arns</a>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>. For more information, see
     *         <a>arns</a>.
     */
    public final String directoryArn() {
        return directoryArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Operations property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOperations() {
        return operations != null && !(operations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of operations that are part of the batch.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOperations} method.
     * </p>
     * 
     * @return A list of operations that are part of the batch.
     */
    public final List<BatchReadOperation> operations() {
        return operations;
    }

    /**
     * <p>
     * Represents the manner and timing in which the successful write or update of an object is reflected in a
     * subsequent read operation of that same object.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #consistencyLevel}
     * will return {@link ConsistencyLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #consistencyLevelAsString}.
     * </p>
     * 
     * @return Represents the manner and timing in which the successful write or update of an object is reflected in a
     *         subsequent read operation of that same object.
     * @see ConsistencyLevel
     */
    public final ConsistencyLevel consistencyLevel() {
        return ConsistencyLevel.fromValue(consistencyLevel);
    }

    /**
     * <p>
     * Represents the manner and timing in which the successful write or update of an object is reflected in a
     * subsequent read operation of that same object.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #consistencyLevel}
     * will return {@link ConsistencyLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #consistencyLevelAsString}.
     * </p>
     * 
     * @return Represents the manner and timing in which the successful write or update of an object is reflected in a
     *         subsequent read operation of that same object.
     * @see ConsistencyLevel
     */
    public final String consistencyLevelAsString() {
        return consistencyLevel;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(directoryArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasOperations() ? operations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(consistencyLevelAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchReadRequest)) {
            return false;
        }
        BatchReadRequest other = (BatchReadRequest) obj;
        return Objects.equals(directoryArn(), other.directoryArn()) && hasOperations() == other.hasOperations()
                && Objects.equals(operations(), other.operations())
                && Objects.equals(consistencyLevelAsString(), other.consistencyLevelAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchReadRequest").add("DirectoryArn", directoryArn())
                .add("Operations", hasOperations() ? operations() : null).add("ConsistencyLevel", consistencyLevelAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DirectoryArn":
            return Optional.ofNullable(clazz.cast(directoryArn()));
        case "Operations":
            return Optional.ofNullable(clazz.cast(operations()));
        case "ConsistencyLevel":
            return Optional.ofNullable(clazz.cast(consistencyLevelAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchReadRequest, T> g) {
        return obj -> g.apply((BatchReadRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudDirectoryRequest.Builder, SdkPojo, CopyableBuilder<Builder, BatchReadRequest> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>. For more information, see
         * <a>arns</a>.
         * </p>
         * 
         * @param directoryArn
         *        The Amazon Resource Name (ARN) that is associated with the <a>Directory</a>. For more information, see
         *        <a>arns</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryArn(String directoryArn);

        /**
         * <p>
         * A list of operations that are part of the batch.
         * </p>
         * 
         * @param operations
         *        A list of operations that are part of the batch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operations(Collection<BatchReadOperation> operations);

        /**
         * <p>
         * A list of operations that are part of the batch.
         * </p>
         * 
         * @param operations
         *        A list of operations that are part of the batch.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operations(BatchReadOperation... operations);

        /**
         * <p>
         * A list of operations that are part of the batch.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<BatchReadOperation>.Builder}
         * avoiding the need to create one manually via {@link List<BatchReadOperation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<BatchReadOperation>.Builder#build()} is called immediately
         * and its result is passed to {@link #operations(List<BatchReadOperation>)}.
         * 
         * @param operations
         *        a consumer that will call methods on {@link List<BatchReadOperation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operations(List<BatchReadOperation>)
         */
        Builder operations(Consumer<BatchReadOperation.Builder>... operations);

        /**
         * <p>
         * Represents the manner and timing in which the successful write or update of an object is reflected in a
         * subsequent read operation of that same object.
         * </p>
         * 
         * @param consistencyLevel
         *        Represents the manner and timing in which the successful write or update of an object is reflected in
         *        a subsequent read operation of that same object.
         * @see ConsistencyLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConsistencyLevel
         */
        Builder consistencyLevel(String consistencyLevel);

        /**
         * <p>
         * Represents the manner and timing in which the successful write or update of an object is reflected in a
         * subsequent read operation of that same object.
         * </p>
         * 
         * @param consistencyLevel
         *        Represents the manner and timing in which the successful write or update of an object is reflected in
         *        a subsequent read operation of that same object.
         * @see ConsistencyLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConsistencyLevel
         */
        Builder consistencyLevel(ConsistencyLevel consistencyLevel);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudDirectoryRequest.BuilderImpl implements Builder {
        private String directoryArn;

        private List<BatchReadOperation> operations = DefaultSdkAutoConstructList.getInstance();

        private String consistencyLevel;

        private BuilderImpl() {
        }

        private BuilderImpl(BatchReadRequest model) {
            super(model);
            directoryArn(model.directoryArn);
            operations(model.operations);
            consistencyLevel(model.consistencyLevel);
        }

        public final String getDirectoryArn() {
            return directoryArn;
        }

        public final void setDirectoryArn(String directoryArn) {
            this.directoryArn = directoryArn;
        }

        @Override
        public final Builder directoryArn(String directoryArn) {
            this.directoryArn = directoryArn;
            return this;
        }

        public final List<BatchReadOperation.Builder> getOperations() {
            List<BatchReadOperation.Builder> result = BatchReadOperationListCopier.copyToBuilder(this.operations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOperations(Collection<BatchReadOperation.BuilderImpl> operations) {
            this.operations = BatchReadOperationListCopier.copyFromBuilder(operations);
        }

        @Override
        public final Builder operations(Collection<BatchReadOperation> operations) {
            this.operations = BatchReadOperationListCopier.copy(operations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operations(BatchReadOperation... operations) {
            operations(Arrays.asList(operations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operations(Consumer<BatchReadOperation.Builder>... operations) {
            operations(Stream.of(operations).map(c -> BatchReadOperation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getConsistencyLevel() {
            return consistencyLevel;
        }

        public final void setConsistencyLevel(String consistencyLevel) {
            this.consistencyLevel = consistencyLevel;
        }

        @Override
        public final Builder consistencyLevel(String consistencyLevel) {
            this.consistencyLevel = consistencyLevel;
            return this;
        }

        @Override
        public final Builder consistencyLevel(ConsistencyLevel consistencyLevel) {
            this.consistencyLevel(consistencyLevel == null ? null : consistencyLevel.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public BatchReadRequest build() {
            return new BatchReadRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
