/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains information about a stack set. A stack set enables you to provision stacks into AWS
 * accounts and across regions by using a single CloudFormation template. In the stack set, you specify the template to
 * use, as well as any parameters and capabilities that the template requires.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSet implements ToCopyableBuilder<StackSet.Builder, StackSet> {
    private final String stackSetName;

    private final String stackSetId;

    private final String description;

    private final String status;

    private final String templateBody;

    private final List<Parameter> parameters;

    private final List<String> capabilities;

    private final List<Tag> tags;

    private StackSet(BuilderImpl builder) {
        this.stackSetName = builder.stackSetName;
        this.stackSetId = builder.stackSetId;
        this.description = builder.description;
        this.status = builder.status;
        this.templateBody = builder.templateBody;
        this.parameters = builder.parameters;
        this.capabilities = builder.capabilities;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name that's associated with the stack set.
     * </p>
     * 
     * @return The name that's associated with the stack set.
     */
    public String stackSetName() {
        return stackSetName;
    }

    /**
     * <p>
     * The ID of the stack set.
     * </p>
     * 
     * @return The ID of the stack set.
     */
    public String stackSetId() {
        return stackSetId;
    }

    /**
     * <p>
     * A description of the stack set that you specify when the stack set is created or updated.
     * </p>
     * 
     * @return A description of the stack set that you specify when the stack set is created or updated.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The status of the stack set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack set.
     * @see StackSetStatus
     */
    public StackSetStatus status() {
        return StackSetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the stack set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack set.
     * @see StackSetStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The structure that contains the body of the template that was used to create or update the stack set.
     * </p>
     * 
     * @return The structure that contains the body of the template that was used to create or update the stack set.
     */
    public String templateBody() {
        return templateBody;
    }

    /**
     * <p>
     * A list of input parameters for a stack set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of input parameters for a stack set.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
     * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
     * users. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The capabilities that are allowed in the stack set. Some stack set templates might include resources that
     *         can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
     *         Management (IAM) users. For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     */
    public List<Capability> capabilities() {
        return TypeConverter.convert(capabilities, Capability::fromValue);
    }

    /**
     * <p>
     * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
     * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
     * users. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The capabilities that are allowed in the stack set. Some stack set templates might include resources that
     *         can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
     *         Management (IAM) users. For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     */
    public List<String> capabilitiesAsStrings() {
        return capabilities;
    }

    /**
     * <p>
     * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of tags that specify information about the stack set. A maximum number of 50 tags can be
     *         specified.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackSetName());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(templateBody());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(capabilitiesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSet)) {
            return false;
        }
        StackSet other = (StackSet) obj;
        return Objects.equals(stackSetName(), other.stackSetName()) && Objects.equals(stackSetId(), other.stackSetId())
                && Objects.equals(description(), other.description()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(templateBody(), other.templateBody()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(capabilitiesAsStrings(), other.capabilitiesAsStrings()) && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        return ToString.builder("StackSet").add("StackSetName", stackSetName()).add("StackSetId", stackSetId())
                .add("Description", description()).add("Status", statusAsString()).add("TemplateBody", templateBody())
                .add("Parameters", parameters()).add("Capabilities", capabilitiesAsStrings()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetName":
            return Optional.ofNullable(clazz.cast(stackSetName()));
        case "StackSetId":
            return Optional.ofNullable(clazz.cast(stackSetId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "TemplateBody":
            return Optional.ofNullable(clazz.cast(templateBody()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "Capabilities":
            return Optional.ofNullable(clazz.cast(capabilitiesAsStrings()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, StackSet> {
        /**
         * <p>
         * The name that's associated with the stack set.
         * </p>
         * 
         * @param stackSetName
         *        The name that's associated with the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetName(String stackSetName);

        /**
         * <p>
         * The ID of the stack set.
         * </p>
         * 
         * @param stackSetId
         *        The ID of the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetId(String stackSetId);

        /**
         * <p>
         * A description of the stack set that you specify when the stack set is created or updated.
         * </p>
         * 
         * @param description
         *        A description of the stack set that you specify when the stack set is created or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The status of the stack set.
         * </p>
         * 
         * @param status
         *        The status of the stack set.
         * @see StackSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the stack set.
         * </p>
         * 
         * @param status
         *        The status of the stack set.
         * @see StackSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetStatus
         */
        Builder status(StackSetStatus status);

        /**
         * <p>
         * The structure that contains the body of the template that was used to create or update the stack set.
         * </p>
         * 
         * @param templateBody
         *        The structure that contains the body of the template that was used to create or update the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateBody(String templateBody);

        /**
         * <p>
         * A list of input parameters for a stack set.
         * </p>
         * 
         * @param parameters
         *        A list of input parameters for a stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * A list of input parameters for a stack set.
         * </p>
         * 
         * @param parameters
         *        A list of input parameters for a stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * A list of input parameters for a stack set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parameter>.Builder} avoiding the need to
         * create one manually via {@link List<Parameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parameter>.Builder#build()} is called immediately and its
         * result is passed to {@link #parameters(List<Parameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<Parameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<Parameter>)
         */
        Builder parameters(Consumer<Parameter.Builder>... parameters);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(Collection<String> capabilities);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(String... capabilities);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<Capability> capabilities);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Capability... capabilities);

        /**
         * <p>
         * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
         * </p>
         * 
         * @param tags
         *        A list of tags that specify information about the stack set. A maximum number of 50 tags can be
         *        specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
         * </p>
         * 
         * @param tags
         *        A list of tags that specify information about the stack set. A maximum number of 50 tags can be
         *        specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String stackSetName;

        private String stackSetId;

        private String description;

        private String status;

        private String templateBody;

        private List<Parameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private List<String> capabilities = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StackSet model) {
            stackSetName(model.stackSetName);
            stackSetId(model.stackSetId);
            description(model.description);
            status(model.status);
            templateBody(model.templateBody);
            parameters(model.parameters);
            capabilitiesWithStrings(model.capabilities);
            tags(model.tags);
        }

        public final String getStackSetName() {
            return stackSetName;
        }

        @Override
        public final Builder stackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
            return this;
        }

        public final void setStackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
        }

        public final String getStackSetId() {
            return stackSetId;
        }

        @Override
        public final Builder stackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
            return this;
        }

        public final void setStackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StackSetStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getTemplateBody() {
            return templateBody;
        }

        @Override
        public final Builder templateBody(String templateBody) {
            this.templateBody = templateBody;
            return this;
        }

        public final void setTemplateBody(String templateBody) {
            this.templateBody = templateBody;
        }

        public final Collection<Parameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<Parameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> Parameter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        public final Collection<String> getCapabilities() {
            return capabilities;
        }

        @Override
        public final Builder capabilitiesWithStrings(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilitiesWithStrings(String... capabilities) {
            capabilitiesWithStrings(Arrays.asList(capabilities));
            return this;
        }

        @Override
        public final Builder capabilities(Collection<Capability> capabilities) {
            this.capabilities = CapabilitiesCopier.copyEnumToString(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(Capability... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public StackSet build() {
            return new StackSet(this);
        }
    }
}
