/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure that contains information about a stack set operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSetOperation implements ToCopyableBuilder<StackSetOperation.Builder, StackSetOperation> {
    private final String operationId;

    private final String stackSetId;

    private final String action;

    private final String status;

    private final StackSetOperationPreferences operationPreferences;

    private final Boolean retainStacks;

    private final Instant creationTimestamp;

    private final Instant endTimestamp;

    private StackSetOperation(BuilderImpl builder) {
        this.operationId = builder.operationId;
        this.stackSetId = builder.stackSetId;
        this.action = builder.action;
        this.status = builder.status;
        this.operationPreferences = builder.operationPreferences;
        this.retainStacks = builder.retainStacks;
        this.creationTimestamp = builder.creationTimestamp;
        this.endTimestamp = builder.endTimestamp;
    }

    /**
     * <p>
     * The unique ID of a stack set operation.
     * </p>
     * 
     * @return The unique ID of a stack set operation.
     */
    public String operationId() {
        return operationId;
    }

    /**
     * <p>
     * The ID of the stack set.
     * </p>
     * 
     * @return The ID of the stack set.
     */
    public String stackSetId() {
        return stackSetId;
    }

    /**
     * <p>
     * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
     * delete operations affect only the specified stack set instances that are associated with the specified stack set.
     * Update operations affect both the stack set itself, as well as <i>all</i> associated stack set instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackSetOperationAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create
     *         and delete operations affect only the specified stack set instances that are associated with the
     *         specified stack set. Update operations affect both the stack set itself, as well as <i>all</i> associated
     *         stack set instances.
     * @see StackSetOperationAction
     */
    public StackSetOperationAction action() {
        return StackSetOperationAction.fromValue(action);
    }

    /**
     * <p>
     * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
     * delete operations affect only the specified stack set instances that are associated with the specified stack set.
     * Update operations affect both the stack set itself, as well as <i>all</i> associated stack set instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackSetOperationAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create
     *         and delete operations affect only the specified stack set instances that are associated with the
     *         specified stack set. Update operations affect both the stack set itself, as well as <i>all</i> associated
     *         stack set instances.
     * @see StackSetOperationAction
     */
    public String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The status of the operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
     * you've set for an operation is applied for each region during stack create and update operations. If the number
     * of failed stacks within a region exceeds the failure tolerance, the status of the operation in the region is set
     * to <code>FAILED</code>. This in turn sets the status of the operation as a whole to <code>FAILED</code>, and AWS
     * CloudFormation cancels the operation in any remaining regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code>: The operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has cancelled the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without exceeding
     * the failure tolerance for the operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the operation. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value
     *         that you've set for an operation is applied for each region during stack create and update operations. If
     *         the number of failed stacks within a region exceeds the failure tolerance, the status of the operation in
     *         the region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
     *         <code>FAILED</code>, and AWS CloudFormation cancels the operation in any remaining regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code>: The operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has cancelled the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
     *         exceeding the failure tolerance for the operation.
     *         </p>
     *         </li>
     * @see StackSetOperationStatus
     */
    public StackSetOperationStatus status() {
        return StackSetOperationStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
     * you've set for an operation is applied for each region during stack create and update operations. If the number
     * of failed stacks within a region exceeds the failure tolerance, the status of the operation in the region is set
     * to <code>FAILED</code>. This in turn sets the status of the operation as a whole to <code>FAILED</code>, and AWS
     * CloudFormation cancels the operation in any remaining regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code>: The operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has cancelled the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without exceeding
     * the failure tolerance for the operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the operation. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value
     *         that you've set for an operation is applied for each region during stack create and update operations. If
     *         the number of failed stacks within a region exceeds the failure tolerance, the status of the operation in
     *         the region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
     *         <code>FAILED</code>, and AWS CloudFormation cancels the operation in any remaining regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code>: The operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has cancelled the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
     *         exceeding the failure tolerance for the operation.
     *         </p>
     *         </li>
     * @see StackSetOperationStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The preferences for how AWS CloudFormation performs this stack set operation.
     * </p>
     * 
     * @return The preferences for how AWS CloudFormation performs this stack set operation.
     */
    public StackSetOperationPreferences operationPreferences() {
        return operationPreferences;
    }

    /**
     * <p>
     * For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack instances from
     * the specified stack set, but doesn't delete the stacks. You can't reassociate a retained stack, or add an
     * existing, saved stack to a new stack set.
     * </p>
     * 
     * @return For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack
     *         instances from the specified stack set, but doesn't delete the stacks. You can't reassociate a retained
     *         stack, or add an existing, saved stack to a new stack set.
     */
    public Boolean retainStacks() {
        return retainStacks;
    }

    /**
     * <p>
     * The time at which the operation was initiated. Note that the creation times for the stack set operation might
     * differ from the creation time of the individual stacks themselves. This is because AWS CloudFormation needs to
     * perform preparatory work for the operation, such as dispatching the work to the requested regions, before
     * actually creating the first stacks.
     * </p>
     * 
     * @return The time at which the operation was initiated. Note that the creation times for the stack set operation
     *         might differ from the creation time of the individual stacks themselves. This is because AWS
     *         CloudFormation needs to perform preparatory work for the operation, such as dispatching the work to the
     *         requested regions, before actually creating the first stacks.
     */
    public Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * <p>
     * The time at which the stack set operation ended, across all accounts and regions specified. Note that this
     * doesn't necessarily mean that the stack set operation was successful, or even attempted, in each account or
     * region.
     * </p>
     * 
     * @return The time at which the stack set operation ended, across all accounts and regions specified. Note that
     *         this doesn't necessarily mean that the stack set operation was successful, or even attempted, in each
     *         account or region.
     */
    public Instant endTimestamp() {
        return endTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(operationId());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetId());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(operationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(retainStacks());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(endTimestamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSetOperation)) {
            return false;
        }
        StackSetOperation other = (StackSetOperation) obj;
        return Objects.equals(operationId(), other.operationId()) && Objects.equals(stackSetId(), other.stackSetId())
                && Objects.equals(actionAsString(), other.actionAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(operationPreferences(), other.operationPreferences())
                && Objects.equals(retainStacks(), other.retainStacks())
                && Objects.equals(creationTimestamp(), other.creationTimestamp())
                && Objects.equals(endTimestamp(), other.endTimestamp());
    }

    @Override
    public String toString() {
        return ToString.builder("StackSetOperation").add("OperationId", operationId()).add("StackSetId", stackSetId())
                .add("Action", actionAsString()).add("Status", statusAsString())
                .add("OperationPreferences", operationPreferences()).add("RetainStacks", retainStacks())
                .add("CreationTimestamp", creationTimestamp()).add("EndTimestamp", endTimestamp()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OperationId":
            return Optional.ofNullable(clazz.cast(operationId()));
        case "StackSetId":
            return Optional.ofNullable(clazz.cast(stackSetId()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "OperationPreferences":
            return Optional.ofNullable(clazz.cast(operationPreferences()));
        case "RetainStacks":
            return Optional.ofNullable(clazz.cast(retainStacks()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "EndTimestamp":
            return Optional.ofNullable(clazz.cast(endTimestamp()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, StackSetOperation> {
        /**
         * <p>
         * The unique ID of a stack set operation.
         * </p>
         * 
         * @param operationId
         *        The unique ID of a stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationId(String operationId);

        /**
         * <p>
         * The ID of the stack set.
         * </p>
         * 
         * @param stackSetId
         *        The ID of the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetId(String stackSetId);

        /**
         * <p>
         * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
         * delete operations affect only the specified stack set instances that are associated with the specified stack
         * set. Update operations affect both the stack set itself, as well as <i>all</i> associated stack set
         * instances.
         * </p>
         * 
         * @param action
         *        The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>.
         *        Create and delete operations affect only the specified stack set instances that are associated with
         *        the specified stack set. Update operations affect both the stack set itself, as well as <i>all</i>
         *        associated stack set instances.
         * @see StackSetOperationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationAction
         */
        Builder action(String action);

        /**
         * <p>
         * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
         * delete operations affect only the specified stack set instances that are associated with the specified stack
         * set. Update operations affect both the stack set itself, as well as <i>all</i> associated stack set
         * instances.
         * </p>
         * 
         * @param action
         *        The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>.
         *        Create and delete operations affect only the specified stack set instances that are associated with
         *        the specified stack set. Update operations affect both the stack set itself, as well as <i>all</i>
         *        associated stack set instances.
         * @see StackSetOperationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationAction
         */
        Builder action(StackSetOperationAction action);

        /**
         * <p>
         * The status of the operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
         * you've set for an operation is applied for each region during stack create and update operations. If the
         * number of failed stacks within a region exceeds the failure tolerance, the status of the operation in the
         * region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
         * <code>FAILED</code>, and AWS CloudFormation cancels the operation in any remaining regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code>: The operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has cancelled the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         * exceeding the failure tolerance for the operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the operation. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance
         *        value that you've set for an operation is applied for each region during stack create and update
         *        operations. If the number of failed stacks within a region exceeds the failure tolerance, the status
         *        of the operation in the region is set to <code>FAILED</code>. This in turn sets the status of the
         *        operation as a whole to <code>FAILED</code>, and AWS CloudFormation cancels the operation in any
         *        remaining regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code>: The operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has cancelled the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         *        exceeding the failure tolerance for the operation.
         *        </p>
         *        </li>
         * @see StackSetOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
         * you've set for an operation is applied for each region during stack create and update operations. If the
         * number of failed stacks within a region exceeds the failure tolerance, the status of the operation in the
         * region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
         * <code>FAILED</code>, and AWS CloudFormation cancels the operation in any remaining regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code>: The operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has cancelled the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         * exceeding the failure tolerance for the operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the operation. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance
         *        value that you've set for an operation is applied for each region during stack create and update
         *        operations. If the number of failed stacks within a region exceeds the failure tolerance, the status
         *        of the operation in the region is set to <code>FAILED</code>. This in turn sets the status of the
         *        operation as a whole to <code>FAILED</code>, and AWS CloudFormation cancels the operation in any
         *        remaining regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code>: The operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has cancelled the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         *        exceeding the failure tolerance for the operation.
         *        </p>
         *        </li>
         * @see StackSetOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationStatus
         */
        Builder status(StackSetOperationStatus status);

        /**
         * <p>
         * The preferences for how AWS CloudFormation performs this stack set operation.
         * </p>
         * 
         * @param operationPreferences
         *        The preferences for how AWS CloudFormation performs this stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationPreferences(StackSetOperationPreferences operationPreferences);

        /**
         * <p>
         * The preferences for how AWS CloudFormation performs this stack set operation.
         * </p>
         * This is a convenience that creates an instance of the {@link StackSetOperationPreferences.Builder} avoiding
         * the need to create one manually via {@link StackSetOperationPreferences#builder()}.
         *
         * When the {@link Consumer} completes, {@link StackSetOperationPreferences.Builder#build()} is called
         * immediately and its result is passed to {@link #operationPreferences(StackSetOperationPreferences)}.
         * 
         * @param operationPreferences
         *        a consumer that will call methods on {@link StackSetOperationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operationPreferences(StackSetOperationPreferences)
         */
        default Builder operationPreferences(Consumer<StackSetOperationPreferences.Builder> operationPreferences) {
            return operationPreferences(StackSetOperationPreferences.builder().applyMutation(operationPreferences).build());
        }

        /**
         * <p>
         * For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack instances
         * from the specified stack set, but doesn't delete the stacks. You can't reassociate a retained stack, or add
         * an existing, saved stack to a new stack set.
         * </p>
         * 
         * @param retainStacks
         *        For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack
         *        instances from the specified stack set, but doesn't delete the stacks. You can't reassociate a
         *        retained stack, or add an existing, saved stack to a new stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retainStacks(Boolean retainStacks);

        /**
         * <p>
         * The time at which the operation was initiated. Note that the creation times for the stack set operation might
         * differ from the creation time of the individual stacks themselves. This is because AWS CloudFormation needs
         * to perform preparatory work for the operation, such as dispatching the work to the requested regions, before
         * actually creating the first stacks.
         * </p>
         * 
         * @param creationTimestamp
         *        The time at which the operation was initiated. Note that the creation times for the stack set
         *        operation might differ from the creation time of the individual stacks themselves. This is because AWS
         *        CloudFormation needs to perform preparatory work for the operation, such as dispatching the work to
         *        the requested regions, before actually creating the first stacks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * The time at which the stack set operation ended, across all accounts and regions specified. Note that this
         * doesn't necessarily mean that the stack set operation was successful, or even attempted, in each account or
         * region.
         * </p>
         * 
         * @param endTimestamp
         *        The time at which the stack set operation ended, across all accounts and regions specified. Note that
         *        this doesn't necessarily mean that the stack set operation was successful, or even attempted, in each
         *        account or region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTimestamp(Instant endTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String operationId;

        private String stackSetId;

        private String action;

        private String status;

        private StackSetOperationPreferences operationPreferences;

        private Boolean retainStacks;

        private Instant creationTimestamp;

        private Instant endTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(StackSetOperation model) {
            operationId(model.operationId);
            stackSetId(model.stackSetId);
            action(model.action);
            status(model.status);
            operationPreferences(model.operationPreferences);
            retainStacks(model.retainStacks);
            creationTimestamp(model.creationTimestamp);
            endTimestamp(model.endTimestamp);
        }

        public final String getOperationId() {
            return operationId;
        }

        @Override
        public final Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public final void setOperationId(String operationId) {
            this.operationId = operationId;
        }

        public final String getStackSetId() {
            return stackSetId;
        }

        @Override
        public final Builder stackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
            return this;
        }

        public final void setStackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
        }

        public final String getAction() {
            return action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(StackSetOperationAction action) {
            this.action(action.toString());
            return this;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StackSetOperationStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final StackSetOperationPreferences.Builder getOperationPreferences() {
            return operationPreferences != null ? operationPreferences.toBuilder() : null;
        }

        @Override
        public final Builder operationPreferences(StackSetOperationPreferences operationPreferences) {
            this.operationPreferences = operationPreferences;
            return this;
        }

        public final void setOperationPreferences(StackSetOperationPreferences.BuilderImpl operationPreferences) {
            this.operationPreferences = operationPreferences != null ? operationPreferences.build() : null;
        }

        public final Boolean getRetainStacks() {
            return retainStacks;
        }

        @Override
        public final Builder retainStacks(Boolean retainStacks) {
            this.retainStacks = retainStacks;
            return this;
        }

        public final void setRetainStacks(Boolean retainStacks) {
            this.retainStacks = retainStacks;
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        public final Instant getEndTimestamp() {
            return endTimestamp;
        }

        @Override
        public final Builder endTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
            return this;
        }

        public final void setEndTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
        }

        @Override
        public StackSetOperation build() {
            return new StackSetOperation(this);
        }
    }
}
