/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AmazonWebServiceRequest;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input for the <a>CreateChangeSet</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class CreateChangeSetRequest extends AmazonWebServiceRequest implements
        ToCopyableBuilder<CreateChangeSetRequest.Builder, CreateChangeSetRequest> {
    private final String stackName;

    private final String templateBody;

    private final String templateURL;

    private final Boolean usePreviousTemplate;

    private final List<Parameter> parameters;

    private final List<String> capabilities;

    private final List<String> resourceTypes;

    private final String roleARN;

    private final RollbackConfiguration rollbackConfiguration;

    private final List<String> notificationARNs;

    private final List<Tag> tags;

    private final String changeSetName;

    private final String clientToken;

    private final String description;

    private final String changeSetType;

    private CreateChangeSetRequest(BuilderImpl builder) {
        this.stackName = builder.stackName;
        this.templateBody = builder.templateBody;
        this.templateURL = builder.templateURL;
        this.usePreviousTemplate = builder.usePreviousTemplate;
        this.parameters = builder.parameters;
        this.capabilities = builder.capabilities;
        this.resourceTypes = builder.resourceTypes;
        this.roleARN = builder.roleARN;
        this.rollbackConfiguration = builder.rollbackConfiguration;
        this.notificationARNs = builder.notificationARNs;
        this.tags = builder.tags;
        this.changeSetName = builder.changeSetName;
        this.clientToken = builder.clientToken;
        this.description = builder.description;
        this.changeSetType = builder.changeSetType;
    }

    /**
     * <p>
     * The name or the unique ID of the stack for which you are creating a change set. AWS CloudFormation generates the
     * change set by comparing this stack's information with the information that you submit, such as a modified
     * template or different parameter input values.
     * </p>
     * 
     * @return The name or the unique ID of the stack for which you are creating a change set. AWS CloudFormation
     *         generates the change set by comparing this stack's information with the information that you submit, such
     *         as a modified template or different parameter input values.
     */
    public String stackName() {
        return stackName;
    }

    /**
     * <p>
     * A structure that contains the body of the revised template, with a minimum length of 1 byte and a maximum length
     * of 51,200 bytes. AWS CloudFormation generates the change set by comparing this template with the template of the
     * stack that you specified.
     * </p>
     * <p>
     * Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
     * </p>
     * 
     * @return A structure that contains the body of the revised template, with a minimum length of 1 byte and a maximum
     *         length of 51,200 bytes. AWS CloudFormation generates the change set by comparing this template with the
     *         template of the stack that you specified.</p>
     *         <p>
     *         Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
     */
    public String templateBody() {
        return templateBody;
    }

    /**
     * <p>
     * The location of the file that contains the revised template. The URL must point to a template (max size: 460,800
     * bytes) that is located in an S3 bucket. AWS CloudFormation generates the change set by comparing this template
     * with the stack that you specified.
     * </p>
     * <p>
     * Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
     * </p>
     * 
     * @return The location of the file that contains the revised template. The URL must point to a template (max size:
     *         460,800 bytes) that is located in an S3 bucket. AWS CloudFormation generates the change set by comparing
     *         this template with the stack that you specified.</p>
     *         <p>
     *         Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
     */
    public String templateURL() {
        return templateURL;
    }

    /**
     * <p>
     * Whether to reuse the template that is associated with the stack to create the change set.
     * </p>
     * 
     * @return Whether to reuse the template that is associated with the stack to create the change set.
     */
    public Boolean usePreviousTemplate() {
        return usePreviousTemplate;
    }

    /**
     * <p>
     * A list of <code>Parameter</code> structures that specify input parameters for the change set. For more
     * information, see the <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a> data
     * type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of <code>Parameter</code> structures that specify input parameters for the change set. For more
     *         information, see the <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a>
     *         data type.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack templates
     * might include resources that can affect permissions in your AWS account, for example, by creating new AWS
     * Identity and Access Management (IAM) users. For those stacks, you must explicitly acknowledge their capabilities
     * by specifying this parameter.
     * </p>
     * <p>
     * The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The following
     * resources require you to specify this parameter: <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
     * AWS::IAM::AccessKey</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
     * AWS::IAM::Group</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
     * AWS::IAM::InstanceProfile</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
     * AWS::IAM::Policy</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
     * AWS::IAM::Role</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
     * AWS::IAM::User</a>, and <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html">
     * AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend that you review
     * all permissions associated with them and edit their permissions if necessary.
     * </p>
     * <p>
     * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names, you
     * must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this action returns an
     * <code>InsufficientCapabilities</code> error.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack
     *         templates might include resources that can affect permissions in your AWS account, for example, by
     *         creating new AWS Identity and Access Management (IAM) users. For those stacks, you must explicitly
     *         acknowledge their capabilities by specifying this parameter.</p>
     *         <p>
     *         The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The
     *         following resources require you to specify this parameter: <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
     *         AWS::IAM::AccessKey</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
     *         AWS::IAM::Group</a>, <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
     *         AWS::IAM::InstanceProfile</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
     *         AWS::IAM::Policy</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
     *         AWS::IAM::Role</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
     *         AWS::IAM::User</a>, and <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html">
     *         AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend that you
     *         review all permissions associated with them and edit their permissions if necessary.
     *         </p>
     *         <p>
     *         If you have IAM resources, you can specify either capability. If you have IAM resources with custom
     *         names, you must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this
     *         action returns an <code>InsufficientCapabilities</code> error.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
     */
    public List<Capability> capabilities() {
        return TypeConverter.convert(capabilities, Capability::fromValue);
    }

    /**
     * <p>
     * A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack templates
     * might include resources that can affect permissions in your AWS account, for example, by creating new AWS
     * Identity and Access Management (IAM) users. For those stacks, you must explicitly acknowledge their capabilities
     * by specifying this parameter.
     * </p>
     * <p>
     * The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The following
     * resources require you to specify this parameter: <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
     * AWS::IAM::AccessKey</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
     * AWS::IAM::Group</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
     * AWS::IAM::InstanceProfile</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
     * AWS::IAM::Policy</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
     * AWS::IAM::Role</a>, <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
     * AWS::IAM::User</a>, and <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html">
     * AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend that you review
     * all permissions associated with them and edit their permissions if necessary.
     * </p>
     * <p>
     * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names, you
     * must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this action returns an
     * <code>InsufficientCapabilities</code> error.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack
     *         templates might include resources that can affect permissions in your AWS account, for example, by
     *         creating new AWS Identity and Access Management (IAM) users. For those stacks, you must explicitly
     *         acknowledge their capabilities by specifying this parameter.</p>
     *         <p>
     *         The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The
     *         following resources require you to specify this parameter: <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
     *         AWS::IAM::AccessKey</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
     *         AWS::IAM::Group</a>, <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
     *         AWS::IAM::InstanceProfile</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
     *         AWS::IAM::Policy</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
     *         AWS::IAM::Role</a>, <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
     *         AWS::IAM::User</a>, and <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html">
     *         AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend that you
     *         review all permissions associated with them and edit their permissions if necessary.
     *         </p>
     *         <p>
     *         If you have IAM resources, you can specify either capability. If you have IAM resources with custom
     *         names, you must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this
     *         action returns an <code>InsufficientCapabilities</code> error.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
     */
    public List<String> capabilitiesStrings() {
        return capabilities;
    }

    /**
     * <p>
     * The template resource types that you have permissions to work with if you execute this change set, such as
     * <code>AWS::EC2::Instance</code>, <code>AWS::EC2::*</code>, or <code>Custom::MyCustomInstance</code>.
     * </p>
     * <p>
     * If the list of resource types doesn't include a resource type that you're updating, the stack update fails. By
     * default, AWS CloudFormation grants permissions to all resource types. AWS Identity and Access Management (IAM)
     * uses this parameter for condition keys in IAM policies for AWS CloudFormation. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html">Controlling Access
     * with AWS Identity and Access Management</a> in the AWS CloudFormation User Guide.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The template resource types that you have permissions to work with if you execute this change set, such
     *         as <code>AWS::EC2::Instance</code>, <code>AWS::EC2::*</code>, or <code>Custom::MyCustomInstance</code>
     *         .</p>
     *         <p>
     *         If the list of resource types doesn't include a resource type that you're updating, the stack update
     *         fails. By default, AWS CloudFormation grants permissions to all resource types. AWS Identity and Access
     *         Management (IAM) uses this parameter for condition keys in IAM policies for AWS CloudFormation. For more
     *         information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html">Controlling
     *         Access with AWS Identity and Access Management</a> in the AWS CloudFormation User Guide.
     */
    public List<String> resourceTypes() {
        return resourceTypes;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS CloudFormation
     * assumes when executing the change set. AWS CloudFormation uses the role's credentials to make calls on your
     * behalf. AWS CloudFormation uses this role for all future operations on the stack. As long as users have
     * permission to operate on the stack, AWS CloudFormation uses this role even if the users don't have permission to
     * pass it. Ensure that the role grants least privilege.
     * </p>
     * <p>
     * If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the stack. If
     * no role is available, AWS CloudFormation uses a temporary session that is generated from your user credentials.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS
     *         CloudFormation assumes when executing the change set. AWS CloudFormation uses the role's credentials to
     *         make calls on your behalf. AWS CloudFormation uses this role for all future operations on the stack. As
     *         long as users have permission to operate on the stack, AWS CloudFormation uses this role even if the
     *         users don't have permission to pass it. Ensure that the role grants least privilege.</p>
     *         <p>
     *         If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the
     *         stack. If no role is available, AWS CloudFormation uses a temporary session that is generated from your
     *         user credentials.
     */
    public String roleARN() {
        return roleARN;
    }

    /**
     * <p>
     * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and for
     * the specified monitoring period afterwards.
     * </p>
     * 
     * @return The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
     *         and for the specified monitoring period afterwards.
     */
    public RollbackConfiguration rollbackConfiguration() {
        return rollbackConfiguration;
    }

    /**
     * <p>
     * The Amazon Resource Names (ARNs) of Amazon Simple Notification Service (Amazon SNS) topics that AWS
     * CloudFormation associates with the stack. To remove all associated notification topics, specify an empty list.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The Amazon Resource Names (ARNs) of Amazon Simple Notification Service (Amazon SNS) topics that AWS
     *         CloudFormation associates with the stack. To remove all associated notification topics, specify an empty
     *         list.
     */
    public List<String> notificationARNs() {
        return notificationARNs;
    }

    /**
     * <p>
     * Key-value pairs to associate with this stack. AWS CloudFormation also propagates these tags to resources in the
     * stack. You can specify a maximum of 50 tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Key-value pairs to associate with this stack. AWS CloudFormation also propagates these tags to resources
     *         in the stack. You can specify a maximum of 50 tags.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The name of the change set. The name must be unique among all change sets that are associated with the specified
     * stack.
     * </p>
     * <p>
     * A change set name can contain only alphanumeric, case sensitive characters and hyphens. It must start with an
     * alphabetic character and cannot exceed 128 characters.
     * </p>
     * 
     * @return The name of the change set. The name must be unique among all change sets that are associated with the
     *         specified stack.</p>
     *         <p>
     *         A change set name can contain only alphanumeric, case sensitive characters and hyphens. It must start
     *         with an alphabetic character and cannot exceed 128 characters.
     */
    public String changeSetName() {
        return changeSetName;
    }

    /**
     * <p>
     * A unique identifier for this <code>CreateChangeSet</code> request. Specify this token if you plan to retry
     * requests so that AWS CloudFormation knows that you're not attempting to create another change set with the same
     * name. You might retry <code>CreateChangeSet</code> requests to ensure that AWS CloudFormation successfully
     * received them.
     * </p>
     * 
     * @return A unique identifier for this <code>CreateChangeSet</code> request. Specify this token if you plan to
     *         retry requests so that AWS CloudFormation knows that you're not attempting to create another change set
     *         with the same name. You might retry <code>CreateChangeSet</code> requests to ensure that AWS
     *         CloudFormation successfully received them.
     */
    public String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * A description to help you identify this change set.
     * </p>
     * 
     * @return A description to help you identify this change set.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>. To create
     * a change set for an existing stack, specify <code>UPDATE</code>.
     * </p>
     * <p>
     * If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID, but no
     * template or resources. The stack will be in the <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995">
     * <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
     * </p>
     * <p>
     * By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code> type to
     * create a change set for a new stack or the <code>CREATE</code> type to create a change set for an existing stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #changeSetType}
     * will return {@link ChangeSetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #changeSetTypeString}.
     * </p>
     * 
     * @return The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>. To
     *         create a change set for an existing stack, specify <code>UPDATE</code>.</p>
     *         <p>
     *         If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID,
     *         but no template or resources. The stack will be in the <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995"
     *         > <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
     *         </p>
     *         <p>
     *         By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code> type
     *         to create a change set for a new stack or the <code>CREATE</code> type to create a change set for an
     *         existing stack.
     * @see ChangeSetType
     */
    public ChangeSetType changeSetType() {
        return ChangeSetType.fromValue(changeSetType);
    }

    /**
     * <p>
     * The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>. To create
     * a change set for an existing stack, specify <code>UPDATE</code>.
     * </p>
     * <p>
     * If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID, but no
     * template or resources. The stack will be in the <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995">
     * <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
     * </p>
     * <p>
     * By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code> type to
     * create a change set for a new stack or the <code>CREATE</code> type to create a change set for an existing stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #changeSetType}
     * will return {@link ChangeSetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #changeSetTypeString}.
     * </p>
     * 
     * @return The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>. To
     *         create a change set for an existing stack, specify <code>UPDATE</code>.</p>
     *         <p>
     *         If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID,
     *         but no template or resources. The stack will be in the <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995"
     *         > <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
     *         </p>
     *         <p>
     *         By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code> type
     *         to create a change set for a new stack or the <code>CREATE</code> type to create a change set for an
     *         existing stack.
     * @see ChangeSetType
     */
    public String changeSetTypeString() {
        return changeSetType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackName());
        hashCode = 31 * hashCode + Objects.hashCode(templateBody());
        hashCode = 31 * hashCode + Objects.hashCode(templateURL());
        hashCode = 31 * hashCode + Objects.hashCode(usePreviousTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(capabilitiesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypes());
        hashCode = 31 * hashCode + Objects.hashCode(roleARN());
        hashCode = 31 * hashCode + Objects.hashCode(rollbackConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(notificationARNs());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(changeSetName());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(changeSetTypeString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateChangeSetRequest)) {
            return false;
        }
        CreateChangeSetRequest other = (CreateChangeSetRequest) obj;
        return Objects.equals(stackName(), other.stackName()) && Objects.equals(templateBody(), other.templateBody())
                && Objects.equals(templateURL(), other.templateURL())
                && Objects.equals(usePreviousTemplate(), other.usePreviousTemplate())
                && Objects.equals(parameters(), other.parameters())
                && Objects.equals(capabilitiesStrings(), other.capabilitiesStrings())
                && Objects.equals(resourceTypes(), other.resourceTypes()) && Objects.equals(roleARN(), other.roleARN())
                && Objects.equals(rollbackConfiguration(), other.rollbackConfiguration())
                && Objects.equals(notificationARNs(), other.notificationARNs()) && Objects.equals(tags(), other.tags())
                && Objects.equals(changeSetName(), other.changeSetName()) && Objects.equals(clientToken(), other.clientToken())
                && Objects.equals(description(), other.description())
                && Objects.equals(changeSetTypeString(), other.changeSetTypeString());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (stackName() != null) {
            sb.append("StackName: ").append(stackName()).append(",");
        }
        if (templateBody() != null) {
            sb.append("TemplateBody: ").append(templateBody()).append(",");
        }
        if (templateURL() != null) {
            sb.append("TemplateURL: ").append(templateURL()).append(",");
        }
        if (usePreviousTemplate() != null) {
            sb.append("UsePreviousTemplate: ").append(usePreviousTemplate()).append(",");
        }
        if (parameters() != null) {
            sb.append("Parameters: ").append(parameters()).append(",");
        }
        if (capabilitiesStrings() != null) {
            sb.append("Capabilities: ").append(capabilitiesStrings()).append(",");
        }
        if (resourceTypes() != null) {
            sb.append("ResourceTypes: ").append(resourceTypes()).append(",");
        }
        if (roleARN() != null) {
            sb.append("RoleARN: ").append(roleARN()).append(",");
        }
        if (rollbackConfiguration() != null) {
            sb.append("RollbackConfiguration: ").append(rollbackConfiguration()).append(",");
        }
        if (notificationARNs() != null) {
            sb.append("NotificationARNs: ").append(notificationARNs()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (changeSetName() != null) {
            sb.append("ChangeSetName: ").append(changeSetName()).append(",");
        }
        if (clientToken() != null) {
            sb.append("ClientToken: ").append(clientToken()).append(",");
        }
        if (description() != null) {
            sb.append("Description: ").append(description()).append(",");
        }
        if (changeSetTypeString() != null) {
            sb.append("ChangeSetType: ").append(changeSetTypeString()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackName":
            return Optional.of(clazz.cast(stackName()));
        case "TemplateBody":
            return Optional.of(clazz.cast(templateBody()));
        case "TemplateURL":
            return Optional.of(clazz.cast(templateURL()));
        case "UsePreviousTemplate":
            return Optional.of(clazz.cast(usePreviousTemplate()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "Capabilities":
            return Optional.of(clazz.cast(capabilitiesStrings()));
        case "ResourceTypes":
            return Optional.of(clazz.cast(resourceTypes()));
        case "RoleARN":
            return Optional.of(clazz.cast(roleARN()));
        case "RollbackConfiguration":
            return Optional.of(clazz.cast(rollbackConfiguration()));
        case "NotificationARNs":
            return Optional.of(clazz.cast(notificationARNs()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        case "ChangeSetName":
            return Optional.of(clazz.cast(changeSetName()));
        case "ClientToken":
            return Optional.of(clazz.cast(clientToken()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "ChangeSetType":
            return Optional.of(clazz.cast(changeSetTypeString()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CreateChangeSetRequest> {
        /**
         * <p>
         * The name or the unique ID of the stack for which you are creating a change set. AWS CloudFormation generates
         * the change set by comparing this stack's information with the information that you submit, such as a modified
         * template or different parameter input values.
         * </p>
         * 
         * @param stackName
         *        The name or the unique ID of the stack for which you are creating a change set. AWS CloudFormation
         *        generates the change set by comparing this stack's information with the information that you submit,
         *        such as a modified template or different parameter input values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackName(String stackName);

        /**
         * <p>
         * A structure that contains the body of the revised template, with a minimum length of 1 byte and a maximum
         * length of 51,200 bytes. AWS CloudFormation generates the change set by comparing this template with the
         * template of the stack that you specified.
         * </p>
         * <p>
         * Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
         * </p>
         * 
         * @param templateBody
         *        A structure that contains the body of the revised template, with a minimum length of 1 byte and a
         *        maximum length of 51,200 bytes. AWS CloudFormation generates the change set by comparing this template
         *        with the template of the stack that you specified.</p>
         *        <p>
         *        Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateBody(String templateBody);

        /**
         * <p>
         * The location of the file that contains the revised template. The URL must point to a template (max size:
         * 460,800 bytes) that is located in an S3 bucket. AWS CloudFormation generates the change set by comparing this
         * template with the stack that you specified.
         * </p>
         * <p>
         * Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
         * </p>
         * 
         * @param templateURL
         *        The location of the file that contains the revised template. The URL must point to a template (max
         *        size: 460,800 bytes) that is located in an S3 bucket. AWS CloudFormation generates the change set by
         *        comparing this template with the stack that you specified.</p>
         *        <p>
         *        Conditional: You must specify only <code>TemplateBody</code> or <code>TemplateURL</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateURL(String templateURL);

        /**
         * <p>
         * Whether to reuse the template that is associated with the stack to create the change set.
         * </p>
         * 
         * @param usePreviousTemplate
         *        Whether to reuse the template that is associated with the stack to create the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usePreviousTemplate(Boolean usePreviousTemplate);

        /**
         * <p>
         * A list of <code>Parameter</code> structures that specify input parameters for the change set. For more
         * information, see the <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a> data
         * type.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures that specify input parameters for the change set. For more
         *        information, see the <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html"
         *        >Parameter</a> data type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * A list of <code>Parameter</code> structures that specify input parameters for the change set. For more
         * information, see the <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a> data
         * type.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures that specify input parameters for the change set. For more
         *        information, see the <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html"
         *        >Parameter</a> data type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack
         * templates might include resources that can affect permissions in your AWS account, for example, by creating
         * new AWS Identity and Access Management (IAM) users. For those stacks, you must explicitly acknowledge their
         * capabilities by specifying this parameter.
         * </p>
         * <p>
         * The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The following
         * resources require you to specify this parameter: <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
         * AWS::IAM::AccessKey</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
         * AWS::IAM::Group</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
         * AWS::IAM::InstanceProfile</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
         * AWS::IAM::Policy</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
         * AWS::IAM::Role</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
         * AWS::IAM::User</a>, and <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html">
         * AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend that you
         * review all permissions associated with them and edit their permissions if necessary.
         * </p>
         * <p>
         * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names,
         * you must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this action returns
         * an <code>InsufficientCapabilities</code> error.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
         * </p>
         * 
         * @param capabilities
         *        A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack
         *        templates might include resources that can affect permissions in your AWS account, for example, by
         *        creating new AWS Identity and Access Management (IAM) users. For those stacks, you must explicitly
         *        acknowledge their capabilities by specifying this parameter.</p>
         *        <p>
         *        The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The
         *        following resources require you to specify this parameter: <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
         *        AWS::IAM::AccessKey</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
         *        AWS::IAM::Group</a>, <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
         *        AWS::IAM::InstanceProfile</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
         *        AWS::IAM::Policy</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
         *        AWS::IAM::Role</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
         *        AWS::IAM::User</a>, and <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html"
         *        > AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend
         *        that you review all permissions associated with them and edit their permissions if necessary.
         *        </p>
         *        <p>
         *        If you have IAM resources, you can specify either capability. If you have IAM resources with custom
         *        names, you must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this
         *        action returns an <code>InsufficientCapabilities</code> error.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<String> capabilities);

        /**
         * <p>
         * A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack
         * templates might include resources that can affect permissions in your AWS account, for example, by creating
         * new AWS Identity and Access Management (IAM) users. For those stacks, you must explicitly acknowledge their
         * capabilities by specifying this parameter.
         * </p>
         * <p>
         * The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The following
         * resources require you to specify this parameter: <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
         * AWS::IAM::AccessKey</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
         * AWS::IAM::Group</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
         * AWS::IAM::InstanceProfile</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
         * AWS::IAM::Policy</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
         * AWS::IAM::Role</a>, <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
         * AWS::IAM::User</a>, and <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html">
         * AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend that you
         * review all permissions associated with them and edit their permissions if necessary.
         * </p>
         * <p>
         * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names,
         * you must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this action returns
         * an <code>InsufficientCapabilities</code> error.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
         * </p>
         * 
         * @param capabilities
         *        A list of values that you must specify before AWS CloudFormation can update certain stacks. Some stack
         *        templates might include resources that can affect permissions in your AWS account, for example, by
         *        creating new AWS Identity and Access Management (IAM) users. For those stacks, you must explicitly
         *        acknowledge their capabilities by specifying this parameter.</p>
         *        <p>
         *        The only valid values are <code>CAPABILITY_IAM</code> and <code>CAPABILITY_NAMED_IAM</code>. The
         *        following resources require you to specify this parameter: <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-accesskey.html">
         *        AWS::IAM::AccessKey</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-group.html">
         *        AWS::IAM::Group</a>, <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-instanceprofile.html">
         *        AWS::IAM::InstanceProfile</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-policy.html">
         *        AWS::IAM::Policy</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-iam-role.html">
         *        AWS::IAM::Role</a>, <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-user.html">
         *        AWS::IAM::User</a>, and <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-iam-addusertogroup.html"
         *        > AWS::IAM::UserToGroupAddition</a>. If your stack template contains these resources, we recommend
         *        that you review all permissions associated with them and edit their permissions if necessary.
         *        </p>
         *        <p>
         *        If you have IAM resources, you can specify either capability. If you have IAM resources with custom
         *        names, you must specify <code>CAPABILITY_NAMED_IAM</code>. If you don't specify this parameter, this
         *        action returns an <code>InsufficientCapabilities</code> error.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(String... capabilities);

        /**
         * <p>
         * The template resource types that you have permissions to work with if you execute this change set, such as
         * <code>AWS::EC2::Instance</code>, <code>AWS::EC2::*</code>, or <code>Custom::MyCustomInstance</code>.
         * </p>
         * <p>
         * If the list of resource types doesn't include a resource type that you're updating, the stack update fails.
         * By default, AWS CloudFormation grants permissions to all resource types. AWS Identity and Access Management
         * (IAM) uses this parameter for condition keys in IAM policies for AWS CloudFormation. For more information,
         * see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html">Controlling
         * Access with AWS Identity and Access Management</a> in the AWS CloudFormation User Guide.
         * </p>
         * 
         * @param resourceTypes
         *        The template resource types that you have permissions to work with if you execute this change set,
         *        such as <code>AWS::EC2::Instance</code>, <code>AWS::EC2::*</code>, or
         *        <code>Custom::MyCustomInstance</code>.</p>
         *        <p>
         *        If the list of resource types doesn't include a resource type that you're updating, the stack update
         *        fails. By default, AWS CloudFormation grants permissions to all resource types. AWS Identity and
         *        Access Management (IAM) uses this parameter for condition keys in IAM policies for AWS CloudFormation.
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html"
         *        >Controlling Access with AWS Identity and Access Management</a> in the AWS CloudFormation User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<String> resourceTypes);

        /**
         * <p>
         * The template resource types that you have permissions to work with if you execute this change set, such as
         * <code>AWS::EC2::Instance</code>, <code>AWS::EC2::*</code>, or <code>Custom::MyCustomInstance</code>.
         * </p>
         * <p>
         * If the list of resource types doesn't include a resource type that you're updating, the stack update fails.
         * By default, AWS CloudFormation grants permissions to all resource types. AWS Identity and Access Management
         * (IAM) uses this parameter for condition keys in IAM policies for AWS CloudFormation. For more information,
         * see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html">Controlling
         * Access with AWS Identity and Access Management</a> in the AWS CloudFormation User Guide.
         * </p>
         * 
         * @param resourceTypes
         *        The template resource types that you have permissions to work with if you execute this change set,
         *        such as <code>AWS::EC2::Instance</code>, <code>AWS::EC2::*</code>, or
         *        <code>Custom::MyCustomInstance</code>.</p>
         *        <p>
         *        If the list of resource types doesn't include a resource type that you're updating, the stack update
         *        fails. By default, AWS CloudFormation grants permissions to all resource types. AWS Identity and
         *        Access Management (IAM) uses this parameter for condition keys in IAM policies for AWS CloudFormation.
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html"
         *        >Controlling Access with AWS Identity and Access Management</a> in the AWS CloudFormation User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(String... resourceTypes);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS CloudFormation
         * assumes when executing the change set. AWS CloudFormation uses the role's credentials to make calls on your
         * behalf. AWS CloudFormation uses this role for all future operations on the stack. As long as users have
         * permission to operate on the stack, AWS CloudFormation uses this role even if the users don't have permission
         * to pass it. Ensure that the role grants least privilege.
         * </p>
         * <p>
         * If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the stack.
         * If no role is available, AWS CloudFormation uses a temporary session that is generated from your user
         * credentials.
         * </p>
         * 
         * @param roleARN
         *        The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS
         *        CloudFormation assumes when executing the change set. AWS CloudFormation uses the role's credentials
         *        to make calls on your behalf. AWS CloudFormation uses this role for all future operations on the
         *        stack. As long as users have permission to operate on the stack, AWS CloudFormation uses this role
         *        even if the users don't have permission to pass it. Ensure that the role grants least privilege.</p>
         *        <p>
         *        If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the
         *        stack. If no role is available, AWS CloudFormation uses a temporary session that is generated from
         *        your user credentials.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleARN(String roleARN);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * 
         * @param rollbackConfiguration
         *        The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
         *        and for the specified monitoring period afterwards.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * This is a convenience that creates an instance of the {@link RollbackConfiguration.Builder} avoiding the need
         * to create one manually via {@link RollbackConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link RollbackConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #rollbackConfiguration(RollbackConfiguration)}.
         * 
         * @param rollbackConfiguration
         *        a consumer that will call methods on {@link RollbackConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rollbackConfiguration(RollbackConfiguration)
         */
        default Builder rollbackConfiguration(Consumer<RollbackConfiguration.Builder> rollbackConfiguration) {
            return rollbackConfiguration(RollbackConfiguration.builder().apply(rollbackConfiguration).build());
        }

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of Amazon Simple Notification Service (Amazon SNS) topics that AWS
         * CloudFormation associates with the stack. To remove all associated notification topics, specify an empty
         * list.
         * </p>
         * 
         * @param notificationARNs
         *        The Amazon Resource Names (ARNs) of Amazon Simple Notification Service (Amazon SNS) topics that AWS
         *        CloudFormation associates with the stack. To remove all associated notification topics, specify an
         *        empty list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(Collection<String> notificationARNs);

        /**
         * <p>
         * The Amazon Resource Names (ARNs) of Amazon Simple Notification Service (Amazon SNS) topics that AWS
         * CloudFormation associates with the stack. To remove all associated notification topics, specify an empty
         * list.
         * </p>
         * 
         * @param notificationARNs
         *        The Amazon Resource Names (ARNs) of Amazon Simple Notification Service (Amazon SNS) topics that AWS
         *        CloudFormation associates with the stack. To remove all associated notification topics, specify an
         *        empty list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(String... notificationARNs);

        /**
         * <p>
         * Key-value pairs to associate with this stack. AWS CloudFormation also propagates these tags to resources in
         * the stack. You can specify a maximum of 50 tags.
         * </p>
         * 
         * @param tags
         *        Key-value pairs to associate with this stack. AWS CloudFormation also propagates these tags to
         *        resources in the stack. You can specify a maximum of 50 tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Key-value pairs to associate with this stack. AWS CloudFormation also propagates these tags to resources in
         * the stack. You can specify a maximum of 50 tags.
         * </p>
         * 
         * @param tags
         *        Key-value pairs to associate with this stack. AWS CloudFormation also propagates these tags to
         *        resources in the stack. You can specify a maximum of 50 tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The name of the change set. The name must be unique among all change sets that are associated with the
         * specified stack.
         * </p>
         * <p>
         * A change set name can contain only alphanumeric, case sensitive characters and hyphens. It must start with an
         * alphabetic character and cannot exceed 128 characters.
         * </p>
         * 
         * @param changeSetName
         *        The name of the change set. The name must be unique among all change sets that are associated with the
         *        specified stack.</p>
         *        <p>
         *        A change set name can contain only alphanumeric, case sensitive characters and hyphens. It must start
         *        with an alphabetic character and cannot exceed 128 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeSetName(String changeSetName);

        /**
         * <p>
         * A unique identifier for this <code>CreateChangeSet</code> request. Specify this token if you plan to retry
         * requests so that AWS CloudFormation knows that you're not attempting to create another change set with the
         * same name. You might retry <code>CreateChangeSet</code> requests to ensure that AWS CloudFormation
         * successfully received them.
         * </p>
         * 
         * @param clientToken
         *        A unique identifier for this <code>CreateChangeSet</code> request. Specify this token if you plan to
         *        retry requests so that AWS CloudFormation knows that you're not attempting to create another change
         *        set with the same name. You might retry <code>CreateChangeSet</code> requests to ensure that AWS
         *        CloudFormation successfully received them.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * A description to help you identify this change set.
         * </p>
         * 
         * @param description
         *        A description to help you identify this change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>. To
         * create a change set for an existing stack, specify <code>UPDATE</code>.
         * </p>
         * <p>
         * If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID, but no
         * template or resources. The stack will be in the <a href=
         * "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995">
         * <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
         * </p>
         * <p>
         * By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code> type to
         * create a change set for a new stack or the <code>CREATE</code> type to create a change set for an existing
         * stack.
         * </p>
         * 
         * @param changeSetType
         *        The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>.
         *        To create a change set for an existing stack, specify <code>UPDATE</code>.</p>
         *        <p>
         *        If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID,
         *        but no template or resources. The stack will be in the <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995"
         *        > <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
         *        </p>
         *        <p>
         *        By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code>
         *        type to create a change set for a new stack or the <code>CREATE</code> type to create a change set for
         *        an existing stack.
         * @see ChangeSetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeSetType
         */
        Builder changeSetType(String changeSetType);

        /**
         * <p>
         * The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>. To
         * create a change set for an existing stack, specify <code>UPDATE</code>.
         * </p>
         * <p>
         * If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID, but no
         * template or resources. The stack will be in the <a href=
         * "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995">
         * <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
         * </p>
         * <p>
         * By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code> type to
         * create a change set for a new stack or the <code>CREATE</code> type to create a change set for an existing
         * stack.
         * </p>
         * 
         * @param changeSetType
         *        The type of change set operation. To create a change set for a new stack, specify <code>CREATE</code>.
         *        To create a change set for an existing stack, specify <code>UPDATE</code>.</p>
         *        <p>
         *        If you create a change set for a new stack, AWS Cloudformation creates a stack with a unique stack ID,
         *        but no template or resources. The stack will be in the <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-describing-stacks.html#d0e11995"
         *        > <code>REVIEW_IN_PROGRESS</code> </a> state until you execute the change set.
         *        </p>
         *        <p>
         *        By default, AWS CloudFormation specifies <code>UPDATE</code>. You can't use the <code>UPDATE</code>
         *        type to create a change set for a new stack or the <code>CREATE</code> type to create a change set for
         *        an existing stack.
         * @see ChangeSetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeSetType
         */
        Builder changeSetType(ChangeSetType changeSetType);
    }

    static final class BuilderImpl implements Builder {
        private String stackName;

        private String templateBody;

        private String templateURL;

        private Boolean usePreviousTemplate;

        private List<Parameter> parameters;

        private List<String> capabilities;

        private List<String> resourceTypes;

        private String roleARN;

        private RollbackConfiguration rollbackConfiguration;

        private List<String> notificationARNs;

        private List<Tag> tags;

        private String changeSetName;

        private String clientToken;

        private String description;

        private String changeSetType;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateChangeSetRequest model) {
            stackName(model.stackName);
            templateBody(model.templateBody);
            templateURL(model.templateURL);
            usePreviousTemplate(model.usePreviousTemplate);
            parameters(model.parameters);
            capabilities(model.capabilities);
            resourceTypes(model.resourceTypes);
            roleARN(model.roleARN);
            rollbackConfiguration(model.rollbackConfiguration);
            notificationARNs(model.notificationARNs);
            tags(model.tags);
            changeSetName(model.changeSetName);
            clientToken(model.clientToken);
            description(model.description);
            changeSetType(model.changeSetType);
        }

        public final String getStackName() {
            return stackName;
        }

        @Override
        public final Builder stackName(String stackName) {
            this.stackName = stackName;
            return this;
        }

        public final void setStackName(String stackName) {
            this.stackName = stackName;
        }

        public final String getTemplateBody() {
            return templateBody;
        }

        @Override
        public final Builder templateBody(String templateBody) {
            this.templateBody = templateBody;
            return this;
        }

        public final void setTemplateBody(String templateBody) {
            this.templateBody = templateBody;
        }

        public final String getTemplateURL() {
            return templateURL;
        }

        @Override
        public final Builder templateURL(String templateURL) {
            this.templateURL = templateURL;
            return this;
        }

        public final void setTemplateURL(String templateURL) {
            this.templateURL = templateURL;
        }

        public final Boolean getUsePreviousTemplate() {
            return usePreviousTemplate;
        }

        @Override
        public final Builder usePreviousTemplate(Boolean usePreviousTemplate) {
            this.usePreviousTemplate = usePreviousTemplate;
            return this;
        }

        public final void setUsePreviousTemplate(Boolean usePreviousTemplate) {
            this.usePreviousTemplate = usePreviousTemplate;
        }

        public final Collection<Parameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        public final Collection<String> getCapabilities() {
            return capabilities;
        }

        @Override
        public final Builder capabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(String... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
        }

        public final Collection<String> getResourceTypes() {
            return resourceTypes;
        }

        @Override
        public final Builder resourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypesCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(String... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = ResourceTypesCopier.copy(resourceTypes);
        }

        public final String getRoleARN() {
            return roleARN;
        }

        @Override
        public final Builder roleARN(String roleARN) {
            this.roleARN = roleARN;
            return this;
        }

        public final void setRoleARN(String roleARN) {
            this.roleARN = roleARN;
        }

        public final RollbackConfiguration.Builder getRollbackConfiguration() {
            return rollbackConfiguration != null ? rollbackConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration;
            return this;
        }

        public final void setRollbackConfiguration(RollbackConfiguration.BuilderImpl rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration != null ? rollbackConfiguration.build() : null;
        }

        public final Collection<String> getNotificationARNs() {
            return notificationARNs;
        }

        @Override
        public final Builder notificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationARNs(String... notificationARNs) {
            notificationARNs(Arrays.asList(notificationARNs));
            return this;
        }

        public final void setNotificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final String getChangeSetName() {
            return changeSetName;
        }

        @Override
        public final Builder changeSetName(String changeSetName) {
            this.changeSetName = changeSetName;
            return this;
        }

        public final void setChangeSetName(String changeSetName) {
            this.changeSetName = changeSetName;
        }

        public final String getClientToken() {
            return clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getChangeSetType() {
            return changeSetType;
        }

        @Override
        public final Builder changeSetType(String changeSetType) {
            this.changeSetType = changeSetType;
            return this;
        }

        @Override
        public final Builder changeSetType(ChangeSetType changeSetType) {
            this.changeSetType(changeSetType.toString());
            return this;
        }

        public final void setChangeSetType(String changeSetType) {
            this.changeSetType = changeSetType;
        }

        @Override
        public CreateChangeSetRequest build() {
            return new CreateChangeSetRequest(this);
        }
    }
}
