/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AmazonWebServiceRequest;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public class CreateStackInstancesRequest extends AmazonWebServiceRequest implements
        ToCopyableBuilder<CreateStackInstancesRequest.Builder, CreateStackInstancesRequest> {
    private final String stackSetName;

    private final List<String> accounts;

    private final List<String> regions;

    private final StackSetOperationPreferences operationPreferences;

    private final String operationId;

    private CreateStackInstancesRequest(BuilderImpl builder) {
        this.stackSetName = builder.stackSetName;
        this.accounts = builder.accounts;
        this.regions = builder.regions;
        this.operationPreferences = builder.operationPreferences;
        this.operationId = builder.operationId;
    }

    /**
     * <p>
     * The name or unique ID of the stack set that you want to create stack instances from.
     * </p>
     * 
     * @return The name or unique ID of the stack set that you want to create stack instances from.
     */
    public String stackSetName() {
        return stackSetName;
    }

    /**
     * <p>
     * The names of one or more AWS accounts that you want to create stack instances in the specified region(s) for.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The names of one or more AWS accounts that you want to create stack instances in the specified region(s)
     *         for.
     */
    public List<String> accounts() {
        return accounts;
    }

    /**
     * <p>
     * The names of one or more regions where you want to create stack instances using the specified AWS account(s).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The names of one or more regions where you want to create stack instances using the specified AWS
     *         account(s).
     */
    public List<String> regions() {
        return regions;
    }

    /**
     * <p>
     * Preferences for how AWS CloudFormation performs this stack set operation.
     * </p>
     * 
     * @return Preferences for how AWS CloudFormation performs this stack set operation.
     */
    public StackSetOperationPreferences operationPreferences() {
        return operationPreferences;
    }

    /**
     * <p>
     * The unique identifier for this stack set operation.
     * </p>
     * <p>
     * The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs the stack set
     * operation only once, even if you retry the request multiple times. You might retry stack set operation requests
     * to ensure that AWS CloudFormation successfully received them.
     * </p>
     * <p>
     * If you don't specify an operation ID, the SDK generates one automatically.
     * </p>
     * <p>
     * Repeating this stack set operation with a new operation ID retries all stack instances whose status is
     * <code>OUTDATED</code>.
     * </p>
     * 
     * @return The unique identifier for this stack set operation. </p>
     *         <p>
     *         The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs the
     *         stack set operation only once, even if you retry the request multiple times. You might retry stack set
     *         operation requests to ensure that AWS CloudFormation successfully received them.
     *         </p>
     *         <p>
     *         If you don't specify an operation ID, the SDK generates one automatically.
     *         </p>
     *         <p>
     *         Repeating this stack set operation with a new operation ID retries all stack instances whose status is
     *         <code>OUTDATED</code>.
     */
    public String operationId() {
        return operationId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackSetName());
        hashCode = 31 * hashCode + Objects.hashCode(accounts());
        hashCode = 31 * hashCode + Objects.hashCode(regions());
        hashCode = 31 * hashCode + Objects.hashCode(operationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(operationId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateStackInstancesRequest)) {
            return false;
        }
        CreateStackInstancesRequest other = (CreateStackInstancesRequest) obj;
        return Objects.equals(stackSetName(), other.stackSetName()) && Objects.equals(accounts(), other.accounts())
                && Objects.equals(regions(), other.regions())
                && Objects.equals(operationPreferences(), other.operationPreferences())
                && Objects.equals(operationId(), other.operationId());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (stackSetName() != null) {
            sb.append("StackSetName: ").append(stackSetName()).append(",");
        }
        if (accounts() != null) {
            sb.append("Accounts: ").append(accounts()).append(",");
        }
        if (regions() != null) {
            sb.append("Regions: ").append(regions()).append(",");
        }
        if (operationPreferences() != null) {
            sb.append("OperationPreferences: ").append(operationPreferences()).append(",");
        }
        if (operationId() != null) {
            sb.append("OperationId: ").append(operationId()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetName":
            return Optional.of(clazz.cast(stackSetName()));
        case "Accounts":
            return Optional.of(clazz.cast(accounts()));
        case "Regions":
            return Optional.of(clazz.cast(regions()));
        case "OperationPreferences":
            return Optional.of(clazz.cast(operationPreferences()));
        case "OperationId":
            return Optional.of(clazz.cast(operationId()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, CreateStackInstancesRequest> {
        /**
         * <p>
         * The name or unique ID of the stack set that you want to create stack instances from.
         * </p>
         * 
         * @param stackSetName
         *        The name or unique ID of the stack set that you want to create stack instances from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetName(String stackSetName);

        /**
         * <p>
         * The names of one or more AWS accounts that you want to create stack instances in the specified region(s) for.
         * </p>
         * 
         * @param accounts
         *        The names of one or more AWS accounts that you want to create stack instances in the specified
         *        region(s) for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(Collection<String> accounts);

        /**
         * <p>
         * The names of one or more AWS accounts that you want to create stack instances in the specified region(s) for.
         * </p>
         * 
         * @param accounts
         *        The names of one or more AWS accounts that you want to create stack instances in the specified
         *        region(s) for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accounts(String... accounts);

        /**
         * <p>
         * The names of one or more regions where you want to create stack instances using the specified AWS account(s).
         * </p>
         * 
         * @param regions
         *        The names of one or more regions where you want to create stack instances using the specified AWS
         *        account(s).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(Collection<String> regions);

        /**
         * <p>
         * The names of one or more regions where you want to create stack instances using the specified AWS account(s).
         * </p>
         * 
         * @param regions
         *        The names of one or more regions where you want to create stack instances using the specified AWS
         *        account(s).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regions(String... regions);

        /**
         * <p>
         * Preferences for how AWS CloudFormation performs this stack set operation.
         * </p>
         * 
         * @param operationPreferences
         *        Preferences for how AWS CloudFormation performs this stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationPreferences(StackSetOperationPreferences operationPreferences);

        /**
         * <p>
         * Preferences for how AWS CloudFormation performs this stack set operation.
         * </p>
         * This is a convenience that creates an instance of the {@link StackSetOperationPreferences.Builder} avoiding
         * the need to create one manually via {@link StackSetOperationPreferences#builder()}.
         *
         * When the {@link Consumer} completes, {@link StackSetOperationPreferences.Builder#build()} is called
         * immediately and its result is passed to {@link #operationPreferences(StackSetOperationPreferences)}.
         * 
         * @param operationPreferences
         *        a consumer that will call methods on {@link StackSetOperationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operationPreferences(StackSetOperationPreferences)
         */
        default Builder operationPreferences(Consumer<StackSetOperationPreferences.Builder> operationPreferences) {
            return operationPreferences(StackSetOperationPreferences.builder().apply(operationPreferences).build());
        }

        /**
         * <p>
         * The unique identifier for this stack set operation.
         * </p>
         * <p>
         * The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs the stack
         * set operation only once, even if you retry the request multiple times. You might retry stack set operation
         * requests to ensure that AWS CloudFormation successfully received them.
         * </p>
         * <p>
         * If you don't specify an operation ID, the SDK generates one automatically.
         * </p>
         * <p>
         * Repeating this stack set operation with a new operation ID retries all stack instances whose status is
         * <code>OUTDATED</code>.
         * </p>
         * 
         * @param operationId
         *        The unique identifier for this stack set operation. </p>
         *        <p>
         *        The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs
         *        the stack set operation only once, even if you retry the request multiple times. You might retry stack
         *        set operation requests to ensure that AWS CloudFormation successfully received them.
         *        </p>
         *        <p>
         *        If you don't specify an operation ID, the SDK generates one automatically.
         *        </p>
         *        <p>
         *        Repeating this stack set operation with a new operation ID retries all stack instances whose status is
         *        <code>OUTDATED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationId(String operationId);
    }

    static final class BuilderImpl implements Builder {
        private String stackSetName;

        private List<String> accounts;

        private List<String> regions;

        private StackSetOperationPreferences operationPreferences;

        private String operationId;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateStackInstancesRequest model) {
            stackSetName(model.stackSetName);
            accounts(model.accounts);
            regions(model.regions);
            operationPreferences(model.operationPreferences);
            operationId(model.operationId);
        }

        public final String getStackSetName() {
            return stackSetName;
        }

        @Override
        public final Builder stackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
            return this;
        }

        public final void setStackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
        }

        public final Collection<String> getAccounts() {
            return accounts;
        }

        @Override
        public final Builder accounts(Collection<String> accounts) {
            this.accounts = AccountListCopier.copy(accounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accounts(String... accounts) {
            accounts(Arrays.asList(accounts));
            return this;
        }

        public final void setAccounts(Collection<String> accounts) {
            this.accounts = AccountListCopier.copy(accounts);
        }

        public final Collection<String> getRegions() {
            return regions;
        }

        @Override
        public final Builder regions(Collection<String> regions) {
            this.regions = RegionListCopier.copy(regions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regions(String... regions) {
            regions(Arrays.asList(regions));
            return this;
        }

        public final void setRegions(Collection<String> regions) {
            this.regions = RegionListCopier.copy(regions);
        }

        public final StackSetOperationPreferences.Builder getOperationPreferences() {
            return operationPreferences != null ? operationPreferences.toBuilder() : null;
        }

        @Override
        public final Builder operationPreferences(StackSetOperationPreferences operationPreferences) {
            this.operationPreferences = operationPreferences;
            return this;
        }

        public final void setOperationPreferences(StackSetOperationPreferences.BuilderImpl operationPreferences) {
            this.operationPreferences = operationPreferences != null ? operationPreferences.build() : null;
        }

        public final String getOperationId() {
            return operationId;
        }

        @Override
        public final Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public final void setOperationId(String operationId) {
            this.operationId = operationId;
        }

        @Override
        public CreateStackInstancesRequest build() {
            return new CreateStackInstancesRequest(this);
        }
    }
}
