/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Structure containing the rollback triggers for AWS CloudFormation to monitor during stack creation and updating
 * operations, and for the specified monitoring period afterwards.
 * </p>
 * <p>
 * Rollback triggers enable you to have AWS CloudFormation monitor the state of your application during stack creation
 * and updating, and to roll back that operation if the application breaches the threshold of any of the alarms you've
 * specified. For each rollback trigger you create, you specify the Cloudwatch alarm that CloudFormation should monitor.
 * CloudFormation monitors the specified alarms during the stack create or update operation, and for the specified
 * amount of time after all resources have been deployed. If any of the alarms goes to ALERT state during the stack
 * operation or the monitoring period, CloudFormation rolls back the entire stack operation. If the monitoring period
 * expires without any alarms going to ALERT state, CloudFormation proceeds to dispose of old resources as usual.
 * </p>
 * <p>
 * By default, CloudFormation only rolls back stack operations if an alarm goes to ALERT state, not INSUFFICIENT_DATA
 * state. To have CloudFormation roll back the stack operation if an alarm goes to INSUFFICIENT_DATA state as well, edit
 * the CloudWatch alarm to treat missing data as <code>breaching</code>. For more information, see <a
 * href="http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/AlarmThatSendsEmail.html">Configuring How
 * CloudWatch Alarms Treats Missing Data</a>.
 * </p>
 * <p>
 * AWS CloudFormation does not monitor rollback triggers when it rolls back a stack during an update operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class RollbackConfiguration implements ToCopyableBuilder<RollbackConfiguration.Builder, RollbackConfiguration> {
    private final List<RollbackTrigger> rollbackTriggers;

    private final Integer monitoringTimeInMinutes;

    private RollbackConfiguration(BuilderImpl builder) {
        this.rollbackTriggers = builder.rollbackTriggers;
        this.monitoringTimeInMinutes = builder.monitoringTimeInMinutes;
    }

    /**
     * <p>
     * The triggers to monitor during stack creation or update actions.
     * </p>
     * <p>
     * By default, AWS CloudFormation saves the rollback triggers specified for a stack and applies them to any
     * subsequent update operations for the stack, unless you specify otherwise. If you do specify rollback triggers for
     * this parameter, those triggers replace any list of triggers previously specified for the stack. This means:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you don't specify this parameter, AWS CloudFormation uses the rollback triggers previously specified for this
     * stack, if any.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify any rollback triggers using this parameter, you must specify all the triggers that you want used
     * for this stack, even triggers you've specifed before (for example, when creating the stack or during a previous
     * stack update). Any triggers that you don't include in the updated list of triggers are no longer applied to the
     * stack.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify an empty list, AWS CloudFormation removes all currently specified triggers.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If a specified Cloudwatch alarm is missing, the entire stack operation fails and is rolled back.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The triggers to monitor during stack creation or update actions. </p>
     *         <p>
     *         By default, AWS CloudFormation saves the rollback triggers specified for a stack and applies them to any
     *         subsequent update operations for the stack, unless you specify otherwise. If you do specify rollback
     *         triggers for this parameter, those triggers replace any list of triggers previously specified for the
     *         stack. This means:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you don't specify this parameter, AWS CloudFormation uses the rollback triggers previously specified
     *         for this stack, if any.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify any rollback triggers using this parameter, you must specify all the triggers that you
     *         want used for this stack, even triggers you've specifed before (for example, when creating the stack or
     *         during a previous stack update). Any triggers that you don't include in the updated list of triggers are
     *         no longer applied to the stack.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify an empty list, AWS CloudFormation removes all currently specified triggers.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If a specified Cloudwatch alarm is missing, the entire stack operation fails and is rolled back.
     */
    public List<RollbackTrigger> rollbackTriggers() {
        return rollbackTriggers;
    }

    /**
     * <p>
     * The amount of time, in minutes, during which CloudFormation should monitor all the rollback triggers after the
     * stack creation or update operation deploys all necessary resources. If any of the alarms goes to ALERT state
     * during the stack operation or this monitoring period, CloudFormation rolls back the entire stack operation. Then,
     * for update operations, if the monitoring period expires without any alarms going to ALERT state CloudFormation
     * proceeds to dispose of old resources as usual.
     * </p>
     * <p>
     * If you specify a monitoring period but do not specify any rollback triggers, CloudFormation still waits the
     * specified period of time before cleaning up old resources for update operations. You can use this monitoring
     * period to perform any manual stack validation desired, and manually cancel the stack creation or update (using <a
     * href
     * ="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html">CancelUpdateStack
     * </a>, for example) as necessary.
     * </p>
     * <p>
     * If you specify 0 for this parameter, CloudFormation still monitors the specified rollback triggers during stack
     * creation and update operations. Then, for update operations, it begins disposing of old resources immediately
     * once the operation completes.
     * </p>
     * 
     * @return The amount of time, in minutes, during which CloudFormation should monitor all the rollback triggers
     *         after the stack creation or update operation deploys all necessary resources. If any of the alarms goes
     *         to ALERT state during the stack operation or this monitoring period, CloudFormation rolls back the entire
     *         stack operation. Then, for update operations, if the monitoring period expires without any alarms going
     *         to ALERT state CloudFormation proceeds to dispose of old resources as usual.</p>
     *         <p>
     *         If you specify a monitoring period but do not specify any rollback triggers, CloudFormation still waits
     *         the specified period of time before cleaning up old resources for update operations. You can use this
     *         monitoring period to perform any manual stack validation desired, and manually cancel the stack creation
     *         or update (using <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html"
     *         >CancelUpdateStack</a>, for example) as necessary.
     *         </p>
     *         <p>
     *         If you specify 0 for this parameter, CloudFormation still monitors the specified rollback triggers during
     *         stack creation and update operations. Then, for update operations, it begins disposing of old resources
     *         immediately once the operation completes.
     */
    public Integer monitoringTimeInMinutes() {
        return monitoringTimeInMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(rollbackTriggers());
        hashCode = 31 * hashCode + Objects.hashCode(monitoringTimeInMinutes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RollbackConfiguration)) {
            return false;
        }
        RollbackConfiguration other = (RollbackConfiguration) obj;
        return Objects.equals(rollbackTriggers(), other.rollbackTriggers())
                && Objects.equals(monitoringTimeInMinutes(), other.monitoringTimeInMinutes());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (rollbackTriggers() != null) {
            sb.append("RollbackTriggers: ").append(rollbackTriggers()).append(",");
        }
        if (monitoringTimeInMinutes() != null) {
            sb.append("MonitoringTimeInMinutes: ").append(monitoringTimeInMinutes()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RollbackTriggers":
            return Optional.of(clazz.cast(rollbackTriggers()));
        case "MonitoringTimeInMinutes":
            return Optional.of(clazz.cast(monitoringTimeInMinutes()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, RollbackConfiguration> {
        /**
         * <p>
         * The triggers to monitor during stack creation or update actions.
         * </p>
         * <p>
         * By default, AWS CloudFormation saves the rollback triggers specified for a stack and applies them to any
         * subsequent update operations for the stack, unless you specify otherwise. If you do specify rollback triggers
         * for this parameter, those triggers replace any list of triggers previously specified for the stack. This
         * means:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you don't specify this parameter, AWS CloudFormation uses the rollback triggers previously specified for
         * this stack, if any.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify any rollback triggers using this parameter, you must specify all the triggers that you want
         * used for this stack, even triggers you've specifed before (for example, when creating the stack or during a
         * previous stack update). Any triggers that you don't include in the updated list of triggers are no longer
         * applied to the stack.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify an empty list, AWS CloudFormation removes all currently specified triggers.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If a specified Cloudwatch alarm is missing, the entire stack operation fails and is rolled back.
         * </p>
         * 
         * @param rollbackTriggers
         *        The triggers to monitor during stack creation or update actions. </p>
         *        <p>
         *        By default, AWS CloudFormation saves the rollback triggers specified for a stack and applies them to
         *        any subsequent update operations for the stack, unless you specify otherwise. If you do specify
         *        rollback triggers for this parameter, those triggers replace any list of triggers previously specified
         *        for the stack. This means:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you don't specify this parameter, AWS CloudFormation uses the rollback triggers previously
         *        specified for this stack, if any.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify any rollback triggers using this parameter, you must specify all the triggers that you
         *        want used for this stack, even triggers you've specifed before (for example, when creating the stack
         *        or during a previous stack update). Any triggers that you don't include in the updated list of
         *        triggers are no longer applied to the stack.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify an empty list, AWS CloudFormation removes all currently specified triggers.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If a specified Cloudwatch alarm is missing, the entire stack operation fails and is rolled back.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackTriggers(Collection<RollbackTrigger> rollbackTriggers);

        /**
         * <p>
         * The triggers to monitor during stack creation or update actions.
         * </p>
         * <p>
         * By default, AWS CloudFormation saves the rollback triggers specified for a stack and applies them to any
         * subsequent update operations for the stack, unless you specify otherwise. If you do specify rollback triggers
         * for this parameter, those triggers replace any list of triggers previously specified for the stack. This
         * means:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you don't specify this parameter, AWS CloudFormation uses the rollback triggers previously specified for
         * this stack, if any.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify any rollback triggers using this parameter, you must specify all the triggers that you want
         * used for this stack, even triggers you've specifed before (for example, when creating the stack or during a
         * previous stack update). Any triggers that you don't include in the updated list of triggers are no longer
         * applied to the stack.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify an empty list, AWS CloudFormation removes all currently specified triggers.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If a specified Cloudwatch alarm is missing, the entire stack operation fails and is rolled back.
         * </p>
         * 
         * @param rollbackTriggers
         *        The triggers to monitor during stack creation or update actions. </p>
         *        <p>
         *        By default, AWS CloudFormation saves the rollback triggers specified for a stack and applies them to
         *        any subsequent update operations for the stack, unless you specify otherwise. If you do specify
         *        rollback triggers for this parameter, those triggers replace any list of triggers previously specified
         *        for the stack. This means:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you don't specify this parameter, AWS CloudFormation uses the rollback triggers previously
         *        specified for this stack, if any.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify any rollback triggers using this parameter, you must specify all the triggers that you
         *        want used for this stack, even triggers you've specifed before (for example, when creating the stack
         *        or during a previous stack update). Any triggers that you don't include in the updated list of
         *        triggers are no longer applied to the stack.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify an empty list, AWS CloudFormation removes all currently specified triggers.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If a specified Cloudwatch alarm is missing, the entire stack operation fails and is rolled back.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackTriggers(RollbackTrigger... rollbackTriggers);

        /**
         * <p>
         * The amount of time, in minutes, during which CloudFormation should monitor all the rollback triggers after
         * the stack creation or update operation deploys all necessary resources. If any of the alarms goes to ALERT
         * state during the stack operation or this monitoring period, CloudFormation rolls back the entire stack
         * operation. Then, for update operations, if the monitoring period expires without any alarms going to ALERT
         * state CloudFormation proceeds to dispose of old resources as usual.
         * </p>
         * <p>
         * If you specify a monitoring period but do not specify any rollback triggers, CloudFormation still waits the
         * specified period of time before cleaning up old resources for update operations. You can use this monitoring
         * period to perform any manual stack validation desired, and manually cancel the stack creation or update
         * (using <a href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html">
         * CancelUpdateStack</a>, for example) as necessary.
         * </p>
         * <p>
         * If you specify 0 for this parameter, CloudFormation still monitors the specified rollback triggers during
         * stack creation and update operations. Then, for update operations, it begins disposing of old resources
         * immediately once the operation completes.
         * </p>
         * 
         * @param monitoringTimeInMinutes
         *        The amount of time, in minutes, during which CloudFormation should monitor all the rollback triggers
         *        after the stack creation or update operation deploys all necessary resources. If any of the alarms
         *        goes to ALERT state during the stack operation or this monitoring period, CloudFormation rolls back
         *        the entire stack operation. Then, for update operations, if the monitoring period expires without any
         *        alarms going to ALERT state CloudFormation proceeds to dispose of old resources as usual.</p>
         *        <p>
         *        If you specify a monitoring period but do not specify any rollback triggers, CloudFormation still
         *        waits the specified period of time before cleaning up old resources for update operations. You can use
         *        this monitoring period to perform any manual stack validation desired, and manually cancel the stack
         *        creation or update (using <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html"
         *        >CancelUpdateStack</a>, for example) as necessary.
         *        </p>
         *        <p>
         *        If you specify 0 for this parameter, CloudFormation still monitors the specified rollback triggers
         *        during stack creation and update operations. Then, for update operations, it begins disposing of old
         *        resources immediately once the operation completes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoringTimeInMinutes(Integer monitoringTimeInMinutes);
    }

    static final class BuilderImpl implements Builder {
        private List<RollbackTrigger> rollbackTriggers;

        private Integer monitoringTimeInMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(RollbackConfiguration model) {
            rollbackTriggers(model.rollbackTriggers);
            monitoringTimeInMinutes(model.monitoringTimeInMinutes);
        }

        public final Collection<RollbackTrigger.Builder> getRollbackTriggers() {
            return rollbackTriggers != null ? rollbackTriggers.stream().map(RollbackTrigger::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder rollbackTriggers(Collection<RollbackTrigger> rollbackTriggers) {
            this.rollbackTriggers = RollbackTriggersCopier.copy(rollbackTriggers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rollbackTriggers(RollbackTrigger... rollbackTriggers) {
            rollbackTriggers(Arrays.asList(rollbackTriggers));
            return this;
        }

        public final void setRollbackTriggers(Collection<RollbackTrigger.BuilderImpl> rollbackTriggers) {
            this.rollbackTriggers = RollbackTriggersCopier.copyFromBuilder(rollbackTriggers);
        }

        public final Integer getMonitoringTimeInMinutes() {
            return monitoringTimeInMinutes;
        }

        @Override
        public final Builder monitoringTimeInMinutes(Integer monitoringTimeInMinutes) {
            this.monitoringTimeInMinutes = monitoringTimeInMinutes;
            return this;
        }

        public final void setMonitoringTimeInMinutes(Integer monitoringTimeInMinutes) {
            this.monitoringTimeInMinutes = monitoringTimeInMinutes;
        }

        @Override
        public RollbackConfiguration build() {
            return new RollbackConfiguration(this);
        }
    }
}
