/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Stack data type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Stack implements ToCopyableBuilder<Stack.Builder, Stack> {
    private final String stackId;

    private final String stackName;

    private final String changeSetId;

    private final String description;

    private final List<Parameter> parameters;

    private final Instant creationTime;

    private final Instant deletionTime;

    private final Instant lastUpdatedTime;

    private final RollbackConfiguration rollbackConfiguration;

    private final String stackStatus;

    private final String stackStatusReason;

    private final Boolean disableRollback;

    private final List<String> notificationARNs;

    private final Integer timeoutInMinutes;

    private final List<String> capabilities;

    private final List<Output> outputs;

    private final String roleARN;

    private final List<Tag> tags;

    private final Boolean enableTerminationProtection;

    private final String parentId;

    private final String rootId;

    private Stack(BuilderImpl builder) {
        this.stackId = builder.stackId;
        this.stackName = builder.stackName;
        this.changeSetId = builder.changeSetId;
        this.description = builder.description;
        this.parameters = builder.parameters;
        this.creationTime = builder.creationTime;
        this.deletionTime = builder.deletionTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.rollbackConfiguration = builder.rollbackConfiguration;
        this.stackStatus = builder.stackStatus;
        this.stackStatusReason = builder.stackStatusReason;
        this.disableRollback = builder.disableRollback;
        this.notificationARNs = builder.notificationARNs;
        this.timeoutInMinutes = builder.timeoutInMinutes;
        this.capabilities = builder.capabilities;
        this.outputs = builder.outputs;
        this.roleARN = builder.roleARN;
        this.tags = builder.tags;
        this.enableTerminationProtection = builder.enableTerminationProtection;
        this.parentId = builder.parentId;
        this.rootId = builder.rootId;
    }

    /**
     * <p>
     * Unique identifier of the stack.
     * </p>
     * 
     * @return Unique identifier of the stack.
     */
    public String stackId() {
        return stackId;
    }

    /**
     * <p>
     * The name associated with the stack.
     * </p>
     * 
     * @return The name associated with the stack.
     */
    public String stackName() {
        return stackName;
    }

    /**
     * <p>
     * The unique ID of the change set.
     * </p>
     * 
     * @return The unique ID of the change set.
     */
    public String changeSetId() {
        return changeSetId;
    }

    /**
     * <p>
     * A user-defined description associated with the stack.
     * </p>
     * 
     * @return A user-defined description associated with the stack.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * A list of <code>Parameter</code> structures.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of <code>Parameter</code> structures.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The time at which the stack was created.
     * </p>
     * 
     * @return The time at which the stack was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time the stack was deleted.
     * </p>
     * 
     * @return The time the stack was deleted.
     */
    public Instant deletionTime() {
        return deletionTime;
    }

    /**
     * <p>
     * The time the stack was last updated. This field will only be returned if the stack has been updated at least
     * once.
     * </p>
     * 
     * @return The time the stack was last updated. This field will only be returned if the stack has been updated at
     *         least once.
     */
    public Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and for
     * the specified monitoring period afterwards.
     * </p>
     * 
     * @return The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
     *         and for the specified monitoring period afterwards.
     */
    public RollbackConfiguration rollbackConfiguration() {
        return rollbackConfiguration;
    }

    /**
     * <p>
     * Current status of the stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackStatus} will
     * return {@link StackStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stackStatusString}.
     * </p>
     * 
     * @return Current status of the stack.
     * @see StackStatus
     */
    public StackStatus stackStatus() {
        return StackStatus.fromValue(stackStatus);
    }

    /**
     * <p>
     * Current status of the stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackStatus} will
     * return {@link StackStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stackStatusString}.
     * </p>
     * 
     * @return Current status of the stack.
     * @see StackStatus
     */
    public String stackStatusString() {
        return stackStatus;
    }

    /**
     * <p>
     * Success/failure message associated with the stack status.
     * </p>
     * 
     * @return Success/failure message associated with the stack status.
     */
    public String stackStatusReason() {
        return stackStatusReason;
    }

    /**
     * <p>
     * Boolean to enable or disable rollback on stack creation failures:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>true</code>: disable rollback
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>false</code>: enable rollback
     * </p>
     * </li>
     * </ul>
     * 
     * @return Boolean to enable or disable rollback on stack creation failures:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>true</code>: disable rollback
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>false</code>: enable rollback
     *         </p>
     *         </li>
     */
    public Boolean disableRollback() {
        return disableRollback;
    }

    /**
     * <p>
     * SNS topic ARNs to which stack related events are published.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return SNS topic ARNs to which stack related events are published.
     */
    public List<String> notificationARNs() {
        return notificationARNs;
    }

    /**
     * <p>
     * The amount of time within which stack creation should complete.
     * </p>
     * 
     * @return The amount of time within which stack creation should complete.
     */
    public Integer timeoutInMinutes() {
        return timeoutInMinutes;
    }

    /**
     * <p>
     * The capabilities allowed in the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The capabilities allowed in the stack.
     */
    public List<Capability> capabilities() {
        return TypeConverter.convert(capabilities, Capability::fromValue);
    }

    /**
     * <p>
     * The capabilities allowed in the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The capabilities allowed in the stack.
     */
    public List<String> capabilitiesStrings() {
        return capabilities;
    }

    /**
     * <p>
     * A list of output structures.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of output structures.
     */
    public List<Output> outputs() {
        return outputs;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated with the
     * stack. During a stack operation, AWS CloudFormation uses this role's credentials to make calls on your behalf.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated
     *         with the stack. During a stack operation, AWS CloudFormation uses this role's credentials to make calls
     *         on your behalf.
     */
    public String roleARN() {
        return roleARN;
    }

    /**
     * <p>
     * A list of <code>Tag</code>s that specify information about the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of <code>Tag</code>s that specify information about the stack.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Whether termination protection is enabled for the stack.
     * </p>
     * <p>
     * For <a href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">nested
     * stacks</a>, termination protection is set on the root stack and cannot be changed directly on the nested stack.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html">Protecting a
     * Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return Whether termination protection is enabled for the stack.</p>
     *         <p>
     *         For <a href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">
     *         nested stacks</a>, termination protection is set on the root stack and cannot be changed directly on the
     *         nested stack. For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html"
     *         >Protecting a Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public Boolean enableTerminationProtection() {
        return enableTerminationProtection;
    }

    /**
     * <p>
     * For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of this
     * stack. For the first level of nested stacks, the root stack is also the parent stack.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working with
     * Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of
     *         this stack. For the first level of nested stacks, the root stack is also the parent stack.</p>
     *         <p>
     *         For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
     *         with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public String parentId() {
        return parentId;
    }

    /**
     * <p>
     * For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level stack to
     * which the nested stack ultimately belongs.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working with
     * Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level stack
     *         to which the nested stack ultimately belongs.</p>
     *         <p>
     *         For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
     *         with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public String rootId() {
        return rootId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackId());
        hashCode = 31 * hashCode + Objects.hashCode(stackName());
        hashCode = 31 * hashCode + Objects.hashCode(changeSetId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(deletionTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(rollbackConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(stackStatusString());
        hashCode = 31 * hashCode + Objects.hashCode(stackStatusReason());
        hashCode = 31 * hashCode + Objects.hashCode(disableRollback());
        hashCode = 31 * hashCode + Objects.hashCode(notificationARNs());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(capabilitiesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(outputs());
        hashCode = 31 * hashCode + Objects.hashCode(roleARN());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(enableTerminationProtection());
        hashCode = 31 * hashCode + Objects.hashCode(parentId());
        hashCode = 31 * hashCode + Objects.hashCode(rootId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Stack)) {
            return false;
        }
        Stack other = (Stack) obj;
        return Objects.equals(stackId(), other.stackId()) && Objects.equals(stackName(), other.stackName())
                && Objects.equals(changeSetId(), other.changeSetId()) && Objects.equals(description(), other.description())
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(deletionTime(), other.deletionTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(rollbackConfiguration(), other.rollbackConfiguration())
                && Objects.equals(stackStatusString(), other.stackStatusString())
                && Objects.equals(stackStatusReason(), other.stackStatusReason())
                && Objects.equals(disableRollback(), other.disableRollback())
                && Objects.equals(notificationARNs(), other.notificationARNs())
                && Objects.equals(timeoutInMinutes(), other.timeoutInMinutes())
                && Objects.equals(capabilitiesStrings(), other.capabilitiesStrings())
                && Objects.equals(outputs(), other.outputs()) && Objects.equals(roleARN(), other.roleARN())
                && Objects.equals(tags(), other.tags())
                && Objects.equals(enableTerminationProtection(), other.enableTerminationProtection())
                && Objects.equals(parentId(), other.parentId()) && Objects.equals(rootId(), other.rootId());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (stackId() != null) {
            sb.append("StackId: ").append(stackId()).append(",");
        }
        if (stackName() != null) {
            sb.append("StackName: ").append(stackName()).append(",");
        }
        if (changeSetId() != null) {
            sb.append("ChangeSetId: ").append(changeSetId()).append(",");
        }
        if (description() != null) {
            sb.append("Description: ").append(description()).append(",");
        }
        if (parameters() != null) {
            sb.append("Parameters: ").append(parameters()).append(",");
        }
        if (creationTime() != null) {
            sb.append("CreationTime: ").append(creationTime()).append(",");
        }
        if (deletionTime() != null) {
            sb.append("DeletionTime: ").append(deletionTime()).append(",");
        }
        if (lastUpdatedTime() != null) {
            sb.append("LastUpdatedTime: ").append(lastUpdatedTime()).append(",");
        }
        if (rollbackConfiguration() != null) {
            sb.append("RollbackConfiguration: ").append(rollbackConfiguration()).append(",");
        }
        if (stackStatusString() != null) {
            sb.append("StackStatus: ").append(stackStatusString()).append(",");
        }
        if (stackStatusReason() != null) {
            sb.append("StackStatusReason: ").append(stackStatusReason()).append(",");
        }
        if (disableRollback() != null) {
            sb.append("DisableRollback: ").append(disableRollback()).append(",");
        }
        if (notificationARNs() != null) {
            sb.append("NotificationARNs: ").append(notificationARNs()).append(",");
        }
        if (timeoutInMinutes() != null) {
            sb.append("TimeoutInMinutes: ").append(timeoutInMinutes()).append(",");
        }
        if (capabilitiesStrings() != null) {
            sb.append("Capabilities: ").append(capabilitiesStrings()).append(",");
        }
        if (outputs() != null) {
            sb.append("Outputs: ").append(outputs()).append(",");
        }
        if (roleARN() != null) {
            sb.append("RoleARN: ").append(roleARN()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (enableTerminationProtection() != null) {
            sb.append("EnableTerminationProtection: ").append(enableTerminationProtection()).append(",");
        }
        if (parentId() != null) {
            sb.append("ParentId: ").append(parentId()).append(",");
        }
        if (rootId() != null) {
            sb.append("RootId: ").append(rootId()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackId":
            return Optional.of(clazz.cast(stackId()));
        case "StackName":
            return Optional.of(clazz.cast(stackName()));
        case "ChangeSetId":
            return Optional.of(clazz.cast(changeSetId()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "CreationTime":
            return Optional.of(clazz.cast(creationTime()));
        case "DeletionTime":
            return Optional.of(clazz.cast(deletionTime()));
        case "LastUpdatedTime":
            return Optional.of(clazz.cast(lastUpdatedTime()));
        case "RollbackConfiguration":
            return Optional.of(clazz.cast(rollbackConfiguration()));
        case "StackStatus":
            return Optional.of(clazz.cast(stackStatusString()));
        case "StackStatusReason":
            return Optional.of(clazz.cast(stackStatusReason()));
        case "DisableRollback":
            return Optional.of(clazz.cast(disableRollback()));
        case "NotificationARNs":
            return Optional.of(clazz.cast(notificationARNs()));
        case "TimeoutInMinutes":
            return Optional.of(clazz.cast(timeoutInMinutes()));
        case "Capabilities":
            return Optional.of(clazz.cast(capabilitiesStrings()));
        case "Outputs":
            return Optional.of(clazz.cast(outputs()));
        case "RoleARN":
            return Optional.of(clazz.cast(roleARN()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        case "EnableTerminationProtection":
            return Optional.of(clazz.cast(enableTerminationProtection()));
        case "ParentId":
            return Optional.of(clazz.cast(parentId()));
        case "RootId":
            return Optional.of(clazz.cast(rootId()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, Stack> {
        /**
         * <p>
         * Unique identifier of the stack.
         * </p>
         * 
         * @param stackId
         *        Unique identifier of the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackId(String stackId);

        /**
         * <p>
         * The name associated with the stack.
         * </p>
         * 
         * @param stackName
         *        The name associated with the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackName(String stackName);

        /**
         * <p>
         * The unique ID of the change set.
         * </p>
         * 
         * @param changeSetId
         *        The unique ID of the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeSetId(String changeSetId);

        /**
         * <p>
         * A user-defined description associated with the stack.
         * </p>
         * 
         * @param description
         *        A user-defined description associated with the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of <code>Parameter</code> structures.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * A list of <code>Parameter</code> structures.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * The time at which the stack was created.
         * </p>
         * 
         * @param creationTime
         *        The time at which the stack was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time the stack was deleted.
         * </p>
         * 
         * @param deletionTime
         *        The time the stack was deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionTime(Instant deletionTime);

        /**
         * <p>
         * The time the stack was last updated. This field will only be returned if the stack has been updated at least
         * once.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The time the stack was last updated. This field will only be returned if the stack has been updated at
         *        least once.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * 
         * @param rollbackConfiguration
         *        The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
         *        and for the specified monitoring period afterwards.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * This is a convenience that creates an instance of the {@link RollbackConfiguration.Builder} avoiding the need
         * to create one manually via {@link RollbackConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link RollbackConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #rollbackConfiguration(RollbackConfiguration)}.
         * 
         * @param rollbackConfiguration
         *        a consumer that will call methods on {@link RollbackConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rollbackConfiguration(RollbackConfiguration)
         */
        default Builder rollbackConfiguration(Consumer<RollbackConfiguration.Builder> rollbackConfiguration) {
            return rollbackConfiguration(RollbackConfiguration.builder().apply(rollbackConfiguration).build());
        }

        /**
         * <p>
         * Current status of the stack.
         * </p>
         * 
         * @param stackStatus
         *        Current status of the stack.
         * @see StackStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackStatus
         */
        Builder stackStatus(String stackStatus);

        /**
         * <p>
         * Current status of the stack.
         * </p>
         * 
         * @param stackStatus
         *        Current status of the stack.
         * @see StackStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackStatus
         */
        Builder stackStatus(StackStatus stackStatus);

        /**
         * <p>
         * Success/failure message associated with the stack status.
         * </p>
         * 
         * @param stackStatusReason
         *        Success/failure message associated with the stack status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackStatusReason(String stackStatusReason);

        /**
         * <p>
         * Boolean to enable or disable rollback on stack creation failures:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>true</code>: disable rollback
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>false</code>: enable rollback
         * </p>
         * </li>
         * </ul>
         * 
         * @param disableRollback
         *        Boolean to enable or disable rollback on stack creation failures:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>true</code>: disable rollback
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>false</code>: enable rollback
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableRollback(Boolean disableRollback);

        /**
         * <p>
         * SNS topic ARNs to which stack related events are published.
         * </p>
         * 
         * @param notificationARNs
         *        SNS topic ARNs to which stack related events are published.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(Collection<String> notificationARNs);

        /**
         * <p>
         * SNS topic ARNs to which stack related events are published.
         * </p>
         * 
         * @param notificationARNs
         *        SNS topic ARNs to which stack related events are published.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(String... notificationARNs);

        /**
         * <p>
         * The amount of time within which stack creation should complete.
         * </p>
         * 
         * @param timeoutInMinutes
         *        The amount of time within which stack creation should complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInMinutes(Integer timeoutInMinutes);

        /**
         * <p>
         * The capabilities allowed in the stack.
         * </p>
         * 
         * @param capabilities
         *        The capabilities allowed in the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<String> capabilities);

        /**
         * <p>
         * The capabilities allowed in the stack.
         * </p>
         * 
         * @param capabilities
         *        The capabilities allowed in the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(String... capabilities);

        /**
         * <p>
         * A list of output structures.
         * </p>
         * 
         * @param outputs
         *        A list of output structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Collection<Output> outputs);

        /**
         * <p>
         * A list of output structures.
         * </p>
         * 
         * @param outputs
         *        A list of output structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Output... outputs);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated with
         * the stack. During a stack operation, AWS CloudFormation uses this role's credentials to make calls on your
         * behalf.
         * </p>
         * 
         * @param roleARN
         *        The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated
         *        with the stack. During a stack operation, AWS CloudFormation uses this role's credentials to make
         *        calls on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleARN(String roleARN);

        /**
         * <p>
         * A list of <code>Tag</code>s that specify information about the stack.
         * </p>
         * 
         * @param tags
         *        A list of <code>Tag</code>s that specify information about the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of <code>Tag</code>s that specify information about the stack.
         * </p>
         * 
         * @param tags
         *        A list of <code>Tag</code>s that specify information about the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Whether termination protection is enabled for the stack.
         * </p>
         * <p>
         * For <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">nested
         * stacks</a>, termination protection is set on the root stack and cannot be changed directly on the nested
         * stack. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html">Protecting
         * a Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param enableTerminationProtection
         *        Whether termination protection is enabled for the stack.</p>
         *        <p>
         *        For <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html"
         *        >nested stacks</a>, termination protection is set on the root stack and cannot be changed directly on
         *        the nested stack. For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html"
         *        >Protecting a Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableTerminationProtection(Boolean enableTerminationProtection);

        /**
         * <p>
         * For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of this
         * stack. For the first level of nested stacks, the root stack is also the parent stack.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
         * with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param parentId
         *        For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of
         *        this stack. For the first level of nested stacks, the root stack is also the parent stack.</p>
         *        <p>
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html"
         *        >Working with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentId(String parentId);

        /**
         * <p>
         * For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level stack to
         * which the nested stack ultimately belongs.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
         * with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param rootId
         *        For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level
         *        stack to which the nested stack ultimately belongs.</p>
         *        <p>
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html"
         *        >Working with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootId(String rootId);
    }

    static final class BuilderImpl implements Builder {
        private String stackId;

        private String stackName;

        private String changeSetId;

        private String description;

        private List<Parameter> parameters;

        private Instant creationTime;

        private Instant deletionTime;

        private Instant lastUpdatedTime;

        private RollbackConfiguration rollbackConfiguration;

        private String stackStatus;

        private String stackStatusReason;

        private Boolean disableRollback;

        private List<String> notificationARNs;

        private Integer timeoutInMinutes;

        private List<String> capabilities;

        private List<Output> outputs;

        private String roleARN;

        private List<Tag> tags;

        private Boolean enableTerminationProtection;

        private String parentId;

        private String rootId;

        private BuilderImpl() {
        }

        private BuilderImpl(Stack model) {
            stackId(model.stackId);
            stackName(model.stackName);
            changeSetId(model.changeSetId);
            description(model.description);
            parameters(model.parameters);
            creationTime(model.creationTime);
            deletionTime(model.deletionTime);
            lastUpdatedTime(model.lastUpdatedTime);
            rollbackConfiguration(model.rollbackConfiguration);
            stackStatus(model.stackStatus);
            stackStatusReason(model.stackStatusReason);
            disableRollback(model.disableRollback);
            notificationARNs(model.notificationARNs);
            timeoutInMinutes(model.timeoutInMinutes);
            capabilities(model.capabilities);
            outputs(model.outputs);
            roleARN(model.roleARN);
            tags(model.tags);
            enableTerminationProtection(model.enableTerminationProtection);
            parentId(model.parentId);
            rootId(model.rootId);
        }

        public final String getStackId() {
            return stackId;
        }

        @Override
        public final Builder stackId(String stackId) {
            this.stackId = stackId;
            return this;
        }

        public final void setStackId(String stackId) {
            this.stackId = stackId;
        }

        public final String getStackName() {
            return stackName;
        }

        @Override
        public final Builder stackName(String stackName) {
            this.stackName = stackName;
            return this;
        }

        public final void setStackName(String stackName) {
            this.stackName = stackName;
        }

        public final String getChangeSetId() {
            return changeSetId;
        }

        @Override
        public final Builder changeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
            return this;
        }

        public final void setChangeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Parameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getDeletionTime() {
            return deletionTime;
        }

        @Override
        public final Builder deletionTime(Instant deletionTime) {
            this.deletionTime = deletionTime;
            return this;
        }

        public final void setDeletionTime(Instant deletionTime) {
            this.deletionTime = deletionTime;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        public final RollbackConfiguration.Builder getRollbackConfiguration() {
            return rollbackConfiguration != null ? rollbackConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration;
            return this;
        }

        public final void setRollbackConfiguration(RollbackConfiguration.BuilderImpl rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration != null ? rollbackConfiguration.build() : null;
        }

        public final String getStackStatus() {
            return stackStatus;
        }

        @Override
        public final Builder stackStatus(String stackStatus) {
            this.stackStatus = stackStatus;
            return this;
        }

        @Override
        public final Builder stackStatus(StackStatus stackStatus) {
            this.stackStatus(stackStatus.toString());
            return this;
        }

        public final void setStackStatus(String stackStatus) {
            this.stackStatus = stackStatus;
        }

        public final String getStackStatusReason() {
            return stackStatusReason;
        }

        @Override
        public final Builder stackStatusReason(String stackStatusReason) {
            this.stackStatusReason = stackStatusReason;
            return this;
        }

        public final void setStackStatusReason(String stackStatusReason) {
            this.stackStatusReason = stackStatusReason;
        }

        public final Boolean getDisableRollback() {
            return disableRollback;
        }

        @Override
        public final Builder disableRollback(Boolean disableRollback) {
            this.disableRollback = disableRollback;
            return this;
        }

        public final void setDisableRollback(Boolean disableRollback) {
            this.disableRollback = disableRollback;
        }

        public final Collection<String> getNotificationARNs() {
            return notificationARNs;
        }

        @Override
        public final Builder notificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationARNs(String... notificationARNs) {
            notificationARNs(Arrays.asList(notificationARNs));
            return this;
        }

        public final void setNotificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
        }

        public final Integer getTimeoutInMinutes() {
            return timeoutInMinutes;
        }

        @Override
        public final Builder timeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
            return this;
        }

        public final void setTimeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
        }

        public final Collection<String> getCapabilities() {
            return capabilities;
        }

        @Override
        public final Builder capabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(String... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
        }

        public final Collection<Output.Builder> getOutputs() {
            return outputs != null ? outputs.stream().map(Output::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder outputs(Collection<Output> outputs) {
            this.outputs = OutputsCopier.copy(outputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputs(Output... outputs) {
            outputs(Arrays.asList(outputs));
            return this;
        }

        public final void setOutputs(Collection<Output.BuilderImpl> outputs) {
            this.outputs = OutputsCopier.copyFromBuilder(outputs);
        }

        public final String getRoleARN() {
            return roleARN;
        }

        @Override
        public final Builder roleARN(String roleARN) {
            this.roleARN = roleARN;
            return this;
        }

        public final void setRoleARN(String roleARN) {
            this.roleARN = roleARN;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final Boolean getEnableTerminationProtection() {
            return enableTerminationProtection;
        }

        @Override
        public final Builder enableTerminationProtection(Boolean enableTerminationProtection) {
            this.enableTerminationProtection = enableTerminationProtection;
            return this;
        }

        public final void setEnableTerminationProtection(Boolean enableTerminationProtection) {
            this.enableTerminationProtection = enableTerminationProtection;
        }

        public final String getParentId() {
            return parentId;
        }

        @Override
        public final Builder parentId(String parentId) {
            this.parentId = parentId;
            return this;
        }

        public final void setParentId(String parentId) {
            this.parentId = parentId;
        }

        public final String getRootId() {
            return rootId;
        }

        @Override
        public final Builder rootId(String rootId) {
            this.rootId = rootId;
            return this;
        }

        public final void setRootId(String rootId) {
            this.rootId = rootId;
        }

        @Override
        public Stack build() {
            return new Stack(this);
        }
    }
}
