/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AwsRequestOverrideConfig;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public class UpdateStackSetRequest extends CloudFormationRequest implements
        ToCopyableBuilder<UpdateStackSetRequest.Builder, UpdateStackSetRequest> {
    private final String stackSetName;

    private final String description;

    private final String templateBody;

    private final String templateURL;

    private final Boolean usePreviousTemplate;

    private final List<Parameter> parameters;

    private final List<String> capabilities;

    private final List<Tag> tags;

    private final StackSetOperationPreferences operationPreferences;

    private final String operationId;

    private UpdateStackSetRequest(BuilderImpl builder) {
        super(builder);
        this.stackSetName = builder.stackSetName;
        this.description = builder.description;
        this.templateBody = builder.templateBody;
        this.templateURL = builder.templateURL;
        this.usePreviousTemplate = builder.usePreviousTemplate;
        this.parameters = builder.parameters;
        this.capabilities = builder.capabilities;
        this.tags = builder.tags;
        this.operationPreferences = builder.operationPreferences;
        this.operationId = builder.operationId;
    }

    /**
     * <p>
     * The name or unique ID of the stack set that you want to update.
     * </p>
     * 
     * @return The name or unique ID of the stack set that you want to update.
     */
    public String stackSetName() {
        return stackSetName;
    }

    /**
     * <p>
     * A brief description of updates that you are making.
     * </p>
     * 
     * @return A brief description of updates that you are making.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The structure that contains the template body, with a minimum length of 1 byte and a maximum length of 51,200
     * bytes. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template Anatomy</a>
     * in the AWS CloudFormation User Guide.
     * </p>
     * <p>
     * Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
     * <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
     * </p>
     * 
     * @return The structure that contains the template body, with a minimum length of 1 byte and a maximum length of
     *         51,200 bytes. For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template
     *         Anatomy</a> in the AWS CloudFormation User Guide.</p>
     *         <p>
     *         Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
     *         <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
     */
    public String templateBody() {
        return templateBody;
    }

    /**
     * <p>
     * The location of the file that contains the template body. The URL must point to a template (maximum size: 460,800
     * bytes) that is located in an Amazon S3 bucket. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template Anatomy</a>
     * in the AWS CloudFormation User Guide.
     * </p>
     * <p>
     * Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
     * <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
     * </p>
     * 
     * @return The location of the file that contains the template body. The URL must point to a template (maximum size:
     *         460,800 bytes) that is located in an Amazon S3 bucket. For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template
     *         Anatomy</a> in the AWS CloudFormation User Guide.</p>
     *         <p>
     *         Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
     *         <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
     */
    public String templateURL() {
        return templateURL;
    }

    /**
     * <p>
     * Use the existing template that's associated with the stack set that you're updating.
     * </p>
     * <p>
     * Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
     * <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
     * </p>
     * 
     * @return Use the existing template that's associated with the stack set that you're updating.</p>
     *         <p>
     *         Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
     *         <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
     */
    public Boolean usePreviousTemplate() {
        return usePreviousTemplate;
    }

    /**
     * <p>
     * A list of input parameters for the stack set template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of input parameters for the stack set template.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some stack set
     * templates might include resources that can affect permissions in your AWS account—for example, by creating new
     * AWS Identity and Access Management (IAM) users. For those stack sets, you must explicitly acknowledge their
     * capabilities by specifying this parameter.
     * </p>
     * <p>
     * The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you to specify
     * this parameter:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AWS::IAM::AccessKey
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::Group
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::InstanceProfile
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::Policy
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::Role
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::User
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::UserToGroupAddition
     * </p>
     * </li>
     * </ul>
     * <p>
     * If your stack template contains these resources, we recommend that you review all permissions that are associated
     * with them and edit their permissions if necessary.
     * </p>
     * <p>
     * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names, you
     * must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns an
     * <code>InsufficientCapabilities</code> error.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some
     *         stack set templates might include resources that can affect permissions in your AWS account—for example,
     *         by creating new AWS Identity and Access Management (IAM) users. For those stack sets, you must explicitly
     *         acknowledge their capabilities by specifying this parameter.</p>
     *         <p>
     *         The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you to
     *         specify this parameter:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS::IAM::AccessKey
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::Group
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::InstanceProfile
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::Policy
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::Role
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::User
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::UserToGroupAddition
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If your stack template contains these resources, we recommend that you review all permissions that are
     *         associated with them and edit their permissions if necessary.
     *         </p>
     *         <p>
     *         If you have IAM resources, you can specify either capability. If you have IAM resources with custom
     *         names, you must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns an
     *         <code>InsufficientCapabilities</code> error.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     */
    public List<Capability> capabilities() {
        return TypeConverter.convert(capabilities, Capability::fromValue);
    }

    /**
     * <p>
     * A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some stack set
     * templates might include resources that can affect permissions in your AWS account—for example, by creating new
     * AWS Identity and Access Management (IAM) users. For those stack sets, you must explicitly acknowledge their
     * capabilities by specifying this parameter.
     * </p>
     * <p>
     * The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you to specify
     * this parameter:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AWS::IAM::AccessKey
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::Group
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::InstanceProfile
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::Policy
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::Role
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::User
     * </p>
     * </li>
     * <li>
     * <p>
     * AWS::IAM::UserToGroupAddition
     * </p>
     * </li>
     * </ul>
     * <p>
     * If your stack template contains these resources, we recommend that you review all permissions that are associated
     * with them and edit their permissions if necessary.
     * </p>
     * <p>
     * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names, you
     * must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns an
     * <code>InsufficientCapabilities</code> error.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some
     *         stack set templates might include resources that can affect permissions in your AWS account—for example,
     *         by creating new AWS Identity and Access Management (IAM) users. For those stack sets, you must explicitly
     *         acknowledge their capabilities by specifying this parameter.</p>
     *         <p>
     *         The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you to
     *         specify this parameter:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         AWS::IAM::AccessKey
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::Group
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::InstanceProfile
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::Policy
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::Role
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::User
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AWS::IAM::UserToGroupAddition
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If your stack template contains these resources, we recommend that you review all permissions that are
     *         associated with them and edit their permissions if necessary.
     *         </p>
     *         <p>
     *         If you have IAM resources, you can specify either capability. If you have IAM resources with custom
     *         names, you must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns an
     *         <code>InsufficientCapabilities</code> error.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     */
    public List<String> capabilitiesStrings() {
        return capabilities;
    }

    /**
     * <p>
     * The key-value pairs to associate with this stack set and the stacks created from it. AWS CloudFormation also
     * propagates these tags to supported resources that are created in the stacks. You can specify a maximum number of
     * 50 tags.
     * </p>
     * <p>
     * If you specify tags for this parameter, those tags replace any list of tags that are currently associated with
     * this stack set. This means:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you don't specify this parameter, AWS CloudFormation doesn't modify the stack's tags.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <i>any</i> tags using this parameter, you must specify <i>all</i> the tags that you want
     * associated with this stack set, even tags you've specifed before (for example, when creating the stack set or
     * during a previous update of the stack set.). Any tags that you don't include in the updated list of tags are
     * removed from the stack set, and therefore from the stacks and resources as well.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify an empty value, AWS CloudFormation removes all currently associated tags.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify new tags as part of an <code>UpdateStackSet</code> action, AWS CloudFormation checks to see if you
     * have the required IAM permission to tag resources. If you omit tags that are currently associated with the stack
     * set from the list of tags you specify, AWS CloudFormation assumes that you want to remove those tags from the
     * stack set, and checks to see if you have permission to untag resources. If you don't have the necessary
     * permission(s), the entire <code>UpdateStackSet</code> action fails with an <code>access denied</code> error, and
     * the stack set is not updated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The key-value pairs to associate with this stack set and the stacks created from it. AWS CloudFormation
     *         also propagates these tags to supported resources that are created in the stacks. You can specify a
     *         maximum number of 50 tags.</p>
     *         <p>
     *         If you specify tags for this parameter, those tags replace any list of tags that are currently associated
     *         with this stack set. This means:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you don't specify this parameter, AWS CloudFormation doesn't modify the stack's tags.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify <i>any</i> tags using this parameter, you must specify <i>all</i> the tags that you want
     *         associated with this stack set, even tags you've specifed before (for example, when creating the stack
     *         set or during a previous update of the stack set.). Any tags that you don't include in the updated list
     *         of tags are removed from the stack set, and therefore from the stacks and resources as well.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify an empty value, AWS CloudFormation removes all currently associated tags.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify new tags as part of an <code>UpdateStackSet</code> action, AWS CloudFormation checks to
     *         see if you have the required IAM permission to tag resources. If you omit tags that are currently
     *         associated with the stack set from the list of tags you specify, AWS CloudFormation assumes that you want
     *         to remove those tags from the stack set, and checks to see if you have permission to untag resources. If
     *         you don't have the necessary permission(s), the entire <code>UpdateStackSet</code> action fails with an
     *         <code>access denied</code> error, and the stack set is not updated.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Preferences for how AWS CloudFormation performs this stack set operation.
     * </p>
     * 
     * @return Preferences for how AWS CloudFormation performs this stack set operation.
     */
    public StackSetOperationPreferences operationPreferences() {
        return operationPreferences;
    }

    /**
     * <p>
     * The unique ID for this stack set operation.
     * </p>
     * <p>
     * The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs the stack set
     * operation only once, even if you retry the request multiple times. You might retry stack set operation requests
     * to ensure that AWS CloudFormation successfully received them.
     * </p>
     * <p>
     * If you don't specify an operation ID, AWS CloudFormation generates one automatically.
     * </p>
     * <p>
     * Repeating this stack set operation with a new operation ID retries all stack instances whose status is
     * <code>OUTDATED</code>.
     * </p>
     * 
     * @return The unique ID for this stack set operation. </p>
     *         <p>
     *         The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs the
     *         stack set operation only once, even if you retry the request multiple times. You might retry stack set
     *         operation requests to ensure that AWS CloudFormation successfully received them.
     *         </p>
     *         <p>
     *         If you don't specify an operation ID, AWS CloudFormation generates one automatically.
     *         </p>
     *         <p>
     *         Repeating this stack set operation with a new operation ID retries all stack instances whose status is
     *         <code>OUTDATED</code>.
     */
    public String operationId() {
        return operationId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackSetName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(templateBody());
        hashCode = 31 * hashCode + Objects.hashCode(templateURL());
        hashCode = 31 * hashCode + Objects.hashCode(usePreviousTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(capabilitiesStrings());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(operationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(operationId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateStackSetRequest)) {
            return false;
        }
        UpdateStackSetRequest other = (UpdateStackSetRequest) obj;
        return Objects.equals(stackSetName(), other.stackSetName()) && Objects.equals(description(), other.description())
                && Objects.equals(templateBody(), other.templateBody()) && Objects.equals(templateURL(), other.templateURL())
                && Objects.equals(usePreviousTemplate(), other.usePreviousTemplate())
                && Objects.equals(parameters(), other.parameters())
                && Objects.equals(capabilitiesStrings(), other.capabilitiesStrings()) && Objects.equals(tags(), other.tags())
                && Objects.equals(operationPreferences(), other.operationPreferences())
                && Objects.equals(operationId(), other.operationId());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (stackSetName() != null) {
            sb.append("StackSetName: ").append(stackSetName()).append(",");
        }
        if (description() != null) {
            sb.append("Description: ").append(description()).append(",");
        }
        if (templateBody() != null) {
            sb.append("TemplateBody: ").append(templateBody()).append(",");
        }
        if (templateURL() != null) {
            sb.append("TemplateURL: ").append(templateURL()).append(",");
        }
        if (usePreviousTemplate() != null) {
            sb.append("UsePreviousTemplate: ").append(usePreviousTemplate()).append(",");
        }
        if (parameters() != null) {
            sb.append("Parameters: ").append(parameters()).append(",");
        }
        if (capabilitiesStrings() != null) {
            sb.append("Capabilities: ").append(capabilitiesStrings()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (operationPreferences() != null) {
            sb.append("OperationPreferences: ").append(operationPreferences()).append(",");
        }
        if (operationId() != null) {
            sb.append("OperationId: ").append(operationId()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetName":
            return Optional.of(clazz.cast(stackSetName()));
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "TemplateBody":
            return Optional.of(clazz.cast(templateBody()));
        case "TemplateURL":
            return Optional.of(clazz.cast(templateURL()));
        case "UsePreviousTemplate":
            return Optional.of(clazz.cast(usePreviousTemplate()));
        case "Parameters":
            return Optional.of(clazz.cast(parameters()));
        case "Capabilities":
            return Optional.of(clazz.cast(capabilitiesStrings()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        case "OperationPreferences":
            return Optional.of(clazz.cast(operationPreferences()));
        case "OperationId":
            return Optional.of(clazz.cast(operationId()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CloudFormationRequest.Builder, CopyableBuilder<Builder, UpdateStackSetRequest> {
        /**
         * <p>
         * The name or unique ID of the stack set that you want to update.
         * </p>
         * 
         * @param stackSetName
         *        The name or unique ID of the stack set that you want to update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetName(String stackSetName);

        /**
         * <p>
         * A brief description of updates that you are making.
         * </p>
         * 
         * @param description
         *        A brief description of updates that you are making.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The structure that contains the template body, with a minimum length of 1 byte and a maximum length of 51,200
         * bytes. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template
         * Anatomy</a> in the AWS CloudFormation User Guide.
         * </p>
         * <p>
         * Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
         * <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
         * </p>
         * 
         * @param templateBody
         *        The structure that contains the template body, with a minimum length of 1 byte and a maximum length of
         *        51,200 bytes. For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template
         *        Anatomy</a> in the AWS CloudFormation User Guide.</p>
         *        <p>
         *        Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
         *        <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateBody(String templateBody);

        /**
         * <p>
         * The location of the file that contains the template body. The URL must point to a template (maximum size:
         * 460,800 bytes) that is located in an Amazon S3 bucket. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template
         * Anatomy</a> in the AWS CloudFormation User Guide.
         * </p>
         * <p>
         * Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
         * <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
         * </p>
         * 
         * @param templateURL
         *        The location of the file that contains the template body. The URL must point to a template (maximum
         *        size: 460,800 bytes) that is located in an Amazon S3 bucket. For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html">Template
         *        Anatomy</a> in the AWS CloudFormation User Guide.</p>
         *        <p>
         *        Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
         *        <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateURL(String templateURL);

        /**
         * <p>
         * Use the existing template that's associated with the stack set that you're updating.
         * </p>
         * <p>
         * Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
         * <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
         * </p>
         * 
         * @param usePreviousTemplate
         *        Use the existing template that's associated with the stack set that you're updating.</p>
         *        <p>
         *        Conditional: You must specify only one of the following parameters: <code>TemplateBody</code> or
         *        <code>TemplateURL</code>—or set <code>UsePreviousTemplate</code> to true.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usePreviousTemplate(Boolean usePreviousTemplate);

        /**
         * <p>
         * A list of input parameters for the stack set template.
         * </p>
         * 
         * @param parameters
         *        A list of input parameters for the stack set template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * A list of input parameters for the stack set template.
         * </p>
         * 
         * @param parameters
         *        A list of input parameters for the stack set template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some stack
         * set templates might include resources that can affect permissions in your AWS account—for example, by
         * creating new AWS Identity and Access Management (IAM) users. For those stack sets, you must explicitly
         * acknowledge their capabilities by specifying this parameter.
         * </p>
         * <p>
         * The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you to
         * specify this parameter:
         * </p>
         * <ul>
         * <li>
         * <p>
         * AWS::IAM::AccessKey
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::Group
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::InstanceProfile
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::Policy
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::Role
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::User
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::UserToGroupAddition
         * </p>
         * </li>
         * </ul>
         * <p>
         * If your stack template contains these resources, we recommend that you review all permissions that are
         * associated with them and edit their permissions if necessary.
         * </p>
         * <p>
         * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names,
         * you must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns an
         * <code>InsufficientCapabilities</code> error.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some
         *        stack set templates might include resources that can affect permissions in your AWS account—for
         *        example, by creating new AWS Identity and Access Management (IAM) users. For those stack sets, you
         *        must explicitly acknowledge their capabilities by specifying this parameter.</p>
         *        <p>
         *        The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you
         *        to specify this parameter:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        AWS::IAM::AccessKey
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::Group
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::InstanceProfile
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::Policy
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::Role
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::User
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::UserToGroupAddition
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If your stack template contains these resources, we recommend that you review all permissions that are
         *        associated with them and edit their permissions if necessary.
         *        </p>
         *        <p>
         *        If you have IAM resources, you can specify either capability. If you have IAM resources with custom
         *        names, you must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns
         *        an <code>InsufficientCapabilities</code> error.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<String> capabilities);

        /**
         * <p>
         * A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some stack
         * set templates might include resources that can affect permissions in your AWS account—for example, by
         * creating new AWS Identity and Access Management (IAM) users. For those stack sets, you must explicitly
         * acknowledge their capabilities by specifying this parameter.
         * </p>
         * <p>
         * The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you to
         * specify this parameter:
         * </p>
         * <ul>
         * <li>
         * <p>
         * AWS::IAM::AccessKey
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::Group
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::InstanceProfile
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::Policy
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::Role
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::User
         * </p>
         * </li>
         * <li>
         * <p>
         * AWS::IAM::UserToGroupAddition
         * </p>
         * </li>
         * </ul>
         * <p>
         * If your stack template contains these resources, we recommend that you review all permissions that are
         * associated with them and edit their permissions if necessary.
         * </p>
         * <p>
         * If you have IAM resources, you can specify either capability. If you have IAM resources with custom names,
         * you must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns an
         * <code>InsufficientCapabilities</code> error.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        A list of values that you must specify before AWS CloudFormation can create certain stack sets. Some
         *        stack set templates might include resources that can affect permissions in your AWS account—for
         *        example, by creating new AWS Identity and Access Management (IAM) users. For those stack sets, you
         *        must explicitly acknowledge their capabilities by specifying this parameter.</p>
         *        <p>
         *        The only valid values are CAPABILITY_IAM and CAPABILITY_NAMED_IAM. The following resources require you
         *        to specify this parameter:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        AWS::IAM::AccessKey
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::Group
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::InstanceProfile
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::Policy
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::Role
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::User
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AWS::IAM::UserToGroupAddition
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If your stack template contains these resources, we recommend that you review all permissions that are
         *        associated with them and edit their permissions if necessary.
         *        </p>
         *        <p>
         *        If you have IAM resources, you can specify either capability. If you have IAM resources with custom
         *        names, you must specify CAPABILITY_NAMED_IAM. If you don't specify this parameter, this action returns
         *        an <code>InsufficientCapabilities</code> error.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(String... capabilities);

        /**
         * <p>
         * The key-value pairs to associate with this stack set and the stacks created from it. AWS CloudFormation also
         * propagates these tags to supported resources that are created in the stacks. You can specify a maximum number
         * of 50 tags.
         * </p>
         * <p>
         * If you specify tags for this parameter, those tags replace any list of tags that are currently associated
         * with this stack set. This means:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you don't specify this parameter, AWS CloudFormation doesn't modify the stack's tags.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify <i>any</i> tags using this parameter, you must specify <i>all</i> the tags that you want
         * associated with this stack set, even tags you've specifed before (for example, when creating the stack set or
         * during a previous update of the stack set.). Any tags that you don't include in the updated list of tags are
         * removed from the stack set, and therefore from the stacks and resources as well.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify an empty value, AWS CloudFormation removes all currently associated tags.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify new tags as part of an <code>UpdateStackSet</code> action, AWS CloudFormation checks to see if
         * you have the required IAM permission to tag resources. If you omit tags that are currently associated with
         * the stack set from the list of tags you specify, AWS CloudFormation assumes that you want to remove those
         * tags from the stack set, and checks to see if you have permission to untag resources. If you don't have the
         * necessary permission(s), the entire <code>UpdateStackSet</code> action fails with an
         * <code>access denied</code> error, and the stack set is not updated.
         * </p>
         * 
         * @param tags
         *        The key-value pairs to associate with this stack set and the stacks created from it. AWS
         *        CloudFormation also propagates these tags to supported resources that are created in the stacks. You
         *        can specify a maximum number of 50 tags.</p>
         *        <p>
         *        If you specify tags for this parameter, those tags replace any list of tags that are currently
         *        associated with this stack set. This means:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you don't specify this parameter, AWS CloudFormation doesn't modify the stack's tags.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify <i>any</i> tags using this parameter, you must specify <i>all</i> the tags that you
         *        want associated with this stack set, even tags you've specifed before (for example, when creating the
         *        stack set or during a previous update of the stack set.). Any tags that you don't include in the
         *        updated list of tags are removed from the stack set, and therefore from the stacks and resources as
         *        well.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify an empty value, AWS CloudFormation removes all currently associated tags.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify new tags as part of an <code>UpdateStackSet</code> action, AWS CloudFormation checks to
         *        see if you have the required IAM permission to tag resources. If you omit tags that are currently
         *        associated with the stack set from the list of tags you specify, AWS CloudFormation assumes that you
         *        want to remove those tags from the stack set, and checks to see if you have permission to untag
         *        resources. If you don't have the necessary permission(s), the entire <code>UpdateStackSet</code>
         *        action fails with an <code>access denied</code> error, and the stack set is not updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The key-value pairs to associate with this stack set and the stacks created from it. AWS CloudFormation also
         * propagates these tags to supported resources that are created in the stacks. You can specify a maximum number
         * of 50 tags.
         * </p>
         * <p>
         * If you specify tags for this parameter, those tags replace any list of tags that are currently associated
         * with this stack set. This means:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you don't specify this parameter, AWS CloudFormation doesn't modify the stack's tags.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify <i>any</i> tags using this parameter, you must specify <i>all</i> the tags that you want
         * associated with this stack set, even tags you've specifed before (for example, when creating the stack set or
         * during a previous update of the stack set.). Any tags that you don't include in the updated list of tags are
         * removed from the stack set, and therefore from the stacks and resources as well.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify an empty value, AWS CloudFormation removes all currently associated tags.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify new tags as part of an <code>UpdateStackSet</code> action, AWS CloudFormation checks to see if
         * you have the required IAM permission to tag resources. If you omit tags that are currently associated with
         * the stack set from the list of tags you specify, AWS CloudFormation assumes that you want to remove those
         * tags from the stack set, and checks to see if you have permission to untag resources. If you don't have the
         * necessary permission(s), the entire <code>UpdateStackSet</code> action fails with an
         * <code>access denied</code> error, and the stack set is not updated.
         * </p>
         * 
         * @param tags
         *        The key-value pairs to associate with this stack set and the stacks created from it. AWS
         *        CloudFormation also propagates these tags to supported resources that are created in the stacks. You
         *        can specify a maximum number of 50 tags.</p>
         *        <p>
         *        If you specify tags for this parameter, those tags replace any list of tags that are currently
         *        associated with this stack set. This means:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you don't specify this parameter, AWS CloudFormation doesn't modify the stack's tags.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify <i>any</i> tags using this parameter, you must specify <i>all</i> the tags that you
         *        want associated with this stack set, even tags you've specifed before (for example, when creating the
         *        stack set or during a previous update of the stack set.). Any tags that you don't include in the
         *        updated list of tags are removed from the stack set, and therefore from the stacks and resources as
         *        well.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify an empty value, AWS CloudFormation removes all currently associated tags.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify new tags as part of an <code>UpdateStackSet</code> action, AWS CloudFormation checks to
         *        see if you have the required IAM permission to tag resources. If you omit tags that are currently
         *        associated with the stack set from the list of tags you specify, AWS CloudFormation assumes that you
         *        want to remove those tags from the stack set, and checks to see if you have permission to untag
         *        resources. If you don't have the necessary permission(s), the entire <code>UpdateStackSet</code>
         *        action fails with an <code>access denied</code> error, and the stack set is not updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Preferences for how AWS CloudFormation performs this stack set operation.
         * </p>
         * 
         * @param operationPreferences
         *        Preferences for how AWS CloudFormation performs this stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationPreferences(StackSetOperationPreferences operationPreferences);

        /**
         * <p>
         * Preferences for how AWS CloudFormation performs this stack set operation.
         * </p>
         * This is a convenience that creates an instance of the {@link StackSetOperationPreferences.Builder} avoiding
         * the need to create one manually via {@link StackSetOperationPreferences#builder()}.
         *
         * When the {@link Consumer} completes, {@link StackSetOperationPreferences.Builder#build()} is called
         * immediately and its result is passed to {@link #operationPreferences(StackSetOperationPreferences)}.
         * 
         * @param operationPreferences
         *        a consumer that will call methods on {@link StackSetOperationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operationPreferences(StackSetOperationPreferences)
         */
        default Builder operationPreferences(Consumer<StackSetOperationPreferences.Builder> operationPreferences) {
            return operationPreferences(StackSetOperationPreferences.builder().apply(operationPreferences).build());
        }

        /**
         * <p>
         * The unique ID for this stack set operation.
         * </p>
         * <p>
         * The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs the stack
         * set operation only once, even if you retry the request multiple times. You might retry stack set operation
         * requests to ensure that AWS CloudFormation successfully received them.
         * </p>
         * <p>
         * If you don't specify an operation ID, AWS CloudFormation generates one automatically.
         * </p>
         * <p>
         * Repeating this stack set operation with a new operation ID retries all stack instances whose status is
         * <code>OUTDATED</code>.
         * </p>
         * 
         * @param operationId
         *        The unique ID for this stack set operation. </p>
         *        <p>
         *        The operation ID also functions as an idempotency token, to ensure that AWS CloudFormation performs
         *        the stack set operation only once, even if you retry the request multiple times. You might retry stack
         *        set operation requests to ensure that AWS CloudFormation successfully received them.
         *        </p>
         *        <p>
         *        If you don't specify an operation ID, AWS CloudFormation generates one automatically.
         *        </p>
         *        <p>
         *        Repeating this stack set operation with a new operation ID retries all stack instances whose status is
         *        <code>OUTDATED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationId(String operationId);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);
    }

    static final class BuilderImpl extends CloudFormationRequest.BuilderImpl implements Builder {
        private String stackSetName;

        private String description;

        private String templateBody;

        private String templateURL;

        private Boolean usePreviousTemplate;

        private List<Parameter> parameters;

        private List<String> capabilities;

        private List<Tag> tags;

        private StackSetOperationPreferences operationPreferences;

        private String operationId;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateStackSetRequest model) {
            stackSetName(model.stackSetName);
            description(model.description);
            templateBody(model.templateBody);
            templateURL(model.templateURL);
            usePreviousTemplate(model.usePreviousTemplate);
            parameters(model.parameters);
            capabilities(model.capabilities);
            tags(model.tags);
            operationPreferences(model.operationPreferences);
            operationId(model.operationId);
        }

        public final String getStackSetName() {
            return stackSetName;
        }

        @Override
        public final Builder stackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
            return this;
        }

        public final void setStackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getTemplateBody() {
            return templateBody;
        }

        @Override
        public final Builder templateBody(String templateBody) {
            this.templateBody = templateBody;
            return this;
        }

        public final void setTemplateBody(String templateBody) {
            this.templateBody = templateBody;
        }

        public final String getTemplateURL() {
            return templateURL;
        }

        @Override
        public final Builder templateURL(String templateURL) {
            this.templateURL = templateURL;
            return this;
        }

        public final void setTemplateURL(String templateURL) {
            this.templateURL = templateURL;
        }

        public final Boolean getUsePreviousTemplate() {
            return usePreviousTemplate;
        }

        @Override
        public final Builder usePreviousTemplate(Boolean usePreviousTemplate) {
            this.usePreviousTemplate = usePreviousTemplate;
            return this;
        }

        public final void setUsePreviousTemplate(Boolean usePreviousTemplate) {
            this.usePreviousTemplate = usePreviousTemplate;
        }

        public final Collection<Parameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        public final Collection<String> getCapabilities() {
            return capabilities;
        }

        @Override
        public final Builder capabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(String... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final StackSetOperationPreferences.Builder getOperationPreferences() {
            return operationPreferences != null ? operationPreferences.toBuilder() : null;
        }

        @Override
        public final Builder operationPreferences(StackSetOperationPreferences operationPreferences) {
            this.operationPreferences = operationPreferences;
            return this;
        }

        public final void setOperationPreferences(StackSetOperationPreferences.BuilderImpl operationPreferences) {
            this.operationPreferences = operationPreferences != null ? operationPreferences.build() : null;
        }

        public final String getOperationId() {
            return operationId;
        }

        @Override
        public final Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public final void setOperationId(String operationId) {
            this.operationId = operationId;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public UpdateStackSetRequest build() {
            return new UpdateStackSetRequest(this);
        }
    }
}
