/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Stack data type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Stack implements SdkPojo, Serializable, ToCopyableBuilder<Stack.Builder, Stack> {
    private static final SdkField<String> STACK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::stackId)).setter(setter(Builder::stackId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackId").build()).build();

    private static final SdkField<String> STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::stackName)).setter(setter(Builder::stackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackName").build()).build();

    private static final SdkField<String> CHANGE_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::changeSetId)).setter(setter(Builder::changeSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeSetId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<Parameter>> PARAMETERS_FIELD = SdkField
            .<List<Parameter>> builder(MarshallingType.LIST)
            .getter(getter(Stack::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Parameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Stack::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> DELETION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Stack::deletionTime)).setter(setter(Builder::deletionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Stack::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final SdkField<RollbackConfiguration> ROLLBACK_CONFIGURATION_FIELD = SdkField
            .<RollbackConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(Stack::rollbackConfiguration))
            .setter(setter(Builder::rollbackConfiguration)).constructor(RollbackConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RollbackConfiguration").build())
            .build();

    private static final SdkField<String> STACK_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::stackStatusAsString)).setter(setter(Builder::stackStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackStatus").build()).build();

    private static final SdkField<String> STACK_STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::stackStatusReason)).setter(setter(Builder::stackStatusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackStatusReason").build()).build();

    private static final SdkField<Boolean> DISABLE_ROLLBACK_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(Stack::disableRollback)).setter(setter(Builder::disableRollback))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisableRollback").build()).build();

    private static final SdkField<List<String>> NOTIFICATION_AR_NS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Stack::notificationARNs))
            .setter(setter(Builder::notificationARNs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationARNs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> TIMEOUT_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Stack::timeoutInMinutes)).setter(setter(Builder::timeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutInMinutes").build()).build();

    private static final SdkField<List<String>> CAPABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Stack::capabilitiesAsStrings))
            .setter(setter(Builder::capabilitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Output>> OUTPUTS_FIELD = SdkField
            .<List<Output>> builder(MarshallingType.LIST)
            .getter(getter(Stack::outputs))
            .setter(setter(Builder::outputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Outputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Output> builder(MarshallingType.SDK_POJO)
                                            .constructor(Output::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::roleARN)).setter(setter(Builder::roleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleARN").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(Stack::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ENABLE_TERMINATION_PROTECTION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(Stack::enableTerminationProtection))
            .setter(setter(Builder::enableTerminationProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableTerminationProtection")
                    .build()).build();

    private static final SdkField<String> PARENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::parentId)).setter(setter(Builder::parentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentId").build()).build();

    private static final SdkField<String> ROOT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Stack::rootId)).setter(setter(Builder::rootId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RootId").build()).build();

    private static final SdkField<StackDriftInformation> DRIFT_INFORMATION_FIELD = SdkField
            .<StackDriftInformation> builder(MarshallingType.SDK_POJO).getter(getter(Stack::driftInformation))
            .setter(setter(Builder::driftInformation)).constructor(StackDriftInformation::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriftInformation").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_ID_FIELD,
            STACK_NAME_FIELD, CHANGE_SET_ID_FIELD, DESCRIPTION_FIELD, PARAMETERS_FIELD, CREATION_TIME_FIELD, DELETION_TIME_FIELD,
            LAST_UPDATED_TIME_FIELD, ROLLBACK_CONFIGURATION_FIELD, STACK_STATUS_FIELD, STACK_STATUS_REASON_FIELD,
            DISABLE_ROLLBACK_FIELD, NOTIFICATION_AR_NS_FIELD, TIMEOUT_IN_MINUTES_FIELD, CAPABILITIES_FIELD, OUTPUTS_FIELD,
            ROLE_ARN_FIELD, TAGS_FIELD, ENABLE_TERMINATION_PROTECTION_FIELD, PARENT_ID_FIELD, ROOT_ID_FIELD,
            DRIFT_INFORMATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackId;

    private final String stackName;

    private final String changeSetId;

    private final String description;

    private final List<Parameter> parameters;

    private final Instant creationTime;

    private final Instant deletionTime;

    private final Instant lastUpdatedTime;

    private final RollbackConfiguration rollbackConfiguration;

    private final String stackStatus;

    private final String stackStatusReason;

    private final Boolean disableRollback;

    private final List<String> notificationARNs;

    private final Integer timeoutInMinutes;

    private final List<String> capabilities;

    private final List<Output> outputs;

    private final String roleARN;

    private final List<Tag> tags;

    private final Boolean enableTerminationProtection;

    private final String parentId;

    private final String rootId;

    private final StackDriftInformation driftInformation;

    private Stack(BuilderImpl builder) {
        this.stackId = builder.stackId;
        this.stackName = builder.stackName;
        this.changeSetId = builder.changeSetId;
        this.description = builder.description;
        this.parameters = builder.parameters;
        this.creationTime = builder.creationTime;
        this.deletionTime = builder.deletionTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.rollbackConfiguration = builder.rollbackConfiguration;
        this.stackStatus = builder.stackStatus;
        this.stackStatusReason = builder.stackStatusReason;
        this.disableRollback = builder.disableRollback;
        this.notificationARNs = builder.notificationARNs;
        this.timeoutInMinutes = builder.timeoutInMinutes;
        this.capabilities = builder.capabilities;
        this.outputs = builder.outputs;
        this.roleARN = builder.roleARN;
        this.tags = builder.tags;
        this.enableTerminationProtection = builder.enableTerminationProtection;
        this.parentId = builder.parentId;
        this.rootId = builder.rootId;
        this.driftInformation = builder.driftInformation;
    }

    /**
     * <p>
     * Unique identifier of the stack.
     * </p>
     * 
     * @return Unique identifier of the stack.
     */
    public String stackId() {
        return stackId;
    }

    /**
     * <p>
     * The name associated with the stack.
     * </p>
     * 
     * @return The name associated with the stack.
     */
    public String stackName() {
        return stackName;
    }

    /**
     * <p>
     * The unique ID of the change set.
     * </p>
     * 
     * @return The unique ID of the change set.
     */
    public String changeSetId() {
        return changeSetId;
    }

    /**
     * <p>
     * A user-defined description associated with the stack.
     * </p>
     * 
     * @return A user-defined description associated with the stack.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * A list of <code>Parameter</code> structures.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of <code>Parameter</code> structures.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The time at which the stack was created.
     * </p>
     * 
     * @return The time at which the stack was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time the stack was deleted.
     * </p>
     * 
     * @return The time the stack was deleted.
     */
    public Instant deletionTime() {
        return deletionTime;
    }

    /**
     * <p>
     * The time the stack was last updated. This field will only be returned if the stack has been updated at least
     * once.
     * </p>
     * 
     * @return The time the stack was last updated. This field will only be returned if the stack has been updated at
     *         least once.
     */
    public Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and for
     * the specified monitoring period afterwards.
     * </p>
     * 
     * @return The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
     *         and for the specified monitoring period afterwards.
     */
    public RollbackConfiguration rollbackConfiguration() {
        return rollbackConfiguration;
    }

    /**
     * <p>
     * Current status of the stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackStatus} will
     * return {@link StackStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stackStatusAsString}.
     * </p>
     * 
     * @return Current status of the stack.
     * @see StackStatus
     */
    public StackStatus stackStatus() {
        return StackStatus.fromValue(stackStatus);
    }

    /**
     * <p>
     * Current status of the stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackStatus} will
     * return {@link StackStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stackStatusAsString}.
     * </p>
     * 
     * @return Current status of the stack.
     * @see StackStatus
     */
    public String stackStatusAsString() {
        return stackStatus;
    }

    /**
     * <p>
     * Success/failure message associated with the stack status.
     * </p>
     * 
     * @return Success/failure message associated with the stack status.
     */
    public String stackStatusReason() {
        return stackStatusReason;
    }

    /**
     * <p>
     * Boolean to enable or disable rollback on stack creation failures:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>true</code>: disable rollback
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>false</code>: enable rollback
     * </p>
     * </li>
     * </ul>
     * 
     * @return Boolean to enable or disable rollback on stack creation failures:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>true</code>: disable rollback
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>false</code>: enable rollback
     *         </p>
     *         </li>
     */
    public Boolean disableRollback() {
        return disableRollback;
    }

    /**
     * <p>
     * SNS topic ARNs to which stack related events are published.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return SNS topic ARNs to which stack related events are published.
     */
    public List<String> notificationARNs() {
        return notificationARNs;
    }

    /**
     * <p>
     * The amount of time within which stack creation should complete.
     * </p>
     * 
     * @return The amount of time within which stack creation should complete.
     */
    public Integer timeoutInMinutes() {
        return timeoutInMinutes;
    }

    /**
     * <p>
     * The capabilities allowed in the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The capabilities allowed in the stack.
     */
    public List<Capability> capabilities() {
        return TypeConverter.convert(capabilities, Capability::fromValue);
    }

    /**
     * <p>
     * The capabilities allowed in the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The capabilities allowed in the stack.
     */
    public List<String> capabilitiesAsStrings() {
        return capabilities;
    }

    /**
     * <p>
     * A list of output structures.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of output structures.
     */
    public List<Output> outputs() {
        return outputs;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated with the
     * stack. During a stack operation, AWS CloudFormation uses this role's credentials to make calls on your behalf.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated
     *         with the stack. During a stack operation, AWS CloudFormation uses this role's credentials to make calls
     *         on your behalf.
     */
    public String roleARN() {
        return roleARN;
    }

    /**
     * <p>
     * A list of <code>Tag</code>s that specify information about the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of <code>Tag</code>s that specify information about the stack.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Whether termination protection is enabled for the stack.
     * </p>
     * <p>
     * For <a href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">nested
     * stacks</a>, termination protection is set on the root stack and cannot be changed directly on the nested stack.
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html">Protecting a
     * Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return Whether termination protection is enabled for the stack.</p>
     *         <p>
     *         For <a href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">
     *         nested stacks</a>, termination protection is set on the root stack and cannot be changed directly on the
     *         nested stack. For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html"
     *         >Protecting a Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public Boolean enableTerminationProtection() {
        return enableTerminationProtection;
    }

    /**
     * <p>
     * For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of this
     * stack. For the first level of nested stacks, the root stack is also the parent stack.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working with
     * Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of
     *         this stack. For the first level of nested stacks, the root stack is also the parent stack.</p>
     *         <p>
     *         For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
     *         with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public String parentId() {
        return parentId;
    }

    /**
     * <p>
     * For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level stack to
     * which the nested stack ultimately belongs.
     * </p>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working with
     * Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level stack
     *         to which the nested stack ultimately belongs.</p>
     *         <p>
     *         For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
     *         with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
     */
    public String rootId() {
        return rootId;
    }

    /**
     * <p>
     * Information on whether a stack's actual configuration differs, or has <i>drifted</i>, from it's expected
     * configuration, as defined in the stack template and any values specified as template parameters. For more
     * information, see <a href=
     * "http://docs.aws.amazon.com/http:/docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift.html"
     * >Detecting Unregulated Configuration Changes to Stacks and Resources</a>.
     * </p>
     * 
     * @return Information on whether a stack's actual configuration differs, or has <i>drifted</i>, from it's expected
     *         configuration, as defined in the stack template and any values specified as template parameters. For more
     *         information, see <a href=
     *         "http://docs.aws.amazon.com/http:/docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift.html"
     *         >Detecting Unregulated Configuration Changes to Stacks and Resources</a>.
     */
    public StackDriftInformation driftInformation() {
        return driftInformation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackId());
        hashCode = 31 * hashCode + Objects.hashCode(stackName());
        hashCode = 31 * hashCode + Objects.hashCode(changeSetId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(deletionTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(rollbackConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(stackStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stackStatusReason());
        hashCode = 31 * hashCode + Objects.hashCode(disableRollback());
        hashCode = 31 * hashCode + Objects.hashCode(notificationARNs());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(capabilitiesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(outputs());
        hashCode = 31 * hashCode + Objects.hashCode(roleARN());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(enableTerminationProtection());
        hashCode = 31 * hashCode + Objects.hashCode(parentId());
        hashCode = 31 * hashCode + Objects.hashCode(rootId());
        hashCode = 31 * hashCode + Objects.hashCode(driftInformation());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Stack)) {
            return false;
        }
        Stack other = (Stack) obj;
        return Objects.equals(stackId(), other.stackId()) && Objects.equals(stackName(), other.stackName())
                && Objects.equals(changeSetId(), other.changeSetId()) && Objects.equals(description(), other.description())
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(deletionTime(), other.deletionTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(rollbackConfiguration(), other.rollbackConfiguration())
                && Objects.equals(stackStatusAsString(), other.stackStatusAsString())
                && Objects.equals(stackStatusReason(), other.stackStatusReason())
                && Objects.equals(disableRollback(), other.disableRollback())
                && Objects.equals(notificationARNs(), other.notificationARNs())
                && Objects.equals(timeoutInMinutes(), other.timeoutInMinutes())
                && Objects.equals(capabilitiesAsStrings(), other.capabilitiesAsStrings())
                && Objects.equals(outputs(), other.outputs()) && Objects.equals(roleARN(), other.roleARN())
                && Objects.equals(tags(), other.tags())
                && Objects.equals(enableTerminationProtection(), other.enableTerminationProtection())
                && Objects.equals(parentId(), other.parentId()) && Objects.equals(rootId(), other.rootId())
                && Objects.equals(driftInformation(), other.driftInformation());
    }

    @Override
    public String toString() {
        return ToString.builder("Stack").add("StackId", stackId()).add("StackName", stackName())
                .add("ChangeSetId", changeSetId()).add("Description", description()).add("Parameters", parameters())
                .add("CreationTime", creationTime()).add("DeletionTime", deletionTime())
                .add("LastUpdatedTime", lastUpdatedTime()).add("RollbackConfiguration", rollbackConfiguration())
                .add("StackStatus", stackStatusAsString()).add("StackStatusReason", stackStatusReason())
                .add("DisableRollback", disableRollback()).add("NotificationARNs", notificationARNs())
                .add("TimeoutInMinutes", timeoutInMinutes()).add("Capabilities", capabilitiesAsStrings())
                .add("Outputs", outputs()).add("RoleARN", roleARN()).add("Tags", tags())
                .add("EnableTerminationProtection", enableTerminationProtection()).add("ParentId", parentId())
                .add("RootId", rootId()).add("DriftInformation", driftInformation()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackId":
            return Optional.ofNullable(clazz.cast(stackId()));
        case "StackName":
            return Optional.ofNullable(clazz.cast(stackName()));
        case "ChangeSetId":
            return Optional.ofNullable(clazz.cast(changeSetId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "DeletionTime":
            return Optional.ofNullable(clazz.cast(deletionTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "RollbackConfiguration":
            return Optional.ofNullable(clazz.cast(rollbackConfiguration()));
        case "StackStatus":
            return Optional.ofNullable(clazz.cast(stackStatusAsString()));
        case "StackStatusReason":
            return Optional.ofNullable(clazz.cast(stackStatusReason()));
        case "DisableRollback":
            return Optional.ofNullable(clazz.cast(disableRollback()));
        case "NotificationARNs":
            return Optional.ofNullable(clazz.cast(notificationARNs()));
        case "TimeoutInMinutes":
            return Optional.ofNullable(clazz.cast(timeoutInMinutes()));
        case "Capabilities":
            return Optional.ofNullable(clazz.cast(capabilitiesAsStrings()));
        case "Outputs":
            return Optional.ofNullable(clazz.cast(outputs()));
        case "RoleARN":
            return Optional.ofNullable(clazz.cast(roleARN()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "EnableTerminationProtection":
            return Optional.ofNullable(clazz.cast(enableTerminationProtection()));
        case "ParentId":
            return Optional.ofNullable(clazz.cast(parentId()));
        case "RootId":
            return Optional.ofNullable(clazz.cast(rootId()));
        case "DriftInformation":
            return Optional.ofNullable(clazz.cast(driftInformation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Stack, T> g) {
        return obj -> g.apply((Stack) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Stack> {
        /**
         * <p>
         * Unique identifier of the stack.
         * </p>
         * 
         * @param stackId
         *        Unique identifier of the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackId(String stackId);

        /**
         * <p>
         * The name associated with the stack.
         * </p>
         * 
         * @param stackName
         *        The name associated with the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackName(String stackName);

        /**
         * <p>
         * The unique ID of the change set.
         * </p>
         * 
         * @param changeSetId
         *        The unique ID of the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeSetId(String changeSetId);

        /**
         * <p>
         * A user-defined description associated with the stack.
         * </p>
         * 
         * @param description
         *        A user-defined description associated with the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of <code>Parameter</code> structures.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * A list of <code>Parameter</code> structures.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * A list of <code>Parameter</code> structures.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parameter>.Builder} avoiding the need to
         * create one manually via {@link List<Parameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parameter>.Builder#build()} is called immediately and its
         * result is passed to {@link #parameters(List<Parameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<Parameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<Parameter>)
         */
        Builder parameters(Consumer<Parameter.Builder>... parameters);

        /**
         * <p>
         * The time at which the stack was created.
         * </p>
         * 
         * @param creationTime
         *        The time at which the stack was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time the stack was deleted.
         * </p>
         * 
         * @param deletionTime
         *        The time the stack was deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionTime(Instant deletionTime);

        /**
         * <p>
         * The time the stack was last updated. This field will only be returned if the stack has been updated at least
         * once.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The time the stack was last updated. This field will only be returned if the stack has been updated at
         *        least once.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * 
         * @param rollbackConfiguration
         *        The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
         *        and for the specified monitoring period afterwards.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * This is a convenience that creates an instance of the {@link RollbackConfiguration.Builder} avoiding the need
         * to create one manually via {@link RollbackConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link RollbackConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #rollbackConfiguration(RollbackConfiguration)}.
         * 
         * @param rollbackConfiguration
         *        a consumer that will call methods on {@link RollbackConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rollbackConfiguration(RollbackConfiguration)
         */
        default Builder rollbackConfiguration(Consumer<RollbackConfiguration.Builder> rollbackConfiguration) {
            return rollbackConfiguration(RollbackConfiguration.builder().applyMutation(rollbackConfiguration).build());
        }

        /**
         * <p>
         * Current status of the stack.
         * </p>
         * 
         * @param stackStatus
         *        Current status of the stack.
         * @see StackStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackStatus
         */
        Builder stackStatus(String stackStatus);

        /**
         * <p>
         * Current status of the stack.
         * </p>
         * 
         * @param stackStatus
         *        Current status of the stack.
         * @see StackStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackStatus
         */
        Builder stackStatus(StackStatus stackStatus);

        /**
         * <p>
         * Success/failure message associated with the stack status.
         * </p>
         * 
         * @param stackStatusReason
         *        Success/failure message associated with the stack status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackStatusReason(String stackStatusReason);

        /**
         * <p>
         * Boolean to enable or disable rollback on stack creation failures:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>true</code>: disable rollback
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>false</code>: enable rollback
         * </p>
         * </li>
         * </ul>
         * 
         * @param disableRollback
         *        Boolean to enable or disable rollback on stack creation failures:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>true</code>: disable rollback
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>false</code>: enable rollback
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disableRollback(Boolean disableRollback);

        /**
         * <p>
         * SNS topic ARNs to which stack related events are published.
         * </p>
         * 
         * @param notificationARNs
         *        SNS topic ARNs to which stack related events are published.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(Collection<String> notificationARNs);

        /**
         * <p>
         * SNS topic ARNs to which stack related events are published.
         * </p>
         * 
         * @param notificationARNs
         *        SNS topic ARNs to which stack related events are published.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(String... notificationARNs);

        /**
         * <p>
         * The amount of time within which stack creation should complete.
         * </p>
         * 
         * @param timeoutInMinutes
         *        The amount of time within which stack creation should complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInMinutes(Integer timeoutInMinutes);

        /**
         * <p>
         * The capabilities allowed in the stack.
         * </p>
         * 
         * @param capabilities
         *        The capabilities allowed in the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(Collection<String> capabilities);

        /**
         * <p>
         * The capabilities allowed in the stack.
         * </p>
         * 
         * @param capabilities
         *        The capabilities allowed in the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(String... capabilities);

        /**
         * <p>
         * The capabilities allowed in the stack.
         * </p>
         * 
         * @param capabilities
         *        The capabilities allowed in the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<Capability> capabilities);

        /**
         * <p>
         * The capabilities allowed in the stack.
         * </p>
         * 
         * @param capabilities
         *        The capabilities allowed in the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Capability... capabilities);

        /**
         * <p>
         * A list of output structures.
         * </p>
         * 
         * @param outputs
         *        A list of output structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Collection<Output> outputs);

        /**
         * <p>
         * A list of output structures.
         * </p>
         * 
         * @param outputs
         *        A list of output structures.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputs(Output... outputs);

        /**
         * <p>
         * A list of output structures.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Output>.Builder} avoiding the need to
         * create one manually via {@link List<Output>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Output>.Builder#build()} is called immediately and its
         * result is passed to {@link #outputs(List<Output>)}.
         * 
         * @param outputs
         *        a consumer that will call methods on {@link List<Output>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputs(List<Output>)
         */
        Builder outputs(Consumer<Output.Builder>... outputs);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated with
         * the stack. During a stack operation, AWS CloudFormation uses this role's credentials to make calls on your
         * behalf.
         * </p>
         * 
         * @param roleARN
         *        The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that is associated
         *        with the stack. During a stack operation, AWS CloudFormation uses this role's credentials to make
         *        calls on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleARN(String roleARN);

        /**
         * <p>
         * A list of <code>Tag</code>s that specify information about the stack.
         * </p>
         * 
         * @param tags
         *        A list of <code>Tag</code>s that specify information about the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of <code>Tag</code>s that specify information about the stack.
         * </p>
         * 
         * @param tags
         *        A list of <code>Tag</code>s that specify information about the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of <code>Tag</code>s that specify information about the stack.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Whether termination protection is enabled for the stack.
         * </p>
         * <p>
         * For <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">nested
         * stacks</a>, termination protection is set on the root stack and cannot be changed directly on the nested
         * stack. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html">Protecting
         * a Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param enableTerminationProtection
         *        Whether termination protection is enabled for the stack.</p>
         *        <p>
         *        For <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html"
         *        >nested stacks</a>, termination protection is set on the root stack and cannot be changed directly on
         *        the nested stack. For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-protect-stacks.html"
         *        >Protecting a Stack From Being Deleted</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableTerminationProtection(Boolean enableTerminationProtection);

        /**
         * <p>
         * For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of this
         * stack. For the first level of nested stacks, the root stack is also the parent stack.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
         * with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param parentId
         *        For nested stacks--stacks created as resources for another stack--the stack ID of the direct parent of
         *        this stack. For the first level of nested stacks, the root stack is also the parent stack.</p>
         *        <p>
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html"
         *        >Working with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentId(String parentId);

        /**
         * <p>
         * For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level stack to
         * which the nested stack ultimately belongs.
         * </p>
         * <p>
         * For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html">Working
         * with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param rootId
         *        For nested stacks--stacks created as resources for another stack--the stack ID of the the top-level
         *        stack to which the nested stack ultimately belongs.</p>
         *        <p>
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-nested-stacks.html"
         *        >Working with Nested Stacks</a> in the <i>AWS CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rootId(String rootId);

        /**
         * <p>
         * Information on whether a stack's actual configuration differs, or has <i>drifted</i>, from it's expected
         * configuration, as defined in the stack template and any values specified as template parameters. For more
         * information, see <a href=
         * "http://docs.aws.amazon.com/http:/docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift.html"
         * >Detecting Unregulated Configuration Changes to Stacks and Resources</a>.
         * </p>
         * 
         * @param driftInformation
         *        Information on whether a stack's actual configuration differs, or has <i>drifted</i>, from it's
         *        expected configuration, as defined in the stack template and any values specified as template
         *        parameters. For more information, see <a href=
         *        "http://docs.aws.amazon.com/http:/docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift.html"
         *        >Detecting Unregulated Configuration Changes to Stacks and Resources</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder driftInformation(StackDriftInformation driftInformation);

        /**
         * <p>
         * Information on whether a stack's actual configuration differs, or has <i>drifted</i>, from it's expected
         * configuration, as defined in the stack template and any values specified as template parameters. For more
         * information, see <a href=
         * "http://docs.aws.amazon.com/http:/docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift.html"
         * >Detecting Unregulated Configuration Changes to Stacks and Resources</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link StackDriftInformation.Builder} avoiding the need
         * to create one manually via {@link StackDriftInformation#builder()}.
         *
         * When the {@link Consumer} completes, {@link StackDriftInformation.Builder#build()} is called immediately and
         * its result is passed to {@link #driftInformation(StackDriftInformation)}.
         * 
         * @param driftInformation
         *        a consumer that will call methods on {@link StackDriftInformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #driftInformation(StackDriftInformation)
         */
        default Builder driftInformation(Consumer<StackDriftInformation.Builder> driftInformation) {
            return driftInformation(StackDriftInformation.builder().applyMutation(driftInformation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String stackId;

        private String stackName;

        private String changeSetId;

        private String description;

        private List<Parameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private Instant deletionTime;

        private Instant lastUpdatedTime;

        private RollbackConfiguration rollbackConfiguration;

        private String stackStatus;

        private String stackStatusReason;

        private Boolean disableRollback;

        private List<String> notificationARNs = DefaultSdkAutoConstructList.getInstance();

        private Integer timeoutInMinutes;

        private List<String> capabilities = DefaultSdkAutoConstructList.getInstance();

        private List<Output> outputs = DefaultSdkAutoConstructList.getInstance();

        private String roleARN;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Boolean enableTerminationProtection;

        private String parentId;

        private String rootId;

        private StackDriftInformation driftInformation;

        private BuilderImpl() {
        }

        private BuilderImpl(Stack model) {
            stackId(model.stackId);
            stackName(model.stackName);
            changeSetId(model.changeSetId);
            description(model.description);
            parameters(model.parameters);
            creationTime(model.creationTime);
            deletionTime(model.deletionTime);
            lastUpdatedTime(model.lastUpdatedTime);
            rollbackConfiguration(model.rollbackConfiguration);
            stackStatus(model.stackStatus);
            stackStatusReason(model.stackStatusReason);
            disableRollback(model.disableRollback);
            notificationARNs(model.notificationARNs);
            timeoutInMinutes(model.timeoutInMinutes);
            capabilitiesWithStrings(model.capabilities);
            outputs(model.outputs);
            roleARN(model.roleARN);
            tags(model.tags);
            enableTerminationProtection(model.enableTerminationProtection);
            parentId(model.parentId);
            rootId(model.rootId);
            driftInformation(model.driftInformation);
        }

        public final String getStackId() {
            return stackId;
        }

        @Override
        public final Builder stackId(String stackId) {
            this.stackId = stackId;
            return this;
        }

        public final void setStackId(String stackId) {
            this.stackId = stackId;
        }

        public final String getStackName() {
            return stackName;
        }

        @Override
        public final Builder stackName(String stackName) {
            this.stackName = stackName;
            return this;
        }

        public final void setStackName(String stackName) {
            this.stackName = stackName;
        }

        public final String getChangeSetId() {
            return changeSetId;
        }

        @Override
        public final Builder changeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
            return this;
        }

        public final void setChangeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Parameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<Parameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> Parameter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getDeletionTime() {
            return deletionTime;
        }

        @Override
        public final Builder deletionTime(Instant deletionTime) {
            this.deletionTime = deletionTime;
            return this;
        }

        public final void setDeletionTime(Instant deletionTime) {
            this.deletionTime = deletionTime;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        public final RollbackConfiguration.Builder getRollbackConfiguration() {
            return rollbackConfiguration != null ? rollbackConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration;
            return this;
        }

        public final void setRollbackConfiguration(RollbackConfiguration.BuilderImpl rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration != null ? rollbackConfiguration.build() : null;
        }

        public final String getStackStatus() {
            return stackStatus;
        }

        @Override
        public final Builder stackStatus(String stackStatus) {
            this.stackStatus = stackStatus;
            return this;
        }

        @Override
        public final Builder stackStatus(StackStatus stackStatus) {
            this.stackStatus(stackStatus.toString());
            return this;
        }

        public final void setStackStatus(String stackStatus) {
            this.stackStatus = stackStatus;
        }

        public final String getStackStatusReason() {
            return stackStatusReason;
        }

        @Override
        public final Builder stackStatusReason(String stackStatusReason) {
            this.stackStatusReason = stackStatusReason;
            return this;
        }

        public final void setStackStatusReason(String stackStatusReason) {
            this.stackStatusReason = stackStatusReason;
        }

        public final Boolean getDisableRollback() {
            return disableRollback;
        }

        @Override
        public final Builder disableRollback(Boolean disableRollback) {
            this.disableRollback = disableRollback;
            return this;
        }

        public final void setDisableRollback(Boolean disableRollback) {
            this.disableRollback = disableRollback;
        }

        public final Collection<String> getNotificationARNs() {
            return notificationARNs;
        }

        @Override
        public final Builder notificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationARNs(String... notificationARNs) {
            notificationARNs(Arrays.asList(notificationARNs));
            return this;
        }

        public final void setNotificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
        }

        public final Integer getTimeoutInMinutes() {
            return timeoutInMinutes;
        }

        @Override
        public final Builder timeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
            return this;
        }

        public final void setTimeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
        }

        public final Collection<String> getCapabilities() {
            return capabilities;
        }

        @Override
        public final Builder capabilitiesWithStrings(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilitiesWithStrings(String... capabilities) {
            capabilitiesWithStrings(Arrays.asList(capabilities));
            return this;
        }

        @Override
        public final Builder capabilities(Collection<Capability> capabilities) {
            this.capabilities = CapabilitiesCopier.copyEnumToString(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(Capability... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
        }

        public final Collection<Output.Builder> getOutputs() {
            return outputs != null ? outputs.stream().map(Output::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder outputs(Collection<Output> outputs) {
            this.outputs = OutputsCopier.copy(outputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputs(Output... outputs) {
            outputs(Arrays.asList(outputs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder outputs(Consumer<Output.Builder>... outputs) {
            outputs(Stream.of(outputs).map(c -> Output.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setOutputs(Collection<Output.BuilderImpl> outputs) {
            this.outputs = OutputsCopier.copyFromBuilder(outputs);
        }

        public final String getRoleARN() {
            return roleARN;
        }

        @Override
        public final Builder roleARN(String roleARN) {
            this.roleARN = roleARN;
            return this;
        }

        public final void setRoleARN(String roleARN) {
            this.roleARN = roleARN;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final Boolean getEnableTerminationProtection() {
            return enableTerminationProtection;
        }

        @Override
        public final Builder enableTerminationProtection(Boolean enableTerminationProtection) {
            this.enableTerminationProtection = enableTerminationProtection;
            return this;
        }

        public final void setEnableTerminationProtection(Boolean enableTerminationProtection) {
            this.enableTerminationProtection = enableTerminationProtection;
        }

        public final String getParentId() {
            return parentId;
        }

        @Override
        public final Builder parentId(String parentId) {
            this.parentId = parentId;
            return this;
        }

        public final void setParentId(String parentId) {
            this.parentId = parentId;
        }

        public final String getRootId() {
            return rootId;
        }

        @Override
        public final Builder rootId(String rootId) {
            this.rootId = rootId;
            return this;
        }

        public final void setRootId(String rootId) {
            this.rootId = rootId;
        }

        public final StackDriftInformation.Builder getDriftInformation() {
            return driftInformation != null ? driftInformation.toBuilder() : null;
        }

        @Override
        public final Builder driftInformation(StackDriftInformation driftInformation) {
            this.driftInformation = driftInformation;
            return this;
        }

        public final void setDriftInformation(StackDriftInformation.BuilderImpl driftInformation) {
            this.driftInformation = driftInformation != null ? driftInformation.build() : null;
        }

        @Override
        public Stack build() {
            return new Stack(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
