/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input for the <a>SignalResource</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SignalResourceRequest extends CloudFormationRequest implements
        ToCopyableBuilder<SignalResourceRequest.Builder, SignalResourceRequest> {
    private static final SdkField<String> STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignalResourceRequest::stackName)).setter(setter(Builder::stackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackName").build()).build();

    private static final SdkField<String> LOGICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignalResourceRequest::logicalResourceId)).setter(setter(Builder::logicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalResourceId").build()).build();

    private static final SdkField<String> UNIQUE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignalResourceRequest::uniqueId)).setter(setter(Builder::uniqueId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UniqueId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignalResourceRequest::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_NAME_FIELD,
            LOGICAL_RESOURCE_ID_FIELD, UNIQUE_ID_FIELD, STATUS_FIELD));

    private final String stackName;

    private final String logicalResourceId;

    private final String uniqueId;

    private final String status;

    private SignalResourceRequest(BuilderImpl builder) {
        super(builder);
        this.stackName = builder.stackName;
        this.logicalResourceId = builder.logicalResourceId;
        this.uniqueId = builder.uniqueId;
        this.status = builder.status;
    }

    /**
     * <p>
     * The stack name or unique stack ID that includes the resource that you want to signal.
     * </p>
     * 
     * @return The stack name or unique stack ID that includes the resource that you want to signal.
     */
    public String stackName() {
        return stackName;
    }

    /**
     * <p>
     * The logical ID of the resource that you want to signal. The logical ID is the name of the resource that given in
     * the template.
     * </p>
     * 
     * @return The logical ID of the resource that you want to signal. The logical ID is the name of the resource that
     *         given in the template.
     */
    public String logicalResourceId() {
        return logicalResourceId;
    }

    /**
     * <p>
     * A unique ID of the signal. When you signal Amazon EC2 instances or Auto Scaling groups, specify the instance ID
     * that you are signaling as the unique ID. If you send multiple signals to a single resource (such as signaling a
     * wait condition), each signal requires a different unique ID.
     * </p>
     * 
     * @return A unique ID of the signal. When you signal Amazon EC2 instances or Auto Scaling groups, specify the
     *         instance ID that you are signaling as the unique ID. If you send multiple signals to a single resource
     *         (such as signaling a wait condition), each signal requires a different unique ID.
     */
    public String uniqueId() {
        return uniqueId;
    }

    /**
     * <p>
     * The status of the signal, which is either success or failure. A failure signal causes AWS CloudFormation to
     * immediately fail the stack creation or update.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceSignalStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the signal, which is either success or failure. A failure signal causes AWS CloudFormation
     *         to immediately fail the stack creation or update.
     * @see ResourceSignalStatus
     */
    public ResourceSignalStatus status() {
        return ResourceSignalStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the signal, which is either success or failure. A failure signal causes AWS CloudFormation to
     * immediately fail the stack creation or update.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ResourceSignalStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the signal, which is either success or failure. A failure signal causes AWS CloudFormation
     *         to immediately fail the stack creation or update.
     * @see ResourceSignalStatus
     */
    public String statusAsString() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(stackName());
        hashCode = 31 * hashCode + Objects.hashCode(logicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(uniqueId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SignalResourceRequest)) {
            return false;
        }
        SignalResourceRequest other = (SignalResourceRequest) obj;
        return Objects.equals(stackName(), other.stackName()) && Objects.equals(logicalResourceId(), other.logicalResourceId())
                && Objects.equals(uniqueId(), other.uniqueId()) && Objects.equals(statusAsString(), other.statusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SignalResourceRequest").add("StackName", stackName())
                .add("LogicalResourceId", logicalResourceId()).add("UniqueId", uniqueId()).add("Status", statusAsString())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackName":
            return Optional.ofNullable(clazz.cast(stackName()));
        case "LogicalResourceId":
            return Optional.ofNullable(clazz.cast(logicalResourceId()));
        case "UniqueId":
            return Optional.ofNullable(clazz.cast(uniqueId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SignalResourceRequest, T> g) {
        return obj -> g.apply((SignalResourceRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudFormationRequest.Builder, SdkPojo, CopyableBuilder<Builder, SignalResourceRequest> {
        /**
         * <p>
         * The stack name or unique stack ID that includes the resource that you want to signal.
         * </p>
         * 
         * @param stackName
         *        The stack name or unique stack ID that includes the resource that you want to signal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackName(String stackName);

        /**
         * <p>
         * The logical ID of the resource that you want to signal. The logical ID is the name of the resource that given
         * in the template.
         * </p>
         * 
         * @param logicalResourceId
         *        The logical ID of the resource that you want to signal. The logical ID is the name of the resource
         *        that given in the template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceId(String logicalResourceId);

        /**
         * <p>
         * A unique ID of the signal. When you signal Amazon EC2 instances or Auto Scaling groups, specify the instance
         * ID that you are signaling as the unique ID. If you send multiple signals to a single resource (such as
         * signaling a wait condition), each signal requires a different unique ID.
         * </p>
         * 
         * @param uniqueId
         *        A unique ID of the signal. When you signal Amazon EC2 instances or Auto Scaling groups, specify the
         *        instance ID that you are signaling as the unique ID. If you send multiple signals to a single resource
         *        (such as signaling a wait condition), each signal requires a different unique ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uniqueId(String uniqueId);

        /**
         * <p>
         * The status of the signal, which is either success or failure. A failure signal causes AWS CloudFormation to
         * immediately fail the stack creation or update.
         * </p>
         * 
         * @param status
         *        The status of the signal, which is either success or failure. A failure signal causes AWS
         *        CloudFormation to immediately fail the stack creation or update.
         * @see ResourceSignalStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceSignalStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the signal, which is either success or failure. A failure signal causes AWS CloudFormation to
         * immediately fail the stack creation or update.
         * </p>
         * 
         * @param status
         *        The status of the signal, which is either success or failure. A failure signal causes AWS
         *        CloudFormation to immediately fail the stack creation or update.
         * @see ResourceSignalStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceSignalStatus
         */
        Builder status(ResourceSignalStatus status);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudFormationRequest.BuilderImpl implements Builder {
        private String stackName;

        private String logicalResourceId;

        private String uniqueId;

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(SignalResourceRequest model) {
            super(model);
            stackName(model.stackName);
            logicalResourceId(model.logicalResourceId);
            uniqueId(model.uniqueId);
            status(model.status);
        }

        public final String getStackName() {
            return stackName;
        }

        @Override
        public final Builder stackName(String stackName) {
            this.stackName = stackName;
            return this;
        }

        public final void setStackName(String stackName) {
            this.stackName = stackName;
        }

        public final String getLogicalResourceId() {
            return logicalResourceId;
        }

        @Override
        public final Builder logicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
            return this;
        }

        public final void setLogicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
        }

        public final String getUniqueId() {
            return uniqueId;
        }

        @Override
        public final Builder uniqueId(String uniqueId) {
            this.uniqueId = uniqueId;
            return this;
        }

        public final void setUniqueId(String uniqueId) {
            this.uniqueId = uniqueId;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ResourceSignalStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SignalResourceRequest build() {
            return new SignalResourceRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
