/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A structure that contains information about a stack set. A stack set enables you to provision stacks into AWS
 * accounts and across Regions by using a single CloudFormation template. In the stack set, you specify the template to
 * use, as well as any parameters and capabilities that the template requires.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSet implements SdkPojo, Serializable, ToCopyableBuilder<StackSet.Builder, StackSet> {
    private static final SdkField<String> STACK_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackSetName").getter(getter(StackSet::stackSetName)).setter(setter(Builder::stackSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetName").build()).build();

    private static final SdkField<String> STACK_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackSetId").getter(getter(StackSet::stackSetId)).setter(setter(Builder::stackSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(StackSet::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(StackSet::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> TEMPLATE_BODY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateBody").getter(getter(StackSet::templateBody)).setter(setter(Builder::templateBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateBody").build()).build();

    private static final SdkField<List<Parameter>> PARAMETERS_FIELD = SdkField
            .<List<Parameter>> builder(MarshallingType.LIST)
            .memberName("Parameters")
            .getter(getter(StackSet::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Parameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> CAPABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Capabilities")
            .getter(getter(StackSet::capabilitiesAsStrings))
            .setter(setter(Builder::capabilitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(StackSet::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STACK_SET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackSetARN").getter(getter(StackSet::stackSetARN)).setter(setter(Builder::stackSetARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetARN").build()).build();

    private static final SdkField<String> ADMINISTRATION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdministrationRoleARN").getter(getter(StackSet::administrationRoleARN))
            .setter(setter(Builder::administrationRoleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdministrationRoleARN").build())
            .build();

    private static final SdkField<String> EXECUTION_ROLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRoleName").getter(getter(StackSet::executionRoleName))
            .setter(setter(Builder::executionRoleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRoleName").build()).build();

    private static final SdkField<StackSetDriftDetectionDetails> STACK_SET_DRIFT_DETECTION_DETAILS_FIELD = SdkField
            .<StackSetDriftDetectionDetails> builder(MarshallingType.SDK_POJO)
            .memberName("StackSetDriftDetectionDetails")
            .getter(getter(StackSet::stackSetDriftDetectionDetails))
            .setter(setter(Builder::stackSetDriftDetectionDetails))
            .constructor(StackSetDriftDetectionDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetDriftDetectionDetails")
                    .build()).build();

    private static final SdkField<AutoDeployment> AUTO_DEPLOYMENT_FIELD = SdkField
            .<AutoDeployment> builder(MarshallingType.SDK_POJO).memberName("AutoDeployment")
            .getter(getter(StackSet::autoDeployment)).setter(setter(Builder::autoDeployment))
            .constructor(AutoDeployment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoDeployment").build()).build();

    private static final SdkField<String> PERMISSION_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PermissionModel").getter(getter(StackSet::permissionModelAsString))
            .setter(setter(Builder::permissionModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionModel").build()).build();

    private static final SdkField<List<String>> ORGANIZATIONAL_UNIT_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("OrganizationalUnitIds")
            .getter(getter(StackSet::organizationalUnitIds))
            .setter(setter(Builder::organizationalUnitIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrganizationalUnitIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(STACK_SET_NAME_FIELD, STACK_SET_ID_FIELD, DESCRIPTION_FIELD, STATUS_FIELD, TEMPLATE_BODY_FIELD,
                    PARAMETERS_FIELD, CAPABILITIES_FIELD, TAGS_FIELD, STACK_SET_ARN_FIELD, ADMINISTRATION_ROLE_ARN_FIELD,
                    EXECUTION_ROLE_NAME_FIELD, STACK_SET_DRIFT_DETECTION_DETAILS_FIELD, AUTO_DEPLOYMENT_FIELD,
                    PERMISSION_MODEL_FIELD, ORGANIZATIONAL_UNIT_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackSetName;

    private final String stackSetId;

    private final String description;

    private final String status;

    private final String templateBody;

    private final List<Parameter> parameters;

    private final List<String> capabilities;

    private final List<Tag> tags;

    private final String stackSetARN;

    private final String administrationRoleARN;

    private final String executionRoleName;

    private final StackSetDriftDetectionDetails stackSetDriftDetectionDetails;

    private final AutoDeployment autoDeployment;

    private final String permissionModel;

    private final List<String> organizationalUnitIds;

    private StackSet(BuilderImpl builder) {
        this.stackSetName = builder.stackSetName;
        this.stackSetId = builder.stackSetId;
        this.description = builder.description;
        this.status = builder.status;
        this.templateBody = builder.templateBody;
        this.parameters = builder.parameters;
        this.capabilities = builder.capabilities;
        this.tags = builder.tags;
        this.stackSetARN = builder.stackSetARN;
        this.administrationRoleARN = builder.administrationRoleARN;
        this.executionRoleName = builder.executionRoleName;
        this.stackSetDriftDetectionDetails = builder.stackSetDriftDetectionDetails;
        this.autoDeployment = builder.autoDeployment;
        this.permissionModel = builder.permissionModel;
        this.organizationalUnitIds = builder.organizationalUnitIds;
    }

    /**
     * <p>
     * The name that's associated with the stack set.
     * </p>
     * 
     * @return The name that's associated with the stack set.
     */
    public String stackSetName() {
        return stackSetName;
    }

    /**
     * <p>
     * The ID of the stack set.
     * </p>
     * 
     * @return The ID of the stack set.
     */
    public String stackSetId() {
        return stackSetId;
    }

    /**
     * <p>
     * A description of the stack set that you specify when the stack set is created or updated.
     * </p>
     * 
     * @return A description of the stack set that you specify when the stack set is created or updated.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The status of the stack set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack set.
     * @see StackSetStatus
     */
    public StackSetStatus status() {
        return StackSetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the stack set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack set.
     * @see StackSetStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The structure that contains the body of the template that was used to create or update the stack set.
     * </p>
     * 
     * @return The structure that contains the body of the template that was used to create or update the stack set.
     */
    public String templateBody() {
        return templateBody;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of input parameters for a stack set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of input parameters for a stack set.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
     * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
     * users. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCapabilities()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The capabilities that are allowed in the stack set. Some stack set templates might include resources that
     *         can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
     *         Management (IAM) users. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     */
    public List<Capability> capabilities() {
        return CapabilitiesCopier.copyStringToEnum(capabilities);
    }

    /**
     * Returns true if the Capabilities property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasCapabilities() {
        return capabilities != null && !(capabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
     * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
     * users. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCapabilities()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The capabilities that are allowed in the stack set. Some stack set templates might include resources that
     *         can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
     *         Management (IAM) users. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
     *         >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
     */
    public List<String> capabilitiesAsStrings() {
        return capabilities;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of tags that specify information about the stack set. A maximum number of 50 tags can be
     *         specified.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The Amazon Resource Number (ARN) of the stack set.
     * </p>
     * 
     * @return The Amazon Resource Number (ARN) of the stack set.
     */
    public String stackSetARN() {
        return stackSetARN;
    }

    /**
     * <p>
     * The Amazon Resource Number (ARN) of the IAM role used to create or update the stack set.
     * </p>
     * <p>
     * Use customized administrator roles to control which users or groups can manage specific stack sets within the
     * same administrator account. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html">Prerequisites:
     * Granting Permissions for Stack Set Operations</a> in the <i>AWS CloudFormation User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Number (ARN) of the IAM role used to create or update the stack set.</p>
     *         <p>
     *         Use customized administrator roles to control which users or groups can manage specific stack sets within
     *         the same administrator account. For more information, see <a
     *         href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html"
     *         >Prerequisites: Granting Permissions for Stack Set Operations</a> in the <i>AWS CloudFormation User
     *         Guide</i>.
     */
    public String administrationRoleARN() {
        return administrationRoleARN;
    }

    /**
     * <p>
     * The name of the IAM execution role used to create or update the stack set.
     * </p>
     * <p>
     * Use customized execution roles to control which stack resources users and groups can include in their stack sets.
     * </p>
     * 
     * @return The name of the IAM execution role used to create or update the stack set. </p>
     *         <p>
     *         Use customized execution roles to control which stack resources users and groups can include in their
     *         stack sets.
     */
    public String executionRoleName() {
        return executionRoleName;
    }

    /**
     * <p>
     * Detailed information about the drift status of the stack set.
     * </p>
     * <p>
     * For stack sets, contains information about the last <i>completed</i> drift operation performed on the stack set.
     * Information about drift operations currently in progress is not included.
     * </p>
     * 
     * @return Detailed information about the drift status of the stack set.</p>
     *         <p>
     *         For stack sets, contains information about the last <i>completed</i> drift operation performed on the
     *         stack set. Information about drift operations currently in progress is not included.
     */
    public StackSetDriftDetectionDetails stackSetDriftDetectionDetails() {
        return stackSetDriftDetectionDetails;
    }

    /**
     * <p>
     * [<code>Service-managed</code> permissions] Describes whether StackSets automatically deploys to AWS Organizations
     * accounts that are added to a target organization or organizational unit (OU).
     * </p>
     * 
     * @return [<code>Service-managed</code> permissions] Describes whether StackSets automatically deploys to AWS
     *         Organizations accounts that are added to a target organization or organizational unit (OU).
     */
    public AutoDeployment autoDeployment() {
        return autoDeployment;
    }

    /**
     * <p>
     * Describes how the IAM roles required for stack set operations are created.
     * </p>
     * <ul>
     * <li>
     * <p>
     * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
     * deploy to target accounts. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html">Grant
     * Self-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to deploy
     * to accounts managed by AWS Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     * >Grant Service-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionModel}
     * will return {@link PermissionModels#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionModelAsString}.
     * </p>
     * 
     * @return Describes how the IAM roles required for stack set operations are created.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         With <code>self-managed</code> permissions, you must create the administrator and execution roles
     *         required to deploy to target accounts. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
     *         >Grant Self-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
     *         deploy to accounts managed by AWS Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     *         >Grant Service-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     * @see PermissionModels
     */
    public PermissionModels permissionModel() {
        return PermissionModels.fromValue(permissionModel);
    }

    /**
     * <p>
     * Describes how the IAM roles required for stack set operations are created.
     * </p>
     * <ul>
     * <li>
     * <p>
     * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
     * deploy to target accounts. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html">Grant
     * Self-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to deploy
     * to accounts managed by AWS Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     * >Grant Service-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionModel}
     * will return {@link PermissionModels#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionModelAsString}.
     * </p>
     * 
     * @return Describes how the IAM roles required for stack set operations are created.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         With <code>self-managed</code> permissions, you must create the administrator and execution roles
     *         required to deploy to target accounts. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
     *         >Grant Self-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
     *         deploy to accounts managed by AWS Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     *         >Grant Service-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     * @see PermissionModels
     */
    public String permissionModelAsString() {
        return permissionModel;
    }

    /**
     * Returns true if the OrganizationalUnitIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasOrganizationalUnitIds() {
        return organizationalUnitIds != null && !(organizationalUnitIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * [<code>Service-managed</code> permissions] The organization root ID or organizational unit (OU) IDs that you
     * specified for <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeploymentTargets.html"
     * >DeploymentTargets</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOrganizationalUnitIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return [<code>Service-managed</code> permissions] The organization root ID or organizational unit (OU) IDs that
     *         you specified for <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeploymentTargets.html"
     *         >DeploymentTargets</a>.
     */
    public List<String> organizationalUnitIds() {
        return organizationalUnitIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackSetName());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(templateBody());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCapabilities() ? capabilitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stackSetARN());
        hashCode = 31 * hashCode + Objects.hashCode(administrationRoleARN());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleName());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetDriftDetectionDetails());
        hashCode = 31 * hashCode + Objects.hashCode(autoDeployment());
        hashCode = 31 * hashCode + Objects.hashCode(permissionModelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasOrganizationalUnitIds() ? organizationalUnitIds() : null);
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSet)) {
            return false;
        }
        StackSet other = (StackSet) obj;
        return Objects.equals(stackSetName(), other.stackSetName()) && Objects.equals(stackSetId(), other.stackSetId())
                && Objects.equals(description(), other.description()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(templateBody(), other.templateBody()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && hasCapabilities() == other.hasCapabilities()
                && Objects.equals(capabilitiesAsStrings(), other.capabilitiesAsStrings()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(stackSetARN(), other.stackSetARN())
                && Objects.equals(administrationRoleARN(), other.administrationRoleARN())
                && Objects.equals(executionRoleName(), other.executionRoleName())
                && Objects.equals(stackSetDriftDetectionDetails(), other.stackSetDriftDetectionDetails())
                && Objects.equals(autoDeployment(), other.autoDeployment())
                && Objects.equals(permissionModelAsString(), other.permissionModelAsString())
                && hasOrganizationalUnitIds() == other.hasOrganizationalUnitIds()
                && Objects.equals(organizationalUnitIds(), other.organizationalUnitIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("StackSet").add("StackSetName", stackSetName()).add("StackSetId", stackSetId())
                .add("Description", description()).add("Status", statusAsString()).add("TemplateBody", templateBody())
                .add("Parameters", hasParameters() ? parameters() : null)
                .add("Capabilities", hasCapabilities() ? capabilitiesAsStrings() : null).add("Tags", hasTags() ? tags() : null)
                .add("StackSetARN", stackSetARN()).add("AdministrationRoleARN", administrationRoleARN())
                .add("ExecutionRoleName", executionRoleName())
                .add("StackSetDriftDetectionDetails", stackSetDriftDetectionDetails()).add("AutoDeployment", autoDeployment())
                .add("PermissionModel", permissionModelAsString())
                .add("OrganizationalUnitIds", hasOrganizationalUnitIds() ? organizationalUnitIds() : null).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetName":
            return Optional.ofNullable(clazz.cast(stackSetName()));
        case "StackSetId":
            return Optional.ofNullable(clazz.cast(stackSetId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "TemplateBody":
            return Optional.ofNullable(clazz.cast(templateBody()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "Capabilities":
            return Optional.ofNullable(clazz.cast(capabilitiesAsStrings()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "StackSetARN":
            return Optional.ofNullable(clazz.cast(stackSetARN()));
        case "AdministrationRoleARN":
            return Optional.ofNullable(clazz.cast(administrationRoleARN()));
        case "ExecutionRoleName":
            return Optional.ofNullable(clazz.cast(executionRoleName()));
        case "StackSetDriftDetectionDetails":
            return Optional.ofNullable(clazz.cast(stackSetDriftDetectionDetails()));
        case "AutoDeployment":
            return Optional.ofNullable(clazz.cast(autoDeployment()));
        case "PermissionModel":
            return Optional.ofNullable(clazz.cast(permissionModelAsString()));
        case "OrganizationalUnitIds":
            return Optional.ofNullable(clazz.cast(organizationalUnitIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackSet, T> g) {
        return obj -> g.apply((StackSet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackSet> {
        /**
         * <p>
         * The name that's associated with the stack set.
         * </p>
         * 
         * @param stackSetName
         *        The name that's associated with the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetName(String stackSetName);

        /**
         * <p>
         * The ID of the stack set.
         * </p>
         * 
         * @param stackSetId
         *        The ID of the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetId(String stackSetId);

        /**
         * <p>
         * A description of the stack set that you specify when the stack set is created or updated.
         * </p>
         * 
         * @param description
         *        A description of the stack set that you specify when the stack set is created or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The status of the stack set.
         * </p>
         * 
         * @param status
         *        The status of the stack set.
         * @see StackSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the stack set.
         * </p>
         * 
         * @param status
         *        The status of the stack set.
         * @see StackSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetStatus
         */
        Builder status(StackSetStatus status);

        /**
         * <p>
         * The structure that contains the body of the template that was used to create or update the stack set.
         * </p>
         * 
         * @param templateBody
         *        The structure that contains the body of the template that was used to create or update the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateBody(String templateBody);

        /**
         * <p>
         * A list of input parameters for a stack set.
         * </p>
         * 
         * @param parameters
         *        A list of input parameters for a stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * A list of input parameters for a stack set.
         * </p>
         * 
         * @param parameters
         *        A list of input parameters for a stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * A list of input parameters for a stack set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parameter>.Builder} avoiding the need to
         * create one manually via {@link List<Parameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parameter>.Builder#build()} is called immediately and its
         * result is passed to {@link #parameters(List<Parameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<Parameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<Parameter>)
         */
        Builder parameters(Consumer<Parameter.Builder>... parameters);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(Collection<String> capabilities);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(String... capabilities);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<Capability> capabilities);

        /**
         * <p>
         * The capabilities that are allowed in the stack set. Some stack set templates might include resources that can
         * affect permissions in your AWS account—for example, by creating new AWS Identity and Access Management (IAM)
         * users. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         * >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * </p>
         * 
         * @param capabilities
         *        The capabilities that are allowed in the stack set. Some stack set templates might include resources
         *        that can affect permissions in your AWS account—for example, by creating new AWS Identity and Access
         *        Management (IAM) users. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-iam-template.html#capabilities"
         *        >Acknowledging IAM Resources in AWS CloudFormation Templates.</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Capability... capabilities);

        /**
         * <p>
         * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
         * </p>
         * 
         * @param tags
         *        A list of tags that specify information about the stack set. A maximum number of 50 tags can be
         *        specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
         * </p>
         * 
         * @param tags
         *        A list of tags that specify information about the stack set. A maximum number of 50 tags can be
         *        specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of tags that specify information about the stack set. A maximum number of 50 tags can be specified.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The Amazon Resource Number (ARN) of the stack set.
         * </p>
         * 
         * @param stackSetARN
         *        The Amazon Resource Number (ARN) of the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetARN(String stackSetARN);

        /**
         * <p>
         * The Amazon Resource Number (ARN) of the IAM role used to create or update the stack set.
         * </p>
         * <p>
         * Use customized administrator roles to control which users or groups can manage specific stack sets within the
         * same administrator account. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html">Prerequisites:
         * Granting Permissions for Stack Set Operations</a> in the <i>AWS CloudFormation User Guide</i>.
         * </p>
         * 
         * @param administrationRoleARN
         *        The Amazon Resource Number (ARN) of the IAM role used to create or update the stack set.</p>
         *        <p>
         *        Use customized administrator roles to control which users or groups can manage specific stack sets
         *        within the same administrator account. For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html"
         *        >Prerequisites: Granting Permissions for Stack Set Operations</a> in the <i>AWS CloudFormation User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administrationRoleARN(String administrationRoleARN);

        /**
         * <p>
         * The name of the IAM execution role used to create or update the stack set.
         * </p>
         * <p>
         * Use customized execution roles to control which stack resources users and groups can include in their stack
         * sets.
         * </p>
         * 
         * @param executionRoleName
         *        The name of the IAM execution role used to create or update the stack set. </p>
         *        <p>
         *        Use customized execution roles to control which stack resources users and groups can include in their
         *        stack sets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleName(String executionRoleName);

        /**
         * <p>
         * Detailed information about the drift status of the stack set.
         * </p>
         * <p>
         * For stack sets, contains information about the last <i>completed</i> drift operation performed on the stack
         * set. Information about drift operations currently in progress is not included.
         * </p>
         * 
         * @param stackSetDriftDetectionDetails
         *        Detailed information about the drift status of the stack set.</p>
         *        <p>
         *        For stack sets, contains information about the last <i>completed</i> drift operation performed on the
         *        stack set. Information about drift operations currently in progress is not included.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetDriftDetectionDetails(StackSetDriftDetectionDetails stackSetDriftDetectionDetails);

        /**
         * <p>
         * Detailed information about the drift status of the stack set.
         * </p>
         * <p>
         * For stack sets, contains information about the last <i>completed</i> drift operation performed on the stack
         * set. Information about drift operations currently in progress is not included.
         * </p>
         * This is a convenience that creates an instance of the {@link StackSetDriftDetectionDetails.Builder} avoiding
         * the need to create one manually via {@link StackSetDriftDetectionDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link StackSetDriftDetectionDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #stackSetDriftDetectionDetails(StackSetDriftDetectionDetails)}
         * .
         * 
         * @param stackSetDriftDetectionDetails
         *        a consumer that will call methods on {@link StackSetDriftDetectionDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stackSetDriftDetectionDetails(StackSetDriftDetectionDetails)
         */
        default Builder stackSetDriftDetectionDetails(
                Consumer<StackSetDriftDetectionDetails.Builder> stackSetDriftDetectionDetails) {
            return stackSetDriftDetectionDetails(StackSetDriftDetectionDetails.builder()
                    .applyMutation(stackSetDriftDetectionDetails).build());
        }

        /**
         * <p>
         * [<code>Service-managed</code> permissions] Describes whether StackSets automatically deploys to AWS
         * Organizations accounts that are added to a target organization or organizational unit (OU).
         * </p>
         * 
         * @param autoDeployment
         *        [<code>Service-managed</code> permissions] Describes whether StackSets automatically deploys to AWS
         *        Organizations accounts that are added to a target organization or organizational unit (OU).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoDeployment(AutoDeployment autoDeployment);

        /**
         * <p>
         * [<code>Service-managed</code> permissions] Describes whether StackSets automatically deploys to AWS
         * Organizations accounts that are added to a target organization or organizational unit (OU).
         * </p>
         * This is a convenience that creates an instance of the {@link AutoDeployment.Builder} avoiding the need to
         * create one manually via {@link AutoDeployment#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoDeployment.Builder#build()} is called immediately and its
         * result is passed to {@link #autoDeployment(AutoDeployment)}.
         * 
         * @param autoDeployment
         *        a consumer that will call methods on {@link AutoDeployment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoDeployment(AutoDeployment)
         */
        default Builder autoDeployment(Consumer<AutoDeployment.Builder> autoDeployment) {
            return autoDeployment(AutoDeployment.builder().applyMutation(autoDeployment).build());
        }

        /**
         * <p>
         * Describes how the IAM roles required for stack set operations are created.
         * </p>
         * <ul>
         * <li>
         * <p>
         * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
         * deploy to target accounts. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         * >Grant Self-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
         * deploy to accounts managed by AWS Organizations. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html">Grant
         * Service-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permissionModel
         *        Describes how the IAM roles required for stack set operations are created.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        With <code>self-managed</code> permissions, you must create the administrator and execution roles
         *        required to deploy to target accounts. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         *        >Grant Self-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required
         *        to deploy to accounts managed by AWS Organizations. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
         *        >Grant Service-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         * @see PermissionModels
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionModels
         */
        Builder permissionModel(String permissionModel);

        /**
         * <p>
         * Describes how the IAM roles required for stack set operations are created.
         * </p>
         * <ul>
         * <li>
         * <p>
         * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
         * deploy to target accounts. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         * >Grant Self-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
         * deploy to accounts managed by AWS Organizations. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html">Grant
         * Service-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permissionModel
         *        Describes how the IAM roles required for stack set operations are created.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        With <code>self-managed</code> permissions, you must create the administrator and execution roles
         *        required to deploy to target accounts. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         *        >Grant Self-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required
         *        to deploy to accounts managed by AWS Organizations. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
         *        >Grant Service-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         * @see PermissionModels
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionModels
         */
        Builder permissionModel(PermissionModels permissionModel);

        /**
         * <p>
         * [<code>Service-managed</code> permissions] The organization root ID or organizational unit (OU) IDs that you
         * specified for <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeploymentTargets.html"
         * >DeploymentTargets</a>.
         * </p>
         * 
         * @param organizationalUnitIds
         *        [<code>Service-managed</code> permissions] The organization root ID or organizational unit (OU) IDs
         *        that you specified for <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeploymentTargets.html"
         *        >DeploymentTargets</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitIds(Collection<String> organizationalUnitIds);

        /**
         * <p>
         * [<code>Service-managed</code> permissions] The organization root ID or organizational unit (OU) IDs that you
         * specified for <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeploymentTargets.html"
         * >DeploymentTargets</a>.
         * </p>
         * 
         * @param organizationalUnitIds
         *        [<code>Service-managed</code> permissions] The organization root ID or organizational unit (OU) IDs
         *        that you specified for <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeploymentTargets.html"
         *        >DeploymentTargets</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitIds(String... organizationalUnitIds);
    }

    static final class BuilderImpl implements Builder {
        private String stackSetName;

        private String stackSetId;

        private String description;

        private String status;

        private String templateBody;

        private List<Parameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private List<String> capabilities = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String stackSetARN;

        private String administrationRoleARN;

        private String executionRoleName;

        private StackSetDriftDetectionDetails stackSetDriftDetectionDetails;

        private AutoDeployment autoDeployment;

        private String permissionModel;

        private List<String> organizationalUnitIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(StackSet model) {
            stackSetName(model.stackSetName);
            stackSetId(model.stackSetId);
            description(model.description);
            status(model.status);
            templateBody(model.templateBody);
            parameters(model.parameters);
            capabilitiesWithStrings(model.capabilities);
            tags(model.tags);
            stackSetARN(model.stackSetARN);
            administrationRoleARN(model.administrationRoleARN);
            executionRoleName(model.executionRoleName);
            stackSetDriftDetectionDetails(model.stackSetDriftDetectionDetails);
            autoDeployment(model.autoDeployment);
            permissionModel(model.permissionModel);
            organizationalUnitIds(model.organizationalUnitIds);
        }

        public final String getStackSetName() {
            return stackSetName;
        }

        @Override
        public final Builder stackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
            return this;
        }

        public final void setStackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
        }

        public final String getStackSetId() {
            return stackSetId;
        }

        @Override
        public final Builder stackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
            return this;
        }

        public final void setStackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StackSetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getTemplateBody() {
            return templateBody;
        }

        @Override
        public final Builder templateBody(String templateBody) {
            this.templateBody = templateBody;
            return this;
        }

        public final void setTemplateBody(String templateBody) {
            this.templateBody = templateBody;
        }

        public final Collection<Parameter.Builder> getParameters() {
            if (parameters instanceof SdkAutoConstructList) {
                return null;
            }
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<Parameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> Parameter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        public final Collection<String> getCapabilities() {
            if (capabilities instanceof SdkAutoConstructList) {
                return null;
            }
            return capabilities;
        }

        @Override
        public final Builder capabilitiesWithStrings(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilitiesWithStrings(String... capabilities) {
            capabilitiesWithStrings(Arrays.asList(capabilities));
            return this;
        }

        @Override
        public final Builder capabilities(Collection<Capability> capabilities) {
            this.capabilities = CapabilitiesCopier.copyEnumToString(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(Capability... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final String getStackSetARN() {
            return stackSetARN;
        }

        @Override
        public final Builder stackSetARN(String stackSetARN) {
            this.stackSetARN = stackSetARN;
            return this;
        }

        public final void setStackSetARN(String stackSetARN) {
            this.stackSetARN = stackSetARN;
        }

        public final String getAdministrationRoleARN() {
            return administrationRoleARN;
        }

        @Override
        public final Builder administrationRoleARN(String administrationRoleARN) {
            this.administrationRoleARN = administrationRoleARN;
            return this;
        }

        public final void setAdministrationRoleARN(String administrationRoleARN) {
            this.administrationRoleARN = administrationRoleARN;
        }

        public final String getExecutionRoleName() {
            return executionRoleName;
        }

        @Override
        public final Builder executionRoleName(String executionRoleName) {
            this.executionRoleName = executionRoleName;
            return this;
        }

        public final void setExecutionRoleName(String executionRoleName) {
            this.executionRoleName = executionRoleName;
        }

        public final StackSetDriftDetectionDetails.Builder getStackSetDriftDetectionDetails() {
            return stackSetDriftDetectionDetails != null ? stackSetDriftDetectionDetails.toBuilder() : null;
        }

        @Override
        public final Builder stackSetDriftDetectionDetails(StackSetDriftDetectionDetails stackSetDriftDetectionDetails) {
            this.stackSetDriftDetectionDetails = stackSetDriftDetectionDetails;
            return this;
        }

        public final void setStackSetDriftDetectionDetails(StackSetDriftDetectionDetails.BuilderImpl stackSetDriftDetectionDetails) {
            this.stackSetDriftDetectionDetails = stackSetDriftDetectionDetails != null ? stackSetDriftDetectionDetails.build()
                    : null;
        }

        public final AutoDeployment.Builder getAutoDeployment() {
            return autoDeployment != null ? autoDeployment.toBuilder() : null;
        }

        @Override
        public final Builder autoDeployment(AutoDeployment autoDeployment) {
            this.autoDeployment = autoDeployment;
            return this;
        }

        public final void setAutoDeployment(AutoDeployment.BuilderImpl autoDeployment) {
            this.autoDeployment = autoDeployment != null ? autoDeployment.build() : null;
        }

        public final String getPermissionModel() {
            return permissionModel;
        }

        @Override
        public final Builder permissionModel(String permissionModel) {
            this.permissionModel = permissionModel;
            return this;
        }

        @Override
        public final Builder permissionModel(PermissionModels permissionModel) {
            this.permissionModel(permissionModel == null ? null : permissionModel.toString());
            return this;
        }

        public final void setPermissionModel(String permissionModel) {
            this.permissionModel = permissionModel;
        }

        public final Collection<String> getOrganizationalUnitIds() {
            if (organizationalUnitIds instanceof SdkAutoConstructList) {
                return null;
            }
            return organizationalUnitIds;
        }

        @Override
        public final Builder organizationalUnitIds(Collection<String> organizationalUnitIds) {
            this.organizationalUnitIds = OrganizationalUnitIdListCopier.copy(organizationalUnitIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationalUnitIds(String... organizationalUnitIds) {
            organizationalUnitIds(Arrays.asList(organizationalUnitIds));
            return this;
        }

        public final void setOrganizationalUnitIds(Collection<String> organizationalUnitIds) {
            this.organizationalUnitIds = OrganizationalUnitIdListCopier.copy(organizationalUnitIds);
        }

        @Override
        public StackSet build() {
            return new StackSet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
