/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the drift information for a resource that has been checked for drift. This includes actual and expected
 * property values for resources in which AWS CloudFormation has detected drift. Only resource properties explicitly
 * defined in the stack template are checked for drift. For more information, see <a
 * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift.html">Detecting
 * Unregulated Configuration Changes to Stacks and Resources</a>.
 * </p>
 * <p>
 * Resources that do not currently support drift detection cannot be checked. For a list of resources that support drift
 * detection, see <a
 * href="http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift-resource-list.html"
 * >Resources that Support Drift Detection</a>.
 * </p>
 * <p>
 * Use <a>DetectStackResourceDrift</a> to detect drift on individual resources, or <a>DetectStackDrift</a> to detect
 * drift on all resources in a given stack that support drift detection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackResourceDrift implements SdkPojo, Serializable,
        ToCopyableBuilder<StackResourceDrift.Builder, StackResourceDrift> {
    private static final SdkField<String> STACK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackId").getter(getter(StackResourceDrift::stackId)).setter(setter(Builder::stackId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackId").build()).build();

    private static final SdkField<String> LOGICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogicalResourceId").getter(getter(StackResourceDrift::logicalResourceId))
            .setter(setter(Builder::logicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalResourceId").build()).build();

    private static final SdkField<String> PHYSICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhysicalResourceId").getter(getter(StackResourceDrift::physicalResourceId))
            .setter(setter(Builder::physicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhysicalResourceId").build())
            .build();

    private static final SdkField<List<PhysicalResourceIdContextKeyValuePair>> PHYSICAL_RESOURCE_ID_CONTEXT_FIELD = SdkField
            .<List<PhysicalResourceIdContextKeyValuePair>> builder(MarshallingType.LIST)
            .memberName("PhysicalResourceIdContext")
            .getter(getter(StackResourceDrift::physicalResourceIdContext))
            .setter(setter(Builder::physicalResourceIdContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhysicalResourceIdContext").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PhysicalResourceIdContextKeyValuePair> builder(MarshallingType.SDK_POJO)
                                            .constructor(PhysicalResourceIdContextKeyValuePair::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(StackResourceDrift::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> EXPECTED_PROPERTIES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExpectedProperties").getter(getter(StackResourceDrift::expectedProperties))
            .setter(setter(Builder::expectedProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedProperties").build())
            .build();

    private static final SdkField<String> ACTUAL_PROPERTIES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActualProperties").getter(getter(StackResourceDrift::actualProperties))
            .setter(setter(Builder::actualProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActualProperties").build()).build();

    private static final SdkField<List<PropertyDifference>> PROPERTY_DIFFERENCES_FIELD = SdkField
            .<List<PropertyDifference>> builder(MarshallingType.LIST)
            .memberName("PropertyDifferences")
            .getter(getter(StackResourceDrift::propertyDifferences))
            .setter(setter(Builder::propertyDifferences))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PropertyDifferences").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PropertyDifference> builder(MarshallingType.SDK_POJO)
                                            .constructor(PropertyDifference::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STACK_RESOURCE_DRIFT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackResourceDriftStatus").getter(getter(StackResourceDrift::stackResourceDriftStatusAsString))
            .setter(setter(Builder::stackResourceDriftStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackResourceDriftStatus").build())
            .build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(StackResourceDrift::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<ModuleInfo> MODULE_INFO_FIELD = SdkField.<ModuleInfo> builder(MarshallingType.SDK_POJO)
            .memberName("ModuleInfo").getter(getter(StackResourceDrift::moduleInfo)).setter(setter(Builder::moduleInfo))
            .constructor(ModuleInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModuleInfo").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_ID_FIELD,
            LOGICAL_RESOURCE_ID_FIELD, PHYSICAL_RESOURCE_ID_FIELD, PHYSICAL_RESOURCE_ID_CONTEXT_FIELD, RESOURCE_TYPE_FIELD,
            EXPECTED_PROPERTIES_FIELD, ACTUAL_PROPERTIES_FIELD, PROPERTY_DIFFERENCES_FIELD, STACK_RESOURCE_DRIFT_STATUS_FIELD,
            TIMESTAMP_FIELD, MODULE_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackId;

    private final String logicalResourceId;

    private final String physicalResourceId;

    private final List<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext;

    private final String resourceType;

    private final String expectedProperties;

    private final String actualProperties;

    private final List<PropertyDifference> propertyDifferences;

    private final String stackResourceDriftStatus;

    private final Instant timestamp;

    private final ModuleInfo moduleInfo;

    private StackResourceDrift(BuilderImpl builder) {
        this.stackId = builder.stackId;
        this.logicalResourceId = builder.logicalResourceId;
        this.physicalResourceId = builder.physicalResourceId;
        this.physicalResourceIdContext = builder.physicalResourceIdContext;
        this.resourceType = builder.resourceType;
        this.expectedProperties = builder.expectedProperties;
        this.actualProperties = builder.actualProperties;
        this.propertyDifferences = builder.propertyDifferences;
        this.stackResourceDriftStatus = builder.stackResourceDriftStatus;
        this.timestamp = builder.timestamp;
        this.moduleInfo = builder.moduleInfo;
    }

    /**
     * <p>
     * The ID of the stack.
     * </p>
     * 
     * @return The ID of the stack.
     */
    public String stackId() {
        return stackId;
    }

    /**
     * <p>
     * The logical name of the resource specified in the template.
     * </p>
     * 
     * @return The logical name of the resource specified in the template.
     */
    public String logicalResourceId() {
        return logicalResourceId;
    }

    /**
     * <p>
     * The name or unique identifier that corresponds to a physical instance ID of a resource supported by AWS
     * CloudFormation.
     * </p>
     * 
     * @return The name or unique identifier that corresponds to a physical instance ID of a resource supported by AWS
     *         CloudFormation.
     */
    public String physicalResourceId() {
        return physicalResourceId;
    }

    /**
     * Returns true if the PhysicalResourceIdContext property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasPhysicalResourceIdContext() {
        return physicalResourceIdContext != null && !(physicalResourceIdContext instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Context information that enables AWS CloudFormation to uniquely identify a resource. AWS CloudFormation uses
     * context key-value pairs in cases where a resource's logical and physical IDs are not enough to uniquely identify
     * that resource. Each context key-value pair specifies a unique resource that contains the targeted resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPhysicalResourceIdContext()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Context information that enables AWS CloudFormation to uniquely identify a resource. AWS CloudFormation
     *         uses context key-value pairs in cases where a resource's logical and physical IDs are not enough to
     *         uniquely identify that resource. Each context key-value pair specifies a unique resource that contains
     *         the targeted resource.
     */
    public List<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext() {
        return physicalResourceIdContext;
    }

    /**
     * <p>
     * The type of the resource.
     * </p>
     * 
     * @return The type of the resource.
     */
    public String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * A JSON structure containing the expected property values of the stack resource, as defined in the stack template
     * and any values specified as template parameters.
     * </p>
     * <p>
     * For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will not be
     * present.
     * </p>
     * 
     * @return A JSON structure containing the expected property values of the stack resource, as defined in the stack
     *         template and any values specified as template parameters. </p>
     *         <p>
     *         For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will
     *         not be present.
     */
    public String expectedProperties() {
        return expectedProperties;
    }

    /**
     * <p>
     * A JSON structure containing the actual property values of the stack resource.
     * </p>
     * <p>
     * For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will not be
     * present.
     * </p>
     * 
     * @return A JSON structure containing the actual property values of the stack resource.</p>
     *         <p>
     *         For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will
     *         not be present.
     */
    public String actualProperties() {
        return actualProperties;
    }

    /**
     * Returns true if the PropertyDifferences property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasPropertyDifferences() {
        return propertyDifferences != null && !(propertyDifferences instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A collection of the resource properties whose actual values differ from their expected values. These will be
     * present only for resources whose <code>StackResourceDriftStatus</code> is <code>MODIFIED</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPropertyDifferences()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A collection of the resource properties whose actual values differ from their expected values. These will
     *         be present only for resources whose <code>StackResourceDriftStatus</code> is <code>MODIFIED</code>.
     */
    public List<PropertyDifference> propertyDifferences() {
        return propertyDifferences;
    }

    /**
     * <p>
     * Status of the resource's actual configuration compared to its expected configuration
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETED</code>: The resource differs from its expected template configuration because the resource has been
     * deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined in the stack
     * template and any values specified as template parameters).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The resources's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackResourceDriftStatus} will return {@link StackResourceDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #stackResourceDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the resource's actual configuration compared to its expected configuration</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETED</code>: The resource differs from its expected template configuration because the resource
     *         has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined in
     *         the stack template and any values specified as template parameters).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The resources's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
     *         </p>
     *         </li>
     * @see StackResourceDriftStatus
     */
    public StackResourceDriftStatus stackResourceDriftStatus() {
        return StackResourceDriftStatus.fromValue(stackResourceDriftStatus);
    }

    /**
     * <p>
     * Status of the resource's actual configuration compared to its expected configuration
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DELETED</code>: The resource differs from its expected template configuration because the resource has been
     * deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined in the stack
     * template and any values specified as template parameters).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The resources's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #stackResourceDriftStatus} will return {@link StackResourceDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #stackResourceDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the resource's actual configuration compared to its expected configuration</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DELETED</code>: The resource differs from its expected template configuration because the resource
     *         has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined in
     *         the stack template and any values specified as template parameters).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The resources's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
     *         </p>
     *         </li>
     * @see StackResourceDriftStatus
     */
    public String stackResourceDriftStatusAsString() {
        return stackResourceDriftStatus;
    }

    /**
     * <p>
     * Time at which AWS CloudFormation performed drift detection on the stack resource.
     * </p>
     * 
     * @return Time at which AWS CloudFormation performed drift detection on the stack resource.
     */
    public Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * Contains information about the module from which the resource was created, if the resource was created from a
     * module included in the stack template.
     * </p>
     * 
     * @return Contains information about the module from which the resource was created, if the resource was created
     *         from a module included in the stack template.
     */
    public ModuleInfo moduleInfo() {
        return moduleInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackId());
        hashCode = 31 * hashCode + Objects.hashCode(logicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(physicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasPhysicalResourceIdContext() ? physicalResourceIdContext() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(expectedProperties());
        hashCode = 31 * hashCode + Objects.hashCode(actualProperties());
        hashCode = 31 * hashCode + Objects.hashCode(hasPropertyDifferences() ? propertyDifferences() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stackResourceDriftStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(moduleInfo());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackResourceDrift)) {
            return false;
        }
        StackResourceDrift other = (StackResourceDrift) obj;
        return Objects.equals(stackId(), other.stackId()) && Objects.equals(logicalResourceId(), other.logicalResourceId())
                && Objects.equals(physicalResourceId(), other.physicalResourceId())
                && hasPhysicalResourceIdContext() == other.hasPhysicalResourceIdContext()
                && Objects.equals(physicalResourceIdContext(), other.physicalResourceIdContext())
                && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(expectedProperties(), other.expectedProperties())
                && Objects.equals(actualProperties(), other.actualProperties())
                && hasPropertyDifferences() == other.hasPropertyDifferences()
                && Objects.equals(propertyDifferences(), other.propertyDifferences())
                && Objects.equals(stackResourceDriftStatusAsString(), other.stackResourceDriftStatusAsString())
                && Objects.equals(timestamp(), other.timestamp()) && Objects.equals(moduleInfo(), other.moduleInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("StackResourceDrift").add("StackId", stackId()).add("LogicalResourceId", logicalResourceId())
                .add("PhysicalResourceId", physicalResourceId())
                .add("PhysicalResourceIdContext", hasPhysicalResourceIdContext() ? physicalResourceIdContext() : null)
                .add("ResourceType", resourceType()).add("ExpectedProperties", expectedProperties())
                .add("ActualProperties", actualProperties())
                .add("PropertyDifferences", hasPropertyDifferences() ? propertyDifferences() : null)
                .add("StackResourceDriftStatus", stackResourceDriftStatusAsString()).add("Timestamp", timestamp())
                .add("ModuleInfo", moduleInfo()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackId":
            return Optional.ofNullable(clazz.cast(stackId()));
        case "LogicalResourceId":
            return Optional.ofNullable(clazz.cast(logicalResourceId()));
        case "PhysicalResourceId":
            return Optional.ofNullable(clazz.cast(physicalResourceId()));
        case "PhysicalResourceIdContext":
            return Optional.ofNullable(clazz.cast(physicalResourceIdContext()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "ExpectedProperties":
            return Optional.ofNullable(clazz.cast(expectedProperties()));
        case "ActualProperties":
            return Optional.ofNullable(clazz.cast(actualProperties()));
        case "PropertyDifferences":
            return Optional.ofNullable(clazz.cast(propertyDifferences()));
        case "StackResourceDriftStatus":
            return Optional.ofNullable(clazz.cast(stackResourceDriftStatusAsString()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "ModuleInfo":
            return Optional.ofNullable(clazz.cast(moduleInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackResourceDrift, T> g) {
        return obj -> g.apply((StackResourceDrift) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackResourceDrift> {
        /**
         * <p>
         * The ID of the stack.
         * </p>
         * 
         * @param stackId
         *        The ID of the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackId(String stackId);

        /**
         * <p>
         * The logical name of the resource specified in the template.
         * </p>
         * 
         * @param logicalResourceId
         *        The logical name of the resource specified in the template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceId(String logicalResourceId);

        /**
         * <p>
         * The name or unique identifier that corresponds to a physical instance ID of a resource supported by AWS
         * CloudFormation.
         * </p>
         * 
         * @param physicalResourceId
         *        The name or unique identifier that corresponds to a physical instance ID of a resource supported by
         *        AWS CloudFormation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceId(String physicalResourceId);

        /**
         * <p>
         * Context information that enables AWS CloudFormation to uniquely identify a resource. AWS CloudFormation uses
         * context key-value pairs in cases where a resource's logical and physical IDs are not enough to uniquely
         * identify that resource. Each context key-value pair specifies a unique resource that contains the targeted
         * resource.
         * </p>
         * 
         * @param physicalResourceIdContext
         *        Context information that enables AWS CloudFormation to uniquely identify a resource. AWS
         *        CloudFormation uses context key-value pairs in cases where a resource's logical and physical IDs are
         *        not enough to uniquely identify that resource. Each context key-value pair specifies a unique resource
         *        that contains the targeted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceIdContext(Collection<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext);

        /**
         * <p>
         * Context information that enables AWS CloudFormation to uniquely identify a resource. AWS CloudFormation uses
         * context key-value pairs in cases where a resource's logical and physical IDs are not enough to uniquely
         * identify that resource. Each context key-value pair specifies a unique resource that contains the targeted
         * resource.
         * </p>
         * 
         * @param physicalResourceIdContext
         *        Context information that enables AWS CloudFormation to uniquely identify a resource. AWS
         *        CloudFormation uses context key-value pairs in cases where a resource's logical and physical IDs are
         *        not enough to uniquely identify that resource. Each context key-value pair specifies a unique resource
         *        that contains the targeted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceIdContext(PhysicalResourceIdContextKeyValuePair... physicalResourceIdContext);

        /**
         * <p>
         * Context information that enables AWS CloudFormation to uniquely identify a resource. AWS CloudFormation uses
         * context key-value pairs in cases where a resource's logical and physical IDs are not enough to uniquely
         * identify that resource. Each context key-value pair specifies a unique resource that contains the targeted
         * resource.
         * </p>
         * This is a convenience that creates an instance of the {@link List
         * <PhysicalResourceIdContextKeyValuePair>.Builder} avoiding the need to create one manually via {@link List
         * <PhysicalResourceIdContextKeyValuePair>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PhysicalResourceIdContextKeyValuePair>.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #physicalResourceIdContext(List<PhysicalResourceIdContextKeyValuePair>)}.
         * 
         * @param physicalResourceIdContext
         *        a consumer that will call methods on {@link List<PhysicalResourceIdContextKeyValuePair>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #physicalResourceIdContext(List<PhysicalResourceIdContextKeyValuePair>)
         */
        Builder physicalResourceIdContext(Consumer<PhysicalResourceIdContextKeyValuePair.Builder>... physicalResourceIdContext);

        /**
         * <p>
         * The type of the resource.
         * </p>
         * 
         * @param resourceType
         *        The type of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * A JSON structure containing the expected property values of the stack resource, as defined in the stack
         * template and any values specified as template parameters.
         * </p>
         * <p>
         * For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will not be
         * present.
         * </p>
         * 
         * @param expectedProperties
         *        A JSON structure containing the expected property values of the stack resource, as defined in the
         *        stack template and any values specified as template parameters. </p>
         *        <p>
         *        For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will
         *        not be present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedProperties(String expectedProperties);

        /**
         * <p>
         * A JSON structure containing the actual property values of the stack resource.
         * </p>
         * <p>
         * For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will not be
         * present.
         * </p>
         * 
         * @param actualProperties
         *        A JSON structure containing the actual property values of the stack resource.</p>
         *        <p>
         *        For resources whose <code>StackResourceDriftStatus</code> is <code>DELETED</code>, this structure will
         *        not be present.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actualProperties(String actualProperties);

        /**
         * <p>
         * A collection of the resource properties whose actual values differ from their expected values. These will be
         * present only for resources whose <code>StackResourceDriftStatus</code> is <code>MODIFIED</code>.
         * </p>
         * 
         * @param propertyDifferences
         *        A collection of the resource properties whose actual values differ from their expected values. These
         *        will be present only for resources whose <code>StackResourceDriftStatus</code> is
         *        <code>MODIFIED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyDifferences(Collection<PropertyDifference> propertyDifferences);

        /**
         * <p>
         * A collection of the resource properties whose actual values differ from their expected values. These will be
         * present only for resources whose <code>StackResourceDriftStatus</code> is <code>MODIFIED</code>.
         * </p>
         * 
         * @param propertyDifferences
         *        A collection of the resource properties whose actual values differ from their expected values. These
         *        will be present only for resources whose <code>StackResourceDriftStatus</code> is
         *        <code>MODIFIED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder propertyDifferences(PropertyDifference... propertyDifferences);

        /**
         * <p>
         * A collection of the resource properties whose actual values differ from their expected values. These will be
         * present only for resources whose <code>StackResourceDriftStatus</code> is <code>MODIFIED</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PropertyDifference>.Builder} avoiding the
         * need to create one manually via {@link List<PropertyDifference>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PropertyDifference>.Builder#build()} is called immediately
         * and its result is passed to {@link #propertyDifferences(List<PropertyDifference>)}.
         * 
         * @param propertyDifferences
         *        a consumer that will call methods on {@link List<PropertyDifference>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #propertyDifferences(List<PropertyDifference>)
         */
        Builder propertyDifferences(Consumer<PropertyDifference.Builder>... propertyDifferences);

        /**
         * <p>
         * Status of the resource's actual configuration compared to its expected configuration
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETED</code>: The resource differs from its expected template configuration because the resource has
         * been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined in the
         * stack template and any values specified as template parameters).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The resources's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackResourceDriftStatus
         *        Status of the resource's actual configuration compared to its expected configuration</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETED</code>: The resource differs from its expected template configuration because the
         *        resource has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined
         *        in the stack template and any values specified as template parameters).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The resources's actual configuration matches its expected template
         *        configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
         *        </p>
         *        </li>
         * @see StackResourceDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackResourceDriftStatus
         */
        Builder stackResourceDriftStatus(String stackResourceDriftStatus);

        /**
         * <p>
         * Status of the resource's actual configuration compared to its expected configuration
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DELETED</code>: The resource differs from its expected template configuration because the resource has
         * been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined in the
         * stack template and any values specified as template parameters).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The resources's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackResourceDriftStatus
         *        Status of the resource's actual configuration compared to its expected configuration</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DELETED</code>: The resource differs from its expected template configuration because the
         *        resource has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFIED</code>: One or more resource properties differ from their expected values (as defined
         *        in the stack template and any values specified as template parameters).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The resources's actual configuration matches its expected template
         *        configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation does not currently return this value.
         *        </p>
         *        </li>
         * @see StackResourceDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackResourceDriftStatus
         */
        Builder stackResourceDriftStatus(StackResourceDriftStatus stackResourceDriftStatus);

        /**
         * <p>
         * Time at which AWS CloudFormation performed drift detection on the stack resource.
         * </p>
         * 
         * @param timestamp
         *        Time at which AWS CloudFormation performed drift detection on the stack resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * Contains information about the module from which the resource was created, if the resource was created from a
         * module included in the stack template.
         * </p>
         * 
         * @param moduleInfo
         *        Contains information about the module from which the resource was created, if the resource was created
         *        from a module included in the stack template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder moduleInfo(ModuleInfo moduleInfo);

        /**
         * <p>
         * Contains information about the module from which the resource was created, if the resource was created from a
         * module included in the stack template.
         * </p>
         * This is a convenience that creates an instance of the {@link ModuleInfo.Builder} avoiding the need to create
         * one manually via {@link ModuleInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link ModuleInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #moduleInfo(ModuleInfo)}.
         * 
         * @param moduleInfo
         *        a consumer that will call methods on {@link ModuleInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #moduleInfo(ModuleInfo)
         */
        default Builder moduleInfo(Consumer<ModuleInfo.Builder> moduleInfo) {
            return moduleInfo(ModuleInfo.builder().applyMutation(moduleInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String stackId;

        private String logicalResourceId;

        private String physicalResourceId;

        private List<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext = DefaultSdkAutoConstructList.getInstance();

        private String resourceType;

        private String expectedProperties;

        private String actualProperties;

        private List<PropertyDifference> propertyDifferences = DefaultSdkAutoConstructList.getInstance();

        private String stackResourceDriftStatus;

        private Instant timestamp;

        private ModuleInfo moduleInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(StackResourceDrift model) {
            stackId(model.stackId);
            logicalResourceId(model.logicalResourceId);
            physicalResourceId(model.physicalResourceId);
            physicalResourceIdContext(model.physicalResourceIdContext);
            resourceType(model.resourceType);
            expectedProperties(model.expectedProperties);
            actualProperties(model.actualProperties);
            propertyDifferences(model.propertyDifferences);
            stackResourceDriftStatus(model.stackResourceDriftStatus);
            timestamp(model.timestamp);
            moduleInfo(model.moduleInfo);
        }

        public final String getStackId() {
            return stackId;
        }

        @Override
        public final Builder stackId(String stackId) {
            this.stackId = stackId;
            return this;
        }

        public final void setStackId(String stackId) {
            this.stackId = stackId;
        }

        public final String getLogicalResourceId() {
            return logicalResourceId;
        }

        @Override
        public final Builder logicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
            return this;
        }

        public final void setLogicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
        }

        public final String getPhysicalResourceId() {
            return physicalResourceId;
        }

        @Override
        public final Builder physicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
            return this;
        }

        public final void setPhysicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
        }

        public final Collection<PhysicalResourceIdContextKeyValuePair.Builder> getPhysicalResourceIdContext() {
            if (physicalResourceIdContext instanceof SdkAutoConstructList) {
                return null;
            }
            return physicalResourceIdContext != null ? physicalResourceIdContext.stream()
                    .map(PhysicalResourceIdContextKeyValuePair::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder physicalResourceIdContext(Collection<PhysicalResourceIdContextKeyValuePair> physicalResourceIdContext) {
            this.physicalResourceIdContext = PhysicalResourceIdContextCopier.copy(physicalResourceIdContext);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder physicalResourceIdContext(PhysicalResourceIdContextKeyValuePair... physicalResourceIdContext) {
            physicalResourceIdContext(Arrays.asList(physicalResourceIdContext));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder physicalResourceIdContext(
                Consumer<PhysicalResourceIdContextKeyValuePair.Builder>... physicalResourceIdContext) {
            physicalResourceIdContext(Stream.of(physicalResourceIdContext)
                    .map(c -> PhysicalResourceIdContextKeyValuePair.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPhysicalResourceIdContext(
                Collection<PhysicalResourceIdContextKeyValuePair.BuilderImpl> physicalResourceIdContext) {
            this.physicalResourceIdContext = PhysicalResourceIdContextCopier.copyFromBuilder(physicalResourceIdContext);
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final String getExpectedProperties() {
            return expectedProperties;
        }

        @Override
        public final Builder expectedProperties(String expectedProperties) {
            this.expectedProperties = expectedProperties;
            return this;
        }

        public final void setExpectedProperties(String expectedProperties) {
            this.expectedProperties = expectedProperties;
        }

        public final String getActualProperties() {
            return actualProperties;
        }

        @Override
        public final Builder actualProperties(String actualProperties) {
            this.actualProperties = actualProperties;
            return this;
        }

        public final void setActualProperties(String actualProperties) {
            this.actualProperties = actualProperties;
        }

        public final Collection<PropertyDifference.Builder> getPropertyDifferences() {
            if (propertyDifferences instanceof SdkAutoConstructList) {
                return null;
            }
            return propertyDifferences != null ? propertyDifferences.stream().map(PropertyDifference::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder propertyDifferences(Collection<PropertyDifference> propertyDifferences) {
            this.propertyDifferences = PropertyDifferencesCopier.copy(propertyDifferences);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder propertyDifferences(PropertyDifference... propertyDifferences) {
            propertyDifferences(Arrays.asList(propertyDifferences));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder propertyDifferences(Consumer<PropertyDifference.Builder>... propertyDifferences) {
            propertyDifferences(Stream.of(propertyDifferences).map(c -> PropertyDifference.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setPropertyDifferences(Collection<PropertyDifference.BuilderImpl> propertyDifferences) {
            this.propertyDifferences = PropertyDifferencesCopier.copyFromBuilder(propertyDifferences);
        }

        public final String getStackResourceDriftStatus() {
            return stackResourceDriftStatus;
        }

        @Override
        public final Builder stackResourceDriftStatus(String stackResourceDriftStatus) {
            this.stackResourceDriftStatus = stackResourceDriftStatus;
            return this;
        }

        @Override
        public final Builder stackResourceDriftStatus(StackResourceDriftStatus stackResourceDriftStatus) {
            this.stackResourceDriftStatus(stackResourceDriftStatus == null ? null : stackResourceDriftStatus.toString());
            return this;
        }

        public final void setStackResourceDriftStatus(String stackResourceDriftStatus) {
            this.stackResourceDriftStatus = stackResourceDriftStatus;
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        public final ModuleInfo.Builder getModuleInfo() {
            return moduleInfo != null ? moduleInfo.toBuilder() : null;
        }

        @Override
        public final Builder moduleInfo(ModuleInfo moduleInfo) {
            this.moduleInfo = moduleInfo;
            return this;
        }

        public final void setModuleInfo(ModuleInfo.BuilderImpl moduleInfo) {
            this.moduleInfo = moduleInfo != null ? moduleInfo.build() : null;
        }

        @Override
        public StackResourceDrift build() {
            return new StackResourceDrift(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
