/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about the drift status of the stack set.
 * </p>
 * <p>
 * For stack sets, contains information about the last <i>completed</i> drift operation performed on the stack set.
 * Information about drift operations in-progress is not included.
 * </p>
 * <p>
 * For stack set operations, includes information about drift operations currently being performed on the stack set.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-drift.html">Detecting Unmanaged
 * Changes in Stack Sets</a> in the <i>AWS CloudFormation User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSetDriftDetectionDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<StackSetDriftDetectionDetails.Builder, StackSetDriftDetectionDetails> {
    private static final SdkField<String> DRIFT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DriftStatus").getter(getter(StackSetDriftDetectionDetails::driftStatusAsString))
            .setter(setter(Builder::driftStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriftStatus").build()).build();

    private static final SdkField<String> DRIFT_DETECTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DriftDetectionStatus").getter(getter(StackSetDriftDetectionDetails::driftDetectionStatusAsString))
            .setter(setter(Builder::driftDetectionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriftDetectionStatus").build())
            .build();

    private static final SdkField<Instant> LAST_DRIFT_CHECK_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastDriftCheckTimestamp").getter(getter(StackSetDriftDetectionDetails::lastDriftCheckTimestamp))
            .setter(setter(Builder::lastDriftCheckTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastDriftCheckTimestamp").build())
            .build();

    private static final SdkField<Integer> TOTAL_STACK_INSTANCES_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("TotalStackInstancesCount")
            .getter(getter(StackSetDriftDetectionDetails::totalStackInstancesCount))
            .setter(setter(Builder::totalStackInstancesCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalStackInstancesCount").build())
            .build();

    private static final SdkField<Integer> DRIFTED_STACK_INSTANCES_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("DriftedStackInstancesCount")
            .getter(getter(StackSetDriftDetectionDetails::driftedStackInstancesCount))
            .setter(setter(Builder::driftedStackInstancesCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriftedStackInstancesCount").build())
            .build();

    private static final SdkField<Integer> IN_SYNC_STACK_INSTANCES_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("InSyncStackInstancesCount")
            .getter(getter(StackSetDriftDetectionDetails::inSyncStackInstancesCount))
            .setter(setter(Builder::inSyncStackInstancesCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InSyncStackInstancesCount").build())
            .build();

    private static final SdkField<Integer> IN_PROGRESS_STACK_INSTANCES_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("InProgressStackInstancesCount")
            .getter(getter(StackSetDriftDetectionDetails::inProgressStackInstancesCount))
            .setter(setter(Builder::inProgressStackInstancesCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InProgressStackInstancesCount")
                    .build()).build();

    private static final SdkField<Integer> FAILED_STACK_INSTANCES_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("FailedStackInstancesCount")
            .getter(getter(StackSetDriftDetectionDetails::failedStackInstancesCount))
            .setter(setter(Builder::failedStackInstancesCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailedStackInstancesCount").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DRIFT_STATUS_FIELD,
            DRIFT_DETECTION_STATUS_FIELD, LAST_DRIFT_CHECK_TIMESTAMP_FIELD, TOTAL_STACK_INSTANCES_COUNT_FIELD,
            DRIFTED_STACK_INSTANCES_COUNT_FIELD, IN_SYNC_STACK_INSTANCES_COUNT_FIELD, IN_PROGRESS_STACK_INSTANCES_COUNT_FIELD,
            FAILED_STACK_INSTANCES_COUNT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String driftStatus;

    private final String driftDetectionStatus;

    private final Instant lastDriftCheckTimestamp;

    private final Integer totalStackInstancesCount;

    private final Integer driftedStackInstancesCount;

    private final Integer inSyncStackInstancesCount;

    private final Integer inProgressStackInstancesCount;

    private final Integer failedStackInstancesCount;

    private StackSetDriftDetectionDetails(BuilderImpl builder) {
        this.driftStatus = builder.driftStatus;
        this.driftDetectionStatus = builder.driftDetectionStatus;
        this.lastDriftCheckTimestamp = builder.lastDriftCheckTimestamp;
        this.totalStackInstancesCount = builder.totalStackInstancesCount;
        this.driftedStackInstancesCount = builder.driftedStackInstancesCount;
        this.inSyncStackInstancesCount = builder.inSyncStackInstancesCount;
        this.inProgressStackInstancesCount = builder.inProgressStackInstancesCount;
        this.failedStackInstancesCount = builder.failedStackInstancesCount;
    }

    /**
     * <p>
     * Status of the stack set's actual configuration compared to its expected template and parameter configuration. A
     * stack set is considered to have drifted if one or more of its stack instances have drifted from their expected
     * template and parameter configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
     * expected template and parameter configuration. A stack instance is considered to have drifted if one or more of
     * the resources in the associated stack have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     * template and parameter configuration.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driftStatus} will
     * return {@link StackSetDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #driftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack set's actual configuration compared to its expected template and parameter
     *         configuration. A stack set is considered to have drifted if one or more of its stack instances have
     *         drifted from their expected template and parameter configuration.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
     *         the expected template and parameter configuration. A stack instance is considered to have drifted if one
     *         or more of the resources in the associated stack have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     *         template and parameter configuration.
     *         </p>
     *         </li>
     * @see StackSetDriftStatus
     */
    public StackSetDriftStatus driftStatus() {
        return StackSetDriftStatus.fromValue(driftStatus);
    }

    /**
     * <p>
     * Status of the stack set's actual configuration compared to its expected template and parameter configuration. A
     * stack set is considered to have drifted if one or more of its stack instances have drifted from their expected
     * template and parameter configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
     * expected template and parameter configuration. A stack instance is considered to have drifted if one or more of
     * the resources in the associated stack have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     * template and parameter configuration.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driftStatus} will
     * return {@link StackSetDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #driftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack set's actual configuration compared to its expected template and parameter
     *         configuration. A stack set is considered to have drifted if one or more of its stack instances have
     *         drifted from their expected template and parameter configuration.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
     *         the expected template and parameter configuration. A stack instance is considered to have drifted if one
     *         or more of the resources in the associated stack have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     *         template and parameter configuration.
     *         </p>
     *         </li>
     * @see StackSetDriftStatus
     */
    public String driftStatusAsString() {
        return driftStatus;
    }

    /**
     * <p>
     * The status of the stack set drift detection operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPLETED</code>: The drift detection operation completed without failing on any stack instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure tolerance for
     * the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has cancelled the drift detection operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #driftDetectionStatus} will return {@link StackSetDriftDetectionStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #driftDetectionStatusAsString}.
     * </p>
     * 
     * @return The status of the stack set drift detection operation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code>: The drift detection operation completed without failing on any stack instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure
     *         tolerance for the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has cancelled the drift detection operation.
     *         </p>
     *         </li>
     * @see StackSetDriftDetectionStatus
     */
    public StackSetDriftDetectionStatus driftDetectionStatus() {
        return StackSetDriftDetectionStatus.fromValue(driftDetectionStatus);
    }

    /**
     * <p>
     * The status of the stack set drift detection operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPLETED</code>: The drift detection operation completed without failing on any stack instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure tolerance for
     * the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has cancelled the drift detection operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #driftDetectionStatus} will return {@link StackSetDriftDetectionStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #driftDetectionStatusAsString}.
     * </p>
     * 
     * @return The status of the stack set drift detection operation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPLETED</code>: The drift detection operation completed without failing on any stack instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure
     *         tolerance for the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has cancelled the drift detection operation.
     *         </p>
     *         </li>
     * @see StackSetDriftDetectionStatus
     */
    public String driftDetectionStatusAsString() {
        return driftDetectionStatus;
    }

    /**
     * <p>
     * Most recent time when CloudFormation performed a drift detection operation on the stack set. This value will be
     * <code>NULL</code> for any stack set on which drift detection has not yet been performed.
     * </p>
     * 
     * @return Most recent time when CloudFormation performed a drift detection operation on the stack set. This value
     *         will be <code>NULL</code> for any stack set on which drift detection has not yet been performed.
     */
    public Instant lastDriftCheckTimestamp() {
        return lastDriftCheckTimestamp;
    }

    /**
     * <p>
     * The total number of stack instances belonging to this stack set.
     * </p>
     * <p>
     * The total number of stack instances is equal to the total of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Stack instances that match the stack set configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stack instances that have drifted from the stack set configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stack instances where the drift detection operation has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stack instances currently being checked for drift.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The total number of stack instances belonging to this stack set. </p>
     *         <p>
     *         The total number of stack instances is equal to the total of:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Stack instances that match the stack set configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stack instances that have drifted from the stack set configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stack instances where the drift detection operation has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stack instances currently being checked for drift.
     *         </p>
     *         </li>
     */
    public Integer totalStackInstancesCount() {
        return totalStackInstancesCount;
    }

    /**
     * <p>
     * The number of stack instances that have drifted from the expected template and parameter configuration of the
     * stack set. A stack instance is considered to have drifted if one or more of the resources in the associated stack
     * do not match their expected configuration.
     * </p>
     * 
     * @return The number of stack instances that have drifted from the expected template and parameter configuration of
     *         the stack set. A stack instance is considered to have drifted if one or more of the resources in the
     *         associated stack do not match their expected configuration.
     */
    public Integer driftedStackInstancesCount() {
        return driftedStackInstancesCount;
    }

    /**
     * <p>
     * The number of stack instances which match the expected template and parameter configuration of the stack set.
     * </p>
     * 
     * @return The number of stack instances which match the expected template and parameter configuration of the stack
     *         set.
     */
    public Integer inSyncStackInstancesCount() {
        return inSyncStackInstancesCount;
    }

    /**
     * <p>
     * The number of stack instances that are currently being checked for drift.
     * </p>
     * 
     * @return The number of stack instances that are currently being checked for drift.
     */
    public Integer inProgressStackInstancesCount() {
        return inProgressStackInstancesCount;
    }

    /**
     * <p>
     * The number of stack instances for which the drift detection operation failed.
     * </p>
     * 
     * @return The number of stack instances for which the drift detection operation failed.
     */
    public Integer failedStackInstancesCount() {
        return failedStackInstancesCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(driftStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(driftDetectionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastDriftCheckTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(totalStackInstancesCount());
        hashCode = 31 * hashCode + Objects.hashCode(driftedStackInstancesCount());
        hashCode = 31 * hashCode + Objects.hashCode(inSyncStackInstancesCount());
        hashCode = 31 * hashCode + Objects.hashCode(inProgressStackInstancesCount());
        hashCode = 31 * hashCode + Objects.hashCode(failedStackInstancesCount());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSetDriftDetectionDetails)) {
            return false;
        }
        StackSetDriftDetectionDetails other = (StackSetDriftDetectionDetails) obj;
        return Objects.equals(driftStatusAsString(), other.driftStatusAsString())
                && Objects.equals(driftDetectionStatusAsString(), other.driftDetectionStatusAsString())
                && Objects.equals(lastDriftCheckTimestamp(), other.lastDriftCheckTimestamp())
                && Objects.equals(totalStackInstancesCount(), other.totalStackInstancesCount())
                && Objects.equals(driftedStackInstancesCount(), other.driftedStackInstancesCount())
                && Objects.equals(inSyncStackInstancesCount(), other.inSyncStackInstancesCount())
                && Objects.equals(inProgressStackInstancesCount(), other.inProgressStackInstancesCount())
                && Objects.equals(failedStackInstancesCount(), other.failedStackInstancesCount());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("StackSetDriftDetectionDetails").add("DriftStatus", driftStatusAsString())
                .add("DriftDetectionStatus", driftDetectionStatusAsString())
                .add("LastDriftCheckTimestamp", lastDriftCheckTimestamp())
                .add("TotalStackInstancesCount", totalStackInstancesCount())
                .add("DriftedStackInstancesCount", driftedStackInstancesCount())
                .add("InSyncStackInstancesCount", inSyncStackInstancesCount())
                .add("InProgressStackInstancesCount", inProgressStackInstancesCount())
                .add("FailedStackInstancesCount", failedStackInstancesCount()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DriftStatus":
            return Optional.ofNullable(clazz.cast(driftStatusAsString()));
        case "DriftDetectionStatus":
            return Optional.ofNullable(clazz.cast(driftDetectionStatusAsString()));
        case "LastDriftCheckTimestamp":
            return Optional.ofNullable(clazz.cast(lastDriftCheckTimestamp()));
        case "TotalStackInstancesCount":
            return Optional.ofNullable(clazz.cast(totalStackInstancesCount()));
        case "DriftedStackInstancesCount":
            return Optional.ofNullable(clazz.cast(driftedStackInstancesCount()));
        case "InSyncStackInstancesCount":
            return Optional.ofNullable(clazz.cast(inSyncStackInstancesCount()));
        case "InProgressStackInstancesCount":
            return Optional.ofNullable(clazz.cast(inProgressStackInstancesCount()));
        case "FailedStackInstancesCount":
            return Optional.ofNullable(clazz.cast(failedStackInstancesCount()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackSetDriftDetectionDetails, T> g) {
        return obj -> g.apply((StackSetDriftDetectionDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackSetDriftDetectionDetails> {
        /**
         * <p>
         * Status of the stack set's actual configuration compared to its expected template and parameter configuration.
         * A stack set is considered to have drifted if one or more of its stack instances have drifted from their
         * expected template and parameter configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
         * expected template and parameter configuration. A stack instance is considered to have drifted if one or more
         * of the resources in the associated stack have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
         * template and parameter configuration.
         * </p>
         * </li>
         * </ul>
         * 
         * @param driftStatus
         *        Status of the stack set's actual configuration compared to its expected template and parameter
         *        configuration. A stack set is considered to have drifted if one or more of its stack instances have
         *        drifted from their expected template and parameter configuration.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
         *        the expected template and parameter configuration. A stack instance is considered to have drifted if
         *        one or more of the resources in the associated stack have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the
         *        expected template and parameter configuration.
         *        </p>
         *        </li>
         * @see StackSetDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetDriftStatus
         */
        Builder driftStatus(String driftStatus);

        /**
         * <p>
         * Status of the stack set's actual configuration compared to its expected template and parameter configuration.
         * A stack set is considered to have drifted if one or more of its stack instances have drifted from their
         * expected template and parameter configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
         * expected template and parameter configuration. A stack instance is considered to have drifted if one or more
         * of the resources in the associated stack have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
         * template and parameter configuration.
         * </p>
         * </li>
         * </ul>
         * 
         * @param driftStatus
         *        Status of the stack set's actual configuration compared to its expected template and parameter
         *        configuration. A stack set is considered to have drifted if one or more of its stack instances have
         *        drifted from their expected template and parameter configuration.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
         *        the expected template and parameter configuration. A stack instance is considered to have drifted if
         *        one or more of the resources in the associated stack have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation has not checked the stack set for drift.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the
         *        expected template and parameter configuration.
         *        </p>
         *        </li>
         * @see StackSetDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetDriftStatus
         */
        Builder driftStatus(StackSetDriftStatus driftStatus);

        /**
         * <p>
         * The status of the stack set drift detection operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPLETED</code>: The drift detection operation completed without failing on any stack instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure tolerance
         * for the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has cancelled the drift detection operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param driftDetectionStatus
         *        The status of the stack set drift detection operation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code>: The drift detection operation completed without failing on any stack
         *        instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure
         *        tolerance for the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has cancelled the drift detection operation.
         *        </p>
         *        </li>
         * @see StackSetDriftDetectionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetDriftDetectionStatus
         */
        Builder driftDetectionStatus(String driftDetectionStatus);

        /**
         * <p>
         * The status of the stack set drift detection operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPLETED</code>: The drift detection operation completed without failing on any stack instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure tolerance
         * for the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has cancelled the drift detection operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param driftDetectionStatus
         *        The status of the stack set drift detection operation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPLETED</code>: The drift detection operation completed without failing on any stack
         *        instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The drift detection operation exceeded the specified failure tolerance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PARTIAL_SUCCESS</code>: The drift detection operation completed without exceeding the failure
         *        tolerance for the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_PROGRESS</code>: The drift detection operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has cancelled the drift detection operation.
         *        </p>
         *        </li>
         * @see StackSetDriftDetectionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetDriftDetectionStatus
         */
        Builder driftDetectionStatus(StackSetDriftDetectionStatus driftDetectionStatus);

        /**
         * <p>
         * Most recent time when CloudFormation performed a drift detection operation on the stack set. This value will
         * be <code>NULL</code> for any stack set on which drift detection has not yet been performed.
         * </p>
         * 
         * @param lastDriftCheckTimestamp
         *        Most recent time when CloudFormation performed a drift detection operation on the stack set. This
         *        value will be <code>NULL</code> for any stack set on which drift detection has not yet been performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastDriftCheckTimestamp(Instant lastDriftCheckTimestamp);

        /**
         * <p>
         * The total number of stack instances belonging to this stack set.
         * </p>
         * <p>
         * The total number of stack instances is equal to the total of:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Stack instances that match the stack set configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stack instances that have drifted from the stack set configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stack instances where the drift detection operation has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stack instances currently being checked for drift.
         * </p>
         * </li>
         * </ul>
         * 
         * @param totalStackInstancesCount
         *        The total number of stack instances belonging to this stack set. </p>
         *        <p>
         *        The total number of stack instances is equal to the total of:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Stack instances that match the stack set configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stack instances that have drifted from the stack set configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stack instances where the drift detection operation has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stack instances currently being checked for drift.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalStackInstancesCount(Integer totalStackInstancesCount);

        /**
         * <p>
         * The number of stack instances that have drifted from the expected template and parameter configuration of the
         * stack set. A stack instance is considered to have drifted if one or more of the resources in the associated
         * stack do not match their expected configuration.
         * </p>
         * 
         * @param driftedStackInstancesCount
         *        The number of stack instances that have drifted from the expected template and parameter configuration
         *        of the stack set. A stack instance is considered to have drifted if one or more of the resources in
         *        the associated stack do not match their expected configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder driftedStackInstancesCount(Integer driftedStackInstancesCount);

        /**
         * <p>
         * The number of stack instances which match the expected template and parameter configuration of the stack set.
         * </p>
         * 
         * @param inSyncStackInstancesCount
         *        The number of stack instances which match the expected template and parameter configuration of the
         *        stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inSyncStackInstancesCount(Integer inSyncStackInstancesCount);

        /**
         * <p>
         * The number of stack instances that are currently being checked for drift.
         * </p>
         * 
         * @param inProgressStackInstancesCount
         *        The number of stack instances that are currently being checked for drift.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inProgressStackInstancesCount(Integer inProgressStackInstancesCount);

        /**
         * <p>
         * The number of stack instances for which the drift detection operation failed.
         * </p>
         * 
         * @param failedStackInstancesCount
         *        The number of stack instances for which the drift detection operation failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedStackInstancesCount(Integer failedStackInstancesCount);
    }

    static final class BuilderImpl implements Builder {
        private String driftStatus;

        private String driftDetectionStatus;

        private Instant lastDriftCheckTimestamp;

        private Integer totalStackInstancesCount;

        private Integer driftedStackInstancesCount;

        private Integer inSyncStackInstancesCount;

        private Integer inProgressStackInstancesCount;

        private Integer failedStackInstancesCount;

        private BuilderImpl() {
        }

        private BuilderImpl(StackSetDriftDetectionDetails model) {
            driftStatus(model.driftStatus);
            driftDetectionStatus(model.driftDetectionStatus);
            lastDriftCheckTimestamp(model.lastDriftCheckTimestamp);
            totalStackInstancesCount(model.totalStackInstancesCount);
            driftedStackInstancesCount(model.driftedStackInstancesCount);
            inSyncStackInstancesCount(model.inSyncStackInstancesCount);
            inProgressStackInstancesCount(model.inProgressStackInstancesCount);
            failedStackInstancesCount(model.failedStackInstancesCount);
        }

        public final String getDriftStatus() {
            return driftStatus;
        }

        @Override
        public final Builder driftStatus(String driftStatus) {
            this.driftStatus = driftStatus;
            return this;
        }

        @Override
        public final Builder driftStatus(StackSetDriftStatus driftStatus) {
            this.driftStatus(driftStatus == null ? null : driftStatus.toString());
            return this;
        }

        public final void setDriftStatus(String driftStatus) {
            this.driftStatus = driftStatus;
        }

        public final String getDriftDetectionStatus() {
            return driftDetectionStatus;
        }

        @Override
        public final Builder driftDetectionStatus(String driftDetectionStatus) {
            this.driftDetectionStatus = driftDetectionStatus;
            return this;
        }

        @Override
        public final Builder driftDetectionStatus(StackSetDriftDetectionStatus driftDetectionStatus) {
            this.driftDetectionStatus(driftDetectionStatus == null ? null : driftDetectionStatus.toString());
            return this;
        }

        public final void setDriftDetectionStatus(String driftDetectionStatus) {
            this.driftDetectionStatus = driftDetectionStatus;
        }

        public final Instant getLastDriftCheckTimestamp() {
            return lastDriftCheckTimestamp;
        }

        @Override
        public final Builder lastDriftCheckTimestamp(Instant lastDriftCheckTimestamp) {
            this.lastDriftCheckTimestamp = lastDriftCheckTimestamp;
            return this;
        }

        public final void setLastDriftCheckTimestamp(Instant lastDriftCheckTimestamp) {
            this.lastDriftCheckTimestamp = lastDriftCheckTimestamp;
        }

        public final Integer getTotalStackInstancesCount() {
            return totalStackInstancesCount;
        }

        @Override
        public final Builder totalStackInstancesCount(Integer totalStackInstancesCount) {
            this.totalStackInstancesCount = totalStackInstancesCount;
            return this;
        }

        public final void setTotalStackInstancesCount(Integer totalStackInstancesCount) {
            this.totalStackInstancesCount = totalStackInstancesCount;
        }

        public final Integer getDriftedStackInstancesCount() {
            return driftedStackInstancesCount;
        }

        @Override
        public final Builder driftedStackInstancesCount(Integer driftedStackInstancesCount) {
            this.driftedStackInstancesCount = driftedStackInstancesCount;
            return this;
        }

        public final void setDriftedStackInstancesCount(Integer driftedStackInstancesCount) {
            this.driftedStackInstancesCount = driftedStackInstancesCount;
        }

        public final Integer getInSyncStackInstancesCount() {
            return inSyncStackInstancesCount;
        }

        @Override
        public final Builder inSyncStackInstancesCount(Integer inSyncStackInstancesCount) {
            this.inSyncStackInstancesCount = inSyncStackInstancesCount;
            return this;
        }

        public final void setInSyncStackInstancesCount(Integer inSyncStackInstancesCount) {
            this.inSyncStackInstancesCount = inSyncStackInstancesCount;
        }

        public final Integer getInProgressStackInstancesCount() {
            return inProgressStackInstancesCount;
        }

        @Override
        public final Builder inProgressStackInstancesCount(Integer inProgressStackInstancesCount) {
            this.inProgressStackInstancesCount = inProgressStackInstancesCount;
            return this;
        }

        public final void setInProgressStackInstancesCount(Integer inProgressStackInstancesCount) {
            this.inProgressStackInstancesCount = inProgressStackInstancesCount;
        }

        public final Integer getFailedStackInstancesCount() {
            return failedStackInstancesCount;
        }

        @Override
        public final Builder failedStackInstancesCount(Integer failedStackInstancesCount) {
            this.failedStackInstancesCount = failedStackInstancesCount;
            return this;
        }

        public final void setFailedStackInstancesCount(Integer failedStackInstancesCount) {
            this.failedStackInstancesCount = failedStackInstancesCount;
        }

        @Override
        public StackSetDriftDetectionDetails build() {
            return new StackSetDriftDetectionDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
