/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structures that contain summary information about the specified stack set.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSetSummary implements SdkPojo, Serializable, ToCopyableBuilder<StackSetSummary.Builder, StackSetSummary> {
    private static final SdkField<String> STACK_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackSetName").getter(getter(StackSetSummary::stackSetName)).setter(setter(Builder::stackSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetName").build()).build();

    private static final SdkField<String> STACK_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackSetId").getter(getter(StackSetSummary::stackSetId)).setter(setter(Builder::stackSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(StackSetSummary::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(StackSetSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<AutoDeployment> AUTO_DEPLOYMENT_FIELD = SdkField
            .<AutoDeployment> builder(MarshallingType.SDK_POJO).memberName("AutoDeployment")
            .getter(getter(StackSetSummary::autoDeployment)).setter(setter(Builder::autoDeployment))
            .constructor(AutoDeployment::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoDeployment").build()).build();

    private static final SdkField<String> PERMISSION_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PermissionModel").getter(getter(StackSetSummary::permissionModelAsString))
            .setter(setter(Builder::permissionModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PermissionModel").build()).build();

    private static final SdkField<String> DRIFT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DriftStatus").getter(getter(StackSetSummary::driftStatusAsString)).setter(setter(Builder::driftStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DriftStatus").build()).build();

    private static final SdkField<Instant> LAST_DRIFT_CHECK_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastDriftCheckTimestamp").getter(getter(StackSetSummary::lastDriftCheckTimestamp))
            .setter(setter(Builder::lastDriftCheckTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastDriftCheckTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_SET_NAME_FIELD,
            STACK_SET_ID_FIELD, DESCRIPTION_FIELD, STATUS_FIELD, AUTO_DEPLOYMENT_FIELD, PERMISSION_MODEL_FIELD,
            DRIFT_STATUS_FIELD, LAST_DRIFT_CHECK_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackSetName;

    private final String stackSetId;

    private final String description;

    private final String status;

    private final AutoDeployment autoDeployment;

    private final String permissionModel;

    private final String driftStatus;

    private final Instant lastDriftCheckTimestamp;

    private StackSetSummary(BuilderImpl builder) {
        this.stackSetName = builder.stackSetName;
        this.stackSetId = builder.stackSetId;
        this.description = builder.description;
        this.status = builder.status;
        this.autoDeployment = builder.autoDeployment;
        this.permissionModel = builder.permissionModel;
        this.driftStatus = builder.driftStatus;
        this.lastDriftCheckTimestamp = builder.lastDriftCheckTimestamp;
    }

    /**
     * <p>
     * The name of the stack set.
     * </p>
     * 
     * @return The name of the stack set.
     */
    public final String stackSetName() {
        return stackSetName;
    }

    /**
     * <p>
     * The ID of the stack set.
     * </p>
     * 
     * @return The ID of the stack set.
     */
    public final String stackSetId() {
        return stackSetId;
    }

    /**
     * <p>
     * A description of the stack set that you specify when the stack set is created or updated.
     * </p>
     * 
     * @return A description of the stack set that you specify when the stack set is created or updated.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The status of the stack set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack set.
     * @see StackSetStatus
     */
    public final StackSetStatus status() {
        return StackSetStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the stack set.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack set.
     * @see StackSetStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * [Service-managed permissions] Describes whether StackSets automatically deploys to Organizations accounts that
     * are added to a target organizational unit (OU).
     * </p>
     * 
     * @return [Service-managed permissions] Describes whether StackSets automatically deploys to Organizations accounts
     *         that are added to a target organizational unit (OU).
     */
    public final AutoDeployment autoDeployment() {
        return autoDeployment;
    }

    /**
     * <p>
     * Describes how the IAM roles required for stack set operations are created.
     * </p>
     * <ul>
     * <li>
     * <p>
     * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
     * deploy to target accounts. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html">Grant
     * Self-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to deploy
     * to accounts managed by Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     * >Grant Service-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionModel}
     * will return {@link PermissionModels#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionModelAsString}.
     * </p>
     * 
     * @return Describes how the IAM roles required for stack set operations are created.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         With <code>self-managed</code> permissions, you must create the administrator and execution roles
     *         required to deploy to target accounts. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
     *         >Grant Self-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
     *         deploy to accounts managed by Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     *         >Grant Service-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     * @see PermissionModels
     */
    public final PermissionModels permissionModel() {
        return PermissionModels.fromValue(permissionModel);
    }

    /**
     * <p>
     * Describes how the IAM roles required for stack set operations are created.
     * </p>
     * <ul>
     * <li>
     * <p>
     * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
     * deploy to target accounts. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html">Grant
     * Self-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to deploy
     * to accounts managed by Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     * >Grant Service-Managed Stack Set Permissions</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #permissionModel}
     * will return {@link PermissionModels#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #permissionModelAsString}.
     * </p>
     * 
     * @return Describes how the IAM roles required for stack set operations are created.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         With <code>self-managed</code> permissions, you must create the administrator and execution roles
     *         required to deploy to target accounts. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
     *         >Grant Self-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
     *         deploy to accounts managed by Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
     *         >Grant Service-Managed Stack Set Permissions</a>.
     *         </p>
     *         </li>
     * @see PermissionModels
     */
    public final String permissionModelAsString() {
        return permissionModel;
    }

    /**
     * <p>
     * Status of the stack set's actual configuration compared to its expected template and parameter configuration. A
     * stack set is considered to have drifted if one or more of its stack instances have drifted from their expected
     * template and parameter configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
     * expected template and parameter configuration. A stack instance is considered to have drifted if one or more of
     * the resources in the associated stack have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     * template and parameter configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNKNOWN</code>: This value is reserved for future use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driftStatus} will
     * return {@link StackDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #driftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack set's actual configuration compared to its expected template and parameter
     *         configuration. A stack set is considered to have drifted if one or more of its stack instances have
     *         drifted from their expected template and parameter configuration.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
     *         the expected template and parameter configuration. A stack instance is considered to have drifted if one
     *         or more of the resources in the associated stack have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     *         template and parameter configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNKNOWN</code>: This value is reserved for future use.
     *         </p>
     *         </li>
     * @see StackDriftStatus
     */
    public final StackDriftStatus driftStatus() {
        return StackDriftStatus.fromValue(driftStatus);
    }

    /**
     * <p>
     * Status of the stack set's actual configuration compared to its expected template and parameter configuration. A
     * stack set is considered to have drifted if one or more of its stack instances have drifted from their expected
     * template and parameter configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
     * expected template and parameter configuration. A stack instance is considered to have drifted if one or more of
     * the resources in the associated stack have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     * template and parameter configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNKNOWN</code>: This value is reserved for future use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #driftStatus} will
     * return {@link StackDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #driftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack set's actual configuration compared to its expected template and parameter
     *         configuration. A stack set is considered to have drifted if one or more of its stack instances have
     *         drifted from their expected template and parameter configuration.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
     *         the expected template and parameter configuration. A stack instance is considered to have drifted if one
     *         or more of the resources in the associated stack have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
     *         template and parameter configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNKNOWN</code>: This value is reserved for future use.
     *         </p>
     *         </li>
     * @see StackDriftStatus
     */
    public final String driftStatusAsString() {
        return driftStatus;
    }

    /**
     * <p>
     * Most recent time when CloudFormation performed a drift detection operation on the stack set. This value will be
     * <code>NULL</code> for any stack set on which drift detection has not yet been performed.
     * </p>
     * 
     * @return Most recent time when CloudFormation performed a drift detection operation on the stack set. This value
     *         will be <code>NULL</code> for any stack set on which drift detection has not yet been performed.
     */
    public final Instant lastDriftCheckTimestamp() {
        return lastDriftCheckTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackSetName());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(autoDeployment());
        hashCode = 31 * hashCode + Objects.hashCode(permissionModelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(driftStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastDriftCheckTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSetSummary)) {
            return false;
        }
        StackSetSummary other = (StackSetSummary) obj;
        return Objects.equals(stackSetName(), other.stackSetName()) && Objects.equals(stackSetId(), other.stackSetId())
                && Objects.equals(description(), other.description()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(autoDeployment(), other.autoDeployment())
                && Objects.equals(permissionModelAsString(), other.permissionModelAsString())
                && Objects.equals(driftStatusAsString(), other.driftStatusAsString())
                && Objects.equals(lastDriftCheckTimestamp(), other.lastDriftCheckTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackSetSummary").add("StackSetName", stackSetName()).add("StackSetId", stackSetId())
                .add("Description", description()).add("Status", statusAsString()).add("AutoDeployment", autoDeployment())
                .add("PermissionModel", permissionModelAsString()).add("DriftStatus", driftStatusAsString())
                .add("LastDriftCheckTimestamp", lastDriftCheckTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetName":
            return Optional.ofNullable(clazz.cast(stackSetName()));
        case "StackSetId":
            return Optional.ofNullable(clazz.cast(stackSetId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "AutoDeployment":
            return Optional.ofNullable(clazz.cast(autoDeployment()));
        case "PermissionModel":
            return Optional.ofNullable(clazz.cast(permissionModelAsString()));
        case "DriftStatus":
            return Optional.ofNullable(clazz.cast(driftStatusAsString()));
        case "LastDriftCheckTimestamp":
            return Optional.ofNullable(clazz.cast(lastDriftCheckTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackSetSummary, T> g) {
        return obj -> g.apply((StackSetSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackSetSummary> {
        /**
         * <p>
         * The name of the stack set.
         * </p>
         * 
         * @param stackSetName
         *        The name of the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetName(String stackSetName);

        /**
         * <p>
         * The ID of the stack set.
         * </p>
         * 
         * @param stackSetId
         *        The ID of the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetId(String stackSetId);

        /**
         * <p>
         * A description of the stack set that you specify when the stack set is created or updated.
         * </p>
         * 
         * @param description
         *        A description of the stack set that you specify when the stack set is created or updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The status of the stack set.
         * </p>
         * 
         * @param status
         *        The status of the stack set.
         * @see StackSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the stack set.
         * </p>
         * 
         * @param status
         *        The status of the stack set.
         * @see StackSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetStatus
         */
        Builder status(StackSetStatus status);

        /**
         * <p>
         * [Service-managed permissions] Describes whether StackSets automatically deploys to Organizations accounts
         * that are added to a target organizational unit (OU).
         * </p>
         * 
         * @param autoDeployment
         *        [Service-managed permissions] Describes whether StackSets automatically deploys to Organizations
         *        accounts that are added to a target organizational unit (OU).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoDeployment(AutoDeployment autoDeployment);

        /**
         * <p>
         * [Service-managed permissions] Describes whether StackSets automatically deploys to Organizations accounts
         * that are added to a target organizational unit (OU).
         * </p>
         * This is a convenience that creates an instance of the {@link AutoDeployment.Builder} avoiding the need to
         * create one manually via {@link AutoDeployment#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoDeployment.Builder#build()} is called immediately and its
         * result is passed to {@link #autoDeployment(AutoDeployment)}.
         * 
         * @param autoDeployment
         *        a consumer that will call methods on {@link AutoDeployment.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoDeployment(AutoDeployment)
         */
        default Builder autoDeployment(Consumer<AutoDeployment.Builder> autoDeployment) {
            return autoDeployment(AutoDeployment.builder().applyMutation(autoDeployment).build());
        }

        /**
         * <p>
         * Describes how the IAM roles required for stack set operations are created.
         * </p>
         * <ul>
         * <li>
         * <p>
         * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
         * deploy to target accounts. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         * >Grant Self-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
         * deploy to accounts managed by Organizations. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html">Grant
         * Service-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permissionModel
         *        Describes how the IAM roles required for stack set operations are created.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        With <code>self-managed</code> permissions, you must create the administrator and execution roles
         *        required to deploy to target accounts. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         *        >Grant Self-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required
         *        to deploy to accounts managed by Organizations. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
         *        >Grant Service-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         * @see PermissionModels
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionModels
         */
        Builder permissionModel(String permissionModel);

        /**
         * <p>
         * Describes how the IAM roles required for stack set operations are created.
         * </p>
         * <ul>
         * <li>
         * <p>
         * With <code>self-managed</code> permissions, you must create the administrator and execution roles required to
         * deploy to target accounts. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         * >Grant Self-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required to
         * deploy to accounts managed by Organizations. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html">Grant
         * Service-Managed Stack Set Permissions</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param permissionModel
         *        Describes how the IAM roles required for stack set operations are created.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        With <code>self-managed</code> permissions, you must create the administrator and execution roles
         *        required to deploy to target accounts. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-self-managed.html"
         *        >Grant Self-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        With <code>service-managed</code> permissions, StackSets automatically creates the IAM roles required
         *        to deploy to accounts managed by Organizations. For more information, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs-service-managed.html"
         *        >Grant Service-Managed Stack Set Permissions</a>.
         *        </p>
         *        </li>
         * @see PermissionModels
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PermissionModels
         */
        Builder permissionModel(PermissionModels permissionModel);

        /**
         * <p>
         * Status of the stack set's actual configuration compared to its expected template and parameter configuration.
         * A stack set is considered to have drifted if one or more of its stack instances have drifted from their
         * expected template and parameter configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
         * expected template and parameter configuration. A stack instance is considered to have drifted if one or more
         * of the resources in the associated stack have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
         * template and parameter configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNKNOWN</code>: This value is reserved for future use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param driftStatus
         *        Status of the stack set's actual configuration compared to its expected template and parameter
         *        configuration. A stack set is considered to have drifted if one or more of its stack instances have
         *        drifted from their expected template and parameter configuration.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
         *        the expected template and parameter configuration. A stack instance is considered to have drifted if
         *        one or more of the resources in the associated stack have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the
         *        expected template and parameter configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNKNOWN</code>: This value is reserved for future use.
         *        </p>
         *        </li>
         * @see StackDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackDriftStatus
         */
        Builder driftStatus(String driftStatus);

        /**
         * <p>
         * Status of the stack set's actual configuration compared to its expected template and parameter configuration.
         * A stack set is considered to have drifted if one or more of its stack instances have drifted from their
         * expected template and parameter configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from the
         * expected template and parameter configuration. A stack instance is considered to have drifted if one or more
         * of the resources in the associated stack have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the expected
         * template and parameter configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNKNOWN</code>: This value is reserved for future use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param driftStatus
         *        Status of the stack set's actual configuration compared to its expected template and parameter
         *        configuration. A stack set is considered to have drifted if one or more of its stack instances have
         *        drifted from their expected template and parameter configuration.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: One or more of the stack instances belonging to the stack set stack differs from
         *        the expected template and parameter configuration. A stack instance is considered to have drifted if
         *        one or more of the resources in the associated stack have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: CloudFormation has not checked the stack set for drift.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: All of the stack instances belonging to the stack set stack match from the
         *        expected template and parameter configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNKNOWN</code>: This value is reserved for future use.
         *        </p>
         *        </li>
         * @see StackDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackDriftStatus
         */
        Builder driftStatus(StackDriftStatus driftStatus);

        /**
         * <p>
         * Most recent time when CloudFormation performed a drift detection operation on the stack set. This value will
         * be <code>NULL</code> for any stack set on which drift detection has not yet been performed.
         * </p>
         * 
         * @param lastDriftCheckTimestamp
         *        Most recent time when CloudFormation performed a drift detection operation on the stack set. This
         *        value will be <code>NULL</code> for any stack set on which drift detection has not yet been performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastDriftCheckTimestamp(Instant lastDriftCheckTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String stackSetName;

        private String stackSetId;

        private String description;

        private String status;

        private AutoDeployment autoDeployment;

        private String permissionModel;

        private String driftStatus;

        private Instant lastDriftCheckTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(StackSetSummary model) {
            stackSetName(model.stackSetName);
            stackSetId(model.stackSetId);
            description(model.description);
            status(model.status);
            autoDeployment(model.autoDeployment);
            permissionModel(model.permissionModel);
            driftStatus(model.driftStatus);
            lastDriftCheckTimestamp(model.lastDriftCheckTimestamp);
        }

        public final String getStackSetName() {
            return stackSetName;
        }

        public final void setStackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
        }

        @Override
        @Transient
        public final Builder stackSetName(String stackSetName) {
            this.stackSetName = stackSetName;
            return this;
        }

        public final String getStackSetId() {
            return stackSetId;
        }

        public final void setStackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
        }

        @Override
        @Transient
        public final Builder stackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(StackSetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final AutoDeployment.Builder getAutoDeployment() {
            return autoDeployment != null ? autoDeployment.toBuilder() : null;
        }

        public final void setAutoDeployment(AutoDeployment.BuilderImpl autoDeployment) {
            this.autoDeployment = autoDeployment != null ? autoDeployment.build() : null;
        }

        @Override
        @Transient
        public final Builder autoDeployment(AutoDeployment autoDeployment) {
            this.autoDeployment = autoDeployment;
            return this;
        }

        public final String getPermissionModel() {
            return permissionModel;
        }

        public final void setPermissionModel(String permissionModel) {
            this.permissionModel = permissionModel;
        }

        @Override
        @Transient
        public final Builder permissionModel(String permissionModel) {
            this.permissionModel = permissionModel;
            return this;
        }

        @Override
        @Transient
        public final Builder permissionModel(PermissionModels permissionModel) {
            this.permissionModel(permissionModel == null ? null : permissionModel.toString());
            return this;
        }

        public final String getDriftStatus() {
            return driftStatus;
        }

        public final void setDriftStatus(String driftStatus) {
            this.driftStatus = driftStatus;
        }

        @Override
        @Transient
        public final Builder driftStatus(String driftStatus) {
            this.driftStatus = driftStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder driftStatus(StackDriftStatus driftStatus) {
            this.driftStatus(driftStatus == null ? null : driftStatus.toString());
            return this;
        }

        public final Instant getLastDriftCheckTimestamp() {
            return lastDriftCheckTimestamp;
        }

        public final void setLastDriftCheckTimestamp(Instant lastDriftCheckTimestamp) {
            this.lastDriftCheckTimestamp = lastDriftCheckTimestamp;
        }

        @Override
        @Transient
        public final Builder lastDriftCheckTimestamp(Instant lastDriftCheckTimestamp) {
            this.lastDriftCheckTimestamp = lastDriftCheckTimestamp;
            return this;
        }

        @Override
        public StackSetSummary build() {
            return new StackSetSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
