/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the target resource of an import operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceToImport implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceToImport.Builder, ResourceToImport> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ResourceToImport::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> LOGICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogicalResourceId").getter(getter(ResourceToImport::logicalResourceId))
            .setter(setter(Builder::logicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalResourceId").build()).build();

    private static final SdkField<Map<String, String>> RESOURCE_IDENTIFIER_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ResourceIdentifier")
            .getter(getter(ResourceToImport::resourceIdentifier))
            .setter(setter(Builder::resourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdentifier").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            LOGICAL_RESOURCE_ID_FIELD, RESOURCE_IDENTIFIER_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final String logicalResourceId;

    private final Map<String, String> resourceIdentifier;

    private ResourceToImport(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.logicalResourceId = builder.logicalResourceId;
        this.resourceIdentifier = builder.resourceIdentifier;
    }

    /**
     * <p>
     * The type of resource to import into your stack, such as <code>AWS::S3::Bucket</code>. For a list of supported
     * resource types, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
     * >Resources that support import operations</a> in the CloudFormation User Guide.
     * </p>
     * 
     * @return The type of resource to import into your stack, such as <code>AWS::S3::Bucket</code>. For a list of
     *         supported resource types, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
     *         >Resources that support import operations</a> in the CloudFormation User Guide.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * The logical ID of the target resource as specified in the template.
     * </p>
     * 
     * @return The logical ID of the target resource as specified in the template.
     */
    public final String logicalResourceId() {
        return logicalResourceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceIdentifier property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceIdentifier() {
        return resourceIdentifier != null && !(resourceIdentifier instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A key-value pair that identifies the target resource. The key is an identifier property (for example,
     * <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual property value
     * (for example, <code>MyS3Bucket</code>).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceIdentifier} method.
     * </p>
     * 
     * @return A key-value pair that identifies the target resource. The key is an identifier property (for example,
     *         <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual property
     *         value (for example, <code>MyS3Bucket</code>).
     */
    public final Map<String, String> resourceIdentifier() {
        return resourceIdentifier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(logicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIdentifier() ? resourceIdentifier() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceToImport)) {
            return false;
        }
        ResourceToImport other = (ResourceToImport) obj;
        return Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(logicalResourceId(), other.logicalResourceId())
                && hasResourceIdentifier() == other.hasResourceIdentifier()
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceToImport").add("ResourceType", resourceType())
                .add("LogicalResourceId", logicalResourceId())
                .add("ResourceIdentifier", hasResourceIdentifier() ? resourceIdentifier() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "LogicalResourceId":
            return Optional.ofNullable(clazz.cast(logicalResourceId()));
        case "ResourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceToImport, T> g) {
        return obj -> g.apply((ResourceToImport) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceToImport> {
        /**
         * <p>
         * The type of resource to import into your stack, such as <code>AWS::S3::Bucket</code>. For a list of supported
         * resource types, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
         * >Resources that support import operations</a> in the CloudFormation User Guide.
         * </p>
         * 
         * @param resourceType
         *        The type of resource to import into your stack, such as <code>AWS::S3::Bucket</code>. For a list of
         *        supported resource types, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
         *        >Resources that support import operations</a> in the CloudFormation User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The logical ID of the target resource as specified in the template.
         * </p>
         * 
         * @param logicalResourceId
         *        The logical ID of the target resource as specified in the template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceId(String logicalResourceId);

        /**
         * <p>
         * A key-value pair that identifies the target resource. The key is an identifier property (for example,
         * <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual property
         * value (for example, <code>MyS3Bucket</code>).
         * </p>
         * 
         * @param resourceIdentifier
         *        A key-value pair that identifies the target resource. The key is an identifier property (for example,
         *        <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual
         *        property value (for example, <code>MyS3Bucket</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(Map<String, String> resourceIdentifier);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private String logicalResourceId;

        private Map<String, String> resourceIdentifier = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceToImport model) {
            resourceType(model.resourceType);
            logicalResourceId(model.logicalResourceId);
            resourceIdentifier(model.resourceIdentifier);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        @Transient
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final String getLogicalResourceId() {
            return logicalResourceId;
        }

        public final void setLogicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
        }

        @Override
        @Transient
        public final Builder logicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
            return this;
        }

        public final Map<String, String> getResourceIdentifier() {
            if (resourceIdentifier instanceof SdkAutoConstructMap) {
                return null;
            }
            return resourceIdentifier;
        }

        public final void setResourceIdentifier(Map<String, String> resourceIdentifier) {
            this.resourceIdentifier = ResourceIdentifierPropertiesCopier.copy(resourceIdentifier);
        }

        @Override
        @Transient
        public final Builder resourceIdentifier(Map<String, String> resourceIdentifier) {
            this.resourceIdentifier = ResourceIdentifierPropertiesCopier.copy(resourceIdentifier);
            return this;
        }

        @Override
        public ResourceToImport build() {
            return new ResourceToImport(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
