/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structure that contains information about a stack set operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSetOperation implements SdkPojo, Serializable,
        ToCopyableBuilder<StackSetOperation.Builder, StackSetOperation> {
    private static final SdkField<String> OPERATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperationId").getter(getter(StackSetOperation::operationId)).setter(setter(Builder::operationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationId").build()).build();

    private static final SdkField<String> STACK_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackSetId").getter(getter(StackSetOperation::stackSetId)).setter(setter(Builder::stackSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetId").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(StackSetOperation::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(StackSetOperation::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<StackSetOperationPreferences> OPERATION_PREFERENCES_FIELD = SdkField
            .<StackSetOperationPreferences> builder(MarshallingType.SDK_POJO).memberName("OperationPreferences")
            .getter(getter(StackSetOperation::operationPreferences)).setter(setter(Builder::operationPreferences))
            .constructor(StackSetOperationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationPreferences").build())
            .build();

    private static final SdkField<Boolean> RETAIN_STACKS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RetainStacks").getter(getter(StackSetOperation::retainStacks)).setter(setter(Builder::retainStacks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetainStacks").build()).build();

    private static final SdkField<String> ADMINISTRATION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdministrationRoleARN").getter(getter(StackSetOperation::administrationRoleARN))
            .setter(setter(Builder::administrationRoleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdministrationRoleARN").build())
            .build();

    private static final SdkField<String> EXECUTION_ROLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRoleName").getter(getter(StackSetOperation::executionRoleName))
            .setter(setter(Builder::executionRoleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRoleName").build()).build();

    private static final SdkField<Instant> CREATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTimestamp").getter(getter(StackSetOperation::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTimestamp").build()).build();

    private static final SdkField<Instant> END_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTimestamp").getter(getter(StackSetOperation::endTimestamp)).setter(setter(Builder::endTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTimestamp").build()).build();

    private static final SdkField<DeploymentTargets> DEPLOYMENT_TARGETS_FIELD = SdkField
            .<DeploymentTargets> builder(MarshallingType.SDK_POJO).memberName("DeploymentTargets")
            .getter(getter(StackSetOperation::deploymentTargets)).setter(setter(Builder::deploymentTargets))
            .constructor(DeploymentTargets::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeploymentTargets").build()).build();

    private static final SdkField<StackSetDriftDetectionDetails> STACK_SET_DRIFT_DETECTION_DETAILS_FIELD = SdkField
            .<StackSetDriftDetectionDetails> builder(MarshallingType.SDK_POJO)
            .memberName("StackSetDriftDetectionDetails")
            .getter(getter(StackSetOperation::stackSetDriftDetectionDetails))
            .setter(setter(Builder::stackSetDriftDetectionDetails))
            .constructor(StackSetDriftDetectionDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetDriftDetectionDetails")
                    .build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(StackSetOperation::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<StackSetOperationStatusDetails> STATUS_DETAILS_FIELD = SdkField
            .<StackSetOperationStatusDetails> builder(MarshallingType.SDK_POJO).memberName("StatusDetails")
            .getter(getter(StackSetOperation::statusDetails)).setter(setter(Builder::statusDetails))
            .constructor(StackSetOperationStatusDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusDetails").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OPERATION_ID_FIELD,
            STACK_SET_ID_FIELD, ACTION_FIELD, STATUS_FIELD, OPERATION_PREFERENCES_FIELD, RETAIN_STACKS_FIELD,
            ADMINISTRATION_ROLE_ARN_FIELD, EXECUTION_ROLE_NAME_FIELD, CREATION_TIMESTAMP_FIELD, END_TIMESTAMP_FIELD,
            DEPLOYMENT_TARGETS_FIELD, STACK_SET_DRIFT_DETECTION_DETAILS_FIELD, STATUS_REASON_FIELD, STATUS_DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String operationId;

    private final String stackSetId;

    private final String action;

    private final String status;

    private final StackSetOperationPreferences operationPreferences;

    private final Boolean retainStacks;

    private final String administrationRoleARN;

    private final String executionRoleName;

    private final Instant creationTimestamp;

    private final Instant endTimestamp;

    private final DeploymentTargets deploymentTargets;

    private final StackSetDriftDetectionDetails stackSetDriftDetectionDetails;

    private final String statusReason;

    private final StackSetOperationStatusDetails statusDetails;

    private StackSetOperation(BuilderImpl builder) {
        this.operationId = builder.operationId;
        this.stackSetId = builder.stackSetId;
        this.action = builder.action;
        this.status = builder.status;
        this.operationPreferences = builder.operationPreferences;
        this.retainStacks = builder.retainStacks;
        this.administrationRoleARN = builder.administrationRoleARN;
        this.executionRoleName = builder.executionRoleName;
        this.creationTimestamp = builder.creationTimestamp;
        this.endTimestamp = builder.endTimestamp;
        this.deploymentTargets = builder.deploymentTargets;
        this.stackSetDriftDetectionDetails = builder.stackSetDriftDetectionDetails;
        this.statusReason = builder.statusReason;
        this.statusDetails = builder.statusDetails;
    }

    /**
     * <p>
     * The unique ID of a stack set operation.
     * </p>
     * 
     * @return The unique ID of a stack set operation.
     */
    public final String operationId() {
        return operationId;
    }

    /**
     * <p>
     * The ID of the stack set.
     * </p>
     * 
     * @return The ID of the stack set.
     */
    public final String stackSetId() {
        return stackSetId;
    }

    /**
     * <p>
     * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
     * delete operations affect only the specified stack set instances that are associated with the specified stack set.
     * Update operations affect both the stack set itself, in addition to <i>all</i> associated stack set instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackSetOperationAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create
     *         and delete operations affect only the specified stack set instances that are associated with the
     *         specified stack set. Update operations affect both the stack set itself, in addition to <i>all</i>
     *         associated stack set instances.
     * @see StackSetOperationAction
     */
    public final StackSetOperationAction action() {
        return StackSetOperationAction.fromValue(action);
    }

    /**
     * <p>
     * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
     * delete operations affect only the specified stack set instances that are associated with the specified stack set.
     * Update operations affect both the stack set itself, in addition to <i>all</i> associated stack set instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackSetOperationAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create
     *         and delete operations affect only the specified stack set instances that are associated with the
     *         specified stack set. Update operations affect both the stack set itself, in addition to <i>all</i>
     *         associated stack set instances.
     * @see StackSetOperationAction
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The status of the operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
     * you've set for an operation is applied for each Region during stack create and update operations. If the number
     * of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the Region is set
     * to <code>FAILED</code>. This in turn sets the status of the operation as a whole to <code>FAILED</code>, and
     * CloudFormation cancels the operation in any remaining Regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     * operations, the operation is queued to be performed. For more information, see the <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
     * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code>: The operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has canceled the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without exceeding
     * the failure tolerance for the operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the operation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value
     *         that you've set for an operation is applied for each Region during stack create and update operations. If
     *         the number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in
     *         the Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
     *         <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     *         operations, the operation is queued to be performed. For more information, see the <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
     *         >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code>: The operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has canceled the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
     *         exceeding the failure tolerance for the operation.
     *         </p>
     *         </li>
     * @see StackSetOperationStatus
     */
    public final StackSetOperationStatus status() {
        return StackSetOperationStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
     * you've set for an operation is applied for each Region during stack create and update operations. If the number
     * of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the Region is set
     * to <code>FAILED</code>. This in turn sets the status of the operation as a whole to <code>FAILED</code>, and
     * CloudFormation cancels the operation in any remaining Regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     * operations, the operation is queued to be performed. For more information, see the <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
     * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code>: The operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has canceled the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without exceeding
     * the failure tolerance for the operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the operation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value
     *         that you've set for an operation is applied for each Region during stack create and update operations. If
     *         the number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in
     *         the Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
     *         <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     *         operations, the operation is queued to be performed. For more information, see the <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
     *         >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code>: The operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has canceled the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
     *         exceeding the failure tolerance for the operation.
     *         </p>
     *         </li>
     * @see StackSetOperationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The preferences for how CloudFormation performs this stack set operation.
     * </p>
     * 
     * @return The preferences for how CloudFormation performs this stack set operation.
     */
    public final StackSetOperationPreferences operationPreferences() {
        return operationPreferences;
    }

    /**
     * <p>
     * For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack instances from
     * the specified stack set, but doesn't delete the stacks. You can't re-associate a retained stack, or add an
     * existing, saved stack to a new stack set.
     * </p>
     * 
     * @return For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack
     *         instances from the specified stack set, but doesn't delete the stacks. You can't re-associate a retained
     *         stack, or add an existing, saved stack to a new stack set.
     */
    public final Boolean retainStacks() {
        return retainStacks;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role used to perform this stack set operation.
     * </p>
     * <p>
     * Use customized administrator roles to control which users or groups can manage specific stack sets within the
     * same administrator account. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html">Define Permissions
     * for Multiple Administrators</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role used to perform this stack set operation.</p>
     *         <p>
     *         Use customized administrator roles to control which users or groups can manage specific stack sets within
     *         the same administrator account. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html">Define
     *         Permissions for Multiple Administrators</a> in the <i>CloudFormation User Guide</i>.
     */
    public final String administrationRoleARN() {
        return administrationRoleARN;
    }

    /**
     * <p>
     * The name of the IAM execution role used to create or update the stack set.
     * </p>
     * <p>
     * Use customized execution roles to control which stack resources users and groups can include in their stack sets.
     * </p>
     * 
     * @return The name of the IAM execution role used to create or update the stack set.</p>
     *         <p>
     *         Use customized execution roles to control which stack resources users and groups can include in their
     *         stack sets.
     */
    public final String executionRoleName() {
        return executionRoleName;
    }

    /**
     * <p>
     * The time at which the operation was initiated. Note that the creation times for the stack set operation might
     * differ from the creation time of the individual stacks themselves. This is because CloudFormation needs to
     * perform preparatory work for the operation, such as dispatching the work to the requested Regions, before
     * actually creating the first stacks.
     * </p>
     * 
     * @return The time at which the operation was initiated. Note that the creation times for the stack set operation
     *         might differ from the creation time of the individual stacks themselves. This is because CloudFormation
     *         needs to perform preparatory work for the operation, such as dispatching the work to the requested
     *         Regions, before actually creating the first stacks.
     */
    public final Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * <p>
     * The time at which the stack set operation ended, across all accounts and Regions specified. Note that this
     * doesn't necessarily mean that the stack set operation was successful, or even attempted, in each account or
     * Region.
     * </p>
     * 
     * @return The time at which the stack set operation ended, across all accounts and Regions specified. Note that
     *         this doesn't necessarily mean that the stack set operation was successful, or even attempted, in each
     *         account or Region.
     */
    public final Instant endTimestamp() {
        return endTimestamp;
    }

    /**
     * <p>
     * [Service-managed permissions] The Organizations accounts affected by the stack operation.
     * </p>
     * 
     * @return [Service-managed permissions] The Organizations accounts affected by the stack operation.
     */
    public final DeploymentTargets deploymentTargets() {
        return deploymentTargets;
    }

    /**
     * <p>
     * Detailed information about the drift status of the stack set. This includes information about drift operations
     * currently being performed on the stack set.
     * </p>
     * <p>
     * This information will only be present for stack set operations whose <code>Action</code> type is
     * <code>DETECT_DRIFT</code>.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-drift.html">Detecting Unmanaged
     * Changes in Stack Sets</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * 
     * @return Detailed information about the drift status of the stack set. This includes information about drift
     *         operations currently being performed on the stack set.</p>
     *         <p>
     *         This information will only be present for stack set operations whose <code>Action</code> type is
     *         <code>DETECT_DRIFT</code>.
     *         </p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-drift.html">Detecting
     *         Unmanaged Changes in Stack Sets</a> in the <i>CloudFormation User Guide</i>.
     */
    public final StackSetDriftDetectionDetails stackSetDriftDetectionDetails() {
        return stackSetDriftDetectionDetails;
    }

    /**
     * <p>
     * The status of the operation in details.
     * </p>
     * 
     * @return The status of the operation in details.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * Detailed information about the StackSet operation.
     * </p>
     * 
     * @return Detailed information about the StackSet operation.
     */
    public final StackSetOperationStatusDetails statusDetails() {
        return statusDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(operationId());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetId());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(operationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(retainStacks());
        hashCode = 31 * hashCode + Objects.hashCode(administrationRoleARN());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleName());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(endTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(deploymentTargets());
        hashCode = 31 * hashCode + Objects.hashCode(stackSetDriftDetectionDetails());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSetOperation)) {
            return false;
        }
        StackSetOperation other = (StackSetOperation) obj;
        return Objects.equals(operationId(), other.operationId()) && Objects.equals(stackSetId(), other.stackSetId())
                && Objects.equals(actionAsString(), other.actionAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(operationPreferences(), other.operationPreferences())
                && Objects.equals(retainStacks(), other.retainStacks())
                && Objects.equals(administrationRoleARN(), other.administrationRoleARN())
                && Objects.equals(executionRoleName(), other.executionRoleName())
                && Objects.equals(creationTimestamp(), other.creationTimestamp())
                && Objects.equals(endTimestamp(), other.endTimestamp())
                && Objects.equals(deploymentTargets(), other.deploymentTargets())
                && Objects.equals(stackSetDriftDetectionDetails(), other.stackSetDriftDetectionDetails())
                && Objects.equals(statusReason(), other.statusReason()) && Objects.equals(statusDetails(), other.statusDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackSetOperation").add("OperationId", operationId()).add("StackSetId", stackSetId())
                .add("Action", actionAsString()).add("Status", statusAsString())
                .add("OperationPreferences", operationPreferences()).add("RetainStacks", retainStacks())
                .add("AdministrationRoleARN", administrationRoleARN()).add("ExecutionRoleName", executionRoleName())
                .add("CreationTimestamp", creationTimestamp()).add("EndTimestamp", endTimestamp())
                .add("DeploymentTargets", deploymentTargets())
                .add("StackSetDriftDetectionDetails", stackSetDriftDetectionDetails()).add("StatusReason", statusReason())
                .add("StatusDetails", statusDetails()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OperationId":
            return Optional.ofNullable(clazz.cast(operationId()));
        case "StackSetId":
            return Optional.ofNullable(clazz.cast(stackSetId()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "OperationPreferences":
            return Optional.ofNullable(clazz.cast(operationPreferences()));
        case "RetainStacks":
            return Optional.ofNullable(clazz.cast(retainStacks()));
        case "AdministrationRoleARN":
            return Optional.ofNullable(clazz.cast(administrationRoleARN()));
        case "ExecutionRoleName":
            return Optional.ofNullable(clazz.cast(executionRoleName()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "EndTimestamp":
            return Optional.ofNullable(clazz.cast(endTimestamp()));
        case "DeploymentTargets":
            return Optional.ofNullable(clazz.cast(deploymentTargets()));
        case "StackSetDriftDetectionDetails":
            return Optional.ofNullable(clazz.cast(stackSetDriftDetectionDetails()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "StatusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackSetOperation, T> g) {
        return obj -> g.apply((StackSetOperation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackSetOperation> {
        /**
         * <p>
         * The unique ID of a stack set operation.
         * </p>
         * 
         * @param operationId
         *        The unique ID of a stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationId(String operationId);

        /**
         * <p>
         * The ID of the stack set.
         * </p>
         * 
         * @param stackSetId
         *        The ID of the stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetId(String stackSetId);

        /**
         * <p>
         * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
         * delete operations affect only the specified stack set instances that are associated with the specified stack
         * set. Update operations affect both the stack set itself, in addition to <i>all</i> associated stack set
         * instances.
         * </p>
         * 
         * @param action
         *        The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>.
         *        Create and delete operations affect only the specified stack set instances that are associated with
         *        the specified stack set. Update operations affect both the stack set itself, in addition to <i>all</i>
         *        associated stack set instances.
         * @see StackSetOperationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationAction
         */
        Builder action(String action);

        /**
         * <p>
         * The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
         * delete operations affect only the specified stack set instances that are associated with the specified stack
         * set. Update operations affect both the stack set itself, in addition to <i>all</i> associated stack set
         * instances.
         * </p>
         * 
         * @param action
         *        The type of stack set operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>.
         *        Create and delete operations affect only the specified stack set instances that are associated with
         *        the specified stack set. Update operations affect both the stack set itself, in addition to <i>all</i>
         *        associated stack set instances.
         * @see StackSetOperationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationAction
         */
        Builder action(StackSetOperationAction action);

        /**
         * <p>
         * The status of the operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
         * you've set for an operation is applied for each Region during stack create and update operations. If the
         * number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the
         * Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
         * <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
         * operations, the operation is queued to be performed. For more information, see the <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
         * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code>: The operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has canceled the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         * exceeding the failure tolerance for the operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the operation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance
         *        value that you've set for an operation is applied for each Region during stack create and update
         *        operations. If the number of failed stacks within a Region exceeds the failure tolerance, the status
         *        of the operation in the Region is set to <code>FAILED</code>. This in turn sets the status of the
         *        operation as a whole to <code>FAILED</code>, and CloudFormation cancels the operation in any remaining
         *        Regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence
         *        of operations, the operation is queued to be performed. For more information, see the <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
         *        >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code>: The operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has canceled the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         *        exceeding the failure tolerance for the operation.
         *        </p>
         *        </li>
         * @see StackSetOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
         * you've set for an operation is applied for each Region during stack create and update operations. If the
         * number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the
         * Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
         * <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
         * operations, the operation is queued to be performed. For more information, see the <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
         * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code>: The operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has canceled the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         * exceeding the failure tolerance for the operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the operation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance
         *        value that you've set for an operation is applied for each Region during stack create and update
         *        operations. If the number of failed stacks within a Region exceeds the failure tolerance, the status
         *        of the operation in the Region is set to <code>FAILED</code>. This in turn sets the status of the
         *        operation as a whole to <code>FAILED</code>, and CloudFormation cancels the operation in any remaining
         *        Regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence
         *        of operations, the operation is queued to be performed. For more information, see the <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stackset-status-codes"
         *        >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code>: The operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has canceled the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         *        exceeding the failure tolerance for the operation.
         *        </p>
         *        </li>
         * @see StackSetOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationStatus
         */
        Builder status(StackSetOperationStatus status);

        /**
         * <p>
         * The preferences for how CloudFormation performs this stack set operation.
         * </p>
         * 
         * @param operationPreferences
         *        The preferences for how CloudFormation performs this stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationPreferences(StackSetOperationPreferences operationPreferences);

        /**
         * <p>
         * The preferences for how CloudFormation performs this stack set operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link StackSetOperationPreferences.Builder}
         * avoiding the need to create one manually via {@link StackSetOperationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StackSetOperationPreferences.Builder#build()} is called
         * immediately and its result is passed to {@link #operationPreferences(StackSetOperationPreferences)}.
         * 
         * @param operationPreferences
         *        a consumer that will call methods on {@link StackSetOperationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operationPreferences(StackSetOperationPreferences)
         */
        default Builder operationPreferences(Consumer<StackSetOperationPreferences.Builder> operationPreferences) {
            return operationPreferences(StackSetOperationPreferences.builder().applyMutation(operationPreferences).build());
        }

        /**
         * <p>
         * For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack instances
         * from the specified stack set, but doesn't delete the stacks. You can't re-associate a retained stack, or add
         * an existing, saved stack to a new stack set.
         * </p>
         * 
         * @param retainStacks
         *        For stack set operations of action type <code>DELETE</code>, specifies whether to remove the stack
         *        instances from the specified stack set, but doesn't delete the stacks. You can't re-associate a
         *        retained stack, or add an existing, saved stack to a new stack set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retainStacks(Boolean retainStacks);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role used to perform this stack set operation.
         * </p>
         * <p>
         * Use customized administrator roles to control which users or groups can manage specific stack sets within the
         * same administrator account. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html">Define
         * Permissions for Multiple Administrators</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * 
         * @param administrationRoleARN
         *        The Amazon Resource Name (ARN) of the IAM role used to perform this stack set operation.</p>
         *        <p>
         *        Use customized administrator roles to control which users or groups can manage specific stack sets
         *        within the same administrator account. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-prereqs.html">Define
         *        Permissions for Multiple Administrators</a> in the <i>CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder administrationRoleARN(String administrationRoleARN);

        /**
         * <p>
         * The name of the IAM execution role used to create or update the stack set.
         * </p>
         * <p>
         * Use customized execution roles to control which stack resources users and groups can include in their stack
         * sets.
         * </p>
         * 
         * @param executionRoleName
         *        The name of the IAM execution role used to create or update the stack set.</p>
         *        <p>
         *        Use customized execution roles to control which stack resources users and groups can include in their
         *        stack sets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleName(String executionRoleName);

        /**
         * <p>
         * The time at which the operation was initiated. Note that the creation times for the stack set operation might
         * differ from the creation time of the individual stacks themselves. This is because CloudFormation needs to
         * perform preparatory work for the operation, such as dispatching the work to the requested Regions, before
         * actually creating the first stacks.
         * </p>
         * 
         * @param creationTimestamp
         *        The time at which the operation was initiated. Note that the creation times for the stack set
         *        operation might differ from the creation time of the individual stacks themselves. This is because
         *        CloudFormation needs to perform preparatory work for the operation, such as dispatching the work to
         *        the requested Regions, before actually creating the first stacks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * The time at which the stack set operation ended, across all accounts and Regions specified. Note that this
         * doesn't necessarily mean that the stack set operation was successful, or even attempted, in each account or
         * Region.
         * </p>
         * 
         * @param endTimestamp
         *        The time at which the stack set operation ended, across all accounts and Regions specified. Note that
         *        this doesn't necessarily mean that the stack set operation was successful, or even attempted, in each
         *        account or Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTimestamp(Instant endTimestamp);

        /**
         * <p>
         * [Service-managed permissions] The Organizations accounts affected by the stack operation.
         * </p>
         * 
         * @param deploymentTargets
         *        [Service-managed permissions] The Organizations accounts affected by the stack operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentTargets(DeploymentTargets deploymentTargets);

        /**
         * <p>
         * [Service-managed permissions] The Organizations accounts affected by the stack operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeploymentTargets.Builder} avoiding the
         * need to create one manually via {@link DeploymentTargets#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentTargets.Builder#build()} is called immediately and its
         * result is passed to {@link #deploymentTargets(DeploymentTargets)}.
         * 
         * @param deploymentTargets
         *        a consumer that will call methods on {@link DeploymentTargets.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentTargets(DeploymentTargets)
         */
        default Builder deploymentTargets(Consumer<DeploymentTargets.Builder> deploymentTargets) {
            return deploymentTargets(DeploymentTargets.builder().applyMutation(deploymentTargets).build());
        }

        /**
         * <p>
         * Detailed information about the drift status of the stack set. This includes information about drift
         * operations currently being performed on the stack set.
         * </p>
         * <p>
         * This information will only be present for stack set operations whose <code>Action</code> type is
         * <code>DETECT_DRIFT</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-drift.html">Detecting
         * Unmanaged Changes in Stack Sets</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * 
         * @param stackSetDriftDetectionDetails
         *        Detailed information about the drift status of the stack set. This includes information about drift
         *        operations currently being performed on the stack set.</p>
         *        <p>
         *        This information will only be present for stack set operations whose <code>Action</code> type is
         *        <code>DETECT_DRIFT</code>.
         *        </p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-drift.html">Detecting
         *        Unmanaged Changes in Stack Sets</a> in the <i>CloudFormation User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetDriftDetectionDetails(StackSetDriftDetectionDetails stackSetDriftDetectionDetails);

        /**
         * <p>
         * Detailed information about the drift status of the stack set. This includes information about drift
         * operations currently being performed on the stack set.
         * </p>
         * <p>
         * This information will only be present for stack set operations whose <code>Action</code> type is
         * <code>DETECT_DRIFT</code>.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-drift.html">Detecting
         * Unmanaged Changes in Stack Sets</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link StackSetDriftDetectionDetails.Builder}
         * avoiding the need to create one manually via {@link StackSetDriftDetectionDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StackSetDriftDetectionDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #stackSetDriftDetectionDetails(StackSetDriftDetectionDetails)}.
         * 
         * @param stackSetDriftDetectionDetails
         *        a consumer that will call methods on {@link StackSetDriftDetectionDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stackSetDriftDetectionDetails(StackSetDriftDetectionDetails)
         */
        default Builder stackSetDriftDetectionDetails(
                Consumer<StackSetDriftDetectionDetails.Builder> stackSetDriftDetectionDetails) {
            return stackSetDriftDetectionDetails(StackSetDriftDetectionDetails.builder()
                    .applyMutation(stackSetDriftDetectionDetails).build());
        }

        /**
         * <p>
         * The status of the operation in details.
         * </p>
         * 
         * @param statusReason
         *        The status of the operation in details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * Detailed information about the StackSet operation.
         * </p>
         * 
         * @param statusDetails
         *        Detailed information about the StackSet operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(StackSetOperationStatusDetails statusDetails);

        /**
         * <p>
         * Detailed information about the StackSet operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link StackSetOperationStatusDetails.Builder}
         * avoiding the need to create one manually via {@link StackSetOperationStatusDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StackSetOperationStatusDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #statusDetails(StackSetOperationStatusDetails)}.
         * 
         * @param statusDetails
         *        a consumer that will call methods on {@link StackSetOperationStatusDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusDetails(StackSetOperationStatusDetails)
         */
        default Builder statusDetails(Consumer<StackSetOperationStatusDetails.Builder> statusDetails) {
            return statusDetails(StackSetOperationStatusDetails.builder().applyMutation(statusDetails).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String operationId;

        private String stackSetId;

        private String action;

        private String status;

        private StackSetOperationPreferences operationPreferences;

        private Boolean retainStacks;

        private String administrationRoleARN;

        private String executionRoleName;

        private Instant creationTimestamp;

        private Instant endTimestamp;

        private DeploymentTargets deploymentTargets;

        private StackSetDriftDetectionDetails stackSetDriftDetectionDetails;

        private String statusReason;

        private StackSetOperationStatusDetails statusDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(StackSetOperation model) {
            operationId(model.operationId);
            stackSetId(model.stackSetId);
            action(model.action);
            status(model.status);
            operationPreferences(model.operationPreferences);
            retainStacks(model.retainStacks);
            administrationRoleARN(model.administrationRoleARN);
            executionRoleName(model.executionRoleName);
            creationTimestamp(model.creationTimestamp);
            endTimestamp(model.endTimestamp);
            deploymentTargets(model.deploymentTargets);
            stackSetDriftDetectionDetails(model.stackSetDriftDetectionDetails);
            statusReason(model.statusReason);
            statusDetails(model.statusDetails);
        }

        public final String getOperationId() {
            return operationId;
        }

        public final void setOperationId(String operationId) {
            this.operationId = operationId;
        }

        @Override
        public final Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public final String getStackSetId() {
            return stackSetId;
        }

        public final void setStackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
        }

        @Override
        public final Builder stackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(StackSetOperationAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StackSetOperationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final StackSetOperationPreferences.Builder getOperationPreferences() {
            return operationPreferences != null ? operationPreferences.toBuilder() : null;
        }

        public final void setOperationPreferences(StackSetOperationPreferences.BuilderImpl operationPreferences) {
            this.operationPreferences = operationPreferences != null ? operationPreferences.build() : null;
        }

        @Override
        public final Builder operationPreferences(StackSetOperationPreferences operationPreferences) {
            this.operationPreferences = operationPreferences;
            return this;
        }

        public final Boolean getRetainStacks() {
            return retainStacks;
        }

        public final void setRetainStacks(Boolean retainStacks) {
            this.retainStacks = retainStacks;
        }

        @Override
        public final Builder retainStacks(Boolean retainStacks) {
            this.retainStacks = retainStacks;
            return this;
        }

        public final String getAdministrationRoleARN() {
            return administrationRoleARN;
        }

        public final void setAdministrationRoleARN(String administrationRoleARN) {
            this.administrationRoleARN = administrationRoleARN;
        }

        @Override
        public final Builder administrationRoleARN(String administrationRoleARN) {
            this.administrationRoleARN = administrationRoleARN;
            return this;
        }

        public final String getExecutionRoleName() {
            return executionRoleName;
        }

        public final void setExecutionRoleName(String executionRoleName) {
            this.executionRoleName = executionRoleName;
        }

        @Override
        public final Builder executionRoleName(String executionRoleName) {
            this.executionRoleName = executionRoleName;
            return this;
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final Instant getEndTimestamp() {
            return endTimestamp;
        }

        public final void setEndTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
        }

        @Override
        public final Builder endTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
            return this;
        }

        public final DeploymentTargets.Builder getDeploymentTargets() {
            return deploymentTargets != null ? deploymentTargets.toBuilder() : null;
        }

        public final void setDeploymentTargets(DeploymentTargets.BuilderImpl deploymentTargets) {
            this.deploymentTargets = deploymentTargets != null ? deploymentTargets.build() : null;
        }

        @Override
        public final Builder deploymentTargets(DeploymentTargets deploymentTargets) {
            this.deploymentTargets = deploymentTargets;
            return this;
        }

        public final StackSetDriftDetectionDetails.Builder getStackSetDriftDetectionDetails() {
            return stackSetDriftDetectionDetails != null ? stackSetDriftDetectionDetails.toBuilder() : null;
        }

        public final void setStackSetDriftDetectionDetails(StackSetDriftDetectionDetails.BuilderImpl stackSetDriftDetectionDetails) {
            this.stackSetDriftDetectionDetails = stackSetDriftDetectionDetails != null ? stackSetDriftDetectionDetails.build()
                    : null;
        }

        @Override
        public final Builder stackSetDriftDetectionDetails(StackSetDriftDetectionDetails stackSetDriftDetectionDetails) {
            this.stackSetDriftDetectionDetails = stackSetDriftDetectionDetails;
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final StackSetOperationStatusDetails.Builder getStatusDetails() {
            return statusDetails != null ? statusDetails.toBuilder() : null;
        }

        public final void setStatusDetails(StackSetOperationStatusDetails.BuilderImpl statusDetails) {
            this.statusDetails = statusDetails != null ? statusDetails.build() : null;
        }

        @Override
        public final Builder statusDetails(StackSetOperationStatusDetails statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        @Override
        public StackSetOperation build() {
            return new StackSetOperation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
