/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The user-specified preferences for how CloudFormation performs a stack set operation.
 * </p>
 * <p>
 * For more information about maximum concurrent accounts and failure tolerance, see <a href=
 * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-ops-options">Stack
 * set operation options</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSetOperationPreferences implements SdkPojo, Serializable,
        ToCopyableBuilder<StackSetOperationPreferences.Builder, StackSetOperationPreferences> {
    private static final SdkField<String> REGION_CONCURRENCY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RegionConcurrencyType").getter(getter(StackSetOperationPreferences::regionConcurrencyTypeAsString))
            .setter(setter(Builder::regionConcurrencyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegionConcurrencyType").build())
            .build();

    private static final SdkField<List<String>> REGION_ORDER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("RegionOrder")
            .getter(getter(StackSetOperationPreferences::regionOrder))
            .setter(setter(Builder::regionOrder))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegionOrder").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> FAILURE_TOLERANCE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("FailureToleranceCount").getter(getter(StackSetOperationPreferences::failureToleranceCount))
            .setter(setter(Builder::failureToleranceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureToleranceCount").build())
            .build();

    private static final SdkField<Integer> FAILURE_TOLERANCE_PERCENTAGE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("FailureTolerancePercentage")
            .getter(getter(StackSetOperationPreferences::failureTolerancePercentage))
            .setter(setter(Builder::failureTolerancePercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureTolerancePercentage").build())
            .build();

    private static final SdkField<Integer> MAX_CONCURRENT_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxConcurrentCount").getter(getter(StackSetOperationPreferences::maxConcurrentCount))
            .setter(setter(Builder::maxConcurrentCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrentCount").build())
            .build();

    private static final SdkField<Integer> MAX_CONCURRENT_PERCENTAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxConcurrentPercentage").getter(getter(StackSetOperationPreferences::maxConcurrentPercentage))
            .setter(setter(Builder::maxConcurrentPercentage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrentPercentage").build())
            .build();

    private static final SdkField<String> CONCURRENCY_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConcurrencyMode").getter(getter(StackSetOperationPreferences::concurrencyModeAsString))
            .setter(setter(Builder::concurrencyMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConcurrencyMode").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REGION_CONCURRENCY_TYPE_FIELD,
            REGION_ORDER_FIELD, FAILURE_TOLERANCE_COUNT_FIELD, FAILURE_TOLERANCE_PERCENTAGE_FIELD, MAX_CONCURRENT_COUNT_FIELD,
            MAX_CONCURRENT_PERCENTAGE_FIELD, CONCURRENCY_MODE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String regionConcurrencyType;

    private final List<String> regionOrder;

    private final Integer failureToleranceCount;

    private final Integer failureTolerancePercentage;

    private final Integer maxConcurrentCount;

    private final Integer maxConcurrentPercentage;

    private final String concurrencyMode;

    private StackSetOperationPreferences(BuilderImpl builder) {
        this.regionConcurrencyType = builder.regionConcurrencyType;
        this.regionOrder = builder.regionOrder;
        this.failureToleranceCount = builder.failureToleranceCount;
        this.failureTolerancePercentage = builder.failureTolerancePercentage;
        this.maxConcurrentCount = builder.maxConcurrentCount;
        this.maxConcurrentPercentage = builder.maxConcurrentPercentage;
        this.concurrencyMode = builder.concurrencyMode;
    }

    /**
     * <p>
     * The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region at a time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #regionConcurrencyType} will return {@link RegionConcurrencyType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #regionConcurrencyTypeAsString}.
     * </p>
     * 
     * @return The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region at
     *         a time.
     * @see RegionConcurrencyType
     */
    public final RegionConcurrencyType regionConcurrencyType() {
        return RegionConcurrencyType.fromValue(regionConcurrencyType);
    }

    /**
     * <p>
     * The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region at a time.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #regionConcurrencyType} will return {@link RegionConcurrencyType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #regionConcurrencyTypeAsString}.
     * </p>
     * 
     * @return The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region at
     *         a time.
     * @see RegionConcurrencyType
     */
    public final String regionConcurrencyTypeAsString() {
        return regionConcurrencyType;
    }

    /**
     * For responses, this returns true if the service returned a value for the RegionOrder property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRegionOrder() {
        return regionOrder != null && !(regionOrder instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The order of the Regions where you want to perform the stack operation.
     * </p>
     * <note>
     * <p>
     * <code>RegionOrder</code> isn't followed if <code>AutoDeployment</code> is enabled.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRegionOrder} method.
     * </p>
     * 
     * @return The order of the Regions where you want to perform the stack operation.</p> <note>
     *         <p>
     *         <code>RegionOrder</code> isn't followed if <code>AutoDeployment</code> is enabled.
     *         </p>
     */
    public final List<String> regionOrder() {
        return regionOrder;
    }

    /**
     * <p>
     * The number of accounts, per Region, for which this operation can fail before CloudFormation stops the operation
     * in that Region. If the operation is stopped in a Region, CloudFormation doesn't attempt the operation in any
     * subsequent Regions.
     * </p>
     * <p>
     * Conditional: You must specify either <code>FailureToleranceCount</code> or
     * <code>FailureTolerancePercentage</code> (but not both).
     * </p>
     * <p>
     * By default, <code>0</code> is specified.
     * </p>
     * 
     * @return The number of accounts, per Region, for which this operation can fail before CloudFormation stops the
     *         operation in that Region. If the operation is stopped in a Region, CloudFormation doesn't attempt the
     *         operation in any subsequent Regions.</p>
     *         <p>
     *         Conditional: You must specify either <code>FailureToleranceCount</code> or
     *         <code>FailureTolerancePercentage</code> (but not both).
     *         </p>
     *         <p>
     *         By default, <code>0</code> is specified.
     */
    public final Integer failureToleranceCount() {
        return failureToleranceCount;
    }

    /**
     * <p>
     * The percentage of accounts, per Region, for which this stack operation can fail before CloudFormation stops the
     * operation in that Region. If the operation is stopped in a Region, CloudFormation doesn't attempt the operation
     * in any subsequent Regions.
     * </p>
     * <p>
     * When calculating the number of accounts based on the specified percentage, CloudFormation rounds <i>down</i> to
     * the next whole number.
     * </p>
     * <p>
     * Conditional: You must specify either <code>FailureToleranceCount</code> or
     * <code>FailureTolerancePercentage</code>, but not both.
     * </p>
     * <p>
     * By default, <code>0</code> is specified.
     * </p>
     * 
     * @return The percentage of accounts, per Region, for which this stack operation can fail before CloudFormation
     *         stops the operation in that Region. If the operation is stopped in a Region, CloudFormation doesn't
     *         attempt the operation in any subsequent Regions.</p>
     *         <p>
     *         When calculating the number of accounts based on the specified percentage, CloudFormation rounds
     *         <i>down</i> to the next whole number.
     *         </p>
     *         <p>
     *         Conditional: You must specify either <code>FailureToleranceCount</code> or
     *         <code>FailureTolerancePercentage</code>, but not both.
     *         </p>
     *         <p>
     *         By default, <code>0</code> is specified.
     */
    public final Integer failureTolerancePercentage() {
        return failureTolerancePercentage;
    }

    /**
     * <p>
     * The maximum number of accounts in which to perform this operation at one time. This can depend on the value of
     * <code>FailureToleranceCount</code> depending on your <code>ConcurrencyMode</code>.
     * <code>MaxConcurrentCount</code> is at most one more than the <code>FailureToleranceCount</code> if you're using
     * <code>STRICT_FAILURE_TOLERANCE</code>.
     * </p>
     * <p>
     * Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments, under certain
     * circumstances the actual number of accounts acted upon concurrently may be lower due to service throttling.
     * </p>
     * <p>
     * Conditional: You must specify either <code>MaxConcurrentCount</code> or <code>MaxConcurrentPercentage</code>, but
     * not both.
     * </p>
     * <p>
     * By default, <code>1</code> is specified.
     * </p>
     * 
     * @return The maximum number of accounts in which to perform this operation at one time. This can depend on the
     *         value of <code>FailureToleranceCount</code> depending on your <code>ConcurrencyMode</code>.
     *         <code>MaxConcurrentCount</code> is at most one more than the <code>FailureToleranceCount</code> if you're
     *         using <code>STRICT_FAILURE_TOLERANCE</code>.</p>
     *         <p>
     *         Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments, under
     *         certain circumstances the actual number of accounts acted upon concurrently may be lower due to service
     *         throttling.
     *         </p>
     *         <p>
     *         Conditional: You must specify either <code>MaxConcurrentCount</code> or
     *         <code>MaxConcurrentPercentage</code>, but not both.
     *         </p>
     *         <p>
     *         By default, <code>1</code> is specified.
     */
    public final Integer maxConcurrentCount() {
        return maxConcurrentCount;
    }

    /**
     * <p>
     * The maximum percentage of accounts in which to perform this operation at one time.
     * </p>
     * <p>
     * When calculating the number of accounts based on the specified percentage, CloudFormation rounds down to the next
     * whole number. This is true except in cases where rounding down would result is zero. In this case, CloudFormation
     * sets the number as one instead.
     * </p>
     * <p>
     * Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments, under certain
     * circumstances the actual number of accounts acted upon concurrently may be lower due to service throttling.
     * </p>
     * <p>
     * Conditional: You must specify either <code>MaxConcurrentCount</code> or <code>MaxConcurrentPercentage</code>, but
     * not both.
     * </p>
     * <p>
     * By default, <code>1</code> is specified.
     * </p>
     * 
     * @return The maximum percentage of accounts in which to perform this operation at one time.</p>
     *         <p>
     *         When calculating the number of accounts based on the specified percentage, CloudFormation rounds down to
     *         the next whole number. This is true except in cases where rounding down would result is zero. In this
     *         case, CloudFormation sets the number as one instead.
     *         </p>
     *         <p>
     *         Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments, under
     *         certain circumstances the actual number of accounts acted upon concurrently may be lower due to service
     *         throttling.
     *         </p>
     *         <p>
     *         Conditional: You must specify either <code>MaxConcurrentCount</code> or
     *         <code>MaxConcurrentPercentage</code>, but not both.
     *         </p>
     *         <p>
     *         By default, <code>1</code> is specified.
     */
    public final Integer maxConcurrentPercentage() {
        return maxConcurrentPercentage;
    }

    /**
     * <p>
     * Specifies how the concurrency level behaves during the operation execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure the number
     * of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The initial actual
     * concurrency is set to the lower of either the value of the <code>MaxConcurrentCount</code>, or the value of
     * <code>FailureToleranceCount</code> +1. The actual concurrency is then reduced proportionally by the number of
     * failures. This is the default behavior.
     * </p>
     * <p>
     * If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the actual
     * concurrency. This allows stack set operations to run at the concurrency level set by the
     * <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the number of
     * failures.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #concurrencyMode}
     * will return {@link ConcurrencyMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #concurrencyModeAsString}.
     * </p>
     * 
     * @return Specifies how the concurrency level behaves during the operation execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure the
     *         number of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The initial
     *         actual concurrency is set to the lower of either the value of the <code>MaxConcurrentCount</code>, or the
     *         value of <code>FailureToleranceCount</code> +1. The actual concurrency is then reduced proportionally by
     *         the number of failures. This is the default behavior.
     *         </p>
     *         <p>
     *         If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the
     *         actual concurrency. This allows stack set operations to run at the concurrency level set by the
     *         <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the number
     *         of failures.
     *         </p>
     *         </li>
     * @see ConcurrencyMode
     */
    public final ConcurrencyMode concurrencyMode() {
        return ConcurrencyMode.fromValue(concurrencyMode);
    }

    /**
     * <p>
     * Specifies how the concurrency level behaves during the operation execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure the number
     * of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The initial actual
     * concurrency is set to the lower of either the value of the <code>MaxConcurrentCount</code>, or the value of
     * <code>FailureToleranceCount</code> +1. The actual concurrency is then reduced proportionally by the number of
     * failures. This is the default behavior.
     * </p>
     * <p>
     * If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the actual
     * concurrency. This allows stack set operations to run at the concurrency level set by the
     * <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the number of
     * failures.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #concurrencyMode}
     * will return {@link ConcurrencyMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #concurrencyModeAsString}.
     * </p>
     * 
     * @return Specifies how the concurrency level behaves during the operation execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure the
     *         number of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The initial
     *         actual concurrency is set to the lower of either the value of the <code>MaxConcurrentCount</code>, or the
     *         value of <code>FailureToleranceCount</code> +1. The actual concurrency is then reduced proportionally by
     *         the number of failures. This is the default behavior.
     *         </p>
     *         <p>
     *         If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the
     *         actual concurrency. This allows stack set operations to run at the concurrency level set by the
     *         <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the number
     *         of failures.
     *         </p>
     *         </li>
     * @see ConcurrencyMode
     */
    public final String concurrencyModeAsString() {
        return concurrencyMode;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(regionConcurrencyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRegionOrder() ? regionOrder() : null);
        hashCode = 31 * hashCode + Objects.hashCode(failureToleranceCount());
        hashCode = 31 * hashCode + Objects.hashCode(failureTolerancePercentage());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrentCount());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrentPercentage());
        hashCode = 31 * hashCode + Objects.hashCode(concurrencyModeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSetOperationPreferences)) {
            return false;
        }
        StackSetOperationPreferences other = (StackSetOperationPreferences) obj;
        return Objects.equals(regionConcurrencyTypeAsString(), other.regionConcurrencyTypeAsString())
                && hasRegionOrder() == other.hasRegionOrder() && Objects.equals(regionOrder(), other.regionOrder())
                && Objects.equals(failureToleranceCount(), other.failureToleranceCount())
                && Objects.equals(failureTolerancePercentage(), other.failureTolerancePercentage())
                && Objects.equals(maxConcurrentCount(), other.maxConcurrentCount())
                && Objects.equals(maxConcurrentPercentage(), other.maxConcurrentPercentage())
                && Objects.equals(concurrencyModeAsString(), other.concurrencyModeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackSetOperationPreferences").add("RegionConcurrencyType", regionConcurrencyTypeAsString())
                .add("RegionOrder", hasRegionOrder() ? regionOrder() : null)
                .add("FailureToleranceCount", failureToleranceCount())
                .add("FailureTolerancePercentage", failureTolerancePercentage()).add("MaxConcurrentCount", maxConcurrentCount())
                .add("MaxConcurrentPercentage", maxConcurrentPercentage()).add("ConcurrencyMode", concurrencyModeAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RegionConcurrencyType":
            return Optional.ofNullable(clazz.cast(regionConcurrencyTypeAsString()));
        case "RegionOrder":
            return Optional.ofNullable(clazz.cast(regionOrder()));
        case "FailureToleranceCount":
            return Optional.ofNullable(clazz.cast(failureToleranceCount()));
        case "FailureTolerancePercentage":
            return Optional.ofNullable(clazz.cast(failureTolerancePercentage()));
        case "MaxConcurrentCount":
            return Optional.ofNullable(clazz.cast(maxConcurrentCount()));
        case "MaxConcurrentPercentage":
            return Optional.ofNullable(clazz.cast(maxConcurrentPercentage()));
        case "ConcurrencyMode":
            return Optional.ofNullable(clazz.cast(concurrencyModeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackSetOperationPreferences, T> g) {
        return obj -> g.apply((StackSetOperationPreferences) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackSetOperationPreferences> {
        /**
         * <p>
         * The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region at a
         * time.
         * </p>
         * 
         * @param regionConcurrencyType
         *        The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region
         *        at a time.
         * @see RegionConcurrencyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionConcurrencyType
         */
        Builder regionConcurrencyType(String regionConcurrencyType);

        /**
         * <p>
         * The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region at a
         * time.
         * </p>
         * 
         * @param regionConcurrencyType
         *        The concurrency type of deploying StackSets operations in Regions, could be in parallel or one Region
         *        at a time.
         * @see RegionConcurrencyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RegionConcurrencyType
         */
        Builder regionConcurrencyType(RegionConcurrencyType regionConcurrencyType);

        /**
         * <p>
         * The order of the Regions where you want to perform the stack operation.
         * </p>
         * <note>
         * <p>
         * <code>RegionOrder</code> isn't followed if <code>AutoDeployment</code> is enabled.
         * </p>
         * </note>
         * 
         * @param regionOrder
         *        The order of the Regions where you want to perform the stack operation.</p> <note>
         *        <p>
         *        <code>RegionOrder</code> isn't followed if <code>AutoDeployment</code> is enabled.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionOrder(Collection<String> regionOrder);

        /**
         * <p>
         * The order of the Regions where you want to perform the stack operation.
         * </p>
         * <note>
         * <p>
         * <code>RegionOrder</code> isn't followed if <code>AutoDeployment</code> is enabled.
         * </p>
         * </note>
         * 
         * @param regionOrder
         *        The order of the Regions where you want to perform the stack operation.</p> <note>
         *        <p>
         *        <code>RegionOrder</code> isn't followed if <code>AutoDeployment</code> is enabled.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionOrder(String... regionOrder);

        /**
         * <p>
         * The number of accounts, per Region, for which this operation can fail before CloudFormation stops the
         * operation in that Region. If the operation is stopped in a Region, CloudFormation doesn't attempt the
         * operation in any subsequent Regions.
         * </p>
         * <p>
         * Conditional: You must specify either <code>FailureToleranceCount</code> or
         * <code>FailureTolerancePercentage</code> (but not both).
         * </p>
         * <p>
         * By default, <code>0</code> is specified.
         * </p>
         * 
         * @param failureToleranceCount
         *        The number of accounts, per Region, for which this operation can fail before CloudFormation stops the
         *        operation in that Region. If the operation is stopped in a Region, CloudFormation doesn't attempt the
         *        operation in any subsequent Regions.</p>
         *        <p>
         *        Conditional: You must specify either <code>FailureToleranceCount</code> or
         *        <code>FailureTolerancePercentage</code> (but not both).
         *        </p>
         *        <p>
         *        By default, <code>0</code> is specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureToleranceCount(Integer failureToleranceCount);

        /**
         * <p>
         * The percentage of accounts, per Region, for which this stack operation can fail before CloudFormation stops
         * the operation in that Region. If the operation is stopped in a Region, CloudFormation doesn't attempt the
         * operation in any subsequent Regions.
         * </p>
         * <p>
         * When calculating the number of accounts based on the specified percentage, CloudFormation rounds <i>down</i>
         * to the next whole number.
         * </p>
         * <p>
         * Conditional: You must specify either <code>FailureToleranceCount</code> or
         * <code>FailureTolerancePercentage</code>, but not both.
         * </p>
         * <p>
         * By default, <code>0</code> is specified.
         * </p>
         * 
         * @param failureTolerancePercentage
         *        The percentage of accounts, per Region, for which this stack operation can fail before CloudFormation
         *        stops the operation in that Region. If the operation is stopped in a Region, CloudFormation doesn't
         *        attempt the operation in any subsequent Regions.</p>
         *        <p>
         *        When calculating the number of accounts based on the specified percentage, CloudFormation rounds
         *        <i>down</i> to the next whole number.
         *        </p>
         *        <p>
         *        Conditional: You must specify either <code>FailureToleranceCount</code> or
         *        <code>FailureTolerancePercentage</code>, but not both.
         *        </p>
         *        <p>
         *        By default, <code>0</code> is specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureTolerancePercentage(Integer failureTolerancePercentage);

        /**
         * <p>
         * The maximum number of accounts in which to perform this operation at one time. This can depend on the value
         * of <code>FailureToleranceCount</code> depending on your <code>ConcurrencyMode</code>.
         * <code>MaxConcurrentCount</code> is at most one more than the <code>FailureToleranceCount</code> if you're
         * using <code>STRICT_FAILURE_TOLERANCE</code>.
         * </p>
         * <p>
         * Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments, under
         * certain circumstances the actual number of accounts acted upon concurrently may be lower due to service
         * throttling.
         * </p>
         * <p>
         * Conditional: You must specify either <code>MaxConcurrentCount</code> or <code>MaxConcurrentPercentage</code>,
         * but not both.
         * </p>
         * <p>
         * By default, <code>1</code> is specified.
         * </p>
         * 
         * @param maxConcurrentCount
         *        The maximum number of accounts in which to perform this operation at one time. This can depend on the
         *        value of <code>FailureToleranceCount</code> depending on your <code>ConcurrencyMode</code>.
         *        <code>MaxConcurrentCount</code> is at most one more than the <code>FailureToleranceCount</code> if
         *        you're using <code>STRICT_FAILURE_TOLERANCE</code>.</p>
         *        <p>
         *        Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments,
         *        under certain circumstances the actual number of accounts acted upon concurrently may be lower due to
         *        service throttling.
         *        </p>
         *        <p>
         *        Conditional: You must specify either <code>MaxConcurrentCount</code> or
         *        <code>MaxConcurrentPercentage</code>, but not both.
         *        </p>
         *        <p>
         *        By default, <code>1</code> is specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrentCount(Integer maxConcurrentCount);

        /**
         * <p>
         * The maximum percentage of accounts in which to perform this operation at one time.
         * </p>
         * <p>
         * When calculating the number of accounts based on the specified percentage, CloudFormation rounds down to the
         * next whole number. This is true except in cases where rounding down would result is zero. In this case,
         * CloudFormation sets the number as one instead.
         * </p>
         * <p>
         * Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments, under
         * certain circumstances the actual number of accounts acted upon concurrently may be lower due to service
         * throttling.
         * </p>
         * <p>
         * Conditional: You must specify either <code>MaxConcurrentCount</code> or <code>MaxConcurrentPercentage</code>,
         * but not both.
         * </p>
         * <p>
         * By default, <code>1</code> is specified.
         * </p>
         * 
         * @param maxConcurrentPercentage
         *        The maximum percentage of accounts in which to perform this operation at one time.</p>
         *        <p>
         *        When calculating the number of accounts based on the specified percentage, CloudFormation rounds down
         *        to the next whole number. This is true except in cases where rounding down would result is zero. In
         *        this case, CloudFormation sets the number as one instead.
         *        </p>
         *        <p>
         *        Note that this setting lets you specify the <i>maximum</i> for operations. For large deployments,
         *        under certain circumstances the actual number of accounts acted upon concurrently may be lower due to
         *        service throttling.
         *        </p>
         *        <p>
         *        Conditional: You must specify either <code>MaxConcurrentCount</code> or
         *        <code>MaxConcurrentPercentage</code>, but not both.
         *        </p>
         *        <p>
         *        By default, <code>1</code> is specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrentPercentage(Integer maxConcurrentPercentage);

        /**
         * <p>
         * Specifies how the concurrency level behaves during the operation execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure the
         * number of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The initial
         * actual concurrency is set to the lower of either the value of the <code>MaxConcurrentCount</code>, or the
         * value of <code>FailureToleranceCount</code> +1. The actual concurrency is then reduced proportionally by the
         * number of failures. This is the default behavior.
         * </p>
         * <p>
         * If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the actual
         * concurrency. This allows stack set operations to run at the concurrency level set by the
         * <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the number of
         * failures.
         * </p>
         * </li>
         * </ul>
         * 
         * @param concurrencyMode
         *        Specifies how the concurrency level behaves during the operation execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure
         *        the number of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The
         *        initial actual concurrency is set to the lower of either the value of the
         *        <code>MaxConcurrentCount</code>, or the value of <code>FailureToleranceCount</code> +1. The actual
         *        concurrency is then reduced proportionally by the number of failures. This is the default behavior.
         *        </p>
         *        <p>
         *        If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the
         *        actual concurrency. This allows stack set operations to run at the concurrency level set by the
         *        <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the
         *        number of failures.
         *        </p>
         *        </li>
         * @see ConcurrencyMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConcurrencyMode
         */
        Builder concurrencyMode(String concurrencyMode);

        /**
         * <p>
         * Specifies how the concurrency level behaves during the operation execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure the
         * number of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The initial
         * actual concurrency is set to the lower of either the value of the <code>MaxConcurrentCount</code>, or the
         * value of <code>FailureToleranceCount</code> +1. The actual concurrency is then reduced proportionally by the
         * number of failures. This is the default behavior.
         * </p>
         * <p>
         * If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the actual
         * concurrency. This allows stack set operations to run at the concurrency level set by the
         * <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the number of
         * failures.
         * </p>
         * </li>
         * </ul>
         * 
         * @param concurrencyMode
         *        Specifies how the concurrency level behaves during the operation execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>STRICT_FAILURE_TOLERANCE</code>: This option dynamically lowers the concurrency level to ensure
         *        the number of failed accounts never exceeds the value of <code>FailureToleranceCount</code> +1. The
         *        initial actual concurrency is set to the lower of either the value of the
         *        <code>MaxConcurrentCount</code>, or the value of <code>FailureToleranceCount</code> +1. The actual
         *        concurrency is then reduced proportionally by the number of failures. This is the default behavior.
         *        </p>
         *        <p>
         *        If failure tolerance or Maximum concurrent accounts are set to percentages, the behavior is similar.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SOFT_FAILURE_TOLERANCE</code>: This option decouples <code>FailureToleranceCount</code> from the
         *        actual concurrency. This allows stack set operations to run at the concurrency level set by the
         *        <code>MaxConcurrentCount</code> value, or <code>MaxConcurrentPercentage</code>, regardless of the
         *        number of failures.
         *        </p>
         *        </li>
         * @see ConcurrencyMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConcurrencyMode
         */
        Builder concurrencyMode(ConcurrencyMode concurrencyMode);
    }

    static final class BuilderImpl implements Builder {
        private String regionConcurrencyType;

        private List<String> regionOrder = DefaultSdkAutoConstructList.getInstance();

        private Integer failureToleranceCount;

        private Integer failureTolerancePercentage;

        private Integer maxConcurrentCount;

        private Integer maxConcurrentPercentage;

        private String concurrencyMode;

        private BuilderImpl() {
        }

        private BuilderImpl(StackSetOperationPreferences model) {
            regionConcurrencyType(model.regionConcurrencyType);
            regionOrder(model.regionOrder);
            failureToleranceCount(model.failureToleranceCount);
            failureTolerancePercentage(model.failureTolerancePercentage);
            maxConcurrentCount(model.maxConcurrentCount);
            maxConcurrentPercentage(model.maxConcurrentPercentage);
            concurrencyMode(model.concurrencyMode);
        }

        public final String getRegionConcurrencyType() {
            return regionConcurrencyType;
        }

        public final void setRegionConcurrencyType(String regionConcurrencyType) {
            this.regionConcurrencyType = regionConcurrencyType;
        }

        @Override
        public final Builder regionConcurrencyType(String regionConcurrencyType) {
            this.regionConcurrencyType = regionConcurrencyType;
            return this;
        }

        @Override
        public final Builder regionConcurrencyType(RegionConcurrencyType regionConcurrencyType) {
            this.regionConcurrencyType(regionConcurrencyType == null ? null : regionConcurrencyType.toString());
            return this;
        }

        public final Collection<String> getRegionOrder() {
            if (regionOrder instanceof SdkAutoConstructList) {
                return null;
            }
            return regionOrder;
        }

        public final void setRegionOrder(Collection<String> regionOrder) {
            this.regionOrder = RegionListCopier.copy(regionOrder);
        }

        @Override
        public final Builder regionOrder(Collection<String> regionOrder) {
            this.regionOrder = RegionListCopier.copy(regionOrder);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder regionOrder(String... regionOrder) {
            regionOrder(Arrays.asList(regionOrder));
            return this;
        }

        public final Integer getFailureToleranceCount() {
            return failureToleranceCount;
        }

        public final void setFailureToleranceCount(Integer failureToleranceCount) {
            this.failureToleranceCount = failureToleranceCount;
        }

        @Override
        public final Builder failureToleranceCount(Integer failureToleranceCount) {
            this.failureToleranceCount = failureToleranceCount;
            return this;
        }

        public final Integer getFailureTolerancePercentage() {
            return failureTolerancePercentage;
        }

        public final void setFailureTolerancePercentage(Integer failureTolerancePercentage) {
            this.failureTolerancePercentage = failureTolerancePercentage;
        }

        @Override
        public final Builder failureTolerancePercentage(Integer failureTolerancePercentage) {
            this.failureTolerancePercentage = failureTolerancePercentage;
            return this;
        }

        public final Integer getMaxConcurrentCount() {
            return maxConcurrentCount;
        }

        public final void setMaxConcurrentCount(Integer maxConcurrentCount) {
            this.maxConcurrentCount = maxConcurrentCount;
        }

        @Override
        public final Builder maxConcurrentCount(Integer maxConcurrentCount) {
            this.maxConcurrentCount = maxConcurrentCount;
            return this;
        }

        public final Integer getMaxConcurrentPercentage() {
            return maxConcurrentPercentage;
        }

        public final void setMaxConcurrentPercentage(Integer maxConcurrentPercentage) {
            this.maxConcurrentPercentage = maxConcurrentPercentage;
        }

        @Override
        public final Builder maxConcurrentPercentage(Integer maxConcurrentPercentage) {
            this.maxConcurrentPercentage = maxConcurrentPercentage;
            return this;
        }

        public final String getConcurrencyMode() {
            return concurrencyMode;
        }

        public final void setConcurrencyMode(String concurrencyMode) {
            this.concurrencyMode = concurrencyMode;
        }

        @Override
        public final Builder concurrencyMode(String concurrencyMode) {
            this.concurrencyMode = concurrencyMode;
            return this;
        }

        @Override
        public final Builder concurrencyMode(ConcurrencyMode concurrencyMode) {
            this.concurrencyMode(concurrencyMode == null ? null : concurrencyMode.toString());
            return this;
        }

        @Override
        public StackSetOperationPreferences build() {
            return new StackSetOperationPreferences(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
