/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.cloudformation.CloudFormationAsyncClient;
import software.amazon.awssdk.services.cloudformation.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.cloudformation.model.CloudFormationRequest;
import software.amazon.awssdk.services.cloudformation.model.DescribeChangeSetRequest;
import software.amazon.awssdk.services.cloudformation.model.DescribeChangeSetResponse;
import software.amazon.awssdk.services.cloudformation.model.DescribeStacksRequest;
import software.amazon.awssdk.services.cloudformation.model.DescribeStacksResponse;
import software.amazon.awssdk.services.cloudformation.model.DescribeTypeRegistrationRequest;
import software.amazon.awssdk.services.cloudformation.model.DescribeTypeRegistrationResponse;
import software.amazon.awssdk.services.cloudformation.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultCloudFormationAsyncWaiter implements CloudFormationAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final CloudFormationAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeStacksResponse> stackExistsWaiter;

    private final AsyncWaiter<DescribeStacksResponse> stackCreateCompleteWaiter;

    private final AsyncWaiter<DescribeStacksResponse> stackDeleteCompleteWaiter;

    private final AsyncWaiter<DescribeStacksResponse> stackUpdateCompleteWaiter;

    private final AsyncWaiter<DescribeStacksResponse> stackImportCompleteWaiter;

    private final AsyncWaiter<DescribeStacksResponse> stackRollbackCompleteWaiter;

    private final AsyncWaiter<DescribeChangeSetResponse> changeSetCreateCompleteWaiter;

    private final AsyncWaiter<DescribeTypeRegistrationResponse> typeRegistrationCompleteWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultCloudFormationAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = CloudFormationAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.stackExistsWaiter = AsyncWaiter.builder(DescribeStacksResponse.class).acceptors(stackExistsWaiterAcceptors())
                .overrideConfiguration(stackExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.stackCreateCompleteWaiter = AsyncWaiter.builder(DescribeStacksResponse.class)
                .acceptors(stackCreateCompleteWaiterAcceptors())
                .overrideConfiguration(stackCreateCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.stackDeleteCompleteWaiter = AsyncWaiter.builder(DescribeStacksResponse.class)
                .acceptors(stackDeleteCompleteWaiterAcceptors())
                .overrideConfiguration(stackDeleteCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.stackUpdateCompleteWaiter = AsyncWaiter.builder(DescribeStacksResponse.class)
                .acceptors(stackUpdateCompleteWaiterAcceptors())
                .overrideConfiguration(stackUpdateCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.stackImportCompleteWaiter = AsyncWaiter.builder(DescribeStacksResponse.class)
                .acceptors(stackImportCompleteWaiterAcceptors())
                .overrideConfiguration(stackImportCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.stackRollbackCompleteWaiter = AsyncWaiter.builder(DescribeStacksResponse.class)
                .acceptors(stackRollbackCompleteWaiterAcceptors())
                .overrideConfiguration(stackRollbackCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.changeSetCreateCompleteWaiter = AsyncWaiter.builder(DescribeChangeSetResponse.class)
                .acceptors(changeSetCreateCompleteWaiterAcceptors())
                .overrideConfiguration(changeSetCreateCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.typeRegistrationCompleteWaiter = AsyncWaiter.builder(DescribeTypeRegistrationResponse.class)
                .acceptors(typeRegistrationCompleteWaiterAcceptors())
                .overrideConfiguration(typeRegistrationCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChangeSetResponse>> waitUntilChangeSetCreateComplete(
            DescribeChangeSetRequest describeChangeSetRequest) {
        return changeSetCreateCompleteWaiter.runAsync(() -> client
                .describeChangeSet(applyWaitersUserAgent(describeChangeSetRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeChangeSetResponse>> waitUntilChangeSetCreateComplete(
            DescribeChangeSetRequest describeChangeSetRequest, WaiterOverrideConfiguration overrideConfig) {
        return changeSetCreateCompleteWaiter.runAsync(
                () -> client.describeChangeSet(applyWaitersUserAgent(describeChangeSetRequest)),
                changeSetCreateCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackCreateComplete(
            DescribeStacksRequest describeStacksRequest) {
        return stackCreateCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackCreateComplete(
            DescribeStacksRequest describeStacksRequest, WaiterOverrideConfiguration overrideConfig) {
        return stackCreateCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)),
                stackCreateCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackDeleteComplete(
            DescribeStacksRequest describeStacksRequest) {
        return stackDeleteCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackDeleteComplete(
            DescribeStacksRequest describeStacksRequest, WaiterOverrideConfiguration overrideConfig) {
        return stackDeleteCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)),
                stackDeleteCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackExists(
            DescribeStacksRequest describeStacksRequest) {
        return stackExistsWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackExists(
            DescribeStacksRequest describeStacksRequest, WaiterOverrideConfiguration overrideConfig) {
        return stackExistsWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)),
                stackExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackImportComplete(
            DescribeStacksRequest describeStacksRequest) {
        return stackImportCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackImportComplete(
            DescribeStacksRequest describeStacksRequest, WaiterOverrideConfiguration overrideConfig) {
        return stackImportCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)),
                stackImportCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackRollbackComplete(
            DescribeStacksRequest describeStacksRequest) {
        return stackRollbackCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackRollbackComplete(
            DescribeStacksRequest describeStacksRequest, WaiterOverrideConfiguration overrideConfig) {
        return stackRollbackCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)),
                stackRollbackCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackUpdateComplete(
            DescribeStacksRequest describeStacksRequest) {
        return stackUpdateCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeStacksResponse>> waitUntilStackUpdateComplete(
            DescribeStacksRequest describeStacksRequest, WaiterOverrideConfiguration overrideConfig) {
        return stackUpdateCompleteWaiter.runAsync(() -> client.describeStacks(applyWaitersUserAgent(describeStacksRequest)),
                stackUpdateCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTypeRegistrationResponse>> waitUntilTypeRegistrationComplete(
            DescribeTypeRegistrationRequest describeTypeRegistrationRequest) {
        return typeRegistrationCompleteWaiter.runAsync(() -> client
                .describeTypeRegistration(applyWaitersUserAgent(describeTypeRegistrationRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeTypeRegistrationResponse>> waitUntilTypeRegistrationComplete(
            DescribeTypeRegistrationRequest describeTypeRegistrationRequest, WaiterOverrideConfiguration overrideConfig) {
        return typeRegistrationCompleteWaiter.runAsync(
                () -> client.describeTypeRegistration(applyWaitersUserAgent(describeTypeRegistrationRequest)),
                typeRegistrationCompleteWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeStacksResponse>> stackExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeStacksResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationError")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeStacksResponse>> stackCreateCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeStacksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "CREATE_COMPLETE"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_COMPLETE"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_IN_PROGRESS"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_COMPLETE_CLEANUP_IN_PROGRESS"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_FAILED"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_IN_PROGRESS"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_FAILED"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_COMPLETE"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "CREATE_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "DELETE_COMPLETE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=DELETE_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "DELETE_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ROLLBACK_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=ROLLBACK_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ROLLBACK_COMPLETE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=ROLLBACK_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationError"),
                "A waiter acceptor was matched on error condition (ValidationError) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeStacksResponse>> stackDeleteCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeStacksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "DELETE_COMPLETE"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationError")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "DELETE_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "CREATE_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=CREATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ROLLBACK_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=ROLLBACK_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_IN_PROGRESS"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_ROLLBACK_IN_PROGRESS) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_ROLLBACK_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_COMPLETE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_ROLLBACK_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_COMPLETE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_COMPLETE) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeStacksResponse>> stackUpdateCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeStacksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_COMPLETE"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_ROLLBACK_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_COMPLETE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_ROLLBACK_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationError"),
                "A waiter acceptor was matched on error condition (ValidationError) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeStacksResponse>> stackImportCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeStacksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "IMPORT_COMPLETE"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ROLLBACK_COMPLETE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=ROLLBACK_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "ROLLBACK_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=ROLLBACK_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "IMPORT_ROLLBACK_IN_PROGRESS"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=IMPORT_ROLLBACK_IN_PROGRESS) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "IMPORT_ROLLBACK_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=IMPORT_ROLLBACK_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "IMPORT_ROLLBACK_COMPLETE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=IMPORT_ROLLBACK_COMPLETE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationError"),
                "A waiter acceptor was matched on error condition (ValidationError) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeStacksResponse>> stackRollbackCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeStacksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_COMPLETE"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "UPDATE_ROLLBACK_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=UPDATE_ROLLBACK_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("Stacks").flatten().field("StackStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "DELETE_FAILED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (Stacks[].StackStatus=DELETE_FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationError"),
                "A waiter acceptor was matched on error condition (ValidationError) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeChangeSetResponse>> changeSetCreateCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeChangeSetResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "CREATE_COMPLETE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("Status").value(), "FAILED");
        },
                "A waiter acceptor with the matcher (path) was matched on parameter (Status=FAILED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "ValidationError"),
                "A waiter acceptor was matched on error condition (ValidationError) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTypeRegistrationResponse>> typeRegistrationCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTypeRegistrationResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("ProgressStatus").value(), "COMPLETE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("ProgressStatus").value(), "FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (ProgressStatus=FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration stackExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(20);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration stackCreateCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration stackDeleteCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration stackUpdateCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration stackImportCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration stackRollbackCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration changeSetCreateCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration typeRegistrationCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(120);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static CloudFormationAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends CloudFormationRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements CloudFormationAsyncWaiter.Builder {
        private CloudFormationAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public CloudFormationAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public CloudFormationAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public CloudFormationAsyncWaiter.Builder client(CloudFormationAsyncClient client) {
            this.client = client;
            return this;
        }

        public CloudFormationAsyncWaiter build() {
            return new DefaultCloudFormationAsyncWaiter(this);
        }
    }
}
