/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The <code>ResourceChange</code> structure describes the resource and the action that CloudFormation will perform on
 * it if you execute this change set.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceChange implements SdkPojo, Serializable, ToCopyableBuilder<ResourceChange.Builder, ResourceChange> {
    private static final SdkField<String> POLICY_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyAction").getter(getter(ResourceChange::policyActionAsString))
            .setter(setter(Builder::policyAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyAction").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(ResourceChange::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<String> LOGICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogicalResourceId").getter(getter(ResourceChange::logicalResourceId))
            .setter(setter(Builder::logicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalResourceId").build()).build();

    private static final SdkField<String> PHYSICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhysicalResourceId").getter(getter(ResourceChange::physicalResourceId))
            .setter(setter(Builder::physicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhysicalResourceId").build())
            .build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ResourceChange::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> REPLACEMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Replacement").getter(getter(ResourceChange::replacementAsString)).setter(setter(Builder::replacement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Replacement").build()).build();

    private static final SdkField<List<String>> SCOPE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Scope")
            .getter(getter(ResourceChange::scopeAsStrings))
            .setter(setter(Builder::scopeWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scope").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<ResourceChangeDetail>> DETAILS_FIELD = SdkField
            .<List<ResourceChangeDetail>> builder(MarshallingType.LIST)
            .memberName("Details")
            .getter(getter(ResourceChange::details))
            .setter(setter(Builder::details))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Details").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceChangeDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceChangeDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CHANGE_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChangeSetId").getter(getter(ResourceChange::changeSetId)).setter(setter(Builder::changeSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeSetId").build()).build();

    private static final SdkField<ModuleInfo> MODULE_INFO_FIELD = SdkField.<ModuleInfo> builder(MarshallingType.SDK_POJO)
            .memberName("ModuleInfo").getter(getter(ResourceChange::moduleInfo)).setter(setter(Builder::moduleInfo))
            .constructor(ModuleInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModuleInfo").build()).build();

    private static final SdkField<String> BEFORE_CONTEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BeforeContext").getter(getter(ResourceChange::beforeContext)).setter(setter(Builder::beforeContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BeforeContext").build()).build();

    private static final SdkField<String> AFTER_CONTEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AfterContext").getter(getter(ResourceChange::afterContext)).setter(setter(Builder::afterContext))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AfterContext").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_ACTION_FIELD,
            ACTION_FIELD, LOGICAL_RESOURCE_ID_FIELD, PHYSICAL_RESOURCE_ID_FIELD, RESOURCE_TYPE_FIELD, REPLACEMENT_FIELD,
            SCOPE_FIELD, DETAILS_FIELD, CHANGE_SET_ID_FIELD, MODULE_INFO_FIELD, BEFORE_CONTEXT_FIELD, AFTER_CONTEXT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("PolicyAction", POLICY_ACTION_FIELD);
                    put("Action", ACTION_FIELD);
                    put("LogicalResourceId", LOGICAL_RESOURCE_ID_FIELD);
                    put("PhysicalResourceId", PHYSICAL_RESOURCE_ID_FIELD);
                    put("ResourceType", RESOURCE_TYPE_FIELD);
                    put("Replacement", REPLACEMENT_FIELD);
                    put("Scope", SCOPE_FIELD);
                    put("Details", DETAILS_FIELD);
                    put("ChangeSetId", CHANGE_SET_ID_FIELD);
                    put("ModuleInfo", MODULE_INFO_FIELD);
                    put("BeforeContext", BEFORE_CONTEXT_FIELD);
                    put("AfterContext", AFTER_CONTEXT_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String policyAction;

    private final String action;

    private final String logicalResourceId;

    private final String physicalResourceId;

    private final String resourceType;

    private final String replacement;

    private final List<String> scope;

    private final List<ResourceChangeDetail> details;

    private final String changeSetId;

    private final ModuleInfo moduleInfo;

    private final String beforeContext;

    private final String afterContext;

    private ResourceChange(BuilderImpl builder) {
        this.policyAction = builder.policyAction;
        this.action = builder.action;
        this.logicalResourceId = builder.logicalResourceId;
        this.physicalResourceId = builder.physicalResourceId;
        this.resourceType = builder.resourceType;
        this.replacement = builder.replacement;
        this.scope = builder.scope;
        this.details = builder.details;
        this.changeSetId = builder.changeSetId;
        this.moduleInfo = builder.moduleInfo;
        this.beforeContext = builder.beforeContext;
        this.afterContext = builder.afterContext;
    }

    /**
     * <p>
     * The action that will be taken on the physical resource when the change set is executed.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Delete</code> The resource will be deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Retain</code> The resource will be retained.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Snapshot</code> The resource will have a snapshot taken.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyAction} will
     * return {@link PolicyAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyActionAsString}.
     * </p>
     * 
     * @return The action that will be taken on the physical resource when the change set is executed.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Delete</code> The resource will be deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Retain</code> The resource will be retained.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Snapshot</code> The resource will have a snapshot taken.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
     *         </p>
     *         </li>
     * @see PolicyAction
     */
    public final PolicyAction policyAction() {
        return PolicyAction.fromValue(policyAction);
    }

    /**
     * <p>
     * The action that will be taken on the physical resource when the change set is executed.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Delete</code> The resource will be deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Retain</code> The resource will be retained.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Snapshot</code> The resource will have a snapshot taken.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyAction} will
     * return {@link PolicyAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyActionAsString}.
     * </p>
     * 
     * @return The action that will be taken on the physical resource when the change set is executed.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Delete</code> The resource will be deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Retain</code> The resource will be retained.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Snapshot</code> The resource will have a snapshot taken.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
     *         </p>
     *         </li>
     * @see PolicyAction
     */
    public final String policyActionAsString() {
        return policyAction;
    }

    /**
     * <p>
     * The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
     * <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource), <code>Import</code> (imports
     * a resource), or <code>Dynamic</code> (exact action for the resource can't be determined).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link ChangeAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
     *         <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource), <code>Import</code>
     *         (imports a resource), or <code>Dynamic</code> (exact action for the resource can't be determined).
     * @see ChangeAction
     */
    public final ChangeAction action() {
        return ChangeAction.fromValue(action);
    }

    /**
     * <p>
     * The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
     * <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource), <code>Import</code> (imports
     * a resource), or <code>Dynamic</code> (exact action for the resource can't be determined).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link ChangeAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
     *         <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource), <code>Import</code>
     *         (imports a resource), or <code>Dynamic</code> (exact action for the resource can't be determined).
     * @see ChangeAction
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The resource's logical ID, which is defined in the stack's template.
     * </p>
     * 
     * @return The resource's logical ID, which is defined in the stack's template.
     */
    public final String logicalResourceId() {
        return logicalResourceId;
    }

    /**
     * <p>
     * The resource's physical ID (resource name). Resources that you are adding don't have physical IDs because they
     * haven't been created.
     * </p>
     * 
     * @return The resource's physical ID (resource name). Resources that you are adding don't have physical IDs because
     *         they haven't been created.
     */
    public final String physicalResourceId() {
        return physicalResourceId;
    }

    /**
     * <p>
     * The type of CloudFormation resource, such as <code>AWS::S3::Bucket</code>.
     * </p>
     * 
     * @return The type of CloudFormation resource, such as <code>AWS::S3::Bucket</code>.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by creating a new
     * one and deleting the old one. This value depends on the value of the <code>RequiresRecreation</code> property in
     * the <code>ResourceTargetDefinition</code> structure. For example, if the <code>RequiresRecreation</code> field is
     * <code>Always</code> and the <code>Evaluation</code> field is <code>Static</code>, <code>Replacement</code> is
     * <code>True</code>. If the <code>RequiresRecreation</code> field is <code>Always</code> and the
     * <code>Evaluation</code> field is <code>Dynamic</code>, <code>Replacement</code> is <code>Conditional</code>.
     * </p>
     * <p>
     * If you have multiple changes with different <code>RequiresRecreation</code> values, the <code>Replacement</code>
     * value depends on the change with the most impact. A <code>RequiresRecreation</code> value of <code>Always</code>
     * has the most impact, followed by <code>Conditional</code>, and then <code>Never</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replacement} will
     * return {@link Replacement#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #replacementAsString}.
     * </p>
     * 
     * @return For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by
     *         creating a new one and deleting the old one. This value depends on the value of the
     *         <code>RequiresRecreation</code> property in the <code>ResourceTargetDefinition</code> structure. For
     *         example, if the <code>RequiresRecreation</code> field is <code>Always</code> and the
     *         <code>Evaluation</code> field is <code>Static</code>, <code>Replacement</code> is <code>True</code>. If
     *         the <code>RequiresRecreation</code> field is <code>Always</code> and the <code>Evaluation</code> field is
     *         <code>Dynamic</code>, <code>Replacement</code> is <code>Conditional</code>.</p>
     *         <p>
     *         If you have multiple changes with different <code>RequiresRecreation</code> values, the
     *         <code>Replacement</code> value depends on the change with the most impact. A
     *         <code>RequiresRecreation</code> value of <code>Always</code> has the most impact, followed by
     *         <code>Conditional</code>, and then <code>Never</code>.
     * @see Replacement
     */
    public final Replacement replacement() {
        return Replacement.fromValue(replacement);
    }

    /**
     * <p>
     * For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by creating a new
     * one and deleting the old one. This value depends on the value of the <code>RequiresRecreation</code> property in
     * the <code>ResourceTargetDefinition</code> structure. For example, if the <code>RequiresRecreation</code> field is
     * <code>Always</code> and the <code>Evaluation</code> field is <code>Static</code>, <code>Replacement</code> is
     * <code>True</code>. If the <code>RequiresRecreation</code> field is <code>Always</code> and the
     * <code>Evaluation</code> field is <code>Dynamic</code>, <code>Replacement</code> is <code>Conditional</code>.
     * </p>
     * <p>
     * If you have multiple changes with different <code>RequiresRecreation</code> values, the <code>Replacement</code>
     * value depends on the change with the most impact. A <code>RequiresRecreation</code> value of <code>Always</code>
     * has the most impact, followed by <code>Conditional</code>, and then <code>Never</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replacement} will
     * return {@link Replacement#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #replacementAsString}.
     * </p>
     * 
     * @return For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by
     *         creating a new one and deleting the old one. This value depends on the value of the
     *         <code>RequiresRecreation</code> property in the <code>ResourceTargetDefinition</code> structure. For
     *         example, if the <code>RequiresRecreation</code> field is <code>Always</code> and the
     *         <code>Evaluation</code> field is <code>Static</code>, <code>Replacement</code> is <code>True</code>. If
     *         the <code>RequiresRecreation</code> field is <code>Always</code> and the <code>Evaluation</code> field is
     *         <code>Dynamic</code>, <code>Replacement</code> is <code>Conditional</code>.</p>
     *         <p>
     *         If you have multiple changes with different <code>RequiresRecreation</code> values, the
     *         <code>Replacement</code> value depends on the change with the most impact. A
     *         <code>RequiresRecreation</code> value of <code>Always</code> has the most impact, followed by
     *         <code>Conditional</code>, and then <code>Never</code>.
     * @see Replacement
     */
    public final String replacementAsString() {
        return replacement;
    }

    /**
     * <p>
     * For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as a
     * change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScope} method.
     * </p>
     * 
     * @return For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as
     *         a change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>
     *         .
     */
    public final List<ResourceAttribute> scope() {
        return ScopeCopier.copyStringToEnum(scope);
    }

    /**
     * For responses, this returns true if the service returned a value for the Scope property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasScope() {
        return scope != null && !(scope instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as a
     * change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScope} method.
     * </p>
     * 
     * @return For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as
     *         a change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>
     *         .
     */
    public final List<String> scopeAsStrings() {
        return scope;
    }

    /**
     * For responses, this returns true if the service returned a value for the Details property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDetails() {
        return details != null && !(details instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For the <code>Modify</code> action, a list of <code>ResourceChangeDetail</code> structures that describes the
     * changes that CloudFormation will make to the resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDetails} method.
     * </p>
     * 
     * @return For the <code>Modify</code> action, a list of <code>ResourceChangeDetail</code> structures that describes
     *         the changes that CloudFormation will make to the resource.
     */
    public final List<ResourceChangeDetail> details() {
        return details;
    }

    /**
     * <p>
     * The change set ID of the nested change set.
     * </p>
     * 
     * @return The change set ID of the nested change set.
     */
    public final String changeSetId() {
        return changeSetId;
    }

    /**
     * <p>
     * Contains information about the module from which the resource was created, if the resource was created from a
     * module included in the stack template.
     * </p>
     * 
     * @return Contains information about the module from which the resource was created, if the resource was created
     *         from a module included in the stack template.
     */
    public final ModuleInfo moduleInfo() {
        return moduleInfo;
    }

    /**
     * <p>
     * An encoded JSON string containing the context of the resource before the change is executed.
     * </p>
     * 
     * @return An encoded JSON string containing the context of the resource before the change is executed.
     */
    public final String beforeContext() {
        return beforeContext;
    }

    /**
     * <p>
     * An encoded JSON string containing the context of the resource after the change is executed.
     * </p>
     * 
     * @return An encoded JSON string containing the context of the resource after the change is executed.
     */
    public final String afterContext() {
        return afterContext;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(policyActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(logicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(physicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(replacementAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasScope() ? scopeAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDetails() ? details() : null);
        hashCode = 31 * hashCode + Objects.hashCode(changeSetId());
        hashCode = 31 * hashCode + Objects.hashCode(moduleInfo());
        hashCode = 31 * hashCode + Objects.hashCode(beforeContext());
        hashCode = 31 * hashCode + Objects.hashCode(afterContext());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceChange)) {
            return false;
        }
        ResourceChange other = (ResourceChange) obj;
        return Objects.equals(policyActionAsString(), other.policyActionAsString())
                && Objects.equals(actionAsString(), other.actionAsString())
                && Objects.equals(logicalResourceId(), other.logicalResourceId())
                && Objects.equals(physicalResourceId(), other.physicalResourceId())
                && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(replacementAsString(), other.replacementAsString()) && hasScope() == other.hasScope()
                && Objects.equals(scopeAsStrings(), other.scopeAsStrings()) && hasDetails() == other.hasDetails()
                && Objects.equals(details(), other.details()) && Objects.equals(changeSetId(), other.changeSetId())
                && Objects.equals(moduleInfo(), other.moduleInfo()) && Objects.equals(beforeContext(), other.beforeContext())
                && Objects.equals(afterContext(), other.afterContext());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceChange").add("PolicyAction", policyActionAsString()).add("Action", actionAsString())
                .add("LogicalResourceId", logicalResourceId()).add("PhysicalResourceId", physicalResourceId())
                .add("ResourceType", resourceType()).add("Replacement", replacementAsString())
                .add("Scope", hasScope() ? scopeAsStrings() : null).add("Details", hasDetails() ? details() : null)
                .add("ChangeSetId", changeSetId()).add("ModuleInfo", moduleInfo()).add("BeforeContext", beforeContext())
                .add("AfterContext", afterContext()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PolicyAction":
            return Optional.ofNullable(clazz.cast(policyActionAsString()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "LogicalResourceId":
            return Optional.ofNullable(clazz.cast(logicalResourceId()));
        case "PhysicalResourceId":
            return Optional.ofNullable(clazz.cast(physicalResourceId()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "Replacement":
            return Optional.ofNullable(clazz.cast(replacementAsString()));
        case "Scope":
            return Optional.ofNullable(clazz.cast(scopeAsStrings()));
        case "Details":
            return Optional.ofNullable(clazz.cast(details()));
        case "ChangeSetId":
            return Optional.ofNullable(clazz.cast(changeSetId()));
        case "ModuleInfo":
            return Optional.ofNullable(clazz.cast(moduleInfo()));
        case "BeforeContext":
            return Optional.ofNullable(clazz.cast(beforeContext()));
        case "AfterContext":
            return Optional.ofNullable(clazz.cast(afterContext()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ResourceChange, T> g) {
        return obj -> g.apply((ResourceChange) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceChange> {
        /**
         * <p>
         * The action that will be taken on the physical resource when the change set is executed.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Delete</code> The resource will be deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Retain</code> The resource will be retained.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Snapshot</code> The resource will have a snapshot taken.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyAction
         *        The action that will be taken on the physical resource when the change set is executed.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Delete</code> The resource will be deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Retain</code> The resource will be retained.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Snapshot</code> The resource will have a snapshot taken.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
         *        </p>
         *        </li>
         * @see PolicyAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyAction
         */
        Builder policyAction(String policyAction);

        /**
         * <p>
         * The action that will be taken on the physical resource when the change set is executed.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Delete</code> The resource will be deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Retain</code> The resource will be retained.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Snapshot</code> The resource will have a snapshot taken.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
         * </p>
         * </li>
         * </ul>
         * 
         * @param policyAction
         *        The action that will be taken on the physical resource when the change set is executed.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Delete</code> The resource will be deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Retain</code> The resource will be retained.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Snapshot</code> The resource will have a snapshot taken.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReplaceAndDelete</code> The resource will be replaced and then deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReplaceAndRetain</code> The resource will be replaced and then retained.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ReplaceAndSnapshot</code> The resource will be replaced and then have a snapshot taken.
         *        </p>
         *        </li>
         * @see PolicyAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyAction
         */
        Builder policyAction(PolicyAction policyAction);

        /**
         * <p>
         * The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
         * <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource), <code>Import</code>
         * (imports a resource), or <code>Dynamic</code> (exact action for the resource can't be determined).
         * </p>
         * 
         * @param action
         *        The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
         *        <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource),
         *        <code>Import</code> (imports a resource), or <code>Dynamic</code> (exact action for the resource can't
         *        be determined).
         * @see ChangeAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeAction
         */
        Builder action(String action);

        /**
         * <p>
         * The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
         * <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource), <code>Import</code>
         * (imports a resource), or <code>Dynamic</code> (exact action for the resource can't be determined).
         * </p>
         * 
         * @param action
         *        The action that CloudFormation takes on the resource, such as <code>Add</code> (adds a new resource),
         *        <code>Modify</code> (changes a resource), <code>Remove</code> (deletes a resource),
         *        <code>Import</code> (imports a resource), or <code>Dynamic</code> (exact action for the resource can't
         *        be determined).
         * @see ChangeAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeAction
         */
        Builder action(ChangeAction action);

        /**
         * <p>
         * The resource's logical ID, which is defined in the stack's template.
         * </p>
         * 
         * @param logicalResourceId
         *        The resource's logical ID, which is defined in the stack's template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceId(String logicalResourceId);

        /**
         * <p>
         * The resource's physical ID (resource name). Resources that you are adding don't have physical IDs because
         * they haven't been created.
         * </p>
         * 
         * @param physicalResourceId
         *        The resource's physical ID (resource name). Resources that you are adding don't have physical IDs
         *        because they haven't been created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceId(String physicalResourceId);

        /**
         * <p>
         * The type of CloudFormation resource, such as <code>AWS::S3::Bucket</code>.
         * </p>
         * 
         * @param resourceType
         *        The type of CloudFormation resource, such as <code>AWS::S3::Bucket</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by creating a
         * new one and deleting the old one. This value depends on the value of the <code>RequiresRecreation</code>
         * property in the <code>ResourceTargetDefinition</code> structure. For example, if the
         * <code>RequiresRecreation</code> field is <code>Always</code> and the <code>Evaluation</code> field is
         * <code>Static</code>, <code>Replacement</code> is <code>True</code>. If the <code>RequiresRecreation</code>
         * field is <code>Always</code> and the <code>Evaluation</code> field is <code>Dynamic</code>,
         * <code>Replacement</code> is <code>Conditional</code>.
         * </p>
         * <p>
         * If you have multiple changes with different <code>RequiresRecreation</code> values, the
         * <code>Replacement</code> value depends on the change with the most impact. A <code>RequiresRecreation</code>
         * value of <code>Always</code> has the most impact, followed by <code>Conditional</code>, and then
         * <code>Never</code>.
         * </p>
         * 
         * @param replacement
         *        For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by
         *        creating a new one and deleting the old one. This value depends on the value of the
         *        <code>RequiresRecreation</code> property in the <code>ResourceTargetDefinition</code> structure. For
         *        example, if the <code>RequiresRecreation</code> field is <code>Always</code> and the
         *        <code>Evaluation</code> field is <code>Static</code>, <code>Replacement</code> is <code>True</code>.
         *        If the <code>RequiresRecreation</code> field is <code>Always</code> and the <code>Evaluation</code>
         *        field is <code>Dynamic</code>, <code>Replacement</code> is <code>Conditional</code>.</p>
         *        <p>
         *        If you have multiple changes with different <code>RequiresRecreation</code> values, the
         *        <code>Replacement</code> value depends on the change with the most impact. A
         *        <code>RequiresRecreation</code> value of <code>Always</code> has the most impact, followed by
         *        <code>Conditional</code>, and then <code>Never</code>.
         * @see Replacement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Replacement
         */
        Builder replacement(String replacement);

        /**
         * <p>
         * For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by creating a
         * new one and deleting the old one. This value depends on the value of the <code>RequiresRecreation</code>
         * property in the <code>ResourceTargetDefinition</code> structure. For example, if the
         * <code>RequiresRecreation</code> field is <code>Always</code> and the <code>Evaluation</code> field is
         * <code>Static</code>, <code>Replacement</code> is <code>True</code>. If the <code>RequiresRecreation</code>
         * field is <code>Always</code> and the <code>Evaluation</code> field is <code>Dynamic</code>,
         * <code>Replacement</code> is <code>Conditional</code>.
         * </p>
         * <p>
         * If you have multiple changes with different <code>RequiresRecreation</code> values, the
         * <code>Replacement</code> value depends on the change with the most impact. A <code>RequiresRecreation</code>
         * value of <code>Always</code> has the most impact, followed by <code>Conditional</code>, and then
         * <code>Never</code>.
         * </p>
         * 
         * @param replacement
         *        For the <code>Modify</code> action, indicates whether CloudFormation will replace the resource by
         *        creating a new one and deleting the old one. This value depends on the value of the
         *        <code>RequiresRecreation</code> property in the <code>ResourceTargetDefinition</code> structure. For
         *        example, if the <code>RequiresRecreation</code> field is <code>Always</code> and the
         *        <code>Evaluation</code> field is <code>Static</code>, <code>Replacement</code> is <code>True</code>.
         *        If the <code>RequiresRecreation</code> field is <code>Always</code> and the <code>Evaluation</code>
         *        field is <code>Dynamic</code>, <code>Replacement</code> is <code>Conditional</code>.</p>
         *        <p>
         *        If you have multiple changes with different <code>RequiresRecreation</code> values, the
         *        <code>Replacement</code> value depends on the change with the most impact. A
         *        <code>RequiresRecreation</code> value of <code>Always</code> has the most impact, followed by
         *        <code>Conditional</code>, and then <code>Never</code>.
         * @see Replacement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Replacement
         */
        Builder replacement(Replacement replacement);

        /**
         * <p>
         * For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as a
         * change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>.
         * </p>
         * 
         * @param scope
         *        For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such
         *        as a change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or
         *        <code>Tags</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeWithStrings(Collection<String> scope);

        /**
         * <p>
         * For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as a
         * change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>.
         * </p>
         * 
         * @param scope
         *        For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such
         *        as a change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or
         *        <code>Tags</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopeWithStrings(String... scope);

        /**
         * <p>
         * For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as a
         * change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>.
         * </p>
         * 
         * @param scope
         *        For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such
         *        as a change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or
         *        <code>Tags</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scope(Collection<ResourceAttribute> scope);

        /**
         * <p>
         * For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such as a
         * change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or <code>Tags</code>.
         * </p>
         * 
         * @param scope
         *        For the <code>Modify</code> action, indicates which resource attribute is triggering this update, such
         *        as a change in the resource attribute's <code>Metadata</code>, <code>Properties</code>, or
         *        <code>Tags</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scope(ResourceAttribute... scope);

        /**
         * <p>
         * For the <code>Modify</code> action, a list of <code>ResourceChangeDetail</code> structures that describes the
         * changes that CloudFormation will make to the resource.
         * </p>
         * 
         * @param details
         *        For the <code>Modify</code> action, a list of <code>ResourceChangeDetail</code> structures that
         *        describes the changes that CloudFormation will make to the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(Collection<ResourceChangeDetail> details);

        /**
         * <p>
         * For the <code>Modify</code> action, a list of <code>ResourceChangeDetail</code> structures that describes the
         * changes that CloudFormation will make to the resource.
         * </p>
         * 
         * @param details
         *        For the <code>Modify</code> action, a list of <code>ResourceChangeDetail</code> structures that
         *        describes the changes that CloudFormation will make to the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(ResourceChangeDetail... details);

        /**
         * <p>
         * For the <code>Modify</code> action, a list of <code>ResourceChangeDetail</code> structures that describes the
         * changes that CloudFormation will make to the resource.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudformation.model.ResourceChangeDetail.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.cloudformation.model.ResourceChangeDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudformation.model.ResourceChangeDetail.Builder#build()} is called
         * immediately and its result is passed to {@link #details(List<ResourceChangeDetail>)}.
         * 
         * @param details
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudformation.model.ResourceChangeDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #details(java.util.Collection<ResourceChangeDetail>)
         */
        Builder details(Consumer<ResourceChangeDetail.Builder>... details);

        /**
         * <p>
         * The change set ID of the nested change set.
         * </p>
         * 
         * @param changeSetId
         *        The change set ID of the nested change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeSetId(String changeSetId);

        /**
         * <p>
         * Contains information about the module from which the resource was created, if the resource was created from a
         * module included in the stack template.
         * </p>
         * 
         * @param moduleInfo
         *        Contains information about the module from which the resource was created, if the resource was created
         *        from a module included in the stack template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder moduleInfo(ModuleInfo moduleInfo);

        /**
         * <p>
         * Contains information about the module from which the resource was created, if the resource was created from a
         * module included in the stack template.
         * </p>
         * This is a convenience method that creates an instance of the {@link ModuleInfo.Builder} avoiding the need to
         * create one manually via {@link ModuleInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ModuleInfo.Builder#build()} is called immediately and its result
         * is passed to {@link #moduleInfo(ModuleInfo)}.
         * 
         * @param moduleInfo
         *        a consumer that will call methods on {@link ModuleInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #moduleInfo(ModuleInfo)
         */
        default Builder moduleInfo(Consumer<ModuleInfo.Builder> moduleInfo) {
            return moduleInfo(ModuleInfo.builder().applyMutation(moduleInfo).build());
        }

        /**
         * <p>
         * An encoded JSON string containing the context of the resource before the change is executed.
         * </p>
         * 
         * @param beforeContext
         *        An encoded JSON string containing the context of the resource before the change is executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder beforeContext(String beforeContext);

        /**
         * <p>
         * An encoded JSON string containing the context of the resource after the change is executed.
         * </p>
         * 
         * @param afterContext
         *        An encoded JSON string containing the context of the resource after the change is executed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder afterContext(String afterContext);
    }

    static final class BuilderImpl implements Builder {
        private String policyAction;

        private String action;

        private String logicalResourceId;

        private String physicalResourceId;

        private String resourceType;

        private String replacement;

        private List<String> scope = DefaultSdkAutoConstructList.getInstance();

        private List<ResourceChangeDetail> details = DefaultSdkAutoConstructList.getInstance();

        private String changeSetId;

        private ModuleInfo moduleInfo;

        private String beforeContext;

        private String afterContext;

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceChange model) {
            policyAction(model.policyAction);
            action(model.action);
            logicalResourceId(model.logicalResourceId);
            physicalResourceId(model.physicalResourceId);
            resourceType(model.resourceType);
            replacement(model.replacement);
            scopeWithStrings(model.scope);
            details(model.details);
            changeSetId(model.changeSetId);
            moduleInfo(model.moduleInfo);
            beforeContext(model.beforeContext);
            afterContext(model.afterContext);
        }

        public final String getPolicyAction() {
            return policyAction;
        }

        public final void setPolicyAction(String policyAction) {
            this.policyAction = policyAction;
        }

        @Override
        public final Builder policyAction(String policyAction) {
            this.policyAction = policyAction;
            return this;
        }

        @Override
        public final Builder policyAction(PolicyAction policyAction) {
            this.policyAction(policyAction == null ? null : policyAction.toString());
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(ChangeAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final String getLogicalResourceId() {
            return logicalResourceId;
        }

        public final void setLogicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
        }

        @Override
        public final Builder logicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
            return this;
        }

        public final String getPhysicalResourceId() {
            return physicalResourceId;
        }

        public final void setPhysicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
        }

        @Override
        public final Builder physicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final String getReplacement() {
            return replacement;
        }

        public final void setReplacement(String replacement) {
            this.replacement = replacement;
        }

        @Override
        public final Builder replacement(String replacement) {
            this.replacement = replacement;
            return this;
        }

        @Override
        public final Builder replacement(Replacement replacement) {
            this.replacement(replacement == null ? null : replacement.toString());
            return this;
        }

        public final Collection<String> getScope() {
            if (scope instanceof SdkAutoConstructList) {
                return null;
            }
            return scope;
        }

        public final void setScope(Collection<String> scope) {
            this.scope = ScopeCopier.copy(scope);
        }

        @Override
        public final Builder scopeWithStrings(Collection<String> scope) {
            this.scope = ScopeCopier.copy(scope);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scopeWithStrings(String... scope) {
            scopeWithStrings(Arrays.asList(scope));
            return this;
        }

        @Override
        public final Builder scope(Collection<ResourceAttribute> scope) {
            this.scope = ScopeCopier.copyEnumToString(scope);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scope(ResourceAttribute... scope) {
            scope(Arrays.asList(scope));
            return this;
        }

        public final List<ResourceChangeDetail.Builder> getDetails() {
            List<ResourceChangeDetail.Builder> result = ResourceChangeDetailsCopier.copyToBuilder(this.details);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDetails(Collection<ResourceChangeDetail.BuilderImpl> details) {
            this.details = ResourceChangeDetailsCopier.copyFromBuilder(details);
        }

        @Override
        public final Builder details(Collection<ResourceChangeDetail> details) {
            this.details = ResourceChangeDetailsCopier.copy(details);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder details(ResourceChangeDetail... details) {
            details(Arrays.asList(details));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder details(Consumer<ResourceChangeDetail.Builder>... details) {
            details(Stream.of(details).map(c -> ResourceChangeDetail.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getChangeSetId() {
            return changeSetId;
        }

        public final void setChangeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
        }

        @Override
        public final Builder changeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
            return this;
        }

        public final ModuleInfo.Builder getModuleInfo() {
            return moduleInfo != null ? moduleInfo.toBuilder() : null;
        }

        public final void setModuleInfo(ModuleInfo.BuilderImpl moduleInfo) {
            this.moduleInfo = moduleInfo != null ? moduleInfo.build() : null;
        }

        @Override
        public final Builder moduleInfo(ModuleInfo moduleInfo) {
            this.moduleInfo = moduleInfo;
            return this;
        }

        public final String getBeforeContext() {
            return beforeContext;
        }

        public final void setBeforeContext(String beforeContext) {
            this.beforeContext = beforeContext;
        }

        @Override
        public final Builder beforeContext(String beforeContext) {
            this.beforeContext = beforeContext;
            return this;
        }

        public final String getAfterContext() {
            return afterContext;
        }

        public final void setAfterContext(String afterContext) {
            this.afterContext = afterContext;
        }

        @Override
        public final Builder afterContext(String afterContext) {
            this.afterContext = afterContext;
            return this;
        }

        @Override
        public ResourceChange build() {
            return new ResourceChange(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
