/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An AWS CloudFormation stack, in a specific account and region, that's part of a stack set operation. A stack instance
 * is a reference to an attempted or actual stack in a given account within a given region. A stack instance can exist
 * without a stack—for example, if the stack couldn't be created for some reason. A stack instance is associated with
 * only one stack set. Each stack instance contains the ID of its associated stack set, as well as the ID of the actual
 * stack and the stack status.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackInstance implements SdkPojo, Serializable, ToCopyableBuilder<StackInstance.Builder, StackInstance> {
    private static final SdkField<String> STACK_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StackInstance::stackSetId)).setter(setter(Builder::stackSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackSetId").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StackInstance::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<String> ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StackInstance::account)).setter(setter(Builder::account))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Account").build()).build();

    private static final SdkField<String> STACK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StackInstance::stackId)).setter(setter(Builder::stackId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackId").build()).build();

    private static final SdkField<List<Parameter>> PARAMETER_OVERRIDES_FIELD = SdkField
            .<List<Parameter>> builder(MarshallingType.LIST)
            .getter(getter(StackInstance::parameterOverrides))
            .setter(setter(Builder::parameterOverrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParameterOverrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Parameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StackInstance::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StackInstance::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_SET_ID_FIELD,
            REGION_FIELD, ACCOUNT_FIELD, STACK_ID_FIELD, PARAMETER_OVERRIDES_FIELD, STATUS_FIELD, STATUS_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackSetId;

    private final String region;

    private final String account;

    private final String stackId;

    private final List<Parameter> parameterOverrides;

    private final String status;

    private final String statusReason;

    private StackInstance(BuilderImpl builder) {
        this.stackSetId = builder.stackSetId;
        this.region = builder.region;
        this.account = builder.account;
        this.stackId = builder.stackId;
        this.parameterOverrides = builder.parameterOverrides;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
    }

    /**
     * <p>
     * The name or unique ID of the stack set that the stack instance is associated with.
     * </p>
     * 
     * @return The name or unique ID of the stack set that the stack instance is associated with.
     */
    public String stackSetId() {
        return stackSetId;
    }

    /**
     * <p>
     * The name of the AWS region that the stack instance is associated with.
     * </p>
     * 
     * @return The name of the AWS region that the stack instance is associated with.
     */
    public String region() {
        return region;
    }

    /**
     * <p>
     * The name of the AWS account that the stack instance is associated with.
     * </p>
     * 
     * @return The name of the AWS account that the stack instance is associated with.
     */
    public String account() {
        return account;
    }

    /**
     * <p>
     * The ID of the stack instance.
     * </p>
     * 
     * @return The ID of the stack instance.
     */
    public String stackId() {
        return stackId;
    }

    /**
     * <p>
     * A list of parameters from the stack set template whose values have been overridden in this stack instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of parameters from the stack set template whose values have been overridden in this stack
     *         instance.
     */
    public List<Parameter> parameterOverrides() {
        return parameterOverrides;
    }

    /**
     * <p>
     * The status of the stack instance, in terms of its synchronization with its associated stack set.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
     * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You might
     * need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to
     * <code>true</code>, to delete the stack instance, and then delete the stack manually.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or was
     * stopped before the stack was created or updated.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>CURRENT</code>: The stack is currently up to date with the stack set.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack instance, in terms of its synchronization with its associated stack set.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in
     *         an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations.
     *         You might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code>
     *         set to <code>true</code>, to delete the stack instance, and then delete the stack manually.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code>
     *         operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed
     *         or was stopped before the stack was created or updated.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CURRENT</code>: The stack is currently up to date with the stack set.
     *         </p>
     *         </li>
     * @see StackInstanceStatus
     */
    public StackInstanceStatus status() {
        return StackInstanceStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the stack instance, in terms of its synchronization with its associated stack set.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
     * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You might
     * need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to
     * <code>true</code>, to delete the stack instance, and then delete the stack manually.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or was
     * stopped before the stack was created or updated.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * <code>CURRENT</code>: The stack is currently up to date with the stack set.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackInstanceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the stack instance, in terms of its synchronization with its associated stack set.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in
     *         an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations.
     *         You might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code>
     *         set to <code>true</code>, to delete the stack instance, and then delete the stack manually.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code>
     *         operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed
     *         or was stopped before the stack was created or updated.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CURRENT</code>: The stack is currently up to date with the stack set.
     *         </p>
     *         </li>
     * @see StackInstanceStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The explanation for the specific status code that is assigned to this stack instance.
     * </p>
     * 
     * @return The explanation for the specific status code that is assigned to this stack instance.
     */
    public String statusReason() {
        return statusReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackSetId());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(account());
        hashCode = 31 * hashCode + Objects.hashCode(stackId());
        hashCode = 31 * hashCode + Objects.hashCode(parameterOverrides());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackInstance)) {
            return false;
        }
        StackInstance other = (StackInstance) obj;
        return Objects.equals(stackSetId(), other.stackSetId()) && Objects.equals(region(), other.region())
                && Objects.equals(account(), other.account()) && Objects.equals(stackId(), other.stackId())
                && Objects.equals(parameterOverrides(), other.parameterOverrides())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason());
    }

    @Override
    public String toString() {
        return ToString.builder("StackInstance").add("StackSetId", stackSetId()).add("Region", region())
                .add("Account", account()).add("StackId", stackId()).add("ParameterOverrides", parameterOverrides())
                .add("Status", statusAsString()).add("StatusReason", statusReason()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackSetId":
            return Optional.ofNullable(clazz.cast(stackSetId()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        case "Account":
            return Optional.ofNullable(clazz.cast(account()));
        case "StackId":
            return Optional.ofNullable(clazz.cast(stackId()));
        case "ParameterOverrides":
            return Optional.ofNullable(clazz.cast(parameterOverrides()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackInstance, T> g) {
        return obj -> g.apply((StackInstance) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackInstance> {
        /**
         * <p>
         * The name or unique ID of the stack set that the stack instance is associated with.
         * </p>
         * 
         * @param stackSetId
         *        The name or unique ID of the stack set that the stack instance is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackSetId(String stackSetId);

        /**
         * <p>
         * The name of the AWS region that the stack instance is associated with.
         * </p>
         * 
         * @param region
         *        The name of the AWS region that the stack instance is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The name of the AWS account that the stack instance is associated with.
         * </p>
         * 
         * @param account
         *        The name of the AWS account that the stack instance is associated with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder account(String account);

        /**
         * <p>
         * The ID of the stack instance.
         * </p>
         * 
         * @param stackId
         *        The ID of the stack instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackId(String stackId);

        /**
         * <p>
         * A list of parameters from the stack set template whose values have been overridden in this stack instance.
         * </p>
         * 
         * @param parameterOverrides
         *        A list of parameters from the stack set template whose values have been overridden in this stack
         *        instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterOverrides(Collection<Parameter> parameterOverrides);

        /**
         * <p>
         * A list of parameters from the stack set template whose values have been overridden in this stack instance.
         * </p>
         * 
         * @param parameterOverrides
         *        A list of parameters from the stack set template whose values have been overridden in this stack
         *        instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameterOverrides(Parameter... parameterOverrides);

        /**
         * <p>
         * A list of parameters from the stack set template whose values have been overridden in this stack instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parameter>.Builder} avoiding the need to
         * create one manually via {@link List<Parameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parameter>.Builder#build()} is called immediately and its
         * result is passed to {@link #parameterOverrides(List<Parameter>)}.
         * 
         * @param parameterOverrides
         *        a consumer that will call methods on {@link List<Parameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameterOverrides(List<Parameter>)
         */
        Builder parameterOverrides(Consumer<Parameter.Builder>... parameterOverrides);

        /**
         * <p>
         * The status of the stack instance, in terms of its synchronization with its associated stack set.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
         * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You
         * might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to
         * <code>true</code>, to delete the stack instance, and then delete the stack manually.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or
         * was stopped before the stack was created or updated.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>CURRENT</code>: The stack is currently up to date with the stack set.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the stack instance, in terms of its synchronization with its associated stack set.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack
         *        in an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code>
         *        operations. You might need to perform a <code>DeleteStackInstances</code> operation, with
         *        <code>RetainStacks</code> set to <code>true</code>, to delete the stack instance, and then delete the
         *        stack manually.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code>
         *        operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that
         *        failed or was stopped before the stack was created or updated.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CURRENT</code>: The stack is currently up to date with the stack set.
         *        </p>
         *        </li>
         * @see StackInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackInstanceStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the stack instance, in terms of its synchronization with its associated stack set.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack in an
         * unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code> operations. You
         * might need to perform a <code>DeleteStackInstances</code> operation, with <code>RetainStacks</code> set to
         * <code>true</code>, to delete the stack instance, and then delete the stack manually.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that failed or
         * was stopped before the stack was created or updated.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * <code>CURRENT</code>: The stack is currently up to date with the stack set.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the stack instance, in terms of its synchronization with its associated stack set.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INOPERABLE</code>: A <code>DeleteStackInstances</code> operation has failed and left the stack
         *        in an unstable state. Stacks in this state are excluded from further <code>UpdateStackSet</code>
         *        operations. You might need to perform a <code>DeleteStackInstances</code> operation, with
         *        <code>RetainStacks</code> set to <code>true</code>, to delete the stack instance, and then delete the
         *        stack manually.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OUTDATED</code>: The stack isn't currently up to date with the stack set because:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The associated stack failed during a <code>CreateStackSet</code> or <code>UpdateStackSet</code>
         *        operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The stack was part of a <code>CreateStackSet</code> or <code>UpdateStackSet</code> operation that
         *        failed or was stopped before the stack was created or updated.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CURRENT</code>: The stack is currently up to date with the stack set.
         *        </p>
         *        </li>
         * @see StackInstanceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackInstanceStatus
         */
        Builder status(StackInstanceStatus status);

        /**
         * <p>
         * The explanation for the specific status code that is assigned to this stack instance.
         * </p>
         * 
         * @param statusReason
         *        The explanation for the specific status code that is assigned to this stack instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);
    }

    static final class BuilderImpl implements Builder {
        private String stackSetId;

        private String region;

        private String account;

        private String stackId;

        private List<Parameter> parameterOverrides = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String statusReason;

        private BuilderImpl() {
        }

        private BuilderImpl(StackInstance model) {
            stackSetId(model.stackSetId);
            region(model.region);
            account(model.account);
            stackId(model.stackId);
            parameterOverrides(model.parameterOverrides);
            status(model.status);
            statusReason(model.statusReason);
        }

        public final String getStackSetId() {
            return stackSetId;
        }

        @Override
        public final Builder stackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
            return this;
        }

        public final void setStackSetId(String stackSetId) {
            this.stackSetId = stackSetId;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        public final String getAccount() {
            return account;
        }

        @Override
        public final Builder account(String account) {
            this.account = account;
            return this;
        }

        public final void setAccount(String account) {
            this.account = account;
        }

        public final String getStackId() {
            return stackId;
        }

        @Override
        public final Builder stackId(String stackId) {
            this.stackId = stackId;
            return this;
        }

        public final void setStackId(String stackId) {
            this.stackId = stackId;
        }

        public final Collection<Parameter.Builder> getParameterOverrides() {
            return parameterOverrides != null ? parameterOverrides.stream().map(Parameter::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameterOverrides(Collection<Parameter> parameterOverrides) {
            this.parameterOverrides = ParametersCopier.copy(parameterOverrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterOverrides(Parameter... parameterOverrides) {
            parameterOverrides(Arrays.asList(parameterOverrides));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameterOverrides(Consumer<Parameter.Builder>... parameterOverrides) {
            parameterOverrides(Stream.of(parameterOverrides).map(c -> Parameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setParameterOverrides(Collection<Parameter.BuilderImpl> parameterOverrides) {
            this.parameterOverrides = ParametersCopier.copyFromBuilder(parameterOverrides);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StackInstanceStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public StackInstance build() {
            return new StackInstance(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
