/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The structures that contain summary information about the specified operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackSetOperationSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<StackSetOperationSummary.Builder, StackSetOperationSummary> {
    private static final SdkField<String> OPERATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OperationId").getter(getter(StackSetOperationSummary::operationId)).setter(setter(Builder::operationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationId").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(StackSetOperationSummary::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(StackSetOperationSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATION_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTimestamp").getter(getter(StackSetOperationSummary::creationTimestamp))
            .setter(setter(Builder::creationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTimestamp").build()).build();

    private static final SdkField<Instant> END_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTimestamp").getter(getter(StackSetOperationSummary::endTimestamp))
            .setter(setter(Builder::endTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTimestamp").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(StackSetOperationSummary::statusReason))
            .setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<StackSetOperationStatusDetails> STATUS_DETAILS_FIELD = SdkField
            .<StackSetOperationStatusDetails> builder(MarshallingType.SDK_POJO).memberName("StatusDetails")
            .getter(getter(StackSetOperationSummary::statusDetails)).setter(setter(Builder::statusDetails))
            .constructor(StackSetOperationStatusDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusDetails").build()).build();

    private static final SdkField<StackSetOperationPreferences> OPERATION_PREFERENCES_FIELD = SdkField
            .<StackSetOperationPreferences> builder(MarshallingType.SDK_POJO).memberName("OperationPreferences")
            .getter(getter(StackSetOperationSummary::operationPreferences)).setter(setter(Builder::operationPreferences))
            .constructor(StackSetOperationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationPreferences").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OPERATION_ID_FIELD,
            ACTION_FIELD, STATUS_FIELD, CREATION_TIMESTAMP_FIELD, END_TIMESTAMP_FIELD, STATUS_REASON_FIELD, STATUS_DETAILS_FIELD,
            OPERATION_PREFERENCES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String operationId;

    private final String action;

    private final String status;

    private final Instant creationTimestamp;

    private final Instant endTimestamp;

    private final String statusReason;

    private final StackSetOperationStatusDetails statusDetails;

    private final StackSetOperationPreferences operationPreferences;

    private StackSetOperationSummary(BuilderImpl builder) {
        this.operationId = builder.operationId;
        this.action = builder.action;
        this.status = builder.status;
        this.creationTimestamp = builder.creationTimestamp;
        this.endTimestamp = builder.endTimestamp;
        this.statusReason = builder.statusReason;
        this.statusDetails = builder.statusDetails;
        this.operationPreferences = builder.operationPreferences;
    }

    /**
     * <p>
     * The unique ID of the stack set operation.
     * </p>
     * 
     * @return The unique ID of the stack set operation.
     */
    public final String operationId() {
        return operationId;
    }

    /**
     * <p>
     * The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and delete
     * operations affect only the specified stack instances that are associated with the specified stack set. Update
     * operations affect both the stack set itself and <i>all</i> associated stack set instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackSetOperationAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
     *         delete operations affect only the specified stack instances that are associated with the specified stack
     *         set. Update operations affect both the stack set itself and <i>all</i> associated stack set instances.
     * @see StackSetOperationAction
     */
    public final StackSetOperationAction action() {
        return StackSetOperationAction.fromValue(action);
    }

    /**
     * <p>
     * The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and delete
     * operations affect only the specified stack instances that are associated with the specified stack set. Update
     * operations affect both the stack set itself and <i>all</i> associated stack set instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackSetOperationAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
     *         delete operations affect only the specified stack instances that are associated with the specified stack
     *         set. Update operations affect both the stack set itself and <i>all</i> associated stack set instances.
     * @see StackSetOperationAction
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The overall status of the operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
     * you've set for an operation is applied for each Region during stack create and update operations. If the number
     * of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the Region is set
     * to <code>FAILED</code>. This in turn sets the status of the operation as a whole to <code>FAILED</code>, and
     * CloudFormation cancels the operation in any remaining Regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     * operations, the operation is queued to be performed. For more information, see the <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
     * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code>: The operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has canceled the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without exceeding
     * the failure tolerance for the operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The overall status of the operation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value
     *         that you've set for an operation is applied for each Region during stack create and update operations. If
     *         the number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in
     *         the Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
     *         <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     *         operations, the operation is queued to be performed. For more information, see the <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
     *         >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code>: The operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has canceled the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
     *         exceeding the failure tolerance for the operation.
     *         </p>
     *         </li>
     * @see StackSetOperationStatus
     */
    public final StackSetOperationStatus status() {
        return StackSetOperationStatus.fromValue(status);
    }

    /**
     * <p>
     * The overall status of the operation.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
     * you've set for an operation is applied for each Region during stack create and update operations. If the number
     * of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the Region is set
     * to <code>FAILED</code>. This in turn sets the status of the operation as a whole to <code>FAILED</code>, and
     * CloudFormation cancels the operation in any remaining Regions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     * operations, the operation is queued to be performed. For more information, see the <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
     * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RUNNING</code>: The operation is currently being performed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPED</code>: The user has canceled the operation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without exceeding
     * the failure tolerance for the operation.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StackSetOperationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The overall status of the operation.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value
     *         that you've set for an operation is applied for each Region during stack create and update operations. If
     *         the number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in
     *         the Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
     *         <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
     *         operations, the operation is queued to be performed. For more information, see the <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
     *         >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RUNNING</code>: The operation is currently being performed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPED</code>: The user has canceled the operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>STOPPING</code>: The operation is in the process of stopping, at user request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
     *         exceeding the failure tolerance for the operation.
     *         </p>
     *         </li>
     * @see StackSetOperationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time at which the operation was initiated. Note that the creation times for the stack set operation might
     * differ from the creation time of the individual stacks themselves. This is because CloudFormation needs to
     * perform preparatory work for the operation, such as dispatching the work to the requested Regions, before
     * actually creating the first stacks.
     * </p>
     * 
     * @return The time at which the operation was initiated. Note that the creation times for the stack set operation
     *         might differ from the creation time of the individual stacks themselves. This is because CloudFormation
     *         needs to perform preparatory work for the operation, such as dispatching the work to the requested
     *         Regions, before actually creating the first stacks.
     */
    public final Instant creationTimestamp() {
        return creationTimestamp;
    }

    /**
     * <p>
     * The time at which the stack set operation ended, across all accounts and Regions specified. Note that this
     * doesn't necessarily mean that the stack set operation was successful, or even attempted, in each account or
     * Region.
     * </p>
     * 
     * @return The time at which the stack set operation ended, across all accounts and Regions specified. Note that
     *         this doesn't necessarily mean that the stack set operation was successful, or even attempted, in each
     *         account or Region.
     */
    public final Instant endTimestamp() {
        return endTimestamp;
    }

    /**
     * <p>
     * The status of the operation in details.
     * </p>
     * 
     * @return The status of the operation in details.
     */
    public final String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * Detailed information about the stack set operation.
     * </p>
     * 
     * @return Detailed information about the stack set operation.
     */
    public final StackSetOperationStatusDetails statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * The user-specified preferences for how CloudFormation performs a stack set operation.
     * </p>
     * <p>
     * For more information about maximum concurrent accounts and failure tolerance, see <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-ops-options"
     * >Stack set operation options</a>.
     * </p>
     * 
     * @return The user-specified preferences for how CloudFormation performs a stack set operation.</p>
     *         <p>
     *         For more information about maximum concurrent accounts and failure tolerance, see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-ops-options"
     *         >Stack set operation options</a>.
     */
    public final StackSetOperationPreferences operationPreferences() {
        return operationPreferences;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(operationId());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(endTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(operationPreferences());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackSetOperationSummary)) {
            return false;
        }
        StackSetOperationSummary other = (StackSetOperationSummary) obj;
        return Objects.equals(operationId(), other.operationId()) && Objects.equals(actionAsString(), other.actionAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(creationTimestamp(), other.creationTimestamp())
                && Objects.equals(endTimestamp(), other.endTimestamp()) && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(statusDetails(), other.statusDetails())
                && Objects.equals(operationPreferences(), other.operationPreferences());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackSetOperationSummary").add("OperationId", operationId()).add("Action", actionAsString())
                .add("Status", statusAsString()).add("CreationTimestamp", creationTimestamp())
                .add("EndTimestamp", endTimestamp()).add("StatusReason", statusReason()).add("StatusDetails", statusDetails())
                .add("OperationPreferences", operationPreferences()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OperationId":
            return Optional.ofNullable(clazz.cast(operationId()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreationTimestamp":
            return Optional.ofNullable(clazz.cast(creationTimestamp()));
        case "EndTimestamp":
            return Optional.ofNullable(clazz.cast(endTimestamp()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "StatusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        case "OperationPreferences":
            return Optional.ofNullable(clazz.cast(operationPreferences()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("OperationId", OPERATION_ID_FIELD);
        map.put("Action", ACTION_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CreationTimestamp", CREATION_TIMESTAMP_FIELD);
        map.put("EndTimestamp", END_TIMESTAMP_FIELD);
        map.put("StatusReason", STATUS_REASON_FIELD);
        map.put("StatusDetails", STATUS_DETAILS_FIELD);
        map.put("OperationPreferences", OPERATION_PREFERENCES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StackSetOperationSummary, T> g) {
        return obj -> g.apply((StackSetOperationSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackSetOperationSummary> {
        /**
         * <p>
         * The unique ID of the stack set operation.
         * </p>
         * 
         * @param operationId
         *        The unique ID of the stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationId(String operationId);

        /**
         * <p>
         * The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and delete
         * operations affect only the specified stack instances that are associated with the specified stack set. Update
         * operations affect both the stack set itself and <i>all</i> associated stack set instances.
         * </p>
         * 
         * @param action
         *        The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
         *        delete operations affect only the specified stack instances that are associated with the specified
         *        stack set. Update operations affect both the stack set itself and <i>all</i> associated stack set
         *        instances.
         * @see StackSetOperationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationAction
         */
        Builder action(String action);

        /**
         * <p>
         * The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and delete
         * operations affect only the specified stack instances that are associated with the specified stack set. Update
         * operations affect both the stack set itself and <i>all</i> associated stack set instances.
         * </p>
         * 
         * @param action
         *        The type of operation: <code>CREATE</code>, <code>UPDATE</code>, or <code>DELETE</code>. Create and
         *        delete operations affect only the specified stack instances that are associated with the specified
         *        stack set. Update operations affect both the stack set itself and <i>all</i> associated stack set
         *        instances.
         * @see StackSetOperationAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationAction
         */
        Builder action(StackSetOperationAction action);

        /**
         * <p>
         * The overall status of the operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
         * you've set for an operation is applied for each Region during stack create and update operations. If the
         * number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the
         * Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
         * <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
         * operations, the operation is queued to be performed. For more information, see the <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
         * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code>: The operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has canceled the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         * exceeding the failure tolerance for the operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The overall status of the operation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance
         *        value that you've set for an operation is applied for each Region during stack create and update
         *        operations. If the number of failed stacks within a Region exceeds the failure tolerance, the status
         *        of the operation in the Region is set to <code>FAILED</code>. This in turn sets the status of the
         *        operation as a whole to <code>FAILED</code>, and CloudFormation cancels the operation in any remaining
         *        Regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence
         *        of operations, the operation is queued to be performed. For more information, see the <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
         *        >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code>: The operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has canceled the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         *        exceeding the failure tolerance for the operation.
         *        </p>
         *        </li>
         * @see StackSetOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The overall status of the operation.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance value that
         * you've set for an operation is applied for each Region during stack create and update operations. If the
         * number of failed stacks within a Region exceeds the failure tolerance, the status of the operation in the
         * Region is set to <code>FAILED</code>. This in turn sets the status of the operation as a whole to
         * <code>FAILED</code>, and CloudFormation cancels the operation in any remaining Regions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence of
         * operations, the operation is queued to be performed. For more information, see the <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
         * >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RUNNING</code>: The operation is currently being performed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPED</code>: The user has canceled the operation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         * exceeding the failure tolerance for the operation.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The overall status of the operation.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The operation exceeded the specified failure tolerance. The failure tolerance
         *        value that you've set for an operation is applied for each Region during stack create and update
         *        operations. If the number of failed stacks within a Region exceeds the failure tolerance, the status
         *        of the operation in the Region is set to <code>FAILED</code>. This in turn sets the status of the
         *        operation as a whole to <code>FAILED</code>, and CloudFormation cancels the operation in any remaining
         *        Regions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>QUEUED</code>: [Service-managed permissions] For automatic deployments that require a sequence
         *        of operations, the operation is queued to be performed. For more information, see the <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-status-codes"
         *        >stack set operation status codes</a> in the <i>CloudFormation User Guide</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RUNNING</code>: The operation is currently being performed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPED</code>: The user has canceled the operation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>STOPPING</code>: The operation is in the process of stopping, at user request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The operation completed creating or updating all the specified stacks without
         *        exceeding the failure tolerance for the operation.
         *        </p>
         *        </li>
         * @see StackSetOperationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackSetOperationStatus
         */
        Builder status(StackSetOperationStatus status);

        /**
         * <p>
         * The time at which the operation was initiated. Note that the creation times for the stack set operation might
         * differ from the creation time of the individual stacks themselves. This is because CloudFormation needs to
         * perform preparatory work for the operation, such as dispatching the work to the requested Regions, before
         * actually creating the first stacks.
         * </p>
         * 
         * @param creationTimestamp
         *        The time at which the operation was initiated. Note that the creation times for the stack set
         *        operation might differ from the creation time of the individual stacks themselves. This is because
         *        CloudFormation needs to perform preparatory work for the operation, such as dispatching the work to
         *        the requested Regions, before actually creating the first stacks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTimestamp(Instant creationTimestamp);

        /**
         * <p>
         * The time at which the stack set operation ended, across all accounts and Regions specified. Note that this
         * doesn't necessarily mean that the stack set operation was successful, or even attempted, in each account or
         * Region.
         * </p>
         * 
         * @param endTimestamp
         *        The time at which the stack set operation ended, across all accounts and Regions specified. Note that
         *        this doesn't necessarily mean that the stack set operation was successful, or even attempted, in each
         *        account or Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTimestamp(Instant endTimestamp);

        /**
         * <p>
         * The status of the operation in details.
         * </p>
         * 
         * @param statusReason
         *        The status of the operation in details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * Detailed information about the stack set operation.
         * </p>
         * 
         * @param statusDetails
         *        Detailed information about the stack set operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(StackSetOperationStatusDetails statusDetails);

        /**
         * <p>
         * Detailed information about the stack set operation.
         * </p>
         * This is a convenience method that creates an instance of the {@link StackSetOperationStatusDetails.Builder}
         * avoiding the need to create one manually via {@link StackSetOperationStatusDetails#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StackSetOperationStatusDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #statusDetails(StackSetOperationStatusDetails)}.
         * 
         * @param statusDetails
         *        a consumer that will call methods on {@link StackSetOperationStatusDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusDetails(StackSetOperationStatusDetails)
         */
        default Builder statusDetails(Consumer<StackSetOperationStatusDetails.Builder> statusDetails) {
            return statusDetails(StackSetOperationStatusDetails.builder().applyMutation(statusDetails).build());
        }

        /**
         * <p>
         * The user-specified preferences for how CloudFormation performs a stack set operation.
         * </p>
         * <p>
         * For more information about maximum concurrent accounts and failure tolerance, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-ops-options"
         * >Stack set operation options</a>.
         * </p>
         * 
         * @param operationPreferences
         *        The user-specified preferences for how CloudFormation performs a stack set operation.</p>
         *        <p>
         *        For more information about maximum concurrent accounts and failure tolerance, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-ops-options"
         *        >Stack set operation options</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationPreferences(StackSetOperationPreferences operationPreferences);

        /**
         * <p>
         * The user-specified preferences for how CloudFormation performs a stack set operation.
         * </p>
         * <p>
         * For more information about maximum concurrent accounts and failure tolerance, see <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html#stackset-ops-options"
         * >Stack set operation options</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link StackSetOperationPreferences.Builder}
         * avoiding the need to create one manually via {@link StackSetOperationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StackSetOperationPreferences.Builder#build()} is called
         * immediately and its result is passed to {@link #operationPreferences(StackSetOperationPreferences)}.
         * 
         * @param operationPreferences
         *        a consumer that will call methods on {@link StackSetOperationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operationPreferences(StackSetOperationPreferences)
         */
        default Builder operationPreferences(Consumer<StackSetOperationPreferences.Builder> operationPreferences) {
            return operationPreferences(StackSetOperationPreferences.builder().applyMutation(operationPreferences).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String operationId;

        private String action;

        private String status;

        private Instant creationTimestamp;

        private Instant endTimestamp;

        private String statusReason;

        private StackSetOperationStatusDetails statusDetails;

        private StackSetOperationPreferences operationPreferences;

        private BuilderImpl() {
        }

        private BuilderImpl(StackSetOperationSummary model) {
            operationId(model.operationId);
            action(model.action);
            status(model.status);
            creationTimestamp(model.creationTimestamp);
            endTimestamp(model.endTimestamp);
            statusReason(model.statusReason);
            statusDetails(model.statusDetails);
            operationPreferences(model.operationPreferences);
        }

        public final String getOperationId() {
            return operationId;
        }

        public final void setOperationId(String operationId) {
            this.operationId = operationId;
        }

        @Override
        public final Builder operationId(String operationId) {
            this.operationId = operationId;
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(StackSetOperationAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StackSetOperationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreationTimestamp() {
            return creationTimestamp;
        }

        public final void setCreationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
        }

        @Override
        public final Builder creationTimestamp(Instant creationTimestamp) {
            this.creationTimestamp = creationTimestamp;
            return this;
        }

        public final Instant getEndTimestamp() {
            return endTimestamp;
        }

        public final void setEndTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
        }

        @Override
        public final Builder endTimestamp(Instant endTimestamp) {
            this.endTimestamp = endTimestamp;
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final StackSetOperationStatusDetails.Builder getStatusDetails() {
            return statusDetails != null ? statusDetails.toBuilder() : null;
        }

        public final void setStatusDetails(StackSetOperationStatusDetails.BuilderImpl statusDetails) {
            this.statusDetails = statusDetails != null ? statusDetails.build() : null;
        }

        @Override
        public final Builder statusDetails(StackSetOperationStatusDetails statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final StackSetOperationPreferences.Builder getOperationPreferences() {
            return operationPreferences != null ? operationPreferences.toBuilder() : null;
        }

        public final void setOperationPreferences(StackSetOperationPreferences.BuilderImpl operationPreferences) {
            this.operationPreferences = operationPreferences != null ? operationPreferences.build() : null;
        }

        @Override
        public final Builder operationPreferences(StackSetOperationPreferences operationPreferences) {
            this.operationPreferences = operationPreferences;
            return this;
        }

        @Override
        public StackSetOperationSummary build() {
            return new StackSetOperationSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
