/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the stack and the action that CloudFormation will perform on it if you execute the stack refactor.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackRefactorAction implements SdkPojo, Serializable,
        ToCopyableBuilder<StackRefactorAction.Builder, StackRefactorAction> {
    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(StackRefactorAction::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final SdkField<String> ENTITY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Entity")
            .getter(getter(StackRefactorAction::entityAsString)).setter(setter(Builder::entity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Entity").build()).build();

    private static final SdkField<String> PHYSICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PhysicalResourceId").getter(getter(StackRefactorAction::physicalResourceId))
            .setter(setter(Builder::physicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PhysicalResourceId").build())
            .build();

    private static final SdkField<String> RESOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceIdentifier").getter(getter(StackRefactorAction::resourceIdentifier))
            .setter(setter(Builder::resourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdentifier").build())
            .build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(StackRefactorAction::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> DETECTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Detection").getter(getter(StackRefactorAction::detectionAsString)).setter(setter(Builder::detection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Detection").build()).build();

    private static final SdkField<String> DETECTION_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DetectionReason").getter(getter(StackRefactorAction::detectionReason))
            .setter(setter(Builder::detectionReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DetectionReason").build()).build();

    private static final SdkField<List<Tag>> TAG_RESOURCES_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("TagResources")
            .getter(getter(StackRefactorAction::tagResources))
            .setter(setter(Builder::tagResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> UNTAG_RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("UntagResources")
            .getter(getter(StackRefactorAction::untagResources))
            .setter(setter(Builder::untagResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UntagResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ResourceMapping> RESOURCE_MAPPING_FIELD = SdkField
            .<ResourceMapping> builder(MarshallingType.SDK_POJO).memberName("ResourceMapping")
            .getter(getter(StackRefactorAction::resourceMapping)).setter(setter(Builder::resourceMapping))
            .constructor(ResourceMapping::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceMapping").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_FIELD, ENTITY_FIELD,
            PHYSICAL_RESOURCE_ID_FIELD, RESOURCE_IDENTIFIER_FIELD, DESCRIPTION_FIELD, DETECTION_FIELD, DETECTION_REASON_FIELD,
            TAG_RESOURCES_FIELD, UNTAG_RESOURCES_FIELD, RESOURCE_MAPPING_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String action;

    private final String entity;

    private final String physicalResourceId;

    private final String resourceIdentifier;

    private final String description;

    private final String detection;

    private final String detectionReason;

    private final List<Tag> tagResources;

    private final List<String> untagResources;

    private final ResourceMapping resourceMapping;

    private StackRefactorAction(BuilderImpl builder) {
        this.action = builder.action;
        this.entity = builder.entity;
        this.physicalResourceId = builder.physicalResourceId;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.description = builder.description;
        this.detection = builder.detection;
        this.detectionReason = builder.detectionReason;
        this.tagResources = builder.tagResources;
        this.untagResources = builder.untagResources;
        this.resourceMapping = builder.resourceMapping;
    }

    /**
     * <p>
     * The action that CloudFormation takes on the stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackRefactorActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The action that CloudFormation takes on the stack.
     * @see StackRefactorActionType
     */
    public final StackRefactorActionType action() {
        return StackRefactorActionType.fromValue(action);
    }

    /**
     * <p>
     * The action that CloudFormation takes on the stack.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link StackRefactorActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionAsString}.
     * </p>
     * 
     * @return The action that CloudFormation takes on the stack.
     * @see StackRefactorActionType
     */
    public final String actionAsString() {
        return action;
    }

    /**
     * <p>
     * The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
     * <code>Entity</code> types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Stack</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Resource</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #entity} will
     * return {@link StackRefactorActionEntity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #entityAsString}.
     * </p>
     * 
     * @return The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
     *         <code>Entity</code> types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Stack</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Resource</code>
     *         </p>
     *         </li>
     * @see StackRefactorActionEntity
     */
    public final StackRefactorActionEntity entity() {
        return StackRefactorActionEntity.fromValue(entity);
    }

    /**
     * <p>
     * The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
     * <code>Entity</code> types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Stack</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Resource</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #entity} will
     * return {@link StackRefactorActionEntity#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #entityAsString}.
     * </p>
     * 
     * @return The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
     *         <code>Entity</code> types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Stack</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Resource</code>
     *         </p>
     *         </li>
     * @see StackRefactorActionEntity
     */
    public final String entityAsString() {
        return entity;
    }

    /**
     * <p>
     * The name or unique identifier associated with the physical instance of the resource.
     * </p>
     * 
     * @return The name or unique identifier associated with the physical instance of the resource.
     */
    public final String physicalResourceId() {
        return physicalResourceId;
    }

    /**
     * <p>
     * A key-value pair that identifies the target resource. The key is an identifier property (for example,
     * <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual property value
     * (for example, <code>MyS3Bucket</code>).
     * </p>
     * 
     * @return A key-value pair that identifies the target resource. The key is an identifier property (for example,
     *         <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual property
     *         value (for example, <code>MyS3Bucket</code>).
     */
    public final String resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * <p>
     * A description to help you identify the refactor.
     * </p>
     * 
     * @return A description to help you identify the refactor.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The detection type is one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto: CloudFormation figured out the mapping on its own.
     * </p>
     * </li>
     * <li>
     * <p>
     * Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #detection} will
     * return {@link StackRefactorDetection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #detectionAsString}.
     * </p>
     * 
     * @return The detection type is one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Auto: CloudFormation figured out the mapping on its own.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
     *         </p>
     *         </li>
     * @see StackRefactorDetection
     */
    public final StackRefactorDetection detection() {
        return StackRefactorDetection.fromValue(detection);
    }

    /**
     * <p>
     * The detection type is one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto: CloudFormation figured out the mapping on its own.
     * </p>
     * </li>
     * <li>
     * <p>
     * Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #detection} will
     * return {@link StackRefactorDetection#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #detectionAsString}.
     * </p>
     * 
     * @return The detection type is one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Auto: CloudFormation figured out the mapping on its own.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
     *         </p>
     *         </li>
     * @see StackRefactorDetection
     */
    public final String detectionAsString() {
        return detection;
    }

    /**
     * <p>
     * The description of the detection type.
     * </p>
     * 
     * @return The description of the detection type.
     */
    public final String detectionReason() {
        return detectionReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagResources() {
        return tagResources != null && !(tagResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Assigns one or more tags to specified resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagResources} method.
     * </p>
     * 
     * @return Assigns one or more tags to specified resources.
     */
    public final List<Tag> tagResources() {
        return tagResources;
    }

    /**
     * For responses, this returns true if the service returned a value for the UntagResources property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUntagResources() {
        return untagResources != null && !(untagResources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Removes one or more tags to specified resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUntagResources} method.
     * </p>
     * 
     * @return Removes one or more tags to specified resources.
     */
    public final List<String> untagResources() {
        return untagResources;
    }

    /**
     * <p>
     * The mapping for the stack resource <code>Source</code> and stack resource <code>Destination</code>.
     * </p>
     * 
     * @return The mapping for the stack resource <code>Source</code> and stack resource <code>Destination</code>.
     */
    public final ResourceMapping resourceMapping() {
        return resourceMapping;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(entityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(physicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(detectionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(detectionReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagResources() ? tagResources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUntagResources() ? untagResources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceMapping());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackRefactorAction)) {
            return false;
        }
        StackRefactorAction other = (StackRefactorAction) obj;
        return Objects.equals(actionAsString(), other.actionAsString())
                && Objects.equals(entityAsString(), other.entityAsString())
                && Objects.equals(physicalResourceId(), other.physicalResourceId())
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && Objects.equals(description(), other.description())
                && Objects.equals(detectionAsString(), other.detectionAsString())
                && Objects.equals(detectionReason(), other.detectionReason()) && hasTagResources() == other.hasTagResources()
                && Objects.equals(tagResources(), other.tagResources()) && hasUntagResources() == other.hasUntagResources()
                && Objects.equals(untagResources(), other.untagResources())
                && Objects.equals(resourceMapping(), other.resourceMapping());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackRefactorAction").add("Action", actionAsString()).add("Entity", entityAsString())
                .add("PhysicalResourceId", physicalResourceId()).add("ResourceIdentifier", resourceIdentifier())
                .add("Description", description()).add("Detection", detectionAsString())
                .add("DetectionReason", detectionReason()).add("TagResources", hasTagResources() ? tagResources() : null)
                .add("UntagResources", hasUntagResources() ? untagResources() : null).add("ResourceMapping", resourceMapping())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        case "Entity":
            return Optional.ofNullable(clazz.cast(entityAsString()));
        case "PhysicalResourceId":
            return Optional.ofNullable(clazz.cast(physicalResourceId()));
        case "ResourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Detection":
            return Optional.ofNullable(clazz.cast(detectionAsString()));
        case "DetectionReason":
            return Optional.ofNullable(clazz.cast(detectionReason()));
        case "TagResources":
            return Optional.ofNullable(clazz.cast(tagResources()));
        case "UntagResources":
            return Optional.ofNullable(clazz.cast(untagResources()));
        case "ResourceMapping":
            return Optional.ofNullable(clazz.cast(resourceMapping()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Action", ACTION_FIELD);
        map.put("Entity", ENTITY_FIELD);
        map.put("PhysicalResourceId", PHYSICAL_RESOURCE_ID_FIELD);
        map.put("ResourceIdentifier", RESOURCE_IDENTIFIER_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Detection", DETECTION_FIELD);
        map.put("DetectionReason", DETECTION_REASON_FIELD);
        map.put("TagResources", TAG_RESOURCES_FIELD);
        map.put("UntagResources", UNTAG_RESOURCES_FIELD);
        map.put("ResourceMapping", RESOURCE_MAPPING_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StackRefactorAction, T> g) {
        return obj -> g.apply((StackRefactorAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackRefactorAction> {
        /**
         * <p>
         * The action that CloudFormation takes on the stack.
         * </p>
         * 
         * @param action
         *        The action that CloudFormation takes on the stack.
         * @see StackRefactorActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackRefactorActionType
         */
        Builder action(String action);

        /**
         * <p>
         * The action that CloudFormation takes on the stack.
         * </p>
         * 
         * @param action
         *        The action that CloudFormation takes on the stack.
         * @see StackRefactorActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackRefactorActionType
         */
        Builder action(StackRefactorActionType action);

        /**
         * <p>
         * The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
         * <code>Entity</code> types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Stack</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Resource</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param entity
         *        The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
         *        <code>Entity</code> types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Stack</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Resource</code>
         *        </p>
         *        </li>
         * @see StackRefactorActionEntity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackRefactorActionEntity
         */
        Builder entity(String entity);

        /**
         * <p>
         * The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
         * <code>Entity</code> types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Stack</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Resource</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param entity
         *        The type that will be evaluated in the <code>StackRefactorAction</code>. The following are potential
         *        <code>Entity</code> types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Stack</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Resource</code>
         *        </p>
         *        </li>
         * @see StackRefactorActionEntity
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackRefactorActionEntity
         */
        Builder entity(StackRefactorActionEntity entity);

        /**
         * <p>
         * The name or unique identifier associated with the physical instance of the resource.
         * </p>
         * 
         * @param physicalResourceId
         *        The name or unique identifier associated with the physical instance of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder physicalResourceId(String physicalResourceId);

        /**
         * <p>
         * A key-value pair that identifies the target resource. The key is an identifier property (for example,
         * <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual property
         * value (for example, <code>MyS3Bucket</code>).
         * </p>
         * 
         * @param resourceIdentifier
         *        A key-value pair that identifies the target resource. The key is an identifier property (for example,
         *        <code>BucketName</code> for <code>AWS::S3::Bucket</code> resources) and the value is the actual
         *        property value (for example, <code>MyS3Bucket</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(String resourceIdentifier);

        /**
         * <p>
         * A description to help you identify the refactor.
         * </p>
         * 
         * @param description
         *        A description to help you identify the refactor.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The detection type is one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Auto: CloudFormation figured out the mapping on its own.
         * </p>
         * </li>
         * <li>
         * <p>
         * Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param detection
         *        The detection type is one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Auto: CloudFormation figured out the mapping on its own.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
         *        </p>
         *        </li>
         * @see StackRefactorDetection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackRefactorDetection
         */
        Builder detection(String detection);

        /**
         * <p>
         * The detection type is one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Auto: CloudFormation figured out the mapping on its own.
         * </p>
         * </li>
         * <li>
         * <p>
         * Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param detection
         *        The detection type is one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Auto: CloudFormation figured out the mapping on its own.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Manual: The customer provided the mapping in the <code>ResourceMapping</code> parameter.
         *        </p>
         *        </li>
         * @see StackRefactorDetection
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackRefactorDetection
         */
        Builder detection(StackRefactorDetection detection);

        /**
         * <p>
         * The description of the detection type.
         * </p>
         * 
         * @param detectionReason
         *        The description of the detection type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder detectionReason(String detectionReason);

        /**
         * <p>
         * Assigns one or more tags to specified resources.
         * </p>
         * 
         * @param tagResources
         *        Assigns one or more tags to specified resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagResources(Collection<Tag> tagResources);

        /**
         * <p>
         * Assigns one or more tags to specified resources.
         * </p>
         * 
         * @param tagResources
         *        Assigns one or more tags to specified resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagResources(Tag... tagResources);

        /**
         * <p>
         * Assigns one or more tags to specified resources.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudformation.model.Tag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.cloudformation.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudformation.model.Tag.Builder#build()} is called immediately and
         * its result is passed to {@link #tagResources(List<Tag>)}.
         * 
         * @param tagResources
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudformation.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagResources(java.util.Collection<Tag>)
         */
        Builder tagResources(Consumer<Tag.Builder>... tagResources);

        /**
         * <p>
         * Removes one or more tags to specified resources.
         * </p>
         * 
         * @param untagResources
         *        Removes one or more tags to specified resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder untagResources(Collection<String> untagResources);

        /**
         * <p>
         * Removes one or more tags to specified resources.
         * </p>
         * 
         * @param untagResources
         *        Removes one or more tags to specified resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder untagResources(String... untagResources);

        /**
         * <p>
         * The mapping for the stack resource <code>Source</code> and stack resource <code>Destination</code>.
         * </p>
         * 
         * @param resourceMapping
         *        The mapping for the stack resource <code>Source</code> and stack resource <code>Destination</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceMapping(ResourceMapping resourceMapping);

        /**
         * <p>
         * The mapping for the stack resource <code>Source</code> and stack resource <code>Destination</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceMapping.Builder} avoiding the
         * need to create one manually via {@link ResourceMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceMapping.Builder#build()} is called immediately and its
         * result is passed to {@link #resourceMapping(ResourceMapping)}.
         * 
         * @param resourceMapping
         *        a consumer that will call methods on {@link ResourceMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceMapping(ResourceMapping)
         */
        default Builder resourceMapping(Consumer<ResourceMapping.Builder> resourceMapping) {
            return resourceMapping(ResourceMapping.builder().applyMutation(resourceMapping).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String action;

        private String entity;

        private String physicalResourceId;

        private String resourceIdentifier;

        private String description;

        private String detection;

        private String detectionReason;

        private List<Tag> tagResources = DefaultSdkAutoConstructList.getInstance();

        private List<String> untagResources = DefaultSdkAutoConstructList.getInstance();

        private ResourceMapping resourceMapping;

        private BuilderImpl() {
        }

        private BuilderImpl(StackRefactorAction model) {
            action(model.action);
            entity(model.entity);
            physicalResourceId(model.physicalResourceId);
            resourceIdentifier(model.resourceIdentifier);
            description(model.description);
            detection(model.detection);
            detectionReason(model.detectionReason);
            tagResources(model.tagResources);
            untagResources(model.untagResources);
            resourceMapping(model.resourceMapping);
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(StackRefactorActionType action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        public final String getEntity() {
            return entity;
        }

        public final void setEntity(String entity) {
            this.entity = entity;
        }

        @Override
        public final Builder entity(String entity) {
            this.entity = entity;
            return this;
        }

        @Override
        public final Builder entity(StackRefactorActionEntity entity) {
            this.entity(entity == null ? null : entity.toString());
            return this;
        }

        public final String getPhysicalResourceId() {
            return physicalResourceId;
        }

        public final void setPhysicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
        }

        @Override
        public final Builder physicalResourceId(String physicalResourceId) {
            this.physicalResourceId = physicalResourceId;
            return this;
        }

        public final String getResourceIdentifier() {
            return resourceIdentifier;
        }

        public final void setResourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
        }

        @Override
        public final Builder resourceIdentifier(String resourceIdentifier) {
            this.resourceIdentifier = resourceIdentifier;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDetection() {
            return detection;
        }

        public final void setDetection(String detection) {
            this.detection = detection;
        }

        @Override
        public final Builder detection(String detection) {
            this.detection = detection;
            return this;
        }

        @Override
        public final Builder detection(StackRefactorDetection detection) {
            this.detection(detection == null ? null : detection.toString());
            return this;
        }

        public final String getDetectionReason() {
            return detectionReason;
        }

        public final void setDetectionReason(String detectionReason) {
            this.detectionReason = detectionReason;
        }

        @Override
        public final Builder detectionReason(String detectionReason) {
            this.detectionReason = detectionReason;
            return this;
        }

        public final List<Tag.Builder> getTagResources() {
            List<Tag.Builder> result = StackRefactorTagResourcesCopier.copyToBuilder(this.tagResources);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagResources(Collection<Tag.BuilderImpl> tagResources) {
            this.tagResources = StackRefactorTagResourcesCopier.copyFromBuilder(tagResources);
        }

        @Override
        public final Builder tagResources(Collection<Tag> tagResources) {
            this.tagResources = StackRefactorTagResourcesCopier.copy(tagResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagResources(Tag... tagResources) {
            tagResources(Arrays.asList(tagResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagResources(Consumer<Tag.Builder>... tagResources) {
            tagResources(Stream.of(tagResources).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getUntagResources() {
            if (untagResources instanceof SdkAutoConstructList) {
                return null;
            }
            return untagResources;
        }

        public final void setUntagResources(Collection<String> untagResources) {
            this.untagResources = StackRefactorUntagResourcesCopier.copy(untagResources);
        }

        @Override
        public final Builder untagResources(Collection<String> untagResources) {
            this.untagResources = StackRefactorUntagResourcesCopier.copy(untagResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder untagResources(String... untagResources) {
            untagResources(Arrays.asList(untagResources));
            return this;
        }

        public final ResourceMapping.Builder getResourceMapping() {
            return resourceMapping != null ? resourceMapping.toBuilder() : null;
        }

        public final void setResourceMapping(ResourceMapping.BuilderImpl resourceMapping) {
            this.resourceMapping = resourceMapping != null ? resourceMapping.build() : null;
        }

        @Override
        public final Builder resourceMapping(ResourceMapping resourceMapping) {
            this.resourceMapping = resourceMapping;
            return this;
        }

        @Override
        public StackRefactorAction build() {
            return new StackRefactorAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
