/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about whether the stack's actual configuration differs, or has <i>drifted</i>, from its expected
 * configuration, as defined in the stack template and any values specified as template parameters. A stack is
 * considered to have drifted if one or more of its resources have drifted.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackDriftInformation implements SdkPojo, Serializable,
        ToCopyableBuilder<StackDriftInformation.Builder, StackDriftInformation> {
    private static final SdkField<String> STACK_DRIFT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StackDriftStatus").getter(getter(StackDriftInformation::stackDriftStatusAsString))
            .setter(setter(Builder::stackDriftStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackDriftStatus").build()).build();

    private static final SdkField<Instant> LAST_CHECK_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastCheckTimestamp").getter(getter(StackDriftInformation::lastCheckTimestamp))
            .setter(setter(Builder::lastCheckTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastCheckTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_DRIFT_STATUS_FIELD,
            LAST_CHECK_TIMESTAMP_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String stackDriftStatus;

    private final Instant lastCheckTimestamp;

    private StackDriftInformation(BuilderImpl builder) {
        this.stackDriftStatus = builder.stackDriftStatus;
        this.lastCheckTimestamp = builder.lastCheckTimestamp;
    }

    /**
     * <p>
     * Status of the stack's actual configuration compared to its expected template configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to have
     * drifted if one or more of its resources have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected template
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackDriftStatus}
     * will return {@link StackDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stackDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack's actual configuration compared to its expected template configuration.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered
     *         to have drifted if one or more of its resources have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected template
     *         configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
     *         </p>
     *         </li>
     * @see StackDriftStatus
     */
    public final StackDriftStatus stackDriftStatus() {
        return StackDriftStatus.fromValue(stackDriftStatus);
    }

    /**
     * <p>
     * Status of the stack's actual configuration compared to its expected template configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to have
     * drifted if one or more of its resources have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected template
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackDriftStatus}
     * will return {@link StackDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stackDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack's actual configuration compared to its expected template configuration.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered
     *         to have drifted if one or more of its resources have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected template
     *         configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
     *         </p>
     *         </li>
     * @see StackDriftStatus
     */
    public final String stackDriftStatusAsString() {
        return stackDriftStatus;
    }

    /**
     * <p>
     * Most recent time when a drift detection operation was initiated on the stack, or any of its individual resources
     * that support drift detection.
     * </p>
     * 
     * @return Most recent time when a drift detection operation was initiated on the stack, or any of its individual
     *         resources that support drift detection.
     */
    public final Instant lastCheckTimestamp() {
        return lastCheckTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackDriftStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastCheckTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackDriftInformation)) {
            return false;
        }
        StackDriftInformation other = (StackDriftInformation) obj;
        return Objects.equals(stackDriftStatusAsString(), other.stackDriftStatusAsString())
                && Objects.equals(lastCheckTimestamp(), other.lastCheckTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StackDriftInformation").add("StackDriftStatus", stackDriftStatusAsString())
                .add("LastCheckTimestamp", lastCheckTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackDriftStatus":
            return Optional.ofNullable(clazz.cast(stackDriftStatusAsString()));
        case "LastCheckTimestamp":
            return Optional.ofNullable(clazz.cast(lastCheckTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("StackDriftStatus", STACK_DRIFT_STATUS_FIELD);
        map.put("LastCheckTimestamp", LAST_CHECK_TIMESTAMP_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<StackDriftInformation, T> g) {
        return obj -> g.apply((StackDriftInformation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackDriftInformation> {
        /**
         * <p>
         * Status of the stack's actual configuration compared to its expected template configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to
         * have drifted if one or more of its resources have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected template
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackDriftStatus
         *        Status of the stack's actual configuration compared to its expected template configuration.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is
         *        considered to have drifted if one or more of its resources have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected
         *        template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
         *        </p>
         *        </li>
         * @see StackDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackDriftStatus
         */
        Builder stackDriftStatus(String stackDriftStatus);

        /**
         * <p>
         * Status of the stack's actual configuration compared to its expected template configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to
         * have drifted if one or more of its resources have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected template
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackDriftStatus
         *        Status of the stack's actual configuration compared to its expected template configuration.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is
         *        considered to have drifted if one or more of its resources have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: CloudFormation hasn't checked if the stack differs from its expected
         *        template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNKNOWN</code>: CloudFormation could not run drift detection for a resource in the stack.
         *        </p>
         *        </li>
         * @see StackDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackDriftStatus
         */
        Builder stackDriftStatus(StackDriftStatus stackDriftStatus);

        /**
         * <p>
         * Most recent time when a drift detection operation was initiated on the stack, or any of its individual
         * resources that support drift detection.
         * </p>
         * 
         * @param lastCheckTimestamp
         *        Most recent time when a drift detection operation was initiated on the stack, or any of its individual
         *        resources that support drift detection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastCheckTimestamp(Instant lastCheckTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String stackDriftStatus;

        private Instant lastCheckTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(StackDriftInformation model) {
            stackDriftStatus(model.stackDriftStatus);
            lastCheckTimestamp(model.lastCheckTimestamp);
        }

        public final String getStackDriftStatus() {
            return stackDriftStatus;
        }

        public final void setStackDriftStatus(String stackDriftStatus) {
            this.stackDriftStatus = stackDriftStatus;
        }

        @Override
        public final Builder stackDriftStatus(String stackDriftStatus) {
            this.stackDriftStatus = stackDriftStatus;
            return this;
        }

        @Override
        public final Builder stackDriftStatus(StackDriftStatus stackDriftStatus) {
            this.stackDriftStatus(stackDriftStatus == null ? null : stackDriftStatus.toString());
            return this;
        }

        public final Instant getLastCheckTimestamp() {
            return lastCheckTimestamp;
        }

        public final void setLastCheckTimestamp(Instant lastCheckTimestamp) {
            this.lastCheckTimestamp = lastCheckTimestamp;
        }

        @Override
        public final Builder lastCheckTimestamp(Instant lastCheckTimestamp) {
            this.lastCheckTimestamp = lastCheckTimestamp;
            return this;
        }

        @Override
        public StackDriftInformation build() {
            return new StackDriftInformation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
