/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Structure that contains the results of the account gate function which CloudFormation invokes, if present, before
 * proceeding with a StackSet operation in an account and Region.
 * </p>
 * <p>
 * For each account and Region, CloudFormation lets you specify a Lambda function that encapsulates any requirements
 * that must be met before CloudFormation can proceed with a StackSet operation in that account and Region.
 * CloudFormation invokes the function each time a StackSet operation is requested for that account and Region; if the
 * function returns <code>FAILED</code>, CloudFormation cancels the operation in that account and Region, and sets the
 * StackSet operation result status for that account and Region to <code>FAILED</code>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-account-gating.html">Prevent failed
 * StackSets deployments using target account gates</a> in the <i>CloudFormation User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AccountGateResult implements SdkPojo, Serializable,
        ToCopyableBuilder<AccountGateResult.Builder, AccountGateResult> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(AccountGateResult::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusReason").getter(getter(AccountGateResult::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            STATUS_REASON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String status;

    private final String statusReason;

    private AccountGateResult(BuilderImpl builder) {
        this.status = builder.status;
        this.statusReason = builder.statusReason;
    }

    /**
     * <p>
     * The status of the account gate function.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes any
     * requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation in that account
     * and Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet the
     * requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in that account and
     * Region, and sets the StackSet operation result status for that account and Region to <code>FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account and Region,
     * for one of the following reasons:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An account gate function hasn't been specified for the account and Region. CloudFormation proceeds with the
     * StackSet operation in this account and Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks permissions to invoke
     * the function. CloudFormation proceeds with the StackSet operation in this account and Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the StackSet
     * operation in this account and Region.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountGateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account gate function.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes any
     *         requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation in that
     *         account and Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet
     *         the requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in that
     *         account and Region, and sets the StackSet operation result status for that account and Region to
     *         <code>FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account and
     *         Region, for one of the following reasons:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An account gate function hasn't been specified for the account and Region. CloudFormation proceeds with
     *         the StackSet operation in this account and Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks permissions
     *         to invoke the function. CloudFormation proceeds with the StackSet operation in this account and Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the StackSet
     *         operation in this account and Region.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see AccountGateStatus
     */
    public final AccountGateStatus status() {
        return AccountGateStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the account gate function.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes any
     * requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation in that account
     * and Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet the
     * requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in that account and
     * Region, and sets the StackSet operation result status for that account and Region to <code>FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account and Region,
     * for one of the following reasons:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An account gate function hasn't been specified for the account and Region. CloudFormation proceeds with the
     * StackSet operation in this account and Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks permissions to invoke
     * the function. CloudFormation proceeds with the StackSet operation in this account and Region.
     * </p>
     * </li>
     * <li>
     * <p>
     * Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the StackSet
     * operation in this account and Region.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountGateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account gate function.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes any
     *         requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation in that
     *         account and Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet
     *         the requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in that
     *         account and Region, and sets the StackSet operation result status for that account and Region to
     *         <code>FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account and
     *         Region, for one of the following reasons:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An account gate function hasn't been specified for the account and Region. CloudFormation proceeds with
     *         the StackSet operation in this account and Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks permissions
     *         to invoke the function. CloudFormation proceeds with the StackSet operation in this account and Region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the StackSet
     *         operation in this account and Region.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see AccountGateStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The reason for the account gate status assigned to this account and Region for the StackSet operation.
     * </p>
     * 
     * @return The reason for the account gate status assigned to this account and Region for the StackSet operation.
     */
    public final String statusReason() {
        return statusReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccountGateResult)) {
            return false;
        }
        AccountGateResult other = (AccountGateResult) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(statusReason(), other.statusReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AccountGateResult").add("Status", statusAsString()).add("StatusReason", statusReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Status", STATUS_FIELD);
        map.put("StatusReason", STATUS_REASON_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AccountGateResult, T> g) {
        return obj -> g.apply((AccountGateResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AccountGateResult> {
        /**
         * <p>
         * The status of the account gate function.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes any
         * requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation in that
         * account and Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet the
         * requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in that account
         * and Region, and sets the StackSet operation result status for that account and Region to <code>FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account and
         * Region, for one of the following reasons:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An account gate function hasn't been specified for the account and Region. CloudFormation proceeds with the
         * StackSet operation in this account and Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks permissions to
         * invoke the function. CloudFormation proceeds with the StackSet operation in this account and Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the StackSet
         * operation in this account and Region.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the account gate function.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes
         *        any requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation
         *        in that account and Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet
         *        the requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in
         *        that account and Region, and sets the StackSet operation result status for that account and Region to
         *        <code>FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account
         *        and Region, for one of the following reasons:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An account gate function hasn't been specified for the account and Region. CloudFormation proceeds
         *        with the StackSet operation in this account and Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks
         *        permissions to invoke the function. CloudFormation proceeds with the StackSet operation in this
         *        account and Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the
         *        StackSet operation in this account and Region.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see AccountGateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountGateStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the account gate function.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes any
         * requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation in that
         * account and Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet the
         * requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in that account
         * and Region, and sets the StackSet operation result status for that account and Region to <code>FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account and
         * Region, for one of the following reasons:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An account gate function hasn't been specified for the account and Region. CloudFormation proceeds with the
         * StackSet operation in this account and Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks permissions to
         * invoke the function. CloudFormation proceeds with the StackSet operation in this account and Region.
         * </p>
         * </li>
         * <li>
         * <p>
         * Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the StackSet
         * operation in this account and Region.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the account gate function.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The account gate function has determined that the account and Region passes
         *        any requirements for a StackSet operation to occur. CloudFormation proceeds with the stack operation
         *        in that account and Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The account gate function has determined that the account and Region doesn't meet
         *        the requirements for a StackSet operation to occur. CloudFormation cancels the StackSet operation in
         *        that account and Region, and sets the StackSet operation result status for that account and Region to
         *        <code>FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SKIPPED</code>: CloudFormation has skipped calling the account gate function for this account
         *        and Region, for one of the following reasons:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An account gate function hasn't been specified for the account and Region. CloudFormation proceeds
         *        with the StackSet operation in this account and Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>AWSCloudFormationStackSetExecutionRole</code> of the administration account lacks
         *        permissions to invoke the function. CloudFormation proceeds with the StackSet operation in this
         *        account and Region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Either no action is necessary, or no action is possible, on the stack. CloudFormation skips the
         *        StackSet operation in this account and Region.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see AccountGateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountGateStatus
         */
        Builder status(AccountGateStatus status);

        /**
         * <p>
         * The reason for the account gate status assigned to this account and Region for the StackSet operation.
         * </p>
         * 
         * @param statusReason
         *        The reason for the account gate status assigned to this account and Region for the StackSet operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private String statusReason;

        private BuilderImpl() {
        }

        private BuilderImpl(AccountGateResult model) {
            status(model.status);
            statusReason(model.statusReason);
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AccountGateStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        @Override
        public AccountGateResult build() {
            return new AccountGateResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
