/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input for <a>DeleteStack</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeleteStackRequest extends CloudFormationRequest implements
        ToCopyableBuilder<DeleteStackRequest.Builder, DeleteStackRequest> {
    private static final SdkField<String> STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DeleteStackRequest::stackName)).setter(setter(Builder::stackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackName").build()).build();

    private static final SdkField<List<String>> RETAIN_RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DeleteStackRequest::retainResources))
            .setter(setter(Builder::retainResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetainResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DeleteStackRequest::roleARN)).setter(setter(Builder::roleARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleARN").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DeleteStackRequest::clientRequestToken)).setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_NAME_FIELD,
            RETAIN_RESOURCES_FIELD, ROLE_ARN_FIELD, CLIENT_REQUEST_TOKEN_FIELD));

    private final String stackName;

    private final List<String> retainResources;

    private final String roleARN;

    private final String clientRequestToken;

    private DeleteStackRequest(BuilderImpl builder) {
        super(builder);
        this.stackName = builder.stackName;
        this.retainResources = builder.retainResources;
        this.roleARN = builder.roleARN;
        this.clientRequestToken = builder.clientRequestToken;
    }

    /**
     * <p>
     * The name or the unique stack ID that is associated with the stack.
     * </p>
     * 
     * @return The name or the unique stack ID that is associated with the stack.
     */
    public String stackName() {
        return stackName;
    }

    /**
     * <p>
     * For stacks in the <code>DELETE_FAILED</code> state, a list of resource logical IDs that are associated with the
     * resources you want to retain. During deletion, AWS CloudFormation deletes the stack but does not delete the
     * retained resources.
     * </p>
     * <p>
     * Retaining resources is useful when you cannot delete a resource, such as a non-empty S3 bucket, but you want to
     * delete the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return For stacks in the <code>DELETE_FAILED</code> state, a list of resource logical IDs that are associated
     *         with the resources you want to retain. During deletion, AWS CloudFormation deletes the stack but does not
     *         delete the retained resources.</p>
     *         <p>
     *         Retaining resources is useful when you cannot delete a resource, such as a non-empty S3 bucket, but you
     *         want to delete the stack.
     */
    public List<String> retainResources() {
        return retainResources;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS CloudFormation
     * assumes to delete the stack. AWS CloudFormation uses the role's credentials to make calls on your behalf.
     * </p>
     * <p>
     * If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the stack. If
     * no role is available, AWS CloudFormation uses a temporary session that is generated from your user credentials.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS
     *         CloudFormation assumes to delete the stack. AWS CloudFormation uses the role's credentials to make calls
     *         on your behalf.</p>
     *         <p>
     *         If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the
     *         stack. If no role is available, AWS CloudFormation uses a temporary session that is generated from your
     *         user credentials.
     */
    public String roleARN() {
        return roleARN;
    }

    /**
     * <p>
     * A unique identifier for this <code>DeleteStack</code> request. Specify this token if you plan to retry requests
     * so that AWS CloudFormation knows that you're not attempting to delete a stack with the same name. You might retry
     * <code>DeleteStack</code> requests to ensure that AWS CloudFormation successfully received them.
     * </p>
     * <p>
     * All events triggered by a given stack operation are assigned the same client request token, which you can use to
     * track operations. For example, if you execute a <code>CreateStack</code> operation with the token
     * <code>token1</code>, then all the <code>StackEvents</code> generated by that operation will have
     * <code>ClientRequestToken</code> set as <code>token1</code>.
     * </p>
     * <p>
     * In the console, stack operations display the client request token on the Events tab. Stack operations that are
     * initiated from the console use the token format <i>Console-StackOperation-ID</i>, which helps you easily identify
     * the stack operation . For example, if you create a stack using the console, each stack event would be assigned
     * the same token in the following format: <code>Console-CreateStack-7f59c3cf-00d2-40c7-b2ff-e75db0987002</code>.
     * </p>
     * 
     * @return A unique identifier for this <code>DeleteStack</code> request. Specify this token if you plan to retry
     *         requests so that AWS CloudFormation knows that you're not attempting to delete a stack with the same
     *         name. You might retry <code>DeleteStack</code> requests to ensure that AWS CloudFormation successfully
     *         received them.</p>
     *         <p>
     *         All events triggered by a given stack operation are assigned the same client request token, which you can
     *         use to track operations. For example, if you execute a <code>CreateStack</code> operation with the token
     *         <code>token1</code>, then all the <code>StackEvents</code> generated by that operation will have
     *         <code>ClientRequestToken</code> set as <code>token1</code>.
     *         </p>
     *         <p>
     *         In the console, stack operations display the client request token on the Events tab. Stack operations
     *         that are initiated from the console use the token format <i>Console-StackOperation-ID</i>, which helps
     *         you easily identify the stack operation . For example, if you create a stack using the console, each
     *         stack event would be assigned the same token in the following format:
     *         <code>Console-CreateStack-7f59c3cf-00d2-40c7-b2ff-e75db0987002</code>.
     */
    public String clientRequestToken() {
        return clientRequestToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackName());
        hashCode = 31 * hashCode + Objects.hashCode(retainResources());
        hashCode = 31 * hashCode + Objects.hashCode(roleARN());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeleteStackRequest)) {
            return false;
        }
        DeleteStackRequest other = (DeleteStackRequest) obj;
        return Objects.equals(stackName(), other.stackName()) && Objects.equals(retainResources(), other.retainResources())
                && Objects.equals(roleARN(), other.roleARN()) && Objects.equals(clientRequestToken(), other.clientRequestToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DeleteStackRequest").add("StackName", stackName()).add("RetainResources", retainResources())
                .add("RoleARN", roleARN()).add("ClientRequestToken", clientRequestToken()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackName":
            return Optional.ofNullable(clazz.cast(stackName()));
        case "RetainResources":
            return Optional.ofNullable(clazz.cast(retainResources()));
        case "RoleARN":
            return Optional.ofNullable(clazz.cast(roleARN()));
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeleteStackRequest, T> g) {
        return obj -> g.apply((DeleteStackRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudFormationRequest.Builder, SdkPojo, CopyableBuilder<Builder, DeleteStackRequest> {
        /**
         * <p>
         * The name or the unique stack ID that is associated with the stack.
         * </p>
         * 
         * @param stackName
         *        The name or the unique stack ID that is associated with the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackName(String stackName);

        /**
         * <p>
         * For stacks in the <code>DELETE_FAILED</code> state, a list of resource logical IDs that are associated with
         * the resources you want to retain. During deletion, AWS CloudFormation deletes the stack but does not delete
         * the retained resources.
         * </p>
         * <p>
         * Retaining resources is useful when you cannot delete a resource, such as a non-empty S3 bucket, but you want
         * to delete the stack.
         * </p>
         * 
         * @param retainResources
         *        For stacks in the <code>DELETE_FAILED</code> state, a list of resource logical IDs that are associated
         *        with the resources you want to retain. During deletion, AWS CloudFormation deletes the stack but does
         *        not delete the retained resources.</p>
         *        <p>
         *        Retaining resources is useful when you cannot delete a resource, such as a non-empty S3 bucket, but
         *        you want to delete the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retainResources(Collection<String> retainResources);

        /**
         * <p>
         * For stacks in the <code>DELETE_FAILED</code> state, a list of resource logical IDs that are associated with
         * the resources you want to retain. During deletion, AWS CloudFormation deletes the stack but does not delete
         * the retained resources.
         * </p>
         * <p>
         * Retaining resources is useful when you cannot delete a resource, such as a non-empty S3 bucket, but you want
         * to delete the stack.
         * </p>
         * 
         * @param retainResources
         *        For stacks in the <code>DELETE_FAILED</code> state, a list of resource logical IDs that are associated
         *        with the resources you want to retain. During deletion, AWS CloudFormation deletes the stack but does
         *        not delete the retained resources.</p>
         *        <p>
         *        Retaining resources is useful when you cannot delete a resource, such as a non-empty S3 bucket, but
         *        you want to delete the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retainResources(String... retainResources);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS CloudFormation
         * assumes to delete the stack. AWS CloudFormation uses the role's credentials to make calls on your behalf.
         * </p>
         * <p>
         * If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the stack.
         * If no role is available, AWS CloudFormation uses a temporary session that is generated from your user
         * credentials.
         * </p>
         * 
         * @param roleARN
         *        The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that AWS
         *        CloudFormation assumes to delete the stack. AWS CloudFormation uses the role's credentials to make
         *        calls on your behalf.</p>
         *        <p>
         *        If you don't specify a value, AWS CloudFormation uses the role that was previously associated with the
         *        stack. If no role is available, AWS CloudFormation uses a temporary session that is generated from
         *        your user credentials.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleARN(String roleARN);

        /**
         * <p>
         * A unique identifier for this <code>DeleteStack</code> request. Specify this token if you plan to retry
         * requests so that AWS CloudFormation knows that you're not attempting to delete a stack with the same name.
         * You might retry <code>DeleteStack</code> requests to ensure that AWS CloudFormation successfully received
         * them.
         * </p>
         * <p>
         * All events triggered by a given stack operation are assigned the same client request token, which you can use
         * to track operations. For example, if you execute a <code>CreateStack</code> operation with the token
         * <code>token1</code>, then all the <code>StackEvents</code> generated by that operation will have
         * <code>ClientRequestToken</code> set as <code>token1</code>.
         * </p>
         * <p>
         * In the console, stack operations display the client request token on the Events tab. Stack operations that
         * are initiated from the console use the token format <i>Console-StackOperation-ID</i>, which helps you easily
         * identify the stack operation . For example, if you create a stack using the console, each stack event would
         * be assigned the same token in the following format:
         * <code>Console-CreateStack-7f59c3cf-00d2-40c7-b2ff-e75db0987002</code>.
         * </p>
         * 
         * @param clientRequestToken
         *        A unique identifier for this <code>DeleteStack</code> request. Specify this token if you plan to retry
         *        requests so that AWS CloudFormation knows that you're not attempting to delete a stack with the same
         *        name. You might retry <code>DeleteStack</code> requests to ensure that AWS CloudFormation successfully
         *        received them.</p>
         *        <p>
         *        All events triggered by a given stack operation are assigned the same client request token, which you
         *        can use to track operations. For example, if you execute a <code>CreateStack</code> operation with the
         *        token <code>token1</code>, then all the <code>StackEvents</code> generated by that operation will have
         *        <code>ClientRequestToken</code> set as <code>token1</code>.
         *        </p>
         *        <p>
         *        In the console, stack operations display the client request token on the Events tab. Stack operations
         *        that are initiated from the console use the token format <i>Console-StackOperation-ID</i>, which helps
         *        you easily identify the stack operation . For example, if you create a stack using the console, each
         *        stack event would be assigned the same token in the following format:
         *        <code>Console-CreateStack-7f59c3cf-00d2-40c7-b2ff-e75db0987002</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudFormationRequest.BuilderImpl implements Builder {
        private String stackName;

        private List<String> retainResources = DefaultSdkAutoConstructList.getInstance();

        private String roleARN;

        private String clientRequestToken;

        private BuilderImpl() {
        }

        private BuilderImpl(DeleteStackRequest model) {
            super(model);
            stackName(model.stackName);
            retainResources(model.retainResources);
            roleARN(model.roleARN);
            clientRequestToken(model.clientRequestToken);
        }

        public final String getStackName() {
            return stackName;
        }

        @Override
        public final Builder stackName(String stackName) {
            this.stackName = stackName;
            return this;
        }

        public final void setStackName(String stackName) {
            this.stackName = stackName;
        }

        public final Collection<String> getRetainResources() {
            return retainResources;
        }

        @Override
        public final Builder retainResources(Collection<String> retainResources) {
            this.retainResources = RetainResourcesCopier.copy(retainResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder retainResources(String... retainResources) {
            retainResources(Arrays.asList(retainResources));
            return this;
        }

        public final void setRetainResources(Collection<String> retainResources) {
            this.retainResources = RetainResourcesCopier.copy(retainResources);
        }

        public final String getRoleARN() {
            return roleARN;
        }

        @Override
        public final Builder roleARN(String roleARN) {
            this.roleARN = roleARN;
            return this;
        }

        public final void setRoleARN(String roleARN) {
            this.roleARN = roleARN;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DeleteStackRequest build() {
            return new DeleteStackRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
