/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Structure that contains the results of the account gate function which AWS CloudFormation invokes, if present, before
 * proceeding with a stack set operation in an account and region.
 * </p>
 * <p>
 * For each account and region, AWS CloudFormation lets you specify a Lamdba function that encapsulates any requirements
 * that must be met before CloudFormation can proceed with a stack set operation in that account and region.
 * CloudFormation invokes the function each time a stack set operation is requested for that account and region; if the
 * function returns <code>FAILED</code>, CloudFormation cancels the operation in that account and region, and sets the
 * stack set operation result status for that account and region to <code>FAILED</code>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-account-gating.html">Configuring a
 * target account gate</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AccountGateResult implements SdkPojo, Serializable,
        ToCopyableBuilder<AccountGateResult.Builder, AccountGateResult> {
    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AccountGateResult::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AccountGateResult::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_FIELD,
            STATUS_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String status;

    private final String statusReason;

    private AccountGateResult(BuilderImpl builder) {
        this.status = builder.status;
        this.statusReason = builder.statusReason;
    }

    /**
     * <p>
     * The status of the account gate function.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes any
     * requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack operation in that
     * account and region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The account gate function has determined that the account and region does not meet the
     * requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set operation in that
     * account and region, and sets the stack set operation result status for that account and region to
     * <code>FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this account and
     * region, for one of the following reasons:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An account gate function has not been specified for the account and region. AWS CloudFormation proceeds with the
     * stack set operation in this account and region.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks permissions
     * to invoke the function. AWS CloudFormation proceeds with the stack set operation in this account and region.
     * </p>
     * </li>
     * <li>
     * <p>
     * Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the stack set
     * operation in this account and region.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountGateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account gate function.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes any
     *         requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack operation in
     *         that account and region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The account gate function has determined that the account and region does not meet
     *         the requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set operation
     *         in that account and region, and sets the stack set operation result status for that account and region to
     *         <code>FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this account
     *         and region, for one of the following reasons:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An account gate function has not been specified for the account and region. AWS CloudFormation proceeds
     *         with the stack set operation in this account and region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks
     *         permissions to invoke the function. AWS CloudFormation proceeds with the stack set operation in this
     *         account and region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the stack
     *         set operation in this account and region.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see AccountGateStatus
     */
    public AccountGateStatus status() {
        return AccountGateStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the account gate function.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes any
     * requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack operation in that
     * account and region.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code>: The account gate function has determined that the account and region does not meet the
     * requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set operation in that
     * account and region, and sets the stack set operation result status for that account and region to
     * <code>FAILED</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this account and
     * region, for one of the following reasons:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An account gate function has not been specified for the account and region. AWS CloudFormation proceeds with the
     * stack set operation in this account and region.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks permissions
     * to invoke the function. AWS CloudFormation proceeds with the stack set operation in this account and region.
     * </p>
     * </li>
     * <li>
     * <p>
     * Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the stack set
     * operation in this account and region.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccountGateStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the account gate function.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes any
     *         requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack operation in
     *         that account and region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code>: The account gate function has determined that the account and region does not meet
     *         the requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set operation
     *         in that account and region, and sets the stack set operation result status for that account and region to
     *         <code>FAILED</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this account
     *         and region, for one of the following reasons:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An account gate function has not been specified for the account and region. AWS CloudFormation proceeds
     *         with the stack set operation in this account and region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks
     *         permissions to invoke the function. AWS CloudFormation proceeds with the stack set operation in this
     *         account and region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the stack
     *         set operation in this account and region.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see AccountGateStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The reason for the account gate status assigned to this account and region for the stack set operation.
     * </p>
     * 
     * @return The reason for the account gate status assigned to this account and region for the stack set operation.
     */
    public String statusReason() {
        return statusReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccountGateResult)) {
            return false;
        }
        AccountGateResult other = (AccountGateResult) obj;
        return Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(statusReason(), other.statusReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AccountGateResult").add("Status", statusAsString()).add("StatusReason", statusReason()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AccountGateResult, T> g) {
        return obj -> g.apply((AccountGateResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AccountGateResult> {
        /**
         * <p>
         * The status of the account gate function.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes any
         * requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack operation in that
         * account and region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The account gate function has determined that the account and region does not meet the
         * requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set operation in that
         * account and region, and sets the stack set operation result status for that account and region to
         * <code>FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this account and
         * region, for one of the following reasons:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An account gate function has not been specified for the account and region. AWS CloudFormation proceeds with
         * the stack set operation in this account and region.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks
         * permissions to invoke the function. AWS CloudFormation proceeds with the stack set operation in this account
         * and region.
         * </p>
         * </li>
         * <li>
         * <p>
         * Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the stack set
         * operation in this account and region.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the account gate function.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes
         *        any requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack
         *        operation in that account and region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The account gate function has determined that the account and region does not
         *        meet the requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set
         *        operation in that account and region, and sets the stack set operation result status for that account
         *        and region to <code>FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this
         *        account and region, for one of the following reasons:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An account gate function has not been specified for the account and region. AWS CloudFormation
         *        proceeds with the stack set operation in this account and region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks
         *        permissions to invoke the function. AWS CloudFormation proceeds with the stack set operation in this
         *        account and region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the
         *        stack set operation in this account and region.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see AccountGateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountGateStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the account gate function.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes any
         * requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack operation in that
         * account and region.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code>: The account gate function has determined that the account and region does not meet the
         * requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set operation in that
         * account and region, and sets the stack set operation result status for that account and region to
         * <code>FAILED</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this account and
         * region, for one of the following reasons:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An account gate function has not been specified for the account and region. AWS CloudFormation proceeds with
         * the stack set operation in this account and region.
         * </p>
         * </li>
         * <li>
         * <p>
         * The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks
         * permissions to invoke the function. AWS CloudFormation proceeds with the stack set operation in this account
         * and region.
         * </p>
         * </li>
         * <li>
         * <p>
         * Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the stack set
         * operation in this account and region.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the account gate function.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code>: The account gate function has determined that the account and region passes
         *        any requirements for a stack set operation to occur. AWS CloudFormation proceeds with the stack
         *        operation in that account and region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code>: The account gate function has determined that the account and region does not
         *        meet the requirements for a stack set operation to occur. AWS CloudFormation cancels the stack set
         *        operation in that account and region, and sets the stack set operation result status for that account
         *        and region to <code>FAILED</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SKIPPED</code>: AWS CloudFormation has skipped calling the account gate function for this
         *        account and region, for one of the following reasons:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An account gate function has not been specified for the account and region. AWS CloudFormation
         *        proceeds with the stack set operation in this account and region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>AWSCloudFormationStackSetExecutionRole</code> of the stack set adminstration account lacks
         *        permissions to invoke the function. AWS CloudFormation proceeds with the stack set operation in this
         *        account and region.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Either no action is necessary, or no action is possible, on the stack. AWS CloudFormation skips the
         *        stack set operation in this account and region.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see AccountGateStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountGateStatus
         */
        Builder status(AccountGateStatus status);

        /**
         * <p>
         * The reason for the account gate status assigned to this account and region for the stack set operation.
         * </p>
         * 
         * @param statusReason
         *        The reason for the account gate status assigned to this account and region for the stack set
         *        operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);
    }

    static final class BuilderImpl implements Builder {
        private String status;

        private String statusReason;

        private BuilderImpl() {
        }

        private BuilderImpl(AccountGateResult model) {
            status(model.status);
            statusReason(model.statusReason);
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AccountGateStatus status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        @Override
        public AccountGateResult build() {
            return new AccountGateResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
