/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about whether the stack's actual configuration differs, or has <i>drifted</i>, from its expected
 * configuration, as defined in the stack template and any values specified as template parameters. A stack is
 * considered to have drifted if one or more of its resources have drifted.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StackDriftInformationSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<StackDriftInformationSummary.Builder, StackDriftInformationSummary> {
    private static final SdkField<String> STACK_DRIFT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(StackDriftInformationSummary::stackDriftStatusAsString)).setter(setter(Builder::stackDriftStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackDriftStatus").build()).build();

    private static final SdkField<Instant> LAST_CHECK_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(StackDriftInformationSummary::lastCheckTimestamp)).setter(setter(Builder::lastCheckTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastCheckTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STACK_DRIFT_STATUS_FIELD,
            LAST_CHECK_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String stackDriftStatus;

    private final Instant lastCheckTimestamp;

    private StackDriftInformationSummary(BuilderImpl builder) {
        this.stackDriftStatus = builder.stackDriftStatus;
        this.lastCheckTimestamp = builder.lastCheckTimestamp;
    }

    /**
     * <p>
     * Status of the stack's actual configuration compared to its expected template configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to have
     * drifted if one or more of its resources have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected template
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNKNOWN</code>: This value is reserved for future use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackDriftStatus}
     * will return {@link StackDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stackDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack's actual configuration compared to its expected template configuration. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered
     *         to have drifted if one or more of its resources have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected
     *         template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNKNOWN</code>: This value is reserved for future use.
     *         </p>
     *         </li>
     * @see StackDriftStatus
     */
    public StackDriftStatus stackDriftStatus() {
        return StackDriftStatus.fromValue(stackDriftStatus);
    }

    /**
     * <p>
     * Status of the stack's actual configuration compared to its expected template configuration.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to have
     * drifted if one or more of its resources have drifted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected template
     * configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UNKNOWN</code>: This value is reserved for future use.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stackDriftStatus}
     * will return {@link StackDriftStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stackDriftStatusAsString}.
     * </p>
     * 
     * @return Status of the stack's actual configuration compared to its expected template configuration. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered
     *         to have drifted if one or more of its resources have drifted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected
     *         template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UNKNOWN</code>: This value is reserved for future use.
     *         </p>
     *         </li>
     * @see StackDriftStatus
     */
    public String stackDriftStatusAsString() {
        return stackDriftStatus;
    }

    /**
     * <p>
     * Most recent time when a drift detection operation was initiated on the stack, or any of its individual resources
     * that support drift detection.
     * </p>
     * 
     * @return Most recent time when a drift detection operation was initiated on the stack, or any of its individual
     *         resources that support drift detection.
     */
    public Instant lastCheckTimestamp() {
        return lastCheckTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(stackDriftStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastCheckTimestamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StackDriftInformationSummary)) {
            return false;
        }
        StackDriftInformationSummary other = (StackDriftInformationSummary) obj;
        return Objects.equals(stackDriftStatusAsString(), other.stackDriftStatusAsString())
                && Objects.equals(lastCheckTimestamp(), other.lastCheckTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("StackDriftInformationSummary").add("StackDriftStatus", stackDriftStatusAsString())
                .add("LastCheckTimestamp", lastCheckTimestamp()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StackDriftStatus":
            return Optional.ofNullable(clazz.cast(stackDriftStatusAsString()));
        case "LastCheckTimestamp":
            return Optional.ofNullable(clazz.cast(lastCheckTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StackDriftInformationSummary, T> g) {
        return obj -> g.apply((StackDriftInformationSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StackDriftInformationSummary> {
        /**
         * <p>
         * Status of the stack's actual configuration compared to its expected template configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to
         * have drifted if one or more of its resources have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected template
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNKNOWN</code>: This value is reserved for future use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackDriftStatus
         *        Status of the stack's actual configuration compared to its expected template configuration. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is
         *        considered to have drifted if one or more of its resources have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected
         *        template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNKNOWN</code>: This value is reserved for future use.
         *        </p>
         *        </li>
         * @see StackDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackDriftStatus
         */
        Builder stackDriftStatus(String stackDriftStatus);

        /**
         * <p>
         * Status of the stack's actual configuration compared to its expected template configuration.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is considered to
         * have drifted if one or more of its resources have drifted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected template
         * configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UNKNOWN</code>: This value is reserved for future use.
         * </p>
         * </li>
         * </ul>
         * 
         * @param stackDriftStatus
         *        Status of the stack's actual configuration compared to its expected template configuration. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>DRIFTED</code>: The stack differs from its expected template configuration. A stack is
         *        considered to have drifted if one or more of its resources have drifted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CHECKED</code>: AWS CloudFormation has not checked if the stack differs from its expected
         *        template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IN_SYNC</code>: The stack's actual configuration matches its expected template configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UNKNOWN</code>: This value is reserved for future use.
         *        </p>
         *        </li>
         * @see StackDriftStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StackDriftStatus
         */
        Builder stackDriftStatus(StackDriftStatus stackDriftStatus);

        /**
         * <p>
         * Most recent time when a drift detection operation was initiated on the stack, or any of its individual
         * resources that support drift detection.
         * </p>
         * 
         * @param lastCheckTimestamp
         *        Most recent time when a drift detection operation was initiated on the stack, or any of its individual
         *        resources that support drift detection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastCheckTimestamp(Instant lastCheckTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String stackDriftStatus;

        private Instant lastCheckTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(StackDriftInformationSummary model) {
            stackDriftStatus(model.stackDriftStatus);
            lastCheckTimestamp(model.lastCheckTimestamp);
        }

        public final String getStackDriftStatusAsString() {
            return stackDriftStatus;
        }

        @Override
        public final Builder stackDriftStatus(String stackDriftStatus) {
            this.stackDriftStatus = stackDriftStatus;
            return this;
        }

        @Override
        public final Builder stackDriftStatus(StackDriftStatus stackDriftStatus) {
            this.stackDriftStatus(stackDriftStatus.toString());
            return this;
        }

        public final void setStackDriftStatus(String stackDriftStatus) {
            this.stackDriftStatus = stackDriftStatus;
        }

        public final Instant getLastCheckTimestamp() {
            return lastCheckTimestamp;
        }

        @Override
        public final Builder lastCheckTimestamp(Instant lastCheckTimestamp) {
            this.lastCheckTimestamp = lastCheckTimestamp;
            return this;
        }

        public final void setLastCheckTimestamp(Instant lastCheckTimestamp) {
            this.lastCheckTimestamp = lastCheckTimestamp;
        }

        @Override
        public StackDriftInformationSummary build() {
            return new StackDriftInformationSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
