/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfrontkeyvaluestore.endpoints.internal;

import java.net.URI;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.cloudfrontkeyvaluestore.endpoints.CloudFrontKeyValueStoreEndpointParams;
import software.amazon.awssdk.services.cloudfrontkeyvaluestore.endpoints.CloudFrontKeyValueStoreEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.MapUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultCloudFrontKeyValueStoreEndpointProvider implements CloudFrontKeyValueStoreEndpointProvider {
    private static final Logger LOG = Logger.loggerFor(DefaultCloudFrontKeyValueStoreEndpointProvider.class);

    private static final EndpointRuleset ENDPOINT_RULE_SET = ruleSet();

    private final EndpointAuthSchemeStrategy endpointAuthSchemeStrategy;

    public DefaultCloudFrontKeyValueStoreEndpointProvider() {
        EndpointAuthSchemeStrategyFactory endpointAuthSchemeStrategyFactory = new DefaultEndpointAuthSchemeStrategyFactory();
        this.endpointAuthSchemeStrategy = endpointAuthSchemeStrategyFactory.endpointAuthSchemeStrategy();
    }

    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(CloudFrontKeyValueStoreEndpointParams endpointParams) {
        Validate.notNull(endpointParams.useFips(), "Parameter 'UseFIPS' must not be null");
        Value res = new DefaultRuleEngine().evaluate(ENDPOINT_RULE_SET, toIdentifierValueMap(endpointParams));
        try {
            return CompletableFuture.completedFuture(valueAsEndpointOrThrow(res));
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static Map<Identifier, Value> toIdentifierValueMap(CloudFrontKeyValueStoreEndpointParams params) {
        Map<Identifier, Value> paramsMap = new HashMap<>();
        if (params.kvsArn() != null) {
            paramsMap.put(Identifier.of("KvsARN"), Value.fromStr(params.kvsArn()));
        }
        if (params.region() != null) {
            paramsMap.put(Identifier.of("Region"), Value.fromStr(params.region().id()));
        }
        if (params.useFips() != null) {
            paramsMap.put(Identifier.of("UseFIPS"), Value.fromBool(params.useFips()));
        }
        if (params.endpoint() != null) {
            paramsMap.put(Identifier.of("Endpoint"), Value.fromStr(params.endpoint()));
        }
        return paramsMap;
    }

    Endpoint valueAsEndpointOrThrow(Value value) {
        if (value instanceof Value.Endpoint) {
            Value.Endpoint endpoint = value.expectEndpoint();
            Endpoint.Builder builder = Endpoint.builder();
            builder.url(URI.create(endpoint.getUrl()));
            Map<String, List<String>> headers = endpoint.getHeaders();
            if (headers != null) {
                headers.forEach((name, values) -> values.forEach(v -> builder.putHeader(name, v)));
            }
            addKnownProperties(builder, endpoint.getProperties());
            return builder.build();
        } else if (value instanceof Value.Str) {
            String errorMsg = value.expectString();
            if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
            }
            throw SdkClientException.create(errorMsg);
        } else {
            throw SdkClientException.create("Rule engine return neither an endpoint result or error value. Returned value was: "
                    + value);
        }
    }

    private static Rule endpointRule_14() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{parsedArn#accountId}.{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4a"), Identifier.of("signingName"),
                                        Literal.fromStr("cloudfront-keyvaluestore"), Identifier.of("signingRegionSet"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromStr("*")))))))).build());
    }

    private static Rule endpointRule_13() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build()).treeRule(Arrays.asList(endpointRule_14()));
    }

    private static Rule endpointRule_15() {
        return Rule.builder().error("Provided endpoint is not a valid URL");
    }

    private static Rule endpointRule_12() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_13(), endpointRule_15()));
    }

    private static Rule endpointRule_16() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{parsedArn#accountId}.cloudfront-kvs.global.api.aws"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4a"), Identifier.of("signingName"),
                                        Literal.fromStr("cloudfront-keyvaluestore"), Identifier.of("signingRegionSet"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromStr("*")))))))).build());
    }

    private static Rule endpointRule_11() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("partitionResult")),
                                                                Expr.of("name"))).build().validate(),
                                                Expr.of("{parsedArn#partition}"))).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_12(), endpointRule_16()));
    }

    private static Rule endpointRule_17() {
        return Rule.builder().error(
                "Client was configured for partition `{partitionResult#name}` but Kvs ARN has `{parsedArn#partition}`");
    }

    private static Rule endpointRule_10() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("partitionResult").build())
                .treeRule(Arrays.asList(endpointRule_11(), endpointRule_17()));
    }

    private static Rule endpointRule_9() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Region")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_10()));
    }

    private static Rule endpointRule_20() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("{url#scheme}://{parsedArn#accountId}.{url#authority}{url#path}"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4a"), Identifier.of("signingName"),
                                        Literal.fromStr("cloudfront-keyvaluestore"), Identifier.of("signingRegionSet"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromStr("*")))))))).build());
    }

    private static Rule endpointRule_19() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("parseURL").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint"))))
                                        .build().validate()).result("url").build()).treeRule(Arrays.asList(endpointRule_20()));
    }

    private static Rule endpointRule_21() {
        return Rule.builder().error("Provided endpoint is not a valid URL");
    }

    private static Rule endpointRule_18() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_19(), endpointRule_21()));
    }

    private static Rule endpointRule_22() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("https://{parsedArn#accountId}.cloudfront-kvs.global.api.aws"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4a"), Identifier.of("signingName"),
                                        Literal.fromStr("cloudfront-keyvaluestore"), Identifier.of("signingRegionSet"),
                                        Literal.fromTuple(Arrays.asList(Literal.fromStr("*")))))))).build());
    }

    private static Rule endpointRule_8() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("parsedArn")),
                                                                Expr.of("partition"))).build().validate(), Expr.of("aws")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_9(), endpointRule_18(), endpointRule_22()));
    }

    private static Rule endpointRule_23() {
        return Rule.builder().error("CloudFront-KeyValueStore is not supported in partition `{parsedArn#partition}`");
    }

    private static Rule endpointRule_7() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of("key-value-store")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_8(), endpointRule_23()));
    }

    private static Rule endpointRule_24() {
        return Rule.builder().error("ARN resource type is invalid. Expected `key-value-store`, found: `{arnType}`");
    }

    private static Rule endpointRule_6() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("stringEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("arnType")), Expr.of(""))).build()
                                                .validate())).build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_7(), endpointRule_24()));
    }

    private static Rule endpointRule_25() {
        return Rule.builder().error("No resource type found in the KVS ARN. Resource type must be `key-value-store`.");
    }

    private static Rule endpointRule_5() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("getAttr")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("parsedArn")), Expr.of("resourceId[0]")))
                                        .build().validate()).result("arnType").build())
                .treeRule(Arrays.asList(endpointRule_6(), endpointRule_25()));
    }

    private static Rule endpointRule_26() {
        return Rule.builder().error("No resource type found in the KVS ARN. Resource type must be `key-value-store`.");
    }

    private static Rule endpointRule_4() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("parsedArn")),
                                                                Expr.of("region"))).build().validate(), Expr.of(""))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_5(), endpointRule_26()));
    }

    private static Rule endpointRule_27() {
        return Rule.builder().error("Provided ARN must be a global resource ARN. Found: `{parsedArn#region}`");
    }

    private static Rule endpointRule_3() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("parsedArn")),
                                                                Expr.of("service"))).build().validate(), Expr.of("cloudfront")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_4(), endpointRule_27()));
    }

    private static Rule endpointRule_28() {
        return Rule.builder().error("Provided ARN is not a valid CloudFront Service ARN. Found: `{parsedArn#service}`");
    }

    private static Rule endpointRule_2() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.parseArn").argv(Arrays.asList(Expr.ref(Identifier.of("KvsARN"))))
                                        .build().validate()).result("parsedArn").build())
                .treeRule(Arrays.asList(endpointRule_3(), endpointRule_28()));
    }

    private static Rule endpointRule_29() {
        return Rule.builder().error("KVS ARN must be a valid ARN");
    }

    private static Rule endpointRule_1() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("KvsARN")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_2(), endpointRule_29()));
    }

    private static Rule endpointRule_30() {
        return Rule.builder().error("KVS ARN must be provided to use this service");
    }

    private static Rule endpointRule_0() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(false))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_1(), endpointRule_30()));
    }

    private static Rule endpointRule_31() {
        return Rule.builder().error("Invalid Configuration: FIPS is not supported with CloudFront-KeyValueStore.");
    }

    private static EndpointRuleset ruleSet() {
        return EndpointRuleset
                .builder()
                .version("1.0")
                .serviceId(null)
                .parameters(
                        Parameters
                                .builder()
                                .addParameter(
                                        Parameter.builder().name("KvsARN").type(ParameterType.fromValue("String"))
                                                .required(false).documentation("The ARN of the Key Value Store").build())
                                .addParameter(
                                        Parameter.builder().name("Region").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("AWS::Region")
                                                .documentation("The AWS region used to dispatch the request.").build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseFIPS")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::UseFIPS")
                                                .documentation(
                                                        "When true, send this request to the FIPS-compliant regional endpoint. If the configured endpoint does not have a FIPS compliant endpoint, dispatching the request will return an error.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter.builder().name("Endpoint").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("SDK::Endpoint")
                                                .documentation("Override the endpoint used to send this request").build())
                                .build()).addRule(endpointRule_0()).addRule(endpointRule_31()).build();
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }

    private void addKnownProperties(Endpoint.Builder builder, Map<String, Value> properties) {
        properties.forEach((n, v) -> {
            switch (n) {
            case "authSchemes":
                builder.putAttribute(AwsEndpointAttribute.AUTH_SCHEMES, endpointAuthSchemeStrategy.createAuthSchemes(v));
                break;
            default:
                LOG.debug(() -> "Ignoring unknown endpoint property: " + n);
                break;
            }
        });
    }
}
