/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfrontkeyvaluestore.auth.scheme.internal;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.EndpointAuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4aAuthScheme;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.http.auth.aws.scheme.AwsV4AuthScheme;
import software.amazon.awssdk.http.auth.aws.scheme.AwsV4aAuthScheme;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4HttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4aHttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.RegionSet;
import software.amazon.awssdk.http.auth.spi.scheme.AuthSchemeOption;
import software.amazon.awssdk.services.cloudfrontkeyvaluestore.auth.scheme.CloudFrontKeyValueStoreAuthSchemeParams;
import software.amazon.awssdk.services.cloudfrontkeyvaluestore.auth.scheme.CloudFrontKeyValueStoreAuthSchemeProvider;
import software.amazon.awssdk.services.cloudfrontkeyvaluestore.endpoints.CloudFrontKeyValueStoreEndpointParams;
import software.amazon.awssdk.services.cloudfrontkeyvaluestore.endpoints.CloudFrontKeyValueStoreEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultCloudFrontKeyValueStoreAuthSchemeProvider implements CloudFrontKeyValueStoreAuthSchemeProvider {
    private static final DefaultCloudFrontKeyValueStoreAuthSchemeProvider DEFAULT = new DefaultCloudFrontKeyValueStoreAuthSchemeProvider();

    private static final CloudFrontKeyValueStoreAuthSchemeProvider MODELED_RESOLVER = ModeledCloudFrontKeyValueStoreAuthSchemeProvider
            .create();

    private static final CloudFrontKeyValueStoreEndpointProvider DELEGATE = CloudFrontKeyValueStoreEndpointProvider
            .defaultProvider();

    private DefaultCloudFrontKeyValueStoreAuthSchemeProvider() {
    }

    public static CloudFrontKeyValueStoreAuthSchemeProvider create() {
        return DEFAULT;
    }

    @Override
    public List<AuthSchemeOption> resolveAuthScheme(CloudFrontKeyValueStoreAuthSchemeParams params) {
        CloudFrontKeyValueStoreEndpointParams endpointParameters = CloudFrontKeyValueStoreEndpointParams.builder()
                .kvsArn(params.kvsArn()).region(params.region()).useFips(params.useFips()).endpoint(params.endpoint()).build();
        Endpoint endpoint = CompletableFutureUtils.joinLikeSync(endpointProvider(params).resolveEndpoint(endpointParameters));
        List<EndpointAuthScheme> authSchemes = endpoint.attribute(AwsEndpointAttribute.AUTH_SCHEMES);
        if (authSchemes == null) {
            return MODELED_RESOLVER.resolveAuthScheme(params);
        }
        List<AuthSchemeOption> options = new ArrayList<>();
        for (EndpointAuthScheme authScheme : authSchemes) {
            String name = authScheme.name();
            switch (name) {
            case "sigv4":
                SigV4AuthScheme sigv4AuthScheme = Validate.isInstanceOf(SigV4AuthScheme.class, authScheme,
                        "Expecting auth scheme of class SigV4AuthScheme, got instead object of class %s", authScheme.getClass()
                                .getName());
                AuthSchemeOption sigv4AuthSchemeOption = AuthSchemeOption.builder().schemeId(AwsV4AuthScheme.SCHEME_ID)
                        .putSignerProperty(AwsV4HttpSigner.SERVICE_SIGNING_NAME, sigv4AuthScheme.signingName())
                        .putSignerProperty(AwsV4HttpSigner.REGION_NAME, sigv4AuthScheme.signingRegion())
                        .putSignerProperty(AwsV4HttpSigner.DOUBLE_URL_ENCODE, !sigv4AuthScheme.disableDoubleEncoding()).build();
                options.add(sigv4AuthSchemeOption);
                break;
            case "sigv4a":
                SigV4aAuthScheme sigv4aAuthScheme = Validate.isInstanceOf(SigV4aAuthScheme.class, authScheme,
                        "Expecting auth scheme of class SigV4AuthScheme, got instead object of class %s", authScheme.getClass()
                                .getName());
                RegionSet regionSet = RegionSet.create(sigv4aAuthScheme.signingRegionSet());
                AuthSchemeOption sigv4aAuthSchemeOption = AuthSchemeOption.builder().schemeId(AwsV4aAuthScheme.SCHEME_ID)
                        .putSignerProperty(AwsV4HttpSigner.SERVICE_SIGNING_NAME, sigv4aAuthScheme.signingName())
                        .putSignerProperty(AwsV4aHttpSigner.REGION_SET, regionSet)
                        .putSignerProperty(AwsV4HttpSigner.DOUBLE_URL_ENCODE, !sigv4aAuthScheme.disableDoubleEncoding()).build();
                options.add(sigv4aAuthSchemeOption);
                break;
            default:
                throw new IllegalArgumentException("Unknown auth scheme: " + name);
            }
        }
        return Collections.unmodifiableList(options);
    }

    private CloudFrontKeyValueStoreEndpointProvider endpointProvider(CloudFrontKeyValueStoreAuthSchemeParams params) {
        if (params instanceof CloudFrontKeyValueStoreEndpointResolverAware) {
            CloudFrontKeyValueStoreEndpointResolverAware endpointAwareParams = (CloudFrontKeyValueStoreEndpointResolverAware) params;
            CloudFrontKeyValueStoreEndpointProvider endpointProvider = endpointAwareParams.endpointProvider();
            if (endpointProvider != null) {
                return endpointProvider;
            }
        }
        return DELEGATE;
    }
}
