/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudhsm;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudhsm.model.AddTagsToResourceRequest;
import software.amazon.awssdk.services.cloudhsm.model.AddTagsToResourceResponse;
import software.amazon.awssdk.services.cloudhsm.model.CloudHsmException;
import software.amazon.awssdk.services.cloudhsm.model.CloudHsmInternalException;
import software.amazon.awssdk.services.cloudhsm.model.CloudHsmServiceException;
import software.amazon.awssdk.services.cloudhsm.model.CreateHapgRequest;
import software.amazon.awssdk.services.cloudhsm.model.CreateHapgResponse;
import software.amazon.awssdk.services.cloudhsm.model.CreateHsmRequest;
import software.amazon.awssdk.services.cloudhsm.model.CreateHsmResponse;
import software.amazon.awssdk.services.cloudhsm.model.CreateLunaClientRequest;
import software.amazon.awssdk.services.cloudhsm.model.CreateLunaClientResponse;
import software.amazon.awssdk.services.cloudhsm.model.DeleteHapgRequest;
import software.amazon.awssdk.services.cloudhsm.model.DeleteHapgResponse;
import software.amazon.awssdk.services.cloudhsm.model.DeleteHsmRequest;
import software.amazon.awssdk.services.cloudhsm.model.DeleteHsmResponse;
import software.amazon.awssdk.services.cloudhsm.model.DeleteLunaClientRequest;
import software.amazon.awssdk.services.cloudhsm.model.DeleteLunaClientResponse;
import software.amazon.awssdk.services.cloudhsm.model.DescribeHapgRequest;
import software.amazon.awssdk.services.cloudhsm.model.DescribeHapgResponse;
import software.amazon.awssdk.services.cloudhsm.model.DescribeHsmRequest;
import software.amazon.awssdk.services.cloudhsm.model.DescribeHsmResponse;
import software.amazon.awssdk.services.cloudhsm.model.DescribeLunaClientRequest;
import software.amazon.awssdk.services.cloudhsm.model.DescribeLunaClientResponse;
import software.amazon.awssdk.services.cloudhsm.model.GetConfigRequest;
import software.amazon.awssdk.services.cloudhsm.model.GetConfigResponse;
import software.amazon.awssdk.services.cloudhsm.model.InvalidRequestException;
import software.amazon.awssdk.services.cloudhsm.model.ListAvailableZonesRequest;
import software.amazon.awssdk.services.cloudhsm.model.ListAvailableZonesResponse;
import software.amazon.awssdk.services.cloudhsm.model.ListHapgsRequest;
import software.amazon.awssdk.services.cloudhsm.model.ListHapgsResponse;
import software.amazon.awssdk.services.cloudhsm.model.ListHsmsRequest;
import software.amazon.awssdk.services.cloudhsm.model.ListHsmsResponse;
import software.amazon.awssdk.services.cloudhsm.model.ListLunaClientsRequest;
import software.amazon.awssdk.services.cloudhsm.model.ListLunaClientsResponse;
import software.amazon.awssdk.services.cloudhsm.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloudhsm.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloudhsm.model.ModifyHapgRequest;
import software.amazon.awssdk.services.cloudhsm.model.ModifyHapgResponse;
import software.amazon.awssdk.services.cloudhsm.model.ModifyHsmRequest;
import software.amazon.awssdk.services.cloudhsm.model.ModifyHsmResponse;
import software.amazon.awssdk.services.cloudhsm.model.ModifyLunaClientRequest;
import software.amazon.awssdk.services.cloudhsm.model.ModifyLunaClientResponse;
import software.amazon.awssdk.services.cloudhsm.model.RemoveTagsFromResourceRequest;
import software.amazon.awssdk.services.cloudhsm.model.RemoveTagsFromResourceResponse;
import software.amazon.awssdk.services.cloudhsm.transform.AddTagsToResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.CreateHapgRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.CreateHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.CreateLunaClientRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.DeleteHapgRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.DeleteHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.DeleteLunaClientRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.DescribeHapgRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.DescribeHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.DescribeLunaClientRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.GetConfigRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ListAvailableZonesRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ListHapgsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ListHsmsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ListLunaClientsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ModifyHapgRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ModifyHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.ModifyLunaClientRequestMarshaller;
import software.amazon.awssdk.services.cloudhsm.transform.RemoveTagsFromResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudHsmClient}.
 *
 * @see CloudHsmClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudHsmClient implements CloudHsmClient {
    private static final Logger log = Logger.loggerFor(DefaultCloudHsmClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudHsmClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Adds or overwrites one or more tags for the specified AWS CloudHSM resource.
     * </p>
     * <p>
     * Each tag consists of a key and a value. Tag keys must be unique to each resource.
     * </p>
     *
     * @param addTagsToResourceRequest
     * @return Result of the AddTagsToResource operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.AddTagsToResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/AddTagsToResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AddTagsToResourceResponse addTagsToResource(AddTagsToResourceRequest addTagsToResourceRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsToResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsToResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsToResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTagsToResource");

            return clientHandler.execute(new ClientExecutionParams<AddTagsToResourceRequest, AddTagsToResourceResponse>()
                    .withOperationName("AddTagsToResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addTagsToResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AddTagsToResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Creates a high-availability partition group. A high-availability partition group is a group of partitions that
     * spans multiple physical HSMs.
     * </p>
     *
     * @param createHapgRequest
     *        Contains the inputs for the <a>CreateHapgRequest</a> action.
     * @return Result of the CreateHapg operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.CreateHapg
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/CreateHapg" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateHapgResponse createHapg(CreateHapgRequest createHapgRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHapgResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateHapgResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHapgRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHapg");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateHapgRequest, CreateHapgResponse>().withOperationName("CreateHapg")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createHapgRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateHapgRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Creates an uninitialized HSM instance.
     * </p>
     * <p>
     * There is an upfront fee charged for each HSM instance that you create with the <code>CreateHsm</code> operation.
     * If you accidentally provision an HSM and want to request a refund, delete the instance using the <a>DeleteHsm</a>
     * operation, go to the <a href="https://console.aws.amazon.com/support/home">AWS Support Center</a>, create a new
     * case, and select <b>Account and Billing Support</b>.
     * </p>
     * <important>
     * <p>
     * It can take up to 20 minutes to create and provision an HSM. You can monitor the status of the HSM with the
     * <a>DescribeHsm</a> operation. The HSM is ready to be initialized when the status changes to <code>RUNNING</code>.
     * </p>
     * </important>
     *
     * @param createHsmRequest
     *        Contains the inputs for the <code>CreateHsm</code> operation.
     * @return Result of the CreateHsm operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.CreateHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/CreateHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateHsmResponse createHsm(CreateHsmRequest createHsmRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHsm");

            return clientHandler.execute(new ClientExecutionParams<CreateHsmRequest, CreateHsmResponse>()
                    .withOperationName("CreateHsm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createHsmRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Creates an HSM client.
     * </p>
     *
     * @param createLunaClientRequest
     *        Contains the inputs for the <a>CreateLunaClient</a> action.
     * @return Result of the CreateLunaClient operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.CreateLunaClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/CreateLunaClient" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateLunaClientResponse createLunaClient(CreateLunaClientRequest createLunaClientRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateLunaClientResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateLunaClientResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLunaClientRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLunaClient");

            return clientHandler.execute(new ClientExecutionParams<CreateLunaClientRequest, CreateLunaClientResponse>()
                    .withOperationName("CreateLunaClient").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createLunaClientRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateLunaClientRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Deletes a high-availability partition group.
     * </p>
     *
     * @param deleteHapgRequest
     *        Contains the inputs for the <a>DeleteHapg</a> action.
     * @return Result of the DeleteHapg operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.DeleteHapg
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/DeleteHapg" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteHapgResponse deleteHapg(DeleteHapgRequest deleteHapgRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHapgResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHapgResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHapgRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHapg");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteHapgRequest, DeleteHapgResponse>().withOperationName("DeleteHapg")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteHapgRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteHapgRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Deletes an HSM. After completion, this operation cannot be undone and your key material cannot be recovered.
     * </p>
     *
     * @param deleteHsmRequest
     *        Contains the inputs for the <a>DeleteHsm</a> operation.
     * @return Result of the DeleteHsm operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.DeleteHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/DeleteHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteHsmResponse deleteHsm(DeleteHsmRequest deleteHsmRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHsm");

            return clientHandler.execute(new ClientExecutionParams<DeleteHsmRequest, DeleteHsmResponse>()
                    .withOperationName("DeleteHsm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteHsmRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new DeleteHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Deletes a client.
     * </p>
     *
     * @param deleteLunaClientRequest
     * @return Result of the DeleteLunaClient operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.DeleteLunaClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/DeleteLunaClient" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteLunaClientResponse deleteLunaClient(DeleteLunaClientRequest deleteLunaClientRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteLunaClientResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteLunaClientResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLunaClientRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLunaClient");

            return clientHandler.execute(new ClientExecutionParams<DeleteLunaClientRequest, DeleteLunaClientResponse>()
                    .withOperationName("DeleteLunaClient").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteLunaClientRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteLunaClientRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Retrieves information about a high-availability partition group.
     * </p>
     *
     * @param describeHapgRequest
     *        Contains the inputs for the <a>DescribeHapg</a> action.
     * @return Result of the DescribeHapg operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.DescribeHapg
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/DescribeHapg" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeHapgResponse describeHapg(DescribeHapgRequest describeHapgRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeHapgResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeHapgResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHapgRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHapg");

            return clientHandler.execute(new ClientExecutionParams<DescribeHapgRequest, DescribeHapgResponse>()
                    .withOperationName("DescribeHapg").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeHapgRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeHapgRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Retrieves information about an HSM. You can identify the HSM by its ARN or its serial number.
     * </p>
     *
     * @param describeHsmRequest
     *        Contains the inputs for the <a>DescribeHsm</a> operation.
     * @return Result of the DescribeHsm operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.DescribeHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/DescribeHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeHsmResponse describeHsm(DescribeHsmRequest describeHsmRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeHsm");

            return clientHandler.execute(new ClientExecutionParams<DescribeHsmRequest, DescribeHsmResponse>()
                    .withOperationName("DescribeHsm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeHsmRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Retrieves information about an HSM client.
     * </p>
     *
     * @param describeLunaClientRequest
     * @return Result of the DescribeLunaClient operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.DescribeLunaClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/DescribeLunaClient" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeLunaClientResponse describeLunaClient(DescribeLunaClientRequest describeLunaClientRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLunaClientResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeLunaClientResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLunaClientRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLunaClient");

            return clientHandler.execute(new ClientExecutionParams<DescribeLunaClientRequest, DescribeLunaClientResponse>()
                    .withOperationName("DescribeLunaClient").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeLunaClientRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLunaClientRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Gets the configuration files necessary to connect to all high availability partition groups the client is
     * associated with.
     * </p>
     *
     * @param getConfigRequest
     * @return Result of the GetConfig operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.GetConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/GetConfig" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetConfigResponse getConfig(GetConfigRequest getConfigRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConfigResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConfigResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfig");

            return clientHandler.execute(new ClientExecutionParams<GetConfigRequest, GetConfigResponse>()
                    .withOperationName("GetConfig").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getConfigRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetConfigRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Lists the Availability Zones that have available AWS CloudHSM capacity.
     * </p>
     *
     * @param listAvailableZonesRequest
     *        Contains the inputs for the <a>ListAvailableZones</a> action.
     * @return Result of the ListAvailableZones operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ListAvailableZones
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ListAvailableZones" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListAvailableZonesResponse listAvailableZones(ListAvailableZonesRequest listAvailableZonesRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAvailableZonesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAvailableZonesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAvailableZonesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAvailableZones");

            return clientHandler.execute(new ClientExecutionParams<ListAvailableZonesRequest, ListAvailableZonesResponse>()
                    .withOperationName("ListAvailableZones").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAvailableZonesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAvailableZonesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Lists the high-availability partition groups for the account.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> member. If more results are
     * available, the <code>NextToken</code> member of the response contains a token that you pass in the next call to
     * <code>ListHapgs</code> to retrieve the next set of items.
     * </p>
     *
     * @param listHapgsRequest
     * @return Result of the ListHapgs operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ListHapgs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ListHapgs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListHapgsResponse listHapgs(ListHapgsRequest listHapgsRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHapgsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHapgsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHapgsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHapgs");

            return clientHandler.execute(new ClientExecutionParams<ListHapgsRequest, ListHapgsResponse>()
                    .withOperationName("ListHapgs").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listHapgsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListHapgsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Retrieves the identifiers of all of the HSMs provisioned for the current customer.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> member. If more results are
     * available, the <code>NextToken</code> member of the response contains a token that you pass in the next call to
     * <code>ListHsms</code> to retrieve the next set of items.
     * </p>
     *
     * @param listHsmsRequest
     * @return Result of the ListHsms operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ListHsms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ListHsms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListHsmsResponse listHsms(ListHsmsRequest listHsmsRequest) throws CloudHsmServiceException, CloudHsmInternalException,
            InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHsmsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListHsmsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listHsmsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHsms");

            return clientHandler.execute(new ClientExecutionParams<ListHsmsRequest, ListHsmsResponse>()
                    .withOperationName("ListHsms").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listHsmsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListHsmsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Lists all of the clients.
     * </p>
     * <p>
     * This operation supports pagination with the use of the <code>NextToken</code> member. If more results are
     * available, the <code>NextToken</code> member of the response contains a token that you pass in the next call to
     * <code>ListLunaClients</code> to retrieve the next set of items.
     * </p>
     *
     * @param listLunaClientsRequest
     * @return Result of the ListLunaClients operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ListLunaClients
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ListLunaClients" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListLunaClientsResponse listLunaClients(ListLunaClientsRequest listLunaClientsRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListLunaClientsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListLunaClientsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLunaClientsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLunaClients");

            return clientHandler.execute(new ClientExecutionParams<ListLunaClientsRequest, ListLunaClientsResponse>()
                    .withOperationName("ListLunaClients").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listLunaClientsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListLunaClientsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Returns a list of all tags for the specified AWS CloudHSM resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Modifies an existing high-availability partition group.
     * </p>
     *
     * @param modifyHapgRequest
     * @return Result of the ModifyHapg operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ModifyHapg
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ModifyHapg" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ModifyHapgResponse modifyHapg(ModifyHapgRequest modifyHapgRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyHapgResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ModifyHapgResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyHapgRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyHapg");

            return clientHandler
                    .execute(new ClientExecutionParams<ModifyHapgRequest, ModifyHapgResponse>().withOperationName("ModifyHapg")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(modifyHapgRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ModifyHapgRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Modifies an HSM.
     * </p>
     * <important>
     * <p>
     * This operation can result in the HSM being offline for up to 15 minutes while the AWS CloudHSM service is
     * reconfigured. If you are modifying a production HSM, you should ensure that your AWS CloudHSM service is
     * configured for high availability, and consider executing this operation during a maintenance window.
     * </p>
     * </important>
     *
     * @param modifyHsmRequest
     *        Contains the inputs for the <a>ModifyHsm</a> operation.
     * @return Result of the ModifyHsm operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ModifyHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ModifyHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ModifyHsmResponse modifyHsm(ModifyHsmRequest modifyHsmRequest) throws CloudHsmServiceException,
            CloudHsmInternalException, InvalidRequestException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ModifyHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyHsm");

            return clientHandler.execute(new ClientExecutionParams<ModifyHsmRequest, ModifyHsmResponse>()
                    .withOperationName("ModifyHsm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(modifyHsmRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ModifyHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Modifies the certificate used by the client.
     * </p>
     * <p>
     * This action can potentially start a workflow to install the new certificate on the client's HSMs.
     * </p>
     *
     * @param modifyLunaClientRequest
     * @return Result of the ModifyLunaClient operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.ModifyLunaClient
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/ModifyLunaClient" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ModifyLunaClientResponse modifyLunaClient(ModifyLunaClientRequest modifyLunaClientRequest)
            throws CloudHsmServiceException, AwsServiceException, SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyLunaClientResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ModifyLunaClientResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyLunaClientRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyLunaClient");

            return clientHandler.execute(new ClientExecutionParams<ModifyLunaClientRequest, ModifyLunaClientResponse>()
                    .withOperationName("ModifyLunaClient").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(modifyLunaClientRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ModifyLunaClientRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This is documentation for <b>AWS CloudHSM Classic</b>. For more information, see <a
     * href="http://aws.amazon.com/cloudhsm/faqs-classic/">AWS CloudHSM Classic FAQs</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/userguide/">AWS CloudHSM Classic User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/classic/APIReference/">AWS CloudHSM Classic API Reference</a>.
     * </p>
     * <p>
     * <b>For information about the current version of AWS CloudHSM</b>, see <a
     * href="http://aws.amazon.com/cloudhsm/">AWS CloudHSM</a>, the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/">AWS CloudHSM User Guide</a>, and the <a
     * href="https://docs.aws.amazon.com/cloudhsm/latest/APIReference/">AWS CloudHSM API Reference</a>.
     * </p>
     * <p>
     * Removes one or more tags from the specified AWS CloudHSM resource.
     * </p>
     * <p>
     * To remove a tag, specify only the tag key to remove (not the value). To overwrite the value for an existing tag,
     * use <a>AddTagsToResource</a>.
     * </p>
     *
     * @param removeTagsFromResourceRequest
     * @return Result of the RemoveTagsFromResource operation returned by the service.
     * @throws CloudHsmServiceException
     *         Indicates that an exception occurred in the AWS CloudHSM service.
     * @throws CloudHsmInternalException
     *         Indicates that an internal error occurred.
     * @throws InvalidRequestException
     *         Indicates that one or more of the request parameters are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmClient.RemoveTagsFromResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsm-2014-05-30/RemoveTagsFromResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RemoveTagsFromResourceResponse removeTagsFromResource(RemoveTagsFromResourceRequest removeTagsFromResourceRequest)
            throws CloudHsmServiceException, CloudHsmInternalException, InvalidRequestException, AwsServiceException,
            SdkClientException, CloudHsmException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsFromResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RemoveTagsFromResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsFromResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTagsFromResource");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveTagsFromResourceRequest, RemoveTagsFromResourceResponse>()
                            .withOperationName("RemoveTagsFromResource").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(removeTagsFromResourceRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveTagsFromResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudHsmException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInternalException")
                                .exceptionBuilderSupplier(CloudHsmInternalException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmServiceException")
                                .exceptionBuilderSupplier(CloudHsmServiceException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
