/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudhsmv2;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmAccessDeniedException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInternalFailureException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInvalidRequestException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmResourceNotFoundException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmServiceException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmTagException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmV2Exception;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmV2Request;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeBackupsPublisher;
import software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeClustersPublisher;
import software.amazon.awssdk.services.cloudhsmv2.paginators.ListTagsPublisher;
import software.amazon.awssdk.services.cloudhsmv2.transform.CopyBackupToRegionRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeClustersRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.InitializeClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyBackupAttributesRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.RestoreBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudHsmV2AsyncClient}.
 *
 * @see CloudHsmV2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudHsmV2AsyncClient implements CloudHsmV2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudHsmV2AsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudHsmV2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Copy an AWS CloudHSM cluster backup to a different region.
     * </p>
     *
     * @param copyBackupToRegionRequest
     * @return A Java Future containing the result of the CopyBackupToRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.CopyBackupToRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CopyBackupToRegion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CopyBackupToRegionResponse> copyBackupToRegion(CopyBackupToRegionRequest copyBackupToRegionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyBackupToRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyBackupToRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CopyBackupToRegionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CopyBackupToRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CopyBackupToRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CopyBackupToRegionRequest, CopyBackupToRegionResponse>()
                            .withOperationName("CopyBackupToRegion")
                            .withMarshaller(new CopyBackupToRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(copyBackupToRegionRequest));
            CompletableFuture<CopyBackupToRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new AWS CloudHSM cluster.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster")
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new hardware security module (HSM) in the specified AWS CloudHSM cluster.
     * </p>
     *
     * @param createHsmRequest
     * @return A Java Future containing the result of the CreateHsm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.CreateHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHsmResponse> createHsm(CreateHsmRequest createHsmRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHsm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateHsmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHsmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHsmRequest, CreateHsmResponse>().withOperationName("CreateHsm")
                            .withMarshaller(new CreateHsmRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(createHsmRequest));
            CompletableFuture<CreateHsmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified AWS CloudHSM backup. A backup can be restored up to 7 days after the DeleteBackup request is
     * made. For more information on restoring a backup, see <a>RestoreBackup</a>.
     * </p>
     *
     * @param deleteBackupRequest
     * @return A Java Future containing the result of the DeleteBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DeleteBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackupResponse> deleteBackup(DeleteBackupRequest deleteBackupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                            .withOperationName("DeleteBackup").withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBackupRequest));
            CompletableFuture<DeleteBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified AWS CloudHSM cluster. Before you can delete a cluster, you must delete all HSMs in the
     * cluster. To see if the cluster contains any HSMs, use <a>DescribeClusters</a>. To delete an HSM, use
     * <a>DeleteHsm</a>.
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                            .withOperationName("DeleteCluster")
                            .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteClusterRequest));
            CompletableFuture<DeleteClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified HSM. To specify an HSM, you can use its identifier (ID), the IP address of the HSM's
     * elastic network interface (ENI), or the ID of the HSM's ENI. You need to specify only one of these values. To
     * find these values, use <a>DescribeClusters</a>.
     * </p>
     *
     * @param deleteHsmRequest
     * @return A Java Future containing the result of the DeleteHsm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DeleteHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHsmResponse> deleteHsm(DeleteHsmRequest deleteHsmRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHsm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteHsmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHsmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHsmRequest, DeleteHsmResponse>().withOperationName("DeleteHsm")
                            .withMarshaller(new DeleteHsmRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteHsmRequest));
            CompletableFuture<DeleteHsmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about backups of AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the backups.
     * When the response contains only a subset of backups, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeBackups</code> request to get more backups. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more backups to get.
     * </p>
     *
     * @param describeBackupsRequest
     * @return A Java Future containing the result of the DescribeBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeBackups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBackupsResponse> describeBackups(DescribeBackupsRequest describeBackupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                            .withOperationName("DescribeBackups")
                            .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeBackupsRequest));
            CompletableFuture<DescribeBackupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about backups of AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the backups.
     * When the response contains only a subset of backups, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeBackups</code> request to get more backups. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more backups to get.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBackups(software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeBackupsPublisher publisher = client.describeBackupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeBackupsPublisher publisher = client.describeBackupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBackups(software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest)} operation.</b>
     * </p>
     *
     * @param describeBackupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeBackups" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeBackupsPublisher describeBackupsPaginator(DescribeBackupsRequest describeBackupsRequest) {
        return new DescribeBackupsPublisher(this, applyPaginatorUserAgent(describeBackupsRequest));
    }

    /**
     * <p>
     * Gets information about AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the clusters.
     * When the response contains only a subset of clusters, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeClusters</code> request to get more clusters. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more clusters to get.
     * </p>
     *
     * @param describeClustersRequest
     * @return A Java Future containing the result of the DescribeClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DescribeClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClustersResponse> describeClusters(DescribeClustersRequest describeClustersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeClustersRequest, DescribeClustersResponse>()
                            .withOperationName("DescribeClusters")
                            .withMarshaller(new DescribeClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeClustersRequest));
            CompletableFuture<DescribeClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the clusters.
     * When the response contains only a subset of clusters, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeClusters</code> request to get more clusters. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more clusters to get.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeClusters(software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeClustersPublisher publisher = client.describeClustersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeClustersPublisher publisher = client.describeClustersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeClusters(software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeClustersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DescribeClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeClusters" target="_top">AWS
     *      API Documentation</a>
     */
    public DescribeClustersPublisher describeClustersPaginator(DescribeClustersRequest describeClustersRequest) {
        return new DescribeClustersPublisher(this, applyPaginatorUserAgent(describeClustersRequest));
    }

    /**
     * <p>
     * Claims an AWS CloudHSM cluster by submitting the cluster certificate issued by your issuing certificate authority
     * (CA) and the CA's root certificate. Before you can claim a cluster, you must sign the cluster's certificate
     * signing request (CSR) with your issuing CA. To get the cluster's CSR, use <a>DescribeClusters</a>.
     * </p>
     *
     * @param initializeClusterRequest
     * @return A Java Future containing the result of the InitializeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.InitializeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/InitializeCluster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<InitializeClusterResponse> initializeCluster(InitializeClusterRequest initializeClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initializeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitializeCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InitializeClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InitializeClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InitializeClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InitializeClusterRequest, InitializeClusterResponse>()
                            .withOperationName("InitializeCluster")
                            .withMarshaller(new InitializeClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(initializeClusterRequest));
            CompletableFuture<InitializeClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of tags for the specified AWS CloudHSM cluster.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTags</code> request to get more tags. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more tags to get.
     * </p>
     *
     * @param listTagsRequest
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsRequest));
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of tags for the specified AWS CloudHSM cluster.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTags</code> request to get more tags. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more tags to get.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTags(software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.ListTagsPublisher publisher = client.listTagsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.ListTagsPublisher publisher = client.listTagsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudhsmv2.model.ListTagsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudhsmv2.model.ListTagsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTags(software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest)} operation.</b>
     * </p>
     *
     * @param listTagsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTagsPublisher listTagsPaginator(ListTagsRequest listTagsRequest) {
        return new ListTagsPublisher(this, applyPaginatorUserAgent(listTagsRequest));
    }

    /**
     * <p>
     * Modifies attributes for AWS CloudHSM backup.
     * </p>
     *
     * @param modifyBackupAttributesRequest
     * @return A Java Future containing the result of the ModifyBackupAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.ModifyBackupAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyBackupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ModifyBackupAttributesResponse> modifyBackupAttributes(
            ModifyBackupAttributesRequest modifyBackupAttributesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyBackupAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyBackupAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ModifyBackupAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ModifyBackupAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ModifyBackupAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ModifyBackupAttributesRequest, ModifyBackupAttributesResponse>()
                            .withOperationName("ModifyBackupAttributes")
                            .withMarshaller(new ModifyBackupAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(modifyBackupAttributesRequest));
            CompletableFuture<ModifyBackupAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies AWS CloudHSM cluster.
     * </p>
     *
     * @param modifyClusterRequest
     * @return A Java Future containing the result of the ModifyCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.ModifyCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ModifyClusterResponse> modifyCluster(ModifyClusterRequest modifyClusterRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ModifyClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ModifyClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ModifyClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ModifyClusterRequest, ModifyClusterResponse>()
                            .withOperationName("ModifyCluster")
                            .withMarshaller(new ModifyClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(modifyClusterRequest));
            CompletableFuture<ModifyClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores a specified AWS CloudHSM backup that is in the <code>PENDING_DELETION</code> state. For mor information
     * on deleting a backup, see <a>DeleteBackup</a>.
     * </p>
     *
     * @param restoreBackupRequest
     * @return A Java Future containing the result of the RestoreBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.RestoreBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/RestoreBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreBackupResponse> restoreBackup(RestoreBackupRequest restoreBackupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RestoreBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreBackupRequest, RestoreBackupResponse>()
                            .withOperationName("RestoreBackup")
                            .withMarshaller(new RestoreBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(restoreBackupRequest));
            CompletableFuture<RestoreBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or overwrites one or more tags for the specified AWS CloudHSM cluster.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tag or tags from the specified AWS CloudHSM cluster.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an AWS CloudHSM internal
     *         failure. The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudHsmV2Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInternalFailureException")
                                .exceptionBuilderSupplier(CloudHsmInternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmTagException")
                                .exceptionBuilderSupplier(CloudHsmTagException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmServiceException")
                                .exceptionBuilderSupplier(CloudHsmServiceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmAccessDeniedException")
                                .exceptionBuilderSupplier(CloudHsmAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmResourceNotFoundException")
                                .exceptionBuilderSupplier(CloudHsmResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInvalidRequestException")
                                .exceptionBuilderSupplier(CloudHsmInvalidRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends CloudHsmV2Request> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
