/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudhsmv2;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.cloudhsmv2.internal.CloudHsmV2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmAccessDeniedException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInternalFailureException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInvalidRequestException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmResourceNotFoundException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmServiceException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmTagException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmV2Exception;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.transform.CopyBackupToRegionRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeClustersRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.InitializeClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyBackupAttributesRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.RestoreBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudHsmV2AsyncClient}.
 *
 * @see CloudHsmV2AsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudHsmV2AsyncClient implements CloudHsmV2AsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudHsmV2AsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudHsmV2AsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Copy an CloudHSM cluster backup to a different region.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param copyBackupToRegionRequest
     * @return A Java Future containing the result of the CopyBackupToRegion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.CopyBackupToRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CopyBackupToRegion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CopyBackupToRegionResponse> copyBackupToRegion(CopyBackupToRegionRequest copyBackupToRegionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(copyBackupToRegionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyBackupToRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyBackupToRegion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CopyBackupToRegionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CopyBackupToRegionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CopyBackupToRegionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CopyBackupToRegionRequest, CopyBackupToRegionResponse>()
                            .withOperationName("CopyBackupToRegion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CopyBackupToRegionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(copyBackupToRegionRequest));
            CompletableFuture<CopyBackupToRegionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> Yes. To perform this operation with an CloudHSM backup in a different AWS account,
     * specify the full backup ARN in the value of the SourceBackupId parameter.
     * </p>
     *
     * @param createClusterRequest
     * @return A Java Future containing the result of the CreateCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateClusterResponse> createCluster(CreateClusterRequest createClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                            .withOperationName("CreateCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createClusterRequest));
            CompletableFuture<CreateClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new hardware security module (HSM) in the specified CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Service account.
     * </p>
     *
     * @param createHsmRequest
     * @return A Java Future containing the result of the CreateHsm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.CreateHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateHsmResponse> createHsm(CreateHsmRequest createHsmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHsmRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHsm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateHsmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateHsmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateHsmRequest, CreateHsmResponse>().withOperationName("CreateHsm")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateHsmRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createHsmRequest));
            CompletableFuture<CreateHsmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified CloudHSM backup. A backup can be restored up to 7 days after the DeleteBackup request is
     * made. For more information on restoring a backup, see <a>RestoreBackup</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteBackupRequest
     * @return A Java Future containing the result of the DeleteBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DeleteBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackupResponse> deleteBackup(DeleteBackupRequest deleteBackupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                            .withOperationName("DeleteBackup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBackupRequest));
            CompletableFuture<DeleteBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified CloudHSM cluster. Before you can delete a cluster, you must delete all HSMs in the cluster.
     * To see if the cluster contains any HSMs, use <a>DescribeClusters</a>. To delete an HSM, use <a>DeleteHsm</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteClusterRequest
     * @return A Java Future containing the result of the DeleteCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteClusterResponse> deleteCluster(DeleteClusterRequest deleteClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                            .withOperationName("DeleteCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteClusterRequest));
            CompletableFuture<DeleteClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified HSM. To specify an HSM, you can use its identifier (ID), the IP address of the HSM's
     * elastic network interface (ENI), or the ID of the HSM's ENI. You need to specify only one of these values. To
     * find these values, use <a>DescribeClusters</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM hsm in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteHsmRequest
     * @return A Java Future containing the result of the DeleteHsm operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DeleteHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteHsmResponse> deleteHsm(DeleteHsmRequest deleteHsmRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHsmRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHsm");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteHsmResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteHsmResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteHsmRequest, DeleteHsmResponse>().withOperationName("DeleteHsm")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteHsmRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteHsmRequest));
            CompletableFuture<DeleteHsmResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an CloudHSM resource policy. Deleting a resource policy will result in the resource being unshared and
     * removed from any RAM resource shares. Deleting the resource policy attached to a backup will not impact any
     * clusters created from that backup.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return A Java Future containing the result of the DeleteResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteResourcePolicyResponse> deleteResourcePolicy(
            DeleteResourcePolicyRequest deleteResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                            .withOperationName("DeleteResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteResourcePolicyRequest));
            CompletableFuture<DeleteResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about backups of CloudHSM clusters. Lists either the backups you own or the backups shared with
     * you when the Shared parameter is true.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the backups.
     * When the response contains only a subset of backups, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeBackups</code> request to get more backups. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more backups to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> Yes. Customers can describe backups in other Amazon Web Services accounts that are
     * shared with them.
     * </p>
     *
     * @param describeBackupsRequest
     * @return A Java Future containing the result of the DescribeBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeBackups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBackupsResponse> describeBackups(DescribeBackupsRequest describeBackupsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBackupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                            .withOperationName("DescribeBackups").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeBackupsRequest));
            CompletableFuture<DescribeBackupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the clusters.
     * When the response contains only a subset of clusters, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeClusters</code> request to get more clusters. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more clusters to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on CloudHSM clusters in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param describeClustersRequest
     * @return A Java Future containing the result of the DescribeClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.DescribeClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeClustersResponse> describeClusters(DescribeClustersRequest describeClustersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeClustersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeClustersRequest, DescribeClustersResponse>()
                            .withOperationName("DescribeClusters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeClustersRequest));
            CompletableFuture<DescribeClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource policy document attached to a given resource.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return A Java Future containing the result of the GetResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/GetResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourcePolicyResponse> getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                            .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getResourcePolicyRequest));
            CompletableFuture<GetResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Claims an CloudHSM cluster by submitting the cluster certificate issued by your issuing certificate authority
     * (CA) and the CA's root certificate. Before you can claim a cluster, you must sign the cluster's certificate
     * signing request (CSR) with your issuing CA. To get the cluster's CSR, use <a>DescribeClusters</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param initializeClusterRequest
     * @return A Java Future containing the result of the InitializeCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.InitializeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/InitializeCluster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<InitializeClusterResponse> initializeCluster(InitializeClusterRequest initializeClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initializeClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initializeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitializeCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InitializeClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, InitializeClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<InitializeClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<InitializeClusterRequest, InitializeClusterResponse>()
                            .withOperationName("InitializeCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InitializeClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(initializeClusterRequest));
            CompletableFuture<InitializeClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a list of tags for the specified CloudHSM cluster.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTags</code> request to get more tags. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more tags to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param listTagsRequest
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsRequest));
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies attributes for CloudHSM backup.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param modifyBackupAttributesRequest
     * @return A Java Future containing the result of the ModifyBackupAttributes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.ModifyBackupAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyBackupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ModifyBackupAttributesResponse> modifyBackupAttributes(
            ModifyBackupAttributesRequest modifyBackupAttributesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(modifyBackupAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyBackupAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyBackupAttributes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ModifyBackupAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ModifyBackupAttributesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ModifyBackupAttributesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ModifyBackupAttributesRequest, ModifyBackupAttributesResponse>()
                            .withOperationName("ModifyBackupAttributes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ModifyBackupAttributesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(modifyBackupAttributesRequest));
            CompletableFuture<ModifyBackupAttributesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param modifyClusterRequest
     * @return A Java Future containing the result of the ModifyCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.ModifyCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ModifyClusterResponse> modifyCluster(ModifyClusterRequest modifyClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(modifyClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ModifyClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ModifyClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ModifyClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ModifyClusterRequest, ModifyClusterResponse>()
                            .withOperationName("ModifyCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ModifyClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(modifyClusterRequest));
            CompletableFuture<ModifyClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates an CloudHSM resource policy. A resource policy helps you to define the IAM entity (for
     * example, an Amazon Web Services account) that can manage your CloudHSM resources. The following resources support
     * CloudHSM resource policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Backup - The resource policy allows you to describe the backup and restore a cluster from the backup in another
     * Amazon Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In order to share a backup, it must be in a 'READY' state and you must own it.
     * </p>
     * <important>
     * <p>
     * While you can share a backup using the CloudHSM PutResourcePolicy operation, we recommend using Resource Access
     * Manager (RAM) instead. Using RAM provides multiple benefits as it creates the policy for you, allows multiple
     * resources to be shared at one time, and increases the discoverability of shared resources. If you use
     * PutResourcePolicy and want consumers to be able to describe the backups you share with them, you must promote the
     * backup to a standard RAM Resource Share using the RAM PromoteResourceShareCreatedFromPolicy API operation. For
     * more information, see <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/sharing.html"> Working with
     * shared backups</a> in the CloudHSM User Guide
     * </p>
     * </important>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return A Java Future containing the result of the PutResourcePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/PutResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutResourcePolicyResponse> putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutResourcePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutResourcePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                            .withOperationName("PutResourcePolicy").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putResourcePolicyRequest));
            CompletableFuture<PutResourcePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores a specified CloudHSM backup that is in the <code>PENDING_DELETION</code> state. For more information on
     * deleting a backup, see <a>DeleteBackup</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param restoreBackupRequest
     * @return A Java Future containing the result of the RestoreBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.RestoreBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/RestoreBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreBackupResponse> restoreBackup(RestoreBackupRequest restoreBackupRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RestoreBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreBackupRequest, RestoreBackupResponse>()
                            .withOperationName("RestoreBackup").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RestoreBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(restoreBackupRequest));
            CompletableFuture<RestoreBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or overwrites one or more tags for the specified CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tag or tags from the specified CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>CloudHsmAccessDeniedException The request was rejected because the requester does not have permission
     *         to perform the requested operation.</li>
     *         <li>CloudHsmInternalFailureException The request was rejected because of an CloudHSM internal failure.
     *         The request can be retried.</li>
     *         <li>CloudHsmInvalidRequestException The request was rejected because it is not a valid request.</li>
     *         <li>CloudHsmResourceNotFoundException The request was rejected because it refers to a resource that
     *         cannot be found.</li>
     *         <li>CloudHsmServiceException The request was rejected because an error occurred.</li>
     *         <li>CloudHsmTagException The request was rejected because of a tagging failure. Verify the tag conditions
     *         in all applicable policies, and then retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudHsmV2Exception Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudHsmV2AsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CloudHsmV2ServiceClientConfiguration serviceClientConfiguration() {
        return new CloudHsmV2ServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudHsmV2Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInternalFailureException")
                                .exceptionBuilderSupplier(CloudHsmInternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmTagException")
                                .exceptionBuilderSupplier(CloudHsmTagException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmServiceException")
                                .exceptionBuilderSupplier(CloudHsmServiceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmAccessDeniedException")
                                .exceptionBuilderSupplier(CloudHsmAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmResourceNotFoundException")
                                .exceptionBuilderSupplier(CloudHsmResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInvalidRequestException")
                                .exceptionBuilderSupplier(CloudHsmInvalidRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        CloudHsmV2ServiceClientConfigurationBuilder serviceConfigBuilder = new CloudHsmV2ServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
