/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudhsmv2;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.cloudhsmv2.internal.CloudHsmV2ServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmAccessDeniedException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInternalFailureException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInvalidRequestException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmResourceNotFoundException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmServiceException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmTagException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmV2Exception;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.transform.CopyBackupToRegionRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeClustersRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.InitializeClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyBackupAttributesRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.RestoreBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudHsmV2Client}.
 *
 * @see CloudHsmV2Client#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudHsmV2Client implements CloudHsmV2Client {
    private static final Logger log = Logger.loggerFor(DefaultCloudHsmV2Client.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudHsmV2Client(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Copy an CloudHSM cluster backup to a different region.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param copyBackupToRegionRequest
     * @return Result of the CopyBackupToRegion operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.CopyBackupToRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CopyBackupToRegion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CopyBackupToRegionResponse copyBackupToRegion(CopyBackupToRegionRequest copyBackupToRegionRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, CloudHsmTagException, AwsServiceException,
            SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CopyBackupToRegionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CopyBackupToRegionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(copyBackupToRegionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyBackupToRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyBackupToRegion");

            return clientHandler.execute(new ClientExecutionParams<CopyBackupToRegionRequest, CopyBackupToRegionResponse>()
                    .withOperationName("CopyBackupToRegion").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(copyBackupToRegionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CopyBackupToRegionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> Yes. To perform this operation with an CloudHSM backup in a different AWS account,
     * specify the full backup ARN in the value of the SourceBackupId parameter.
     * </p>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.CreateCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");

            return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                    .withOperationName("CreateCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new hardware security module (HSM) in the specified CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Service account.
     * </p>
     *
     * @param createHsmRequest
     * @return Result of the CreateHsm operation returned by the service.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.CreateHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateHsmResponse createHsm(CreateHsmRequest createHsmRequest) throws CloudHsmInternalFailureException,
            CloudHsmServiceException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmAccessDeniedException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createHsmRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHsm");

            return clientHandler.execute(new ClientExecutionParams<CreateHsmRequest, CreateHsmResponse>()
                    .withOperationName("CreateHsm").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createHsmRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified CloudHSM backup. A backup can be restored up to 7 days after the DeleteBackup request is
     * made. For more information on restoring a backup, see <a>RestoreBackup</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteBackupRequest
     * @return Result of the DeleteBackup operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DeleteBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupResponse deleteBackup(DeleteBackupRequest deleteBackupRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                    .withOperationName("DeleteBackup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified CloudHSM cluster. Before you can delete a cluster, you must delete all HSMs in the cluster.
     * To see if the cluster contains any HSMs, use <a>DescribeClusters</a>. To delete an HSM, use <a>DeleteHsm</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteClusterRequest
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DeleteCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteClusterResponse deleteCluster(DeleteClusterRequest deleteClusterRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");

            return clientHandler.execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                    .withOperationName("DeleteCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified HSM. To specify an HSM, you can use its identifier (ID), the IP address of the HSM's
     * elastic network interface (ENI), or the ID of the HSM's ENI. You need to specify only one of these values. To
     * find these values, use <a>DescribeClusters</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM hsm in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteHsmRequest
     * @return Result of the DeleteHsm operation returned by the service.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DeleteHsm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteHsmResponse deleteHsm(DeleteHsmRequest deleteHsmRequest) throws CloudHsmInternalFailureException,
            CloudHsmServiceException, CloudHsmResourceNotFoundException, CloudHsmInvalidRequestException,
            CloudHsmAccessDeniedException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteHsmRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHsm");

            return clientHandler.execute(new ClientExecutionParams<DeleteHsmRequest, DeleteHsmResponse>()
                    .withOperationName("DeleteHsm").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteHsmRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an CloudHSM resource policy. Deleting a resource policy will result in the resource being unshared and
     * removed from any RAM resource shares. Deleting the resource policy attached to a backup will not impact any
     * clusters created from that backup.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param deleteResourcePolicyRequest
     * @return Result of the DeleteResourcePolicy operation returned by the service.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResourcePolicyResponse deleteResourcePolicy(DeleteResourcePolicyRequest deleteResourcePolicyRequest)
            throws CloudHsmInternalFailureException, CloudHsmServiceException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmAccessDeniedException, AwsServiceException, SdkClientException,
            CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                    .withOperationName("DeleteResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about backups of CloudHSM clusters. Lists either the backups you own or the backups shared with
     * you when the Shared parameter is true.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the backups.
     * When the response contains only a subset of backups, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeBackups</code> request to get more backups. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more backups to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> Yes. Customers can describe backups in other Amazon Web Services accounts that are
     * shared with them.
     * </p>
     *
     * @param describeBackupsRequest
     * @return Result of the DescribeBackups operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeBackups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeBackupsResponse describeBackups(DescribeBackupsRequest describeBackupsRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, CloudHsmTagException, AwsServiceException,
            SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBackupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBackupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackups");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                    .withOperationName("DescribeBackups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeBackupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the clusters.
     * When the response contains only a subset of clusters, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeClusters</code> request to get more clusters. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more clusters to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on CloudHSM clusters in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param describeClustersRequest
     * @return Result of the DescribeClusters operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DescribeClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeClustersResponse describeClusters(DescribeClustersRequest describeClustersRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeClustersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeClusters");

            return clientHandler.execute(new ClientExecutionParams<DescribeClustersRequest, DescribeClustersResponse>()
                    .withOperationName("DescribeClusters").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the resource policy document attached to a given resource.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/GetResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws CloudHsmInternalFailureException, CloudHsmServiceException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmAccessDeniedException, AwsServiceException, SdkClientException,
            CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                    .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Claims an CloudHSM cluster by submitting the cluster certificate issued by your issuing certificate authority
     * (CA) and the CA's root certificate. Before you can claim a cluster, you must sign the cluster's certificate
     * signing request (CSR) with your issuing CA. To get the cluster's CSR, use <a>DescribeClusters</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param initializeClusterRequest
     * @return Result of the InitializeCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.InitializeCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/InitializeCluster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public InitializeClusterResponse initializeCluster(InitializeClusterRequest initializeClusterRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, AwsServiceException, SdkClientException,
            CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitializeClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InitializeClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(initializeClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initializeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitializeCluster");

            return clientHandler.execute(new ClientExecutionParams<InitializeClusterRequest, InitializeClusterResponse>()
                    .withOperationName("InitializeCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(initializeClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitializeClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of tags for the specified CloudHSM cluster.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTags</code> request to get more tags. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more tags to get.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param listTagsRequest
     * @return Result of the ListTags operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsResponse listTags(ListTagsRequest listTagsRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");

            return clientHandler.execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>()
                    .withOperationName("ListTags").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies attributes for CloudHSM backup.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param modifyBackupAttributesRequest
     * @return Result of the ModifyBackupAttributes operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.ModifyBackupAttributes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyBackupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyBackupAttributesResponse modifyBackupAttributes(ModifyBackupAttributesRequest modifyBackupAttributesRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, AwsServiceException, SdkClientException,
            CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyBackupAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyBackupAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(modifyBackupAttributesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyBackupAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyBackupAttributes");

            return clientHandler
                    .execute(new ClientExecutionParams<ModifyBackupAttributesRequest, ModifyBackupAttributesResponse>()
                            .withOperationName("ModifyBackupAttributes").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(modifyBackupAttributesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ModifyBackupAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM cluster in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param modifyClusterRequest
     * @return Result of the ModifyCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.ModifyCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ModifyClusterResponse modifyCluster(ModifyClusterRequest modifyClusterRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ModifyClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(modifyClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyCluster");

            return clientHandler.execute(new ClientExecutionParams<ModifyClusterRequest, ModifyClusterResponse>()
                    .withOperationName("ModifyCluster").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(modifyClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ModifyClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates an CloudHSM resource policy. A resource policy helps you to define the IAM entity (for
     * example, an Amazon Web Services account) that can manage your CloudHSM resources. The following resources support
     * CloudHSM resource policies:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Backup - The resource policy allows you to describe the backup and restore a cluster from the backup in another
     * Amazon Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * In order to share a backup, it must be in a 'READY' state and you must own it.
     * </p>
     * <important>
     * <p>
     * While you can share a backup using the CloudHSM PutResourcePolicy operation, we recommend using Resource Access
     * Manager (RAM) instead. Using RAM provides multiple benefits as it creates the policy for you, allows multiple
     * resources to be shared at one time, and increases the discoverability of shared resources. If you use
     * PutResourcePolicy and want consumers to be able to describe the backups you share with them, you must promote the
     * backup to a standard RAM Resource Share using the RAM PromoteResourceShareCreatedFromPolicy API operation. For
     * more information, see <a href="https://docs.aws.amazon.com/cloudhsm/latest/userguide/sharing.html"> Working with
     * shared backups</a> in the CloudHSM User Guide
     * </p>
     * </important>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param putResourcePolicyRequest
     * @return Result of the PutResourcePolicy operation returned by the service.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/PutResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutResourcePolicyResponse putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest)
            throws CloudHsmInternalFailureException, CloudHsmServiceException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmAccessDeniedException, AwsServiceException, SdkClientException,
            CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                    .withOperationName("PutResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Restores a specified CloudHSM backup that is in the <code>PENDING_DELETION</code> state. For more information on
     * deleting a backup, see <a>DeleteBackup</a>.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM backup in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param restoreBackupRequest
     * @return Result of the RestoreBackup operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.RestoreBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/RestoreBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RestoreBackupResponse restoreBackup(RestoreBackupRequest restoreBackupRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RestoreBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreBackup");

            return clientHandler.execute(new ClientExecutionParams<RestoreBackupRequest, RestoreBackupResponse>()
                    .withOperationName("RestoreBackup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(restoreBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RestoreBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or overwrites one or more tags for the specified CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tag or tags from the specified CloudHSM cluster.
     * </p>
     * <p>
     * <b>Cross-account use:</b> No. You cannot perform this operation on an CloudHSM resource in a different Amazon Web
     * Services account.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        CloudHsmV2ServiceClientConfigurationBuilder serviceConfigBuilder = new CloudHsmV2ServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudHsmV2Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInternalFailureException")
                                .exceptionBuilderSupplier(CloudHsmInternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmTagException")
                                .exceptionBuilderSupplier(CloudHsmTagException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmServiceException")
                                .exceptionBuilderSupplier(CloudHsmServiceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmAccessDeniedException")
                                .exceptionBuilderSupplier(CloudHsmAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmResourceNotFoundException")
                                .exceptionBuilderSupplier(CloudHsmResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInvalidRequestException")
                                .exceptionBuilderSupplier(CloudHsmInvalidRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public final CloudHsmV2ServiceClientConfiguration serviceClientConfiguration() {
        return new CloudHsmV2ServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
