/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an event that was returned by a lookup request. The result includes a representation of a
 * CloudTrail event.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventId").getter(getter(Event::eventId)).setter(setter(Builder::eventId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventId").build()).build();

    private static final SdkField<String> EVENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventName").getter(getter(Event::eventName)).setter(setter(Builder::eventName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventName").build()).build();

    private static final SdkField<String> READ_ONLY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReadOnly").getter(getter(Event::readOnly)).setter(setter(Builder::readOnly))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadOnly").build()).build();

    private static final SdkField<String> ACCESS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessKeyId").getter(getter(Event::accessKeyId)).setter(setter(Builder::accessKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessKeyId").build()).build();

    private static final SdkField<Instant> EVENT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EventTime").getter(getter(Event::eventTime)).setter(setter(Builder::eventTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventTime").build()).build();

    private static final SdkField<String> EVENT_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventSource").getter(getter(Event::eventSource)).setter(setter(Builder::eventSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventSource").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(Event::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<List<Resource>> RESOURCES_FIELD = SdkField
            .<List<Resource>> builder(MarshallingType.LIST)
            .memberName("Resources")
            .getter(getter(Event::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Resources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Resource> builder(MarshallingType.SDK_POJO)
                                            .constructor(Resource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CLOUD_TRAIL_EVENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CloudTrailEvent").getter(getter(Event::cloudTrailEvent)).setter(setter(Builder::cloudTrailEvent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudTrailEvent").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_ID_FIELD,
            EVENT_NAME_FIELD, READ_ONLY_FIELD, ACCESS_KEY_ID_FIELD, EVENT_TIME_FIELD, EVENT_SOURCE_FIELD, USERNAME_FIELD,
            RESOURCES_FIELD, CLOUD_TRAIL_EVENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String eventId;

    private final String eventName;

    private final String readOnly;

    private final String accessKeyId;

    private final Instant eventTime;

    private final String eventSource;

    private final String username;

    private final List<Resource> resources;

    private final String cloudTrailEvent;

    private Event(BuilderImpl builder) {
        this.eventId = builder.eventId;
        this.eventName = builder.eventName;
        this.readOnly = builder.readOnly;
        this.accessKeyId = builder.accessKeyId;
        this.eventTime = builder.eventTime;
        this.eventSource = builder.eventSource;
        this.username = builder.username;
        this.resources = builder.resources;
        this.cloudTrailEvent = builder.cloudTrailEvent;
    }

    /**
     * <p>
     * The CloudTrail ID of the event returned.
     * </p>
     * 
     * @return The CloudTrail ID of the event returned.
     */
    public String eventId() {
        return eventId;
    }

    /**
     * <p>
     * The name of the event returned.
     * </p>
     * 
     * @return The name of the event returned.
     */
    public String eventName() {
        return eventName;
    }

    /**
     * <p>
     * Information about whether the event is a write event or a read event.
     * </p>
     * 
     * @return Information about whether the event is a write event or a read event.
     */
    public String readOnly() {
        return readOnly;
    }

    /**
     * <p>
     * The AWS access key ID that was used to sign the request. If the request was made with temporary security
     * credentials, this is the access key ID of the temporary credentials.
     * </p>
     * 
     * @return The AWS access key ID that was used to sign the request. If the request was made with temporary security
     *         credentials, this is the access key ID of the temporary credentials.
     */
    public String accessKeyId() {
        return accessKeyId;
    }

    /**
     * <p>
     * The date and time of the event returned.
     * </p>
     * 
     * @return The date and time of the event returned.
     */
    public Instant eventTime() {
        return eventTime;
    }

    /**
     * <p>
     * The AWS service that the request was made to.
     * </p>
     * 
     * @return The AWS service that the request was made to.
     */
    public String eventSource() {
        return eventSource;
    }

    /**
     * <p>
     * A user name or role name of the requester that called the API in the event returned.
     * </p>
     * 
     * @return A user name or role name of the requester that called the API in the event returned.
     */
    public String username() {
        return username;
    }

    /**
     * Returns true if the Resources property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasResources() {
        return resources != null && !(resources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of resources referenced by the event returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of resources referenced by the event returned.
     */
    public List<Resource> resources() {
        return resources;
    }

    /**
     * <p>
     * A JSON string that contains a representation of the event returned.
     * </p>
     * 
     * @return A JSON string that contains a representation of the event returned.
     */
    public String cloudTrailEvent() {
        return cloudTrailEvent;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventId());
        hashCode = 31 * hashCode + Objects.hashCode(eventName());
        hashCode = 31 * hashCode + Objects.hashCode(readOnly());
        hashCode = 31 * hashCode + Objects.hashCode(accessKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(eventTime());
        hashCode = 31 * hashCode + Objects.hashCode(eventSource());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(hasResources() ? resources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cloudTrailEvent());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(eventId(), other.eventId()) && Objects.equals(eventName(), other.eventName())
                && Objects.equals(readOnly(), other.readOnly()) && Objects.equals(accessKeyId(), other.accessKeyId())
                && Objects.equals(eventTime(), other.eventTime()) && Objects.equals(eventSource(), other.eventSource())
                && Objects.equals(username(), other.username()) && hasResources() == other.hasResources()
                && Objects.equals(resources(), other.resources()) && Objects.equals(cloudTrailEvent(), other.cloudTrailEvent());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Event").add("EventId", eventId()).add("EventName", eventName()).add("ReadOnly", readOnly())
                .add("AccessKeyId", accessKeyId()).add("EventTime", eventTime()).add("EventSource", eventSource())
                .add("Username", username()).add("Resources", hasResources() ? resources() : null)
                .add("CloudTrailEvent", cloudTrailEvent()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventId":
            return Optional.ofNullable(clazz.cast(eventId()));
        case "EventName":
            return Optional.ofNullable(clazz.cast(eventName()));
        case "ReadOnly":
            return Optional.ofNullable(clazz.cast(readOnly()));
        case "AccessKeyId":
            return Optional.ofNullable(clazz.cast(accessKeyId()));
        case "EventTime":
            return Optional.ofNullable(clazz.cast(eventTime()));
        case "EventSource":
            return Optional.ofNullable(clazz.cast(eventSource()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "CloudTrailEvent":
            return Optional.ofNullable(clazz.cast(cloudTrailEvent()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The CloudTrail ID of the event returned.
         * </p>
         * 
         * @param eventId
         *        The CloudTrail ID of the event returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventId(String eventId);

        /**
         * <p>
         * The name of the event returned.
         * </p>
         * 
         * @param eventName
         *        The name of the event returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventName(String eventName);

        /**
         * <p>
         * Information about whether the event is a write event or a read event.
         * </p>
         * 
         * @param readOnly
         *        Information about whether the event is a write event or a read event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readOnly(String readOnly);

        /**
         * <p>
         * The AWS access key ID that was used to sign the request. If the request was made with temporary security
         * credentials, this is the access key ID of the temporary credentials.
         * </p>
         * 
         * @param accessKeyId
         *        The AWS access key ID that was used to sign the request. If the request was made with temporary
         *        security credentials, this is the access key ID of the temporary credentials.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessKeyId(String accessKeyId);

        /**
         * <p>
         * The date and time of the event returned.
         * </p>
         * 
         * @param eventTime
         *        The date and time of the event returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventTime(Instant eventTime);

        /**
         * <p>
         * The AWS service that the request was made to.
         * </p>
         * 
         * @param eventSource
         *        The AWS service that the request was made to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSource(String eventSource);

        /**
         * <p>
         * A user name or role name of the requester that called the API in the event returned.
         * </p>
         * 
         * @param username
         *        A user name or role name of the requester that called the API in the event returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * A list of resources referenced by the event returned.
         * </p>
         * 
         * @param resources
         *        A list of resources referenced by the event returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<Resource> resources);

        /**
         * <p>
         * A list of resources referenced by the event returned.
         * </p>
         * 
         * @param resources
         *        A list of resources referenced by the event returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Resource... resources);

        /**
         * <p>
         * A list of resources referenced by the event returned.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Resource>.Builder} avoiding the need to
         * create one manually via {@link List<Resource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Resource>.Builder#build()} is called immediately and its
         * result is passed to {@link #resources(List<Resource>)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link List<Resource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(List<Resource>)
         */
        Builder resources(Consumer<Resource.Builder>... resources);

        /**
         * <p>
         * A JSON string that contains a representation of the event returned.
         * </p>
         * 
         * @param cloudTrailEvent
         *        A JSON string that contains a representation of the event returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudTrailEvent(String cloudTrailEvent);
    }

    static final class BuilderImpl implements Builder {
        private String eventId;

        private String eventName;

        private String readOnly;

        private String accessKeyId;

        private Instant eventTime;

        private String eventSource;

        private String username;

        private List<Resource> resources = DefaultSdkAutoConstructList.getInstance();

        private String cloudTrailEvent;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            eventId(model.eventId);
            eventName(model.eventName);
            readOnly(model.readOnly);
            accessKeyId(model.accessKeyId);
            eventTime(model.eventTime);
            eventSource(model.eventSource);
            username(model.username);
            resources(model.resources);
            cloudTrailEvent(model.cloudTrailEvent);
        }

        public final String getEventId() {
            return eventId;
        }

        @Override
        public final Builder eventId(String eventId) {
            this.eventId = eventId;
            return this;
        }

        public final void setEventId(String eventId) {
            this.eventId = eventId;
        }

        public final String getEventName() {
            return eventName;
        }

        @Override
        public final Builder eventName(String eventName) {
            this.eventName = eventName;
            return this;
        }

        public final void setEventName(String eventName) {
            this.eventName = eventName;
        }

        public final String getReadOnly() {
            return readOnly;
        }

        @Override
        public final Builder readOnly(String readOnly) {
            this.readOnly = readOnly;
            return this;
        }

        public final void setReadOnly(String readOnly) {
            this.readOnly = readOnly;
        }

        public final String getAccessKeyId() {
            return accessKeyId;
        }

        @Override
        public final Builder accessKeyId(String accessKeyId) {
            this.accessKeyId = accessKeyId;
            return this;
        }

        public final void setAccessKeyId(String accessKeyId) {
            this.accessKeyId = accessKeyId;
        }

        public final Instant getEventTime() {
            return eventTime;
        }

        @Override
        public final Builder eventTime(Instant eventTime) {
            this.eventTime = eventTime;
            return this;
        }

        public final void setEventTime(Instant eventTime) {
            this.eventTime = eventTime;
        }

        public final String getEventSource() {
            return eventSource;
        }

        @Override
        public final Builder eventSource(String eventSource) {
            this.eventSource = eventSource;
            return this;
        }

        public final void setEventSource(String eventSource) {
            this.eventSource = eventSource;
        }

        public final String getUsername() {
            return username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        public final Collection<Resource.Builder> getResources() {
            if (resources instanceof SdkAutoConstructList) {
                return null;
            }
            return resources != null ? resources.stream().map(Resource::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder resources(Collection<Resource> resources) {
            this.resources = ResourceListCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Resource... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(Consumer<Resource.Builder>... resources) {
            resources(Stream.of(resources).map(c -> Resource.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setResources(Collection<Resource.BuilderImpl> resources) {
            this.resources = ResourceListCopier.copyFromBuilder(resources);
        }

        public final String getCloudTrailEvent() {
            return cloudTrailEvent;
        }

        @Override
        public final Builder cloudTrailEvent(String cloudTrailEvent) {
            this.cloudTrailEvent = cloudTrailEvent;
            return this;
        }

        public final void setCloudTrailEvent(String cloudTrailEvent) {
            this.cloudTrailEvent = cloudTrailEvent;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
