/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudtrail.model.AddTagsRequest;
import software.amazon.awssdk.services.cloudtrail.model.AddTagsResponse;
import software.amazon.awssdk.services.cloudtrail.model.CancelQueryRequest;
import software.amazon.awssdk.services.cloudtrail.model.CancelQueryResponse;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailAccessNotEnabledException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailArnInvalidException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailInvalidClientTokenIdException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.CloudWatchLogsDeliveryUnavailableException;
import software.amazon.awssdk.services.cloudtrail.model.ConflictException;
import software.amazon.awssdk.services.cloudtrail.model.CreateEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.CreateEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.CreateTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.CreateTrailResponse;
import software.amazon.awssdk.services.cloudtrail.model.DeleteEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.DeleteEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.DeleteTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.DeleteTrailResponse;
import software.amazon.awssdk.services.cloudtrail.model.DescribeQueryRequest;
import software.amazon.awssdk.services.cloudtrail.model.DescribeQueryResponse;
import software.amazon.awssdk.services.cloudtrail.model.DescribeTrailsRequest;
import software.amazon.awssdk.services.cloudtrail.model.DescribeTrailsResponse;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreAlreadyExistsException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreArnInvalidException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreMaxLimitExceededException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreTerminationProtectedException;
import software.amazon.awssdk.services.cloudtrail.model.GetEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetEventSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetEventSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetInsightSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetInsightSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailStatusRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailStatusResponse;
import software.amazon.awssdk.services.cloudtrail.model.InactiveEventDataStoreException;
import software.amazon.awssdk.services.cloudtrail.model.InactiveQueryException;
import software.amazon.awssdk.services.cloudtrail.model.InsightNotEnabledException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientDependencyServiceAccessPermissionException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientEncryptionPolicyException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientS3BucketPolicyException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientSnsTopicPolicyException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidCloudWatchLogsLogGroupArnException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidCloudWatchLogsRoleArnException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidDateRangeException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidEventCategoryException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidEventDataStoreStatusException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidEventSelectorsException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidHomeRegionException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidInsightSelectorsException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidKmsKeyIdException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidLookupAttributesException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidNextTokenException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidParameterCombinationException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidParameterException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidQueryStatementException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidQueryStatusException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidS3BucketNameException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidS3PrefixException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidSnsTopicNameException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTagParameterException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTimeRangeException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTokenException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTrailNameException;
import software.amazon.awssdk.services.cloudtrail.model.KmsException;
import software.amazon.awssdk.services.cloudtrail.model.KmsKeyNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListQueriesRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListQueriesResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListTagsRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListTagsResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListTrailsRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListTrailsResponse;
import software.amazon.awssdk.services.cloudtrail.model.LookupEventsRequest;
import software.amazon.awssdk.services.cloudtrail.model.LookupEventsResponse;
import software.amazon.awssdk.services.cloudtrail.model.MaxConcurrentQueriesException;
import software.amazon.awssdk.services.cloudtrail.model.MaximumNumberOfTrailsExceededException;
import software.amazon.awssdk.services.cloudtrail.model.NotOrganizationMasterAccountException;
import software.amazon.awssdk.services.cloudtrail.model.OperationNotPermittedException;
import software.amazon.awssdk.services.cloudtrail.model.OrganizationNotInAllFeaturesModeException;
import software.amazon.awssdk.services.cloudtrail.model.OrganizationsNotInUseException;
import software.amazon.awssdk.services.cloudtrail.model.PutEventSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.PutEventSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.PutInsightSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.PutInsightSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.QueryIdNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.RemoveTagsRequest;
import software.amazon.awssdk.services.cloudtrail.model.RemoveTagsResponse;
import software.amazon.awssdk.services.cloudtrail.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.ResourceTypeNotSupportedException;
import software.amazon.awssdk.services.cloudtrail.model.RestoreEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.RestoreEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.S3BucketDoesNotExistException;
import software.amazon.awssdk.services.cloudtrail.model.StartLoggingRequest;
import software.amazon.awssdk.services.cloudtrail.model.StartLoggingResponse;
import software.amazon.awssdk.services.cloudtrail.model.StartQueryRequest;
import software.amazon.awssdk.services.cloudtrail.model.StartQueryResponse;
import software.amazon.awssdk.services.cloudtrail.model.StopLoggingRequest;
import software.amazon.awssdk.services.cloudtrail.model.StopLoggingResponse;
import software.amazon.awssdk.services.cloudtrail.model.TagsLimitExceededException;
import software.amazon.awssdk.services.cloudtrail.model.TrailAlreadyExistsException;
import software.amazon.awssdk.services.cloudtrail.model.TrailNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.TrailNotProvidedException;
import software.amazon.awssdk.services.cloudtrail.model.UnsupportedOperationException;
import software.amazon.awssdk.services.cloudtrail.model.UpdateEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.UpdateEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.UpdateTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.UpdateTrailResponse;
import software.amazon.awssdk.services.cloudtrail.paginators.GetQueryResultsPublisher;
import software.amazon.awssdk.services.cloudtrail.paginators.ListEventDataStoresPublisher;
import software.amazon.awssdk.services.cloudtrail.paginators.ListPublicKeysPublisher;
import software.amazon.awssdk.services.cloudtrail.paginators.ListQueriesPublisher;
import software.amazon.awssdk.services.cloudtrail.paginators.ListTagsPublisher;
import software.amazon.awssdk.services.cloudtrail.paginators.ListTrailsPublisher;
import software.amazon.awssdk.services.cloudtrail.paginators.LookupEventsPublisher;
import software.amazon.awssdk.services.cloudtrail.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.CancelQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.CreateEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.CreateTrailRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DeleteEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DeleteTrailRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DescribeQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DescribeTrailsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetEventSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetInsightSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetQueryResultsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetTrailRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetTrailStatusRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListEventDataStoresRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListPublicKeysRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListQueriesRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListTrailsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.LookupEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.PutEventSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.PutInsightSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.RestoreEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StartLoggingRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StartQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StopLoggingRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.UpdateEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.UpdateTrailRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CloudTrailAsyncClient}.
 *
 * @see CloudTrailAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudTrailAsyncClient implements CloudTrailAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCloudTrailAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudTrailAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds one or more tags to a trail, up to a limit of 50. Overwrites an existing tag's value when a new value is
     * specified for an existing tag key. Tag key names must be unique for a trail; you cannot have two keys with the
     * same name but different values. If you specify a key without a value, the tag will be created with the specified
     * key and a value of null. You can tag a trail that applies to all Amazon Web Services Regions only from the Region
     * in which the trail was created (also known as its home region).
     * </p>
     *
     * @param addTagsRequest
     *        Specifies the tags to add to a trail.
     * @return A Java Future containing the result of the AddTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException This exception is thrown when the specified resource is not found.</li>
     *         <li>CloudTrailArnInvalidException This exception is thrown when an operation is called with a trail ARN
     *         that is not valid. The following is the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code></li>
     *         <li>ResourceTypeNotSupportedException This exception is thrown when the specified resource type is not
     *         supported by CloudTrail.</li>
     *         <li>TagsLimitExceededException The number of tags per trail has exceeded the permitted amount. Currently,
     *         the limit is 50.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InvalidTagParameterException This exception is thrown when the specified tag key or values are not
     *         valid. It can also occur if there are duplicate tags or too many tags on the resource.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>ConflictException This exception is thrown when the specified resource is not ready for an operation.
     *         This can occur when you try to run an operation on a trail before CloudTrail has time to fully load the
     *         trail. If this exception occurs, wait a few minutes, and then try the operation again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.AddTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/AddTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AddTagsResponse> addTags(AddTagsRequest addTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AddTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>().withOperationName("AddTags")
                            .withMarshaller(new AddTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(addTagsRequest));
            CompletableFuture<AddTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels a query if the query is not in a terminated state, such as <code>CANCELLED</code>, <code>FAILED</code>,
     * <code>TIMED_OUT</code>, or <code>FINISHED</code>. You must specify an ARN value for <code>EventDataStore</code>.
     * The ID of the query that you want to cancel is also required. When you run <code>CancelQuery</code>, the query
     * status might show as <code>CANCELLED</code> even if the operation is not yet finished.
     * </p>
     *
     * @param cancelQueryRequest
     * @return A Java Future containing the result of the CancelQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InactiveQueryException The specified query cannot be canceled because it is in the
     *         <code>FINISHED</code>, <code>FAILED</code>, <code>TIMED_OUT</code>, or <code>CANCELLED</code> state.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>QueryIdNotFoundException The query ID does not exist or does not map to a query.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>ConflictException This exception is thrown when the specified resource is not ready for an operation.
     *         This can occur when you try to run an operation on a trail before CloudTrail has time to fully load the
     *         trail. If this exception occurs, wait a few minutes, and then try the operation again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.CancelQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/CancelQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelQueryResponse> cancelQuery(CancelQueryRequest cancelQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelQueryRequest, CancelQueryResponse>()
                            .withOperationName("CancelQuery").withMarshaller(new CancelQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelQueryRequest));
            CompletableFuture<CancelQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new event data store.
     * </p>
     *
     * @param createEventDataStoreRequest
     * @return A Java Future containing the result of the CreateEventDataStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreAlreadyExistsException An event data store with that name already exists.</li>
     *         <li>EventDataStoreMaxLimitExceededException Your account has used the maximum number of event data
     *         stores.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>InvalidTagParameterException This exception is thrown when the specified tag key or values are not
     *         valid. It can also occur if there are duplicate tags or too many tags on the resource.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>ConflictException This exception is thrown when the specified resource is not ready for an operation.
     *         This can occur when you try to run an operation on a trail before CloudTrail has time to fully load the
     *         trail. If this exception occurs, wait a few minutes, and then try the operation again.</li>
     *         <li>CloudTrailAccessNotEnabledException This exception is thrown when trusted access has not been enabled
     *         between CloudTrail and Organizations. For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationsNotInUseException This exception is thrown when the request is made from an Amazon Web
     *         Services account that is not a member of an organization. To make this request, sign in using the
     *         credentials of an account that belongs to an organization.</li>
     *         <li>OrganizationNotInAllFeaturesModeException This exception is thrown when Organizations is not
     *         configured to support all features. All features must be enabled in Organizations to support creating an
     *         organization trail. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.CreateEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/CreateEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEventDataStoreResponse> createEventDataStore(
            CreateEventDataStoreRequest createEventDataStoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEventDataStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEventDataStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEventDataStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEventDataStoreRequest, CreateEventDataStoreResponse>()
                            .withOperationName("CreateEventDataStore")
                            .withMarshaller(new CreateEventDataStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createEventDataStoreRequest));
            CompletableFuture<CreateEventDataStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a trail that specifies the settings for delivery of log data to an Amazon S3 bucket.
     * </p>
     *
     * @param createTrailRequest
     *        Specifies the settings for each trail.
     * @return A Java Future containing the result of the CreateTrail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>MaximumNumberOfTrailsExceededException This exception is thrown when the maximum number of trails is
     *         reached.</li>
     *         <li>TrailAlreadyExistsException This exception is thrown when the specified trail already exists.</li>
     *         <li>S3BucketDoesNotExistException This exception is thrown when the specified S3 bucket does not exist.</li>
     *         <li>InsufficientS3BucketPolicyException This exception is thrown when the policy on the S3 bucket is not
     *         sufficient.</li>
     *         <li>InsufficientSnsTopicPolicyException This exception is thrown when the policy on the Amazon SNS topic
     *         is not sufficient.</li>
     *         <li>InsufficientEncryptionPolicyException This exception is thrown when the policy on the S3 bucket or
     *         KMS key is not sufficient.</li>
     *         <li>InvalidS3BucketNameException This exception is thrown when the provided S3 bucket name is not valid.</li>
     *         <li>InvalidS3PrefixException This exception is thrown when the provided S3 prefix is not valid.</li>
     *         <li>InvalidSnsTopicNameException This exception is thrown when the provided SNS topic name is not valid.</li>
     *         <li>InvalidKmsKeyIdException This exception is thrown when the KMS key ARN is not valid.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>TrailNotProvidedException This exception is no longer in use.</li>
     *         <li>InvalidParameterCombinationException This exception is thrown when the combination of parameters
     *         provided is not valid.</li>
     *         <li>KmsKeyNotFoundException This exception is thrown when the KMS key does not exist, when the S3 bucket
     *         and the KMS key are not in the same region, or when the KMS key associated with the Amazon SNS topic
     *         either does not exist or is not in the same region.</li>
     *         <li>KmsException This exception is thrown when there is an issue with the specified KMS key and the trail
     *         can’t be updated.</li>
     *         <li>InvalidCloudWatchLogsLogGroupArnException This exception is thrown when the provided CloudWatch Logs
     *         log group is not valid.</li>
     *         <li>InvalidCloudWatchLogsRoleArnException This exception is thrown when the provided role is not valid.</li>
     *         <li>CloudWatchLogsDeliveryUnavailableException Cannot set a CloudWatch Logs delivery for this region.</li>
     *         <li>InvalidTagParameterException This exception is thrown when the specified tag key or values are not
     *         valid. It can also occur if there are duplicate tags or too many tags on the resource.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>CloudTrailAccessNotEnabledException This exception is thrown when trusted access has not been enabled
     *         between CloudTrail and Organizations. For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationsNotInUseException This exception is thrown when the request is made from an Amazon Web
     *         Services account that is not a member of an organization. To make this request, sign in using the
     *         credentials of an account that belongs to an organization.</li>
     *         <li>OrganizationNotInAllFeaturesModeException This exception is thrown when Organizations is not
     *         configured to support all features. All features must be enabled in Organizations to support creating an
     *         organization trail. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>CloudTrailInvalidClientTokenIdException This exception is thrown when a call results in the
     *         <code>InvalidClientTokenId</code> error code. This can occur when you are creating or updating a trail to
     *         send notifications to an Amazon SNS topic that is in a suspended Amazon Web Services account.</li>
     *         <li>ConflictException This exception is thrown when the specified resource is not ready for an operation.
     *         This can occur when you try to run an operation on a trail before CloudTrail has time to fully load the
     *         trail. If this exception occurs, wait a few minutes, and then try the operation again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.CreateTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/CreateTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTrailResponse> createTrail(CreateTrailRequest createTrailRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTrail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTrailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTrailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTrailRequest, CreateTrailResponse>()
                            .withOperationName("CreateTrail").withMarshaller(new CreateTrailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTrailRequest));
            CompletableFuture<CreateTrailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disables the event data store specified by <code>EventDataStore</code>, which accepts an event data store ARN.
     * After you run <code>DeleteEventDataStore</code>, the event data store enters a <code>PENDING_DELETION</code>
     * state, and is automatically deleted after a wait period of seven days. <code>TerminationProtectionEnabled</code>
     * must be set to <code>False</code> on the event data store; this operation cannot work if
     * <code>TerminationProtectionEnabled</code> is <code>True</code>.
     * </p>
     * <p>
     * After you run <code>DeleteEventDataStore</code> on an event data store, you cannot run <code>ListQueries</code>,
     * <code>DescribeQuery</code>, or <code>GetQueryResults</code> on queries that are using an event data store in a
     * <code>PENDING_DELETION</code> state. An event data store in the <code>PENDING_DELETION</code> state does not
     * incur costs.
     * </p>
     *
     * @param deleteEventDataStoreRequest
     * @return A Java Future containing the result of the DeleteEventDataStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>EventDataStoreTerminationProtectedException The event data store cannot be deleted because
     *         termination protection is enabled for it.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.DeleteEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DeleteEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventDataStoreResponse> deleteEventDataStore(
            DeleteEventDataStoreRequest deleteEventDataStoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventDataStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEventDataStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEventDataStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventDataStoreRequest, DeleteEventDataStoreResponse>()
                            .withOperationName("DeleteEventDataStore")
                            .withMarshaller(new DeleteEventDataStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteEventDataStoreRequest));
            CompletableFuture<DeleteEventDataStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a trail. This operation must be called from the region in which the trail was created.
     * <code>DeleteTrail</code> cannot be called on the shadow trails (replicated trails in other regions) of a trail
     * that is enabled in all regions.
     * </p>
     *
     * @param deleteTrailRequest
     *        The request that specifies the name of a trail to delete.
     * @return A Java Future containing the result of the DeleteTrail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InvalidHomeRegionException This exception is thrown when an operation is called on a trail from a
     *         region other than the region in which the trail was created.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>ConflictException This exception is thrown when the specified resource is not ready for an operation.
     *         This can occur when you try to run an operation on a trail before CloudTrail has time to fully load the
     *         trail. If this exception occurs, wait a few minutes, and then try the operation again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.DeleteTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DeleteTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTrailResponse> deleteTrail(DeleteTrailRequest deleteTrailRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTrail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTrailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTrailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTrailRequest, DeleteTrailResponse>()
                            .withOperationName("DeleteTrail").withMarshaller(new DeleteTrailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTrailRequest));
            CompletableFuture<DeleteTrailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about a query, including query run time in milliseconds, number of events scanned and matched,
     * and query status. You must specify an ARN for <code>EventDataStore</code>, and a value for <code>QueryID</code>.
     * </p>
     *
     * @param describeQueryRequest
     * @return A Java Future containing the result of the DescribeQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>QueryIdNotFoundException The query ID does not exist or does not map to a query.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.DescribeQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DescribeQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeQueryResponse> describeQuery(DescribeQueryRequest describeQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeQueryRequest, DescribeQueryResponse>()
                            .withOperationName("DescribeQuery")
                            .withMarshaller(new DescribeQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeQueryRequest));
            CompletableFuture<DescribeQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves settings for one or more trails associated with the current region for your account.
     * </p>
     *
     * @param describeTrailsRequest
     *        Returns information about the trail.
     * @return A Java Future containing the result of the DescribeTrails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.DescribeTrails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DescribeTrails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTrailsResponse> describeTrails(DescribeTrailsRequest describeTrailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTrailsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTrailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTrailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTrailsRequest, DescribeTrailsResponse>()
                            .withOperationName("DescribeTrails")
                            .withMarshaller(new DescribeTrailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTrailsRequest));
            CompletableFuture<DescribeTrailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an event data store specified as either an ARN or the ID portion of the ARN.
     * </p>
     *
     * @param getEventDataStoreRequest
     * @return A Java Future containing the result of the GetEventDataStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.GetEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetEventDataStore" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEventDataStoreResponse> getEventDataStore(GetEventDataStoreRequest getEventDataStoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventDataStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEventDataStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEventDataStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEventDataStoreRequest, GetEventDataStoreResponse>()
                            .withOperationName("GetEventDataStore")
                            .withMarshaller(new GetEventDataStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEventDataStoreRequest));
            CompletableFuture<GetEventDataStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the settings for the event selectors that you configured for your trail. The information returned for
     * your event selectors includes the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If your event selector includes read-only events, write-only events, or all events. This applies to both
     * management events and data events.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your event selector includes management events.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your event selector includes data events, the resources on which you are logging data events.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html"
     * >Logging Data and Management Events for Trails </a> in the <i>CloudTrail User Guide</i>.
     * </p>
     *
     * @param getEventSelectorsRequest
     * @return A Java Future containing the result of the GetEventSelectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.GetEventSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetEventSelectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEventSelectorsResponse> getEventSelectors(GetEventSelectorsRequest getEventSelectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventSelectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEventSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEventSelectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEventSelectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEventSelectorsRequest, GetEventSelectorsResponse>()
                            .withOperationName("GetEventSelectors")
                            .withMarshaller(new GetEventSelectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEventSelectorsRequest));
            CompletableFuture<GetEventSelectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the settings for the Insights event selectors that you configured for your trail.
     * <code>GetInsightSelectors</code> shows if CloudTrail Insights event logging is enabled on the trail, and if it
     * is, which insight types are enabled. If you run <code>GetInsightSelectors</code> on a trail that does not have
     * Insights events enabled, the operation throws the exception <code>InsightNotEnabledException</code>
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-insights-events-with-cloudtrail.html"
     * >Logging CloudTrail Insights Events for Trails </a> in the <i>CloudTrail User Guide</i>.
     * </p>
     *
     * @param getInsightSelectorsRequest
     * @return A Java Future containing the result of the GetInsightSelectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>InsightNotEnabledException If you run <code>GetInsightSelectors</code> on a trail that does not have
     *         Insights events enabled, the operation throws the exception <code>InsightNotEnabledException</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.GetInsightSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetInsightSelectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInsightSelectorsResponse> getInsightSelectors(
            GetInsightSelectorsRequest getInsightSelectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightSelectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInsightSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInsightSelectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInsightSelectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInsightSelectorsRequest, GetInsightSelectorsResponse>()
                            .withOperationName("GetInsightSelectors")
                            .withMarshaller(new GetInsightSelectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getInsightSelectorsRequest));
            CompletableFuture<GetInsightSelectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets event data results of a query. You must specify the <code>QueryID</code> value returned by the
     * <code>StartQuery</code> operation, and an ARN for <code>EventDataStore</code>.
     * </p>
     *
     * @param getQueryResultsRequest
     * @return A Java Future containing the result of the GetQueryResults operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>QueryIdNotFoundException The query ID does not exist or does not map to a query.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.GetQueryResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetQueryResults" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueryResultsResponse> getQueryResults(GetQueryResultsRequest getQueryResultsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueryResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResults");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueryResultsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQueryResultsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQueryResultsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueryResultsRequest, GetQueryResultsResponse>()
                            .withOperationName("GetQueryResults")
                            .withMarshaller(new GetQueryResultsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getQueryResultsRequest));
            CompletableFuture<GetQueryResultsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets event data results of a query. You must specify the <code>QueryID</code> value returned by the
     * <code>StartQuery</code> operation, and an ARN for <code>EventDataStore</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getQueryResults(software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.GetQueryResultsPublisher publisher = client.getQueryResultsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getQueryResults(software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsRequest)} operation.</b>
     * </p>
     *
     * @param getQueryResultsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>QueryIdNotFoundException The query ID does not exist or does not map to a query.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.GetQueryResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetQueryResults" target="_top">AWS
     *      API Documentation</a>
     */
    public GetQueryResultsPublisher getQueryResultsPaginator(GetQueryResultsRequest getQueryResultsRequest) {
        return new GetQueryResultsPublisher(this, applyPaginatorUserAgent(getQueryResultsRequest));
    }

    /**
     * <p>
     * Returns settings information for a specified trail.
     * </p>
     *
     * @param getTrailRequest
     * @return A Java Future containing the result of the GetTrail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.GetTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTrailResponse> getTrail(GetTrailRequest getTrailRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTrail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTrailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTrailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTrailRequest, GetTrailResponse>().withOperationName("GetTrail")
                            .withMarshaller(new GetTrailRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTrailRequest));
            CompletableFuture<GetTrailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a JSON-formatted list of information about the specified trail. Fields include information on delivery
     * errors, Amazon SNS and Amazon S3 errors, and start and stop logging times for each trail. This operation returns
     * trail status from a single region. To return trail status from all regions, you must call the operation on each
     * region.
     * </p>
     *
     * @param getTrailStatusRequest
     *        The name of a trail about which you want the current status.
     * @return A Java Future containing the result of the GetTrailStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.GetTrailStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetTrailStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTrailStatusResponse> getTrailStatus(GetTrailStatusRequest getTrailStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTrailStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTrailStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTrailStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetTrailStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTrailStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTrailStatusRequest, GetTrailStatusResponse>()
                            .withOperationName("GetTrailStatus")
                            .withMarshaller(new GetTrailStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getTrailStatusRequest));
            CompletableFuture<GetTrailStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about all event data stores in the account, in the current region.
     * </p>
     *
     * @param listEventDataStoresRequest
     * @return A Java Future containing the result of the ListEventDataStores operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListEventDataStores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListEventDataStores"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventDataStoresResponse> listEventDataStores(
            ListEventDataStoresRequest listEventDataStoresRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventDataStoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventDataStores");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventDataStoresResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventDataStoresResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventDataStoresResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventDataStoresRequest, ListEventDataStoresResponse>()
                            .withOperationName("ListEventDataStores")
                            .withMarshaller(new ListEventDataStoresRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEventDataStoresRequest));
            CompletableFuture<ListEventDataStoresResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about all event data stores in the account, in the current region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEventDataStores(software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListEventDataStoresPublisher publisher = client.listEventDataStoresPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListEventDataStoresPublisher publisher = client.listEventDataStoresPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEventDataStores(software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEventDataStoresRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListEventDataStores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListEventDataStores"
     *      target="_top">AWS API Documentation</a>
     */
    public ListEventDataStoresPublisher listEventDataStoresPaginator(ListEventDataStoresRequest listEventDataStoresRequest) {
        return new ListEventDataStoresPublisher(this, applyPaginatorUserAgent(listEventDataStoresRequest));
    }

    /**
     * <p>
     * Returns all public keys whose private keys were used to sign the digest files within the specified time range.
     * The public key is needed to validate digest files that were signed with its corresponding private key.
     * </p>
     * <note>
     * <p>
     * CloudTrail uses different private and public key pairs per region. Each digest file is signed with a private key
     * unique to its region. When you validate a digest file from a specific region, you must look in the same region
     * for its corresponding public key.
     * </p>
     * </note>
     *
     * @param listPublicKeysRequest
     *        Requests the public keys for a specified time range.
     * @return A Java Future containing the result of the ListPublicKeys operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTimeRangeException Occurs if the timestamp values are not valid. Either the start time occurs
     *         after the end time, or the time range is outside the range of possible values.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>InvalidTokenException Reserved for future use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListPublicKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListPublicKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPublicKeysResponse> listPublicKeys(ListPublicKeysRequest listPublicKeysRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPublicKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPublicKeys");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPublicKeysResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPublicKeysResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPublicKeysResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPublicKeysRequest, ListPublicKeysResponse>()
                            .withOperationName("ListPublicKeys")
                            .withMarshaller(new ListPublicKeysRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPublicKeysRequest));
            CompletableFuture<ListPublicKeysResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all public keys whose private keys were used to sign the digest files within the specified time range.
     * The public key is needed to validate digest files that were signed with its corresponding private key.
     * </p>
     * <note>
     * <p>
     * CloudTrail uses different private and public key pairs per region. Each digest file is signed with a private key
     * unique to its region. When you validate a digest file from a specific region, you must look in the same region
     * for its corresponding public key.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listPublicKeys(software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListPublicKeysPublisher publisher = client.listPublicKeysPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListPublicKeysPublisher publisher = client.listPublicKeysPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPublicKeys(software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysRequest)} operation.</b>
     * </p>
     *
     * @param listPublicKeysRequest
     *        Requests the public keys for a specified time range.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidTimeRangeException Occurs if the timestamp values are not valid. Either the start time occurs
     *         after the end time, or the time range is outside the range of possible values.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>InvalidTokenException Reserved for future use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListPublicKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListPublicKeys" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPublicKeysPublisher listPublicKeysPaginator(ListPublicKeysRequest listPublicKeysRequest) {
        return new ListPublicKeysPublisher(this, applyPaginatorUserAgent(listPublicKeysRequest));
    }

    /**
     * <p>
     * Returns a list of queries and query statuses for the past seven days. You must specify an ARN value for
     * <code>EventDataStore</code>. Optionally, to shorten the list of results, you can specify a time range, formatted
     * as timestamps, by adding <code>StartTime</code> and <code>EndTime</code> parameters, and a
     * <code>QueryStatus</code> value. Valid values for <code>QueryStatus</code> include <code>QUEUED</code>,
     * <code>RUNNING</code>, <code>FINISHED</code>, <code>FAILED</code>, <code>TIMED_OUT</code>, or
     * <code>CANCELLED</code>.
     * </p>
     *
     * @param listQueriesRequest
     * @return A Java Future containing the result of the ListQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InvalidDateRangeException A date range for the query was specified that is not valid. For more
     *         information about writing a query, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-create-edit-query.html">Create or
     *         edit a query</a> in the <i>CloudTrail User Guide</i>.</li>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>InvalidQueryStatusException The query status is not valid for the operation.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListQueries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListQueriesResponse> listQueries(ListQueriesRequest listQueriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQueriesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListQueriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListQueriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQueriesRequest, ListQueriesResponse>()
                            .withOperationName("ListQueries").withMarshaller(new ListQueriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listQueriesRequest));
            CompletableFuture<ListQueriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of queries and query statuses for the past seven days. You must specify an ARN value for
     * <code>EventDataStore</code>. Optionally, to shorten the list of results, you can specify a time range, formatted
     * as timestamps, by adding <code>StartTime</code> and <code>EndTime</code> parameters, and a
     * <code>QueryStatus</code> value. Valid values for <code>QueryStatus</code> include <code>QUEUED</code>,
     * <code>RUNNING</code>, <code>FINISHED</code>, <code>FAILED</code>, <code>TIMED_OUT</code>, or
     * <code>CANCELLED</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listQueries(software.amazon.awssdk.services.cloudtrail.model.ListQueriesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListQueriesPublisher publisher = client.listQueriesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListQueriesPublisher publisher = client.listQueriesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudtrail.model.ListQueriesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudtrail.model.ListQueriesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueries(software.amazon.awssdk.services.cloudtrail.model.ListQueriesRequest)} operation.</b>
     * </p>
     *
     * @param listQueriesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InvalidDateRangeException A date range for the query was specified that is not valid. For more
     *         information about writing a query, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-create-edit-query.html">Create or
     *         edit a query</a> in the <i>CloudTrail User Guide</i>.</li>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>InvalidQueryStatusException The query status is not valid for the operation.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListQueries" target="_top">AWS API
     *      Documentation</a>
     */
    public ListQueriesPublisher listQueriesPaginator(ListQueriesRequest listQueriesRequest) {
        return new ListQueriesPublisher(this, applyPaginatorUserAgent(listQueriesRequest));
    }

    /**
     * <p>
     * Lists the tags for the trail in the current region.
     * </p>
     *
     * @param listTagsRequest
     *        Specifies a list of trail tags to return.
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException This exception is thrown when the specified resource is not found.</li>
     *         <li>CloudTrailArnInvalidException This exception is thrown when an operation is called with a trail ARN
     *         that is not valid. The following is the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code></li>
     *         <li>ResourceTypeNotSupportedException This exception is thrown when the specified resource type is not
     *         supported by CloudTrail.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>InvalidTokenException Reserved for future use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsRequest));
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for the trail in the current region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTags(software.amazon.awssdk.services.cloudtrail.model.ListTagsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTagsPublisher publisher = client.listTagsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTagsPublisher publisher = client.listTagsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudtrail.model.ListTagsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudtrail.model.ListTagsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTags(software.amazon.awssdk.services.cloudtrail.model.ListTagsRequest)} operation.</b>
     * </p>
     *
     * @param listTagsRequest
     *        Specifies a list of trail tags to return.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException This exception is thrown when the specified resource is not found.</li>
     *         <li>CloudTrailArnInvalidException This exception is thrown when an operation is called with a trail ARN
     *         that is not valid. The following is the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code></li>
     *         <li>ResourceTypeNotSupportedException This exception is thrown when the specified resource type is not
     *         supported by CloudTrail.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>InvalidTokenException Reserved for future use.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTagsPublisher listTagsPaginator(ListTagsRequest listTagsRequest) {
        return new ListTagsPublisher(this, applyPaginatorUserAgent(listTagsRequest));
    }

    /**
     * <p>
     * Lists trails that are in the current account.
     * </p>
     *
     * @param listTrailsRequest
     * @return A Java Future containing the result of the ListTrails operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListTrails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTrails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTrailsResponse> listTrails(ListTrailsRequest listTrailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTrailsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTrailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTrailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTrailsRequest, ListTrailsResponse>().withOperationName("ListTrails")
                            .withMarshaller(new ListTrailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTrailsRequest));
            CompletableFuture<ListTrailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists trails that are in the current account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTrails(software.amazon.awssdk.services.cloudtrail.model.ListTrailsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTrailsPublisher publisher = client.listTrailsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTrailsPublisher publisher = client.listTrailsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudtrail.model.ListTrailsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudtrail.model.ListTrailsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTrails(software.amazon.awssdk.services.cloudtrail.model.ListTrailsRequest)} operation.</b>
     * </p>
     *
     * @param listTrailsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.ListTrails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTrails" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTrailsPublisher listTrailsPaginator(ListTrailsRequest listTrailsRequest) {
        return new ListTrailsPublisher(this, applyPaginatorUserAgent(listTrailsRequest));
    }

    /**
     * <p>
     * Looks up <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-management-events"
     * >management events</a> or <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-insights-events"
     * >CloudTrail Insights events</a> that are captured by CloudTrail. You can look up events that occurred in a region
     * within the last 90 days. Lookup supports the following attributes for management events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Web Services access key
     * </p>
     * </li>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * <li>
     * <p>
     * Read only
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource name
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource type
     * </p>
     * </li>
     * <li>
     * <p>
     * User name
     * </p>
     * </li>
     * </ul>
     * <p>
     * Lookup supports the following attributes for Insights events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * </ul>
     * <p>
     * All attributes are optional. The default number of results returned is 50, with a maximum of 50 possible. The
     * response includes a token that you can use to get the next page of results.
     * </p>
     * <important>
     * <p>
     * The rate of lookup requests is limited to two per second, per account, per region. If this limit is exceeded, a
     * throttling error occurs.
     * </p>
     * </important>
     *
     * @param lookupEventsRequest
     *        Contains a request for LookupEvents.
     * @return A Java Future containing the result of the LookupEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidLookupAttributesException Occurs when a lookup attribute is specified that is not valid.</li>
     *         <li>InvalidTimeRangeException Occurs if the timestamp values are not valid. Either the start time occurs
     *         after the end time, or the time range is outside the range of possible values.</li>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>InvalidEventCategoryException Occurs if an event category that is not valid is specified as a value
     *         of <code>EventCategory</code>.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.LookupEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/LookupEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<LookupEventsResponse> lookupEvents(LookupEventsRequest lookupEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, lookupEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "LookupEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<LookupEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    LookupEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<LookupEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<LookupEventsRequest, LookupEventsResponse>()
                            .withOperationName("LookupEvents").withMarshaller(new LookupEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(lookupEventsRequest));
            CompletableFuture<LookupEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Looks up <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-management-events"
     * >management events</a> or <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-insights-events"
     * >CloudTrail Insights events</a> that are captured by CloudTrail. You can look up events that occurred in a region
     * within the last 90 days. Lookup supports the following attributes for management events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Web Services access key
     * </p>
     * </li>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * <li>
     * <p>
     * Read only
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource name
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource type
     * </p>
     * </li>
     * <li>
     * <p>
     * User name
     * </p>
     * </li>
     * </ul>
     * <p>
     * Lookup supports the following attributes for Insights events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * </ul>
     * <p>
     * All attributes are optional. The default number of results returned is 50, with a maximum of 50 possible. The
     * response includes a token that you can use to get the next page of results.
     * </p>
     * <important>
     * <p>
     * The rate of lookup requests is limited to two per second, per account, per region. If this limit is exceeded, a
     * throttling error occurs.
     * </p>
     * </important><br/>
     * <p>
     * This is a variant of {@link #lookupEvents(software.amazon.awssdk.services.cloudtrail.model.LookupEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.LookupEventsPublisher publisher = client.lookupEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.LookupEventsPublisher publisher = client.lookupEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.cloudtrail.model.LookupEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.cloudtrail.model.LookupEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #lookupEvents(software.amazon.awssdk.services.cloudtrail.model.LookupEventsRequest)} operation.</b>
     * </p>
     *
     * @param lookupEventsRequest
     *        Contains a request for LookupEvents.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidLookupAttributesException Occurs when a lookup attribute is specified that is not valid.</li>
     *         <li>InvalidTimeRangeException Occurs if the timestamp values are not valid. Either the start time occurs
     *         after the end time, or the time range is outside the range of possible values.</li>
     *         <li>InvalidMaxResultsException This exception is thrown if the limit specified is not valid.</li>
     *         <li>InvalidNextTokenException A token that is not valid, or a token that was previously used in a request
     *         with different parameters. This exception is thrown if the token is not valid.</li>
     *         <li>InvalidEventCategoryException Occurs if an event category that is not valid is specified as a value
     *         of <code>EventCategory</code>.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.LookupEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/LookupEvents" target="_top">AWS API
     *      Documentation</a>
     */
    public LookupEventsPublisher lookupEventsPaginator(LookupEventsRequest lookupEventsRequest) {
        return new LookupEventsPublisher(this, applyPaginatorUserAgent(lookupEventsRequest));
    }

    /**
     * <p>
     * Configures an event selector or advanced event selectors for your trail. Use event selectors or advanced event
     * selectors to specify management and data event settings for your trail. By default, trails created without
     * specific event selectors are configured to log all read and write management events, and no data events.
     * </p>
     * <p>
     * When an event occurs in your account, CloudTrail evaluates the event selectors or advanced event selectors in all
     * trails. For each trail, if the event matches any event selector, the trail processes and logs the event. If the
     * event doesn't match any event selector, the trail doesn't log the event.
     * </p>
     * <p>
     * Example
     * </p>
     * <ol>
     * <li>
     * <p>
     * You create an event selector for a trail and specify that you want write-only events.
     * </p>
     * </li>
     * <li>
     * <p>
     * The EC2 <code>GetConsoleOutput</code> and <code>RunInstances</code> API operations occur in your account.
     * </p>
     * </li>
     * <li>
     * <p>
     * CloudTrail evaluates whether the events match your event selectors.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>RunInstances</code> is a write-only event and it matches your event selector. The trail logs the event.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>GetConsoleOutput</code> is a read-only event that doesn't match your event selector. The trail doesn't
     * log the event.
     * </p>
     * </li>
     * </ol>
     * <p>
     * The <code>PutEventSelectors</code> operation must be called from the region in which the trail was created;
     * otherwise, an <code>InvalidHomeRegionException</code> exception is thrown.
     * </p>
     * <p>
     * You can configure up to five event selectors for each trail. For more information, see <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html"
     * >Logging data and management events for trails </a> and <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Quotas in
     * CloudTrail</a> in the <i>CloudTrail User Guide</i>.
     * </p>
     * <p>
     * You can add advanced event selectors, and conditions for your advanced event selectors, up to a maximum of 500
     * values for all conditions and selectors on a trail. You can use either <code>AdvancedEventSelectors</code> or
     * <code>EventSelectors</code>, but not both. If you apply <code>AdvancedEventSelectors</code> to a trail, any
     * existing <code>EventSelectors</code> are overwritten. For more information about advanced event selectors, see <a
     * href
     * ="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html">Logging
     * data events for trails</a> in the <i>CloudTrail User Guide</i>.
     * </p>
     *
     * @param putEventSelectorsRequest
     * @return A Java Future containing the result of the PutEventSelectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InvalidHomeRegionException This exception is thrown when an operation is called on a trail from a
     *         region other than the region in which the trail was created.</li>
     *         <li>InvalidEventSelectorsException This exception is thrown when the <code>PutEventSelectors</code>
     *         operation is called with a number of event selectors, advanced event selectors, or data resources that is
     *         not valid. The combination of event selectors or advanced event selectors and data resources is not
     *         valid. A trail can have up to 5 event selectors. If a trail uses advanced event selectors, a maximum of
     *         500 total values for all conditions in all advanced event selectors is allowed. A trail is limited to 250
     *         data resources. These data resources can be distributed across event selectors, but the overall total
     *         cannot exceed 250.</p>
     *         <p>
     *         You can:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify a valid number of event selectors (1 to 5) for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid number of data resources (1 to 250) for an event selector. The limit of number of
     *         resources on an individual event selector is configurable up to 250. However, this upper limit is allowed
     *         only if the total number of data resources does not exceed 250 across all event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify up to 500 values for all conditions in all advanced event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid value for a parameter. For example, specifying the <code>ReadWriteType</code> parameter
     *         with a value of <code>read-only</code> is not valid.
     *         </p>
     *         </li></li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.PutEventSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/PutEventSelectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutEventSelectorsResponse> putEventSelectors(PutEventSelectorsRequest putEventSelectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEventSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEventSelectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutEventSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutEventSelectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutEventSelectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutEventSelectorsRequest, PutEventSelectorsResponse>()
                            .withOperationName("PutEventSelectors")
                            .withMarshaller(new PutEventSelectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putEventSelectorsRequest));
            CompletableFuture<PutEventSelectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lets you enable Insights event logging by specifying the Insights selectors that you want to enable on an
     * existing trail. You also use <code>PutInsightSelectors</code> to turn off Insights event logging, by passing an
     * empty list of insight types. The valid Insights event types in this release are <code>ApiErrorRateInsight</code>
     * and <code>ApiCallRateInsight</code>.
     * </p>
     *
     * @param putInsightSelectorsRequest
     * @return A Java Future containing the result of the PutInsightSelectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InvalidHomeRegionException This exception is thrown when an operation is called on a trail from a
     *         region other than the region in which the trail was created.</li>
     *         <li>InvalidInsightSelectorsException The formatting or syntax of the <code>InsightSelectors</code> JSON
     *         statement in your <code>PutInsightSelectors</code> or <code>GetInsightSelectors</code> request is not
     *         valid, or the specified insight type in the <code>InsightSelectors</code> statement is not a valid
     *         insight type.</li>
     *         <li>InsufficientS3BucketPolicyException This exception is thrown when the policy on the S3 bucket is not
     *         sufficient.</li>
     *         <li>InsufficientEncryptionPolicyException This exception is thrown when the policy on the S3 bucket or
     *         KMS key is not sufficient.</li>
     *         <li>S3BucketDoesNotExistException This exception is thrown when the specified S3 bucket does not exist.</li>
     *         <li>KmsException This exception is thrown when there is an issue with the specified KMS key and the trail
     *         can’t be updated.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.PutInsightSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/PutInsightSelectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutInsightSelectorsResponse> putInsightSelectors(
            PutInsightSelectorsRequest putInsightSelectorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putInsightSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutInsightSelectors");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutInsightSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutInsightSelectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutInsightSelectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutInsightSelectorsRequest, PutInsightSelectorsResponse>()
                            .withOperationName("PutInsightSelectors")
                            .withMarshaller(new PutInsightSelectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putInsightSelectorsRequest));
            CompletableFuture<PutInsightSelectorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from a trail.
     * </p>
     *
     * @param removeTagsRequest
     *        Specifies the tags to remove from a trail.
     * @return A Java Future containing the result of the RemoveTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException This exception is thrown when the specified resource is not found.</li>
     *         <li>CloudTrailArnInvalidException This exception is thrown when an operation is called with a trail ARN
     *         that is not valid. The following is the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code></li>
     *         <li>ResourceTypeNotSupportedException This exception is thrown when the specified resource type is not
     *         supported by CloudTrail.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InvalidTagParameterException This exception is thrown when the specified tag key or values are not
     *         valid. It can also occur if there are duplicate tags or too many tags on the resource.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.RemoveTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/RemoveTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveTagsResponse> removeTags(RemoveTagsRequest removeTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RemoveTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>().withOperationName("RemoveTags")
                            .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeTagsRequest));
            CompletableFuture<RemoveTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores a deleted event data store specified by <code>EventDataStore</code>, which accepts an event data store
     * ARN. You can only restore a deleted event data store within the seven-day wait period after deletion. Restoring
     * an event data store can take several minutes, depending on the size of the event data store.
     * </p>
     *
     * @param restoreEventDataStoreRequest
     * @return A Java Future containing the result of the RestoreEventDataStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>EventDataStoreMaxLimitExceededException Your account has used the maximum number of event data
     *         stores.</li>
     *         <li>InvalidEventDataStoreStatusException The event data store is not in a status that supports the
     *         operation.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>CloudTrailAccessNotEnabledException This exception is thrown when trusted access has not been enabled
     *         between CloudTrail and Organizations. For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationsNotInUseException This exception is thrown when the request is made from an Amazon Web
     *         Services account that is not a member of an organization. To make this request, sign in using the
     *         credentials of an account that belongs to an organization.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationNotInAllFeaturesModeException This exception is thrown when Organizations is not
     *         configured to support all features. All features must be enabled in Organizations to support creating an
     *         organization trail. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.RestoreEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/RestoreEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreEventDataStoreResponse> restoreEventDataStore(
            RestoreEventDataStoreRequest restoreEventDataStoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreEventDataStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreEventDataStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreEventDataStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreEventDataStoreRequest, RestoreEventDataStoreResponse>()
                            .withOperationName("RestoreEventDataStore")
                            .withMarshaller(new RestoreEventDataStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(restoreEventDataStoreRequest));
            CompletableFuture<RestoreEventDataStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts the recording of Amazon Web Services API calls and log file delivery for a trail. For a trail that is
     * enabled in all regions, this operation must be called from the region in which the trail was created. This
     * operation cannot be called on the shadow trails (replicated trails in other regions) of a trail that is enabled
     * in all regions.
     * </p>
     *
     * @param startLoggingRequest
     *        The request to CloudTrail to start logging Amazon Web Services API calls for an account.
     * @return A Java Future containing the result of the StartLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InvalidHomeRegionException This exception is thrown when an operation is called on a trail from a
     *         region other than the region in which the trail was created.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.StartLogging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StartLogging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartLoggingResponse> startLogging(StartLoggingRequest startLoggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLoggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLogging");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartLoggingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartLoggingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartLoggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartLoggingRequest, StartLoggingResponse>()
                            .withOperationName("StartLogging").withMarshaller(new StartLoggingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startLoggingRequest));
            CompletableFuture<StartLoggingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Starts a CloudTrail Lake query. The required <code>QueryStatement</code> parameter provides your SQL query,
     * enclosed in single quotation marks.
     * </p>
     *
     * @param startQueryRequest
     * @return A Java Future containing the result of the StartQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>InvalidQueryStatementException The query that was submitted has validation errors, or uses incorrect
     *         syntax or unsupported keywords. For more information about writing a query, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-create-edit-query.html">Create or
     *         edit a query</a> in the <i>CloudTrail User Guide</i>.</li>
     *         <li>MaxConcurrentQueriesException You are already running the maximum number of concurrent queries. Wait
     *         a minute for some queries to finish, and then run the query again.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.StartQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StartQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StartQueryResponse> startQuery(StartQueryRequest startQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StartQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartQueryRequest, StartQueryResponse>().withOperationName("StartQuery")
                            .withMarshaller(new StartQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startQueryRequest));
            CompletableFuture<StartQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Suspends the recording of Amazon Web Services API calls and log file delivery for the specified trail. Under most
     * circumstances, there is no need to use this action. You can update a trail without stopping it first. This action
     * is the only way to stop recording. For a trail enabled in all regions, this operation must be called from the
     * region in which the trail was created, or an <code>InvalidHomeRegionException</code> will occur. This operation
     * cannot be called on the shadow trails (replicated trails in other regions) of a trail enabled in all regions.
     * </p>
     *
     * @param stopLoggingRequest
     *        Passes the request to CloudTrail to stop logging Amazon Web Services API calls for the specified account.
     * @return A Java Future containing the result of the StopLogging operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>InvalidHomeRegionException This exception is thrown when an operation is called on a trail from a
     *         region other than the region in which the trail was created.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.StopLogging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StopLogging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<StopLoggingResponse> stopLogging(StopLoggingRequest stopLoggingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopLoggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopLogging");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StopLoggingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    StopLoggingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StopLoggingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StopLoggingRequest, StopLoggingResponse>()
                            .withOperationName("StopLogging").withMarshaller(new StopLoggingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(stopLoggingRequest));
            CompletableFuture<StopLoggingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an event data store. The required <code>EventDataStore</code> value is an ARN or the ID portion of the
     * ARN. Other parameters are optional, but at least one optional parameter must be specified, or CloudTrail throws
     * an error. <code>RetentionPeriod</code> is in days, and valid values are integers between 90 and 2555. By default,
     * <code>TerminationProtection</code> is enabled. <code>AdvancedEventSelectors</code> includes or excludes
     * management and data events in your event data store; for more information about
     * <code>AdvancedEventSelectors</code>, see <a>PutEventSelectorsRequest&#36AdvancedEventSelectors</a>.
     * </p>
     *
     * @param updateEventDataStoreRequest
     * @return A Java Future containing the result of the UpdateEventDataStore operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>EventDataStoreArnInvalidException The specified event data store ARN is not valid or does not map to
     *         an event data store in your account.</li>
     *         <li>EventDataStoreNotFoundException The specified event data store was not found.</li>
     *         <li>InactiveEventDataStoreException The event data store against which you ran your query is inactive.</li>
     *         <li>InvalidParameterException The request includes a parameter that is not valid.</li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>CloudTrailAccessNotEnabledException This exception is thrown when trusted access has not been enabled
     *         between CloudTrail and Organizations. For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationsNotInUseException This exception is thrown when the request is made from an Amazon Web
     *         Services account that is not a member of an organization. To make this request, sign in using the
     *         credentials of an account that belongs to an organization.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationNotInAllFeaturesModeException This exception is thrown when Organizations is not
     *         configured to support all features. All features must be enabled in Organizations to support creating an
     *         organization trail. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.UpdateEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/UpdateEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEventDataStoreResponse> updateEventDataStore(
            UpdateEventDataStoreRequest updateEventDataStoreRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventDataStore");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEventDataStoreResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEventDataStoreResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEventDataStoreRequest, UpdateEventDataStoreResponse>()
                            .withOperationName("UpdateEventDataStore")
                            .withMarshaller(new UpdateEventDataStoreRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateEventDataStoreRequest));
            CompletableFuture<UpdateEventDataStoreResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates trail settings that control what events you are logging, and how to handle log files. Changes to a trail
     * do not require stopping the CloudTrail service. Use this action to designate an existing bucket for log delivery.
     * If the existing bucket has previously been a target for CloudTrail log files, an IAM policy exists for the
     * bucket. <code>UpdateTrail</code> must be called from the region in which the trail was created; otherwise, an
     * <code>InvalidHomeRegionException</code> is thrown.
     * </p>
     *
     * @param updateTrailRequest
     *        Specifies settings to update for the trail.
     * @return A Java Future containing the result of the UpdateTrail operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>S3BucketDoesNotExistException This exception is thrown when the specified S3 bucket does not exist.</li>
     *         <li>InsufficientS3BucketPolicyException This exception is thrown when the policy on the S3 bucket is not
     *         sufficient.</li>
     *         <li>InsufficientSnsTopicPolicyException This exception is thrown when the policy on the Amazon SNS topic
     *         is not sufficient.</li>
     *         <li>InsufficientEncryptionPolicyException This exception is thrown when the policy on the S3 bucket or
     *         KMS key is not sufficient.</li>
     *         <li>TrailNotFoundException This exception is thrown when the trail with the given name is not found.</li>
     *         <li>InvalidS3BucketNameException This exception is thrown when the provided S3 bucket name is not valid.</li>
     *         <li>InvalidS3PrefixException This exception is thrown when the provided S3 prefix is not valid.</li>
     *         <li>InvalidSnsTopicNameException This exception is thrown when the provided SNS topic name is not valid.</li>
     *         <li>InvalidKmsKeyIdException This exception is thrown when the KMS key ARN is not valid.</li>
     *         <li>InvalidTrailNameException This exception is thrown when the provided trail name is not valid. Trail
     *         names must meet the following requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li></li>
     *         <li>TrailNotProvidedException This exception is no longer in use.</li>
     *         <li>InvalidEventSelectorsException This exception is thrown when the <code>PutEventSelectors</code>
     *         operation is called with a number of event selectors, advanced event selectors, or data resources that is
     *         not valid. The combination of event selectors or advanced event selectors and data resources is not
     *         valid. A trail can have up to 5 event selectors. If a trail uses advanced event selectors, a maximum of
     *         500 total values for all conditions in all advanced event selectors is allowed. A trail is limited to 250
     *         data resources. These data resources can be distributed across event selectors, but the overall total
     *         cannot exceed 250.</p>
     *         <p>
     *         You can:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify a valid number of event selectors (1 to 5) for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid number of data resources (1 to 250) for an event selector. The limit of number of
     *         resources on an individual event selector is configurable up to 250. However, this upper limit is allowed
     *         only if the total number of data resources does not exceed 250 across all event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify up to 500 values for all conditions in all advanced event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid value for a parameter. For example, specifying the <code>ReadWriteType</code> parameter
     *         with a value of <code>read-only</code> is not valid.
     *         </p>
     *         </li></li>
     *         <li>InvalidParameterCombinationException This exception is thrown when the combination of parameters
     *         provided is not valid.</li>
     *         <li>InvalidHomeRegionException This exception is thrown when an operation is called on a trail from a
     *         region other than the region in which the trail was created.</li>
     *         <li>KmsKeyNotFoundException This exception is thrown when the KMS key does not exist, when the S3 bucket
     *         and the KMS key are not in the same region, or when the KMS key associated with the Amazon SNS topic
     *         either does not exist or is not in the same region.</li>
     *         <li>KmsException This exception is thrown when there is an issue with the specified KMS key and the trail
     *         can’t be updated.</li>
     *         <li>InvalidCloudWatchLogsLogGroupArnException This exception is thrown when the provided CloudWatch Logs
     *         log group is not valid.</li>
     *         <li>InvalidCloudWatchLogsRoleArnException This exception is thrown when the provided role is not valid.</li>
     *         <li>CloudWatchLogsDeliveryUnavailableException Cannot set a CloudWatch Logs delivery for this region.</li>
     *         <li>UnsupportedOperationException This exception is thrown when the requested operation is not supported.
     *         </li>
     *         <li>OperationNotPermittedException This exception is thrown when the requested operation is not
     *         permitted.</li>
     *         <li>CloudTrailAccessNotEnabledException This exception is thrown when trusted access has not been enabled
     *         between CloudTrail and Organizations. For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>InsufficientDependencyServiceAccessPermissionException This exception is thrown when the IAM user or
     *         role that is used to create the organization trail is lacking one or more required permissions for
     *         creating an organization trail in a required service. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationsNotInUseException This exception is thrown when the request is made from an Amazon Web
     *         Services account that is not a member of an organization. To make this request, sign in using the
     *         credentials of an account that belongs to an organization.</li>
     *         <li>NotOrganizationMasterAccountException This exception is thrown when the Amazon Web Services account
     *         making the request to create or update an organization trail is not the management account for an
     *         organization in Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>OrganizationNotInAllFeaturesModeException This exception is thrown when Organizations is not
     *         configured to support all features. All features must be enabled in Organizations to support creating an
     *         organization trail. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.</li>
     *         <li>CloudTrailInvalidClientTokenIdException This exception is thrown when a call results in the
     *         <code>InvalidClientTokenId</code> error code. This can occur when you are creating or updating a trail to
     *         send notifications to an Amazon SNS topic that is in a suspended Amazon Web Services account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CloudTrailException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CloudTrailAsyncClient.UpdateTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/UpdateTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTrailResponse> updateTrail(UpdateTrailRequest updateTrailRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTrail");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTrailResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTrailResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTrailRequest, UpdateTrailResponse>()
                            .withOperationName("UpdateTrail").withMarshaller(new UpdateTrailRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTrailRequest));
            CompletableFuture<UpdateTrailResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudTrailException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientDependencyServiceAccessPermissionException")
                                .exceptionBuilderSupplier(InsufficientDependencyServiceAccessPermissionException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreMaxLimitExceededException")
                                .exceptionBuilderSupplier(EventDataStoreMaxLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreTerminationProtectedException")
                                .exceptionBuilderSupplier(EventDataStoreTerminationProtectedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidKmsKeyIdException")
                                .exceptionBuilderSupplier(InvalidKmsKeyIdException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QueryIdNotFoundException")
                                .exceptionBuilderSupplier(QueryIdNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InactiveEventDataStoreException")
                                .exceptionBuilderSupplier(InactiveEventDataStoreException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TrailNotProvidedException")
                                .exceptionBuilderSupplier(TrailNotProvidedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidQueryStatusException")
                                .exceptionBuilderSupplier(InvalidQueryStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCloudWatchLogsLogGroupArnException")
                                .exceptionBuilderSupplier(InvalidCloudWatchLogsLogGroupArnException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreAlreadyExistsException")
                                .exceptionBuilderSupplier(EventDataStoreAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventCategoryException")
                                .exceptionBuilderSupplier(InvalidEventCategoryException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KmsException").exceptionBuilderSupplier(KmsException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudTrailARNInvalidException")
                                .exceptionBuilderSupplier(CloudTrailArnInvalidException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("S3BucketDoesNotExistException")
                                .exceptionBuilderSupplier(S3BucketDoesNotExistException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagsLimitExceededException")
                                .exceptionBuilderSupplier(TagsLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSnsTopicNameException")
                                .exceptionBuilderSupplier(InvalidSnsTopicNameException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotOrganizationMasterAccountException")
                                .exceptionBuilderSupplier(NotOrganizationMasterAccountException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KmsKeyNotFoundException")
                                .exceptionBuilderSupplier(KmsKeyNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLookupAttributesException")
                                .exceptionBuilderSupplier(InvalidLookupAttributesException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudTrailInvalidClientTokenIdException")
                                .exceptionBuilderSupplier(CloudTrailInvalidClientTokenIdException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OrganizationsNotInUseException")
                                .exceptionBuilderSupplier(OrganizationsNotInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceTypeNotSupportedException")
                                .exceptionBuilderSupplier(ResourceTypeNotSupportedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTokenException")
                                .exceptionBuilderSupplier(InvalidTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCloudWatchLogsRoleArnException")
                                .exceptionBuilderSupplier(InvalidCloudWatchLogsRoleArnException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidQueryStatementException")
                                .exceptionBuilderSupplier(InvalidQueryStatementException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTimeRangeException")
                                .exceptionBuilderSupplier(InvalidTimeRangeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventSelectorsException")
                                .exceptionBuilderSupplier(InvalidEventSelectorsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventDataStoreStatusException")
                                .exceptionBuilderSupplier(InvalidEventDataStoreStatusException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterCombinationException")
                                .exceptionBuilderSupplier(InvalidParameterCombinationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudTrailAccessNotEnabledException")
                                .exceptionBuilderSupplier(CloudTrailAccessNotEnabledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3BucketNameException")
                                .exceptionBuilderSupplier(InvalidS3BucketNameException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudWatchLogsDeliveryUnavailableException")
                                .exceptionBuilderSupplier(CloudWatchLogsDeliveryUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InactiveQueryException")
                                .exceptionBuilderSupplier(InactiveQueryException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTrailNameException")
                                .exceptionBuilderSupplier(InvalidTrailNameException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreNotFoundException")
                                .exceptionBuilderSupplier(EventDataStoreNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientEncryptionPolicyException")
                                .exceptionBuilderSupplier(InsufficientEncryptionPolicyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagParameterException")
                                .exceptionBuilderSupplier(InvalidTagParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TrailNotFoundException")
                                .exceptionBuilderSupplier(TrailNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreARNInvalidException")
                                .exceptionBuilderSupplier(EventDataStoreArnInvalidException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumNumberOfTrailsExceededException")
                                .exceptionBuilderSupplier(MaximumNumberOfTrailsExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3PrefixException")
                                .exceptionBuilderSupplier(InvalidS3PrefixException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInsightSelectorsException")
                                .exceptionBuilderSupplier(InvalidInsightSelectorsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDateRangeException")
                                .exceptionBuilderSupplier(InvalidDateRangeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientSnsTopicPolicyException")
                                .exceptionBuilderSupplier(InsufficientSnsTopicPolicyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaxConcurrentQueriesException")
                                .exceptionBuilderSupplier(MaxConcurrentQueriesException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientS3BucketPolicyException")
                                .exceptionBuilderSupplier(InsufficientS3BucketPolicyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidHomeRegionException")
                                .exceptionBuilderSupplier(InvalidHomeRegionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TrailAlreadyExistsException")
                                .exceptionBuilderSupplier(TrailAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsightNotEnabledException")
                                .exceptionBuilderSupplier(InsightNotEnabledException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends CloudTrailRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
