/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudtrail.model.AccountHasOngoingImportException;
import software.amazon.awssdk.services.cloudtrail.model.AddTagsRequest;
import software.amazon.awssdk.services.cloudtrail.model.AddTagsResponse;
import software.amazon.awssdk.services.cloudtrail.model.CancelQueryRequest;
import software.amazon.awssdk.services.cloudtrail.model.CancelQueryResponse;
import software.amazon.awssdk.services.cloudtrail.model.ChannelArnInvalidException;
import software.amazon.awssdk.services.cloudtrail.model.ChannelNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailAccessNotEnabledException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailArnInvalidException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailInvalidClientTokenIdException;
import software.amazon.awssdk.services.cloudtrail.model.CloudTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.CloudWatchLogsDeliveryUnavailableException;
import software.amazon.awssdk.services.cloudtrail.model.ConflictException;
import software.amazon.awssdk.services.cloudtrail.model.CreateEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.CreateEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.CreateTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.CreateTrailResponse;
import software.amazon.awssdk.services.cloudtrail.model.DeleteEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.DeleteEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.DeleteTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.DeleteTrailResponse;
import software.amazon.awssdk.services.cloudtrail.model.DescribeQueryRequest;
import software.amazon.awssdk.services.cloudtrail.model.DescribeQueryResponse;
import software.amazon.awssdk.services.cloudtrail.model.DescribeTrailsRequest;
import software.amazon.awssdk.services.cloudtrail.model.DescribeTrailsResponse;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreAlreadyExistsException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreArnInvalidException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreHasOngoingImportException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreMaxLimitExceededException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.EventDataStoreTerminationProtectedException;
import software.amazon.awssdk.services.cloudtrail.model.GetChannelRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetChannelResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetEventSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetEventSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetImportRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetImportResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetInsightSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetInsightSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailResponse;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailStatusRequest;
import software.amazon.awssdk.services.cloudtrail.model.GetTrailStatusResponse;
import software.amazon.awssdk.services.cloudtrail.model.ImportNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.InactiveEventDataStoreException;
import software.amazon.awssdk.services.cloudtrail.model.InactiveQueryException;
import software.amazon.awssdk.services.cloudtrail.model.InsightNotEnabledException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientDependencyServiceAccessPermissionException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientEncryptionPolicyException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientS3BucketPolicyException;
import software.amazon.awssdk.services.cloudtrail.model.InsufficientSnsTopicPolicyException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidCloudWatchLogsLogGroupArnException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidCloudWatchLogsRoleArnException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidDateRangeException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidEventCategoryException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidEventDataStoreCategoryException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidEventDataStoreStatusException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidEventSelectorsException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidHomeRegionException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidImportSourceException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidInsightSelectorsException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidKmsKeyIdException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidLookupAttributesException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidMaxResultsException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidNextTokenException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidParameterCombinationException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidParameterException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidQueryStatementException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidQueryStatusException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidS3BucketNameException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidS3PrefixException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidSnsTopicNameException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTagParameterException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTimeRangeException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTokenException;
import software.amazon.awssdk.services.cloudtrail.model.InvalidTrailNameException;
import software.amazon.awssdk.services.cloudtrail.model.KmsException;
import software.amazon.awssdk.services.cloudtrail.model.KmsKeyNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.ListChannelsRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListChannelsResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListImportFailuresRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListImportFailuresResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListImportsRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListImportsResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListQueriesRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListQueriesResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListTagsRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListTagsResponse;
import software.amazon.awssdk.services.cloudtrail.model.ListTrailsRequest;
import software.amazon.awssdk.services.cloudtrail.model.ListTrailsResponse;
import software.amazon.awssdk.services.cloudtrail.model.LookupEventsRequest;
import software.amazon.awssdk.services.cloudtrail.model.LookupEventsResponse;
import software.amazon.awssdk.services.cloudtrail.model.MaxConcurrentQueriesException;
import software.amazon.awssdk.services.cloudtrail.model.MaximumNumberOfTrailsExceededException;
import software.amazon.awssdk.services.cloudtrail.model.NotOrganizationMasterAccountException;
import software.amazon.awssdk.services.cloudtrail.model.OperationNotPermittedException;
import software.amazon.awssdk.services.cloudtrail.model.OrganizationNotInAllFeaturesModeException;
import software.amazon.awssdk.services.cloudtrail.model.OrganizationsNotInUseException;
import software.amazon.awssdk.services.cloudtrail.model.PutEventSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.PutEventSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.PutInsightSelectorsRequest;
import software.amazon.awssdk.services.cloudtrail.model.PutInsightSelectorsResponse;
import software.amazon.awssdk.services.cloudtrail.model.QueryIdNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.RemoveTagsRequest;
import software.amazon.awssdk.services.cloudtrail.model.RemoveTagsResponse;
import software.amazon.awssdk.services.cloudtrail.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.ResourceTypeNotSupportedException;
import software.amazon.awssdk.services.cloudtrail.model.RestoreEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.RestoreEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.S3BucketDoesNotExistException;
import software.amazon.awssdk.services.cloudtrail.model.StartImportRequest;
import software.amazon.awssdk.services.cloudtrail.model.StartImportResponse;
import software.amazon.awssdk.services.cloudtrail.model.StartLoggingRequest;
import software.amazon.awssdk.services.cloudtrail.model.StartLoggingResponse;
import software.amazon.awssdk.services.cloudtrail.model.StartQueryRequest;
import software.amazon.awssdk.services.cloudtrail.model.StartQueryResponse;
import software.amazon.awssdk.services.cloudtrail.model.StopImportRequest;
import software.amazon.awssdk.services.cloudtrail.model.StopImportResponse;
import software.amazon.awssdk.services.cloudtrail.model.StopLoggingRequest;
import software.amazon.awssdk.services.cloudtrail.model.StopLoggingResponse;
import software.amazon.awssdk.services.cloudtrail.model.TagsLimitExceededException;
import software.amazon.awssdk.services.cloudtrail.model.TrailAlreadyExistsException;
import software.amazon.awssdk.services.cloudtrail.model.TrailNotFoundException;
import software.amazon.awssdk.services.cloudtrail.model.TrailNotProvidedException;
import software.amazon.awssdk.services.cloudtrail.model.UnsupportedOperationException;
import software.amazon.awssdk.services.cloudtrail.model.UpdateEventDataStoreRequest;
import software.amazon.awssdk.services.cloudtrail.model.UpdateEventDataStoreResponse;
import software.amazon.awssdk.services.cloudtrail.model.UpdateTrailRequest;
import software.amazon.awssdk.services.cloudtrail.model.UpdateTrailResponse;
import software.amazon.awssdk.services.cloudtrail.paginators.GetQueryResultsIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListChannelsIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListEventDataStoresIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListImportFailuresIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListImportsIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListPublicKeysIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListQueriesIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListTagsIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.ListTrailsIterable;
import software.amazon.awssdk.services.cloudtrail.paginators.LookupEventsIterable;
import software.amazon.awssdk.services.cloudtrail.transform.AddTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.CancelQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.CreateEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.CreateTrailRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DeleteEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DeleteTrailRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DescribeQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.DescribeTrailsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetChannelRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetEventSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetImportRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetInsightSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetQueryResultsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetTrailRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.GetTrailStatusRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListEventDataStoresRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListImportFailuresRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListImportsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListPublicKeysRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListQueriesRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.ListTrailsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.LookupEventsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.PutEventSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.PutInsightSelectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.RemoveTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.RestoreEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StartImportRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StartLoggingRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StartQueryRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StopImportRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.StopLoggingRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.UpdateEventDataStoreRequestMarshaller;
import software.amazon.awssdk.services.cloudtrail.transform.UpdateTrailRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudTrailClient}.
 *
 * @see CloudTrailClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudTrailClient implements CloudTrailClient {
    private static final Logger log = Logger.loggerFor(DefaultCloudTrailClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudTrailClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds one or more tags to a trail or event data store, up to a limit of 50. Overwrites an existing tag's value
     * when a new value is specified for an existing tag key. Tag key names must be unique for a trail; you cannot have
     * two keys with the same name but different values. If you specify a key without a value, the tag will be created
     * with the specified key and a value of null. You can tag a trail or event data store that applies to all Amazon
     * Web Services Regions only from the Region in which the trail or event data store was created (also known as its
     * home region).
     * </p>
     *
     * @param addTagsRequest
     *        Specifies the tags to add to a trail or event data store.
     * @return Result of the AddTags operation returned by the service.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the specified resource is not found.
     * @throws CloudTrailArnInvalidException
     *         This exception is thrown when an operation is called with a trail ARN that is not valid. The following is
     *         the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code>
     * @throws ResourceTypeNotSupportedException
     *         This exception is thrown when the specified resource type is not supported by CloudTrail.
     * @throws TagsLimitExceededException
     *         The number of tags per trail has exceeded the permitted amount. Currently, the limit is 50.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InvalidTagParameterException
     *         This exception is thrown when the specified tag key or values are not valid. It can also occur if there
     *         are duplicate tags or too many tags on the resource.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws ConflictException
     *         This exception is thrown when the specified resource is not ready for an operation. This can occur when
     *         you try to run an operation on a resource before CloudTrail has time to fully load the resource. If this
     *         exception occurs, wait a few minutes, and then try the operation again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.AddTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/AddTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public AddTagsResponse addTags(AddTagsRequest addTagsRequest) throws ResourceNotFoundException,
            CloudTrailArnInvalidException, ResourceTypeNotSupportedException, TagsLimitExceededException,
            InvalidTrailNameException, InvalidTagParameterException, InactiveEventDataStoreException,
            EventDataStoreNotFoundException, UnsupportedOperationException, OperationNotPermittedException,
            NotOrganizationMasterAccountException, ConflictException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AddTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                AddTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddTags");

            return clientHandler.execute(new ClientExecutionParams<AddTagsRequest, AddTagsResponse>()
                    .withOperationName("AddTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(addTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new AddTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels a query if the query is not in a terminated state, such as <code>CANCELLED</code>, <code>FAILED</code>,
     * <code>TIMED_OUT</code>, or <code>FINISHED</code>. You must specify an ARN value for <code>EventDataStore</code>.
     * The ID of the query that you want to cancel is also required. When you run <code>CancelQuery</code>, the query
     * status might show as <code>CANCELLED</code> even if the operation is not yet finished.
     * </p>
     *
     * @param cancelQueryRequest
     * @return Result of the CancelQuery operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InactiveQueryException
     *         The specified query cannot be canceled because it is in the <code>FINISHED</code>, <code>FAILED</code>,
     *         <code>TIMED_OUT</code>, or <code>CANCELLED</code> state.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws QueryIdNotFoundException
     *         The query ID does not exist or does not map to a query.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws ConflictException
     *         This exception is thrown when the specified resource is not ready for an operation. This can occur when
     *         you try to run an operation on a resource before CloudTrail has time to fully load the resource. If this
     *         exception occurs, wait a few minutes, and then try the operation again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.CancelQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/CancelQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CancelQueryResponse cancelQuery(CancelQueryRequest cancelQueryRequest) throws EventDataStoreArnInvalidException,
            EventDataStoreNotFoundException, InactiveEventDataStoreException, InactiveQueryException, InvalidParameterException,
            QueryIdNotFoundException, OperationNotPermittedException, UnsupportedOperationException, ConflictException,
            AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuery");

            return clientHandler.execute(new ClientExecutionParams<CancelQueryRequest, CancelQueryResponse>()
                    .withOperationName("CancelQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new event data store.
     * </p>
     *
     * @param createEventDataStoreRequest
     * @return Result of the CreateEventDataStore operation returned by the service.
     * @throws EventDataStoreAlreadyExistsException
     *         An event data store with that name already exists.
     * @throws EventDataStoreMaxLimitExceededException
     *         Your account has used the maximum number of event data stores.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws InvalidTagParameterException
     *         This exception is thrown when the specified tag key or values are not valid. It can also occur if there
     *         are duplicate tags or too many tags on the resource.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws ConflictException
     *         This exception is thrown when the specified resource is not ready for an operation. This can occur when
     *         you try to run an operation on a resource before CloudTrail has time to fully load the resource. If this
     *         exception occurs, wait a few minutes, and then try the operation again.
     * @throws CloudTrailAccessNotEnabledException
     *         This exception is thrown when trusted access has not been enabled between CloudTrail and Organizations.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws OrganizationsNotInUseException
     *         This exception is thrown when the request is made from an Amazon Web Services account that is not a
     *         member of an organization. To make this request, sign in using the credentials of an account that belongs
     *         to an organization.
     * @throws OrganizationNotInAllFeaturesModeException
     *         This exception is thrown when Organizations is not configured to support all features. All features must
     *         be enabled in Organizations to support creating an organization trail or event data store.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.CreateEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/CreateEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateEventDataStoreResponse createEventDataStore(CreateEventDataStoreRequest createEventDataStoreRequest)
            throws EventDataStoreAlreadyExistsException, EventDataStoreMaxLimitExceededException, InvalidParameterException,
            InvalidTagParameterException, OperationNotPermittedException, UnsupportedOperationException, ConflictException,
            CloudTrailAccessNotEnabledException, InsufficientDependencyServiceAccessPermissionException,
            NotOrganizationMasterAccountException, OrganizationsNotInUseException, OrganizationNotInAllFeaturesModeException,
            AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateEventDataStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEventDataStore");

            return clientHandler.execute(new ClientExecutionParams<CreateEventDataStoreRequest, CreateEventDataStoreResponse>()
                    .withOperationName("CreateEventDataStore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createEventDataStoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEventDataStoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a trail that specifies the settings for delivery of log data to an Amazon S3 bucket.
     * </p>
     *
     * @param createTrailRequest
     *        Specifies the settings for each trail.
     * @return Result of the CreateTrail operation returned by the service.
     * @throws MaximumNumberOfTrailsExceededException
     *         This exception is thrown when the maximum number of trails is reached.
     * @throws TrailAlreadyExistsException
     *         This exception is thrown when the specified trail already exists.
     * @throws S3BucketDoesNotExistException
     *         This exception is thrown when the specified S3 bucket does not exist.
     * @throws InsufficientS3BucketPolicyException
     *         This exception is thrown when the policy on the S3 bucket is not sufficient.
     * @throws InsufficientSnsTopicPolicyException
     *         This exception is thrown when the policy on the Amazon SNS topic is not sufficient.
     * @throws InsufficientEncryptionPolicyException
     *         This exception is thrown when the policy on the S3 bucket or KMS key is not sufficient.
     * @throws InvalidS3BucketNameException
     *         This exception is thrown when the provided S3 bucket name is not valid.
     * @throws InvalidS3PrefixException
     *         This exception is thrown when the provided S3 prefix is not valid.
     * @throws InvalidSnsTopicNameException
     *         This exception is thrown when the provided SNS topic name is not valid.
     * @throws InvalidKmsKeyIdException
     *         This exception is thrown when the KMS key ARN is not valid.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws TrailNotProvidedException
     *         This exception is no longer in use.
     * @throws InvalidParameterCombinationException
     *         This exception is thrown when the combination of parameters provided is not valid.
     * @throws KmsKeyNotFoundException
     *         This exception is thrown when the KMS key does not exist, when the S3 bucket and the KMS key are not in
     *         the same region, or when the KMS key associated with the Amazon SNS topic either does not exist or is not
     *         in the same region.
     * @throws KmsException
     *         This exception is thrown when there is an issue with the specified KMS key and the trail can’t be
     *         updated.
     * @throws InvalidCloudWatchLogsLogGroupArnException
     *         This exception is thrown when the provided CloudWatch Logs log group is not valid.
     * @throws InvalidCloudWatchLogsRoleArnException
     *         This exception is thrown when the provided role is not valid.
     * @throws CloudWatchLogsDeliveryUnavailableException
     *         Cannot set a CloudWatch Logs delivery for this region.
     * @throws InvalidTagParameterException
     *         This exception is thrown when the specified tag key or values are not valid. It can also occur if there
     *         are duplicate tags or too many tags on the resource.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws CloudTrailAccessNotEnabledException
     *         This exception is thrown when trusted access has not been enabled between CloudTrail and Organizations.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws OrganizationsNotInUseException
     *         This exception is thrown when the request is made from an Amazon Web Services account that is not a
     *         member of an organization. To make this request, sign in using the credentials of an account that belongs
     *         to an organization.
     * @throws OrganizationNotInAllFeaturesModeException
     *         This exception is thrown when Organizations is not configured to support all features. All features must
     *         be enabled in Organizations to support creating an organization trail or event data store.
     * @throws CloudTrailInvalidClientTokenIdException
     *         This exception is thrown when a call results in the <code>InvalidClientTokenId</code> error code. This
     *         can occur when you are creating or updating a trail to send notifications to an Amazon SNS topic that is
     *         in a suspended Amazon Web Services account.
     * @throws ConflictException
     *         This exception is thrown when the specified resource is not ready for an operation. This can occur when
     *         you try to run an operation on a resource before CloudTrail has time to fully load the resource. If this
     *         exception occurs, wait a few minutes, and then try the operation again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.CreateTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/CreateTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTrailResponse createTrail(CreateTrailRequest createTrailRequest) throws MaximumNumberOfTrailsExceededException,
            TrailAlreadyExistsException, S3BucketDoesNotExistException, InsufficientS3BucketPolicyException,
            InsufficientSnsTopicPolicyException, InsufficientEncryptionPolicyException, InvalidS3BucketNameException,
            InvalidS3PrefixException, InvalidSnsTopicNameException, InvalidKmsKeyIdException, InvalidTrailNameException,
            TrailNotProvidedException, InvalidParameterCombinationException, KmsKeyNotFoundException, KmsException,
            InvalidCloudWatchLogsLogGroupArnException, InvalidCloudWatchLogsRoleArnException,
            CloudWatchLogsDeliveryUnavailableException, InvalidTagParameterException, UnsupportedOperationException,
            OperationNotPermittedException, CloudTrailAccessNotEnabledException,
            InsufficientDependencyServiceAccessPermissionException, NotOrganizationMasterAccountException,
            OrganizationsNotInUseException, OrganizationNotInAllFeaturesModeException, CloudTrailInvalidClientTokenIdException,
            ConflictException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTrailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTrail");

            return clientHandler.execute(new ClientExecutionParams<CreateTrailRequest, CreateTrailResponse>()
                    .withOperationName("CreateTrail").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createTrailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTrailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables the event data store specified by <code>EventDataStore</code>, which accepts an event data store ARN.
     * After you run <code>DeleteEventDataStore</code>, the event data store enters a <code>PENDING_DELETION</code>
     * state, and is automatically deleted after a wait period of seven days. <code>TerminationProtectionEnabled</code>
     * must be set to <code>False</code> on the event data store; this operation cannot work if
     * <code>TerminationProtectionEnabled</code> is <code>True</code>.
     * </p>
     * <p>
     * After you run <code>DeleteEventDataStore</code> on an event data store, you cannot run <code>ListQueries</code>,
     * <code>DescribeQuery</code>, or <code>GetQueryResults</code> on queries that are using an event data store in a
     * <code>PENDING_DELETION</code> state. An event data store in the <code>PENDING_DELETION</code> state does not
     * incur costs.
     * </p>
     *
     * @param deleteEventDataStoreRequest
     * @return Result of the DeleteEventDataStore operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws EventDataStoreTerminationProtectedException
     *         The event data store cannot be deleted because termination protection is enabled for it.
     * @throws EventDataStoreHasOngoingImportException
     *         This exception is thrown when you try to update or delete an event data store that currently has an
     *         import in progress.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.DeleteEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DeleteEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteEventDataStoreResponse deleteEventDataStore(DeleteEventDataStoreRequest deleteEventDataStoreRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException,
            EventDataStoreTerminationProtectedException, EventDataStoreHasOngoingImportException, InvalidParameterException,
            OperationNotPermittedException, UnsupportedOperationException, NotOrganizationMasterAccountException,
            InsufficientDependencyServiceAccessPermissionException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteEventDataStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventDataStore");

            return clientHandler.execute(new ClientExecutionParams<DeleteEventDataStoreRequest, DeleteEventDataStoreResponse>()
                    .withOperationName("DeleteEventDataStore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteEventDataStoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEventDataStoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a trail. This operation must be called from the region in which the trail was created.
     * <code>DeleteTrail</code> cannot be called on the shadow trails (replicated trails in other regions) of a trail
     * that is enabled in all regions.
     * </p>
     *
     * @param deleteTrailRequest
     *        The request that specifies the name of a trail to delete.
     * @return Result of the DeleteTrail operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InvalidHomeRegionException
     *         This exception is thrown when an operation is called on a trail from a region other than the region in
     *         which the trail was created.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws ConflictException
     *         This exception is thrown when the specified resource is not ready for an operation. This can occur when
     *         you try to run an operation on a resource before CloudTrail has time to fully load the resource. If this
     *         exception occurs, wait a few minutes, and then try the operation again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.DeleteTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DeleteTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTrailResponse deleteTrail(DeleteTrailRequest deleteTrailRequest) throws TrailNotFoundException,
            InvalidTrailNameException, InvalidHomeRegionException, UnsupportedOperationException, OperationNotPermittedException,
            NotOrganizationMasterAccountException, InsufficientDependencyServiceAccessPermissionException, ConflictException,
            AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTrailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTrail");

            return clientHandler.execute(new ClientExecutionParams<DeleteTrailRequest, DeleteTrailResponse>()
                    .withOperationName("DeleteTrail").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteTrailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTrailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns metadata about a query, including query run time in milliseconds, number of events scanned and matched,
     * and query status. You must specify an ARN for <code>EventDataStore</code>, and a value for <code>QueryID</code>.
     * </p>
     *
     * @param describeQueryRequest
     * @return Result of the DescribeQuery operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws QueryIdNotFoundException
     *         The query ID does not exist or does not map to a query.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.DescribeQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DescribeQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeQueryResponse describeQuery(DescribeQueryRequest describeQueryRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException, InactiveEventDataStoreException,
            InvalidParameterException, QueryIdNotFoundException, OperationNotPermittedException, UnsupportedOperationException,
            AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeQuery");

            return clientHandler.execute(new ClientExecutionParams<DescribeQueryRequest, DescribeQueryResponse>()
                    .withOperationName("DescribeQuery").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves settings for one or more trails associated with the current region for your account.
     * </p>
     *
     * @param describeTrailsRequest
     *        Returns information about the trail.
     * @return Result of the DescribeTrails operation returned by the service.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.DescribeTrails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/DescribeTrails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeTrailsResponse describeTrails(DescribeTrailsRequest describeTrailsRequest)
            throws UnsupportedOperationException, OperationNotPermittedException, InvalidTrailNameException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTrailsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTrailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTrailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTrails");

            return clientHandler.execute(new ClientExecutionParams<DescribeTrailsRequest, DescribeTrailsResponse>()
                    .withOperationName("DescribeTrails").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeTrailsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTrailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the specified CloudTrail service-linked channel. Amazon Web Services services create service-linked
     * channels to view CloudTrail events.
     * </p>
     *
     * @param getChannelRequest
     * @return Result of the GetChannel operation returned by the service.
     * @throws ChannelArnInvalidException
     *         The specified channel ARN is not valid or does not map to a channel in your account.
     * @throws ChannelNotFoundException
     *         The specified channel was not found.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetChannel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetChannelResponse getChannel(GetChannelRequest getChannelRequest) throws ChannelArnInvalidException,
            ChannelNotFoundException, OperationNotPermittedException, UnsupportedOperationException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetChannelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannel");

            return clientHandler
                    .execute(new ClientExecutionParams<GetChannelRequest, GetChannelResponse>().withOperationName("GetChannel")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getChannelRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetChannelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about an event data store specified as either an ARN or the ID portion of the ARN.
     * </p>
     *
     * @param getEventDataStoreRequest
     * @return Result of the GetEventDataStore operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetEventDataStore" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetEventDataStoreResponse getEventDataStore(GetEventDataStoreRequest getEventDataStoreRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException, InvalidParameterException,
            OperationNotPermittedException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEventDataStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventDataStore");

            return clientHandler.execute(new ClientExecutionParams<GetEventDataStoreRequest, GetEventDataStoreResponse>()
                    .withOperationName("GetEventDataStore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEventDataStoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEventDataStoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the settings for the event selectors that you configured for your trail. The information returned for
     * your event selectors includes the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If your event selector includes read-only events, write-only events, or all events. This applies to both
     * management events and data events.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your event selector includes management events.
     * </p>
     * </li>
     * <li>
     * <p>
     * If your event selector includes data events, the resources on which you are logging data events.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about logging management and data events, see the following topics in the <i>CloudTrail User
     * Guide</i>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html"
     * >Logging management events for trails </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html">
     * Logging data events for trails </a>
     * </p>
     * </li>
     * </ul>
     *
     * @param getEventSelectorsRequest
     * @return Result of the GetEventSelectors operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetEventSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetEventSelectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetEventSelectorsResponse getEventSelectors(GetEventSelectorsRequest getEventSelectorsRequest)
            throws TrailNotFoundException, InvalidTrailNameException, UnsupportedOperationException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEventSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEventSelectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventSelectors");

            return clientHandler.execute(new ClientExecutionParams<GetEventSelectorsRequest, GetEventSelectorsResponse>()
                    .withOperationName("GetEventSelectors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEventSelectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEventSelectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information for the specified import.
     * </p>
     *
     * @param getImportRequest
     * @return Result of the GetImport operation returned by the service.
     * @throws ImportNotFoundException
     *         The specified import was not found.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetImport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetImportResponse getImport(GetImportRequest getImportRequest) throws ImportNotFoundException,
            InvalidParameterException, OperationNotPermittedException, UnsupportedOperationException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetImportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetImport");

            return clientHandler.execute(new ClientExecutionParams<GetImportRequest, GetImportResponse>()
                    .withOperationName("GetImport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getImportRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetImportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the settings for the Insights event selectors that you configured for your trail.
     * <code>GetInsightSelectors</code> shows if CloudTrail Insights event logging is enabled on the trail, and if it
     * is, which insight types are enabled. If you run <code>GetInsightSelectors</code> on a trail that does not have
     * Insights events enabled, the operation throws the exception <code>InsightNotEnabledException</code>
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-insights-events-with-cloudtrail.html"
     * >Logging CloudTrail Insights Events for Trails </a> in the <i>CloudTrail User Guide</i>.
     * </p>
     *
     * @param getInsightSelectorsRequest
     * @return Result of the GetInsightSelectors operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws InsightNotEnabledException
     *         If you run <code>GetInsightSelectors</code> on a trail that does not have Insights events enabled, the
     *         operation throws the exception <code>InsightNotEnabledException</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetInsightSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetInsightSelectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInsightSelectorsResponse getInsightSelectors(GetInsightSelectorsRequest getInsightSelectorsRequest)
            throws TrailNotFoundException, InvalidTrailNameException, UnsupportedOperationException,
            OperationNotPermittedException, InsightNotEnabledException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInsightSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetInsightSelectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightSelectors");

            return clientHandler.execute(new ClientExecutionParams<GetInsightSelectorsRequest, GetInsightSelectorsResponse>()
                    .withOperationName("GetInsightSelectors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getInsightSelectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInsightSelectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets event data results of a query. You must specify the <code>QueryID</code> value returned by the
     * <code>StartQuery</code> operation, and an ARN for <code>EventDataStore</code>.
     * </p>
     *
     * @param getQueryResultsRequest
     * @return Result of the GetQueryResults operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws QueryIdNotFoundException
     *         The query ID does not exist or does not map to a query.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetQueryResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetQueryResults" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetQueryResultsResponse getQueryResults(GetQueryResultsRequest getQueryResultsRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException, InactiveEventDataStoreException,
            InvalidMaxResultsException, InvalidNextTokenException, InvalidParameterException, QueryIdNotFoundException,
            OperationNotPermittedException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetQueryResultsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetQueryResultsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueryResultsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueryResults");

            return clientHandler.execute(new ClientExecutionParams<GetQueryResultsRequest, GetQueryResultsResponse>()
                    .withOperationName("GetQueryResults").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getQueryResultsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetQueryResultsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets event data results of a query. You must specify the <code>QueryID</code> value returned by the
     * <code>StartQuery</code> operation, and an ARN for <code>EventDataStore</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getQueryResults(software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.GetQueryResultsIterable responses = client.getQueryResultsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.GetQueryResultsIterable responses = client
     *             .getQueryResultsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.GetQueryResultsIterable responses = client.getQueryResultsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getQueryResults(software.amazon.awssdk.services.cloudtrail.model.GetQueryResultsRequest)} operation.</b>
     * </p>
     *
     * @param getQueryResultsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws QueryIdNotFoundException
     *         The query ID does not exist or does not map to a query.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetQueryResults
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetQueryResults" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetQueryResultsIterable getQueryResultsPaginator(GetQueryResultsRequest getQueryResultsRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException, InactiveEventDataStoreException,
            InvalidMaxResultsException, InvalidNextTokenException, InvalidParameterException, QueryIdNotFoundException,
            OperationNotPermittedException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            CloudTrailException {
        return new GetQueryResultsIterable(this, applyPaginatorUserAgent(getQueryResultsRequest));
    }

    /**
     * <p>
     * Returns settings information for a specified trail.
     * </p>
     *
     * @param getTrailRequest
     * @return Result of the GetTrail operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTrailResponse getTrail(GetTrailRequest getTrailRequest) throws TrailNotFoundException, InvalidTrailNameException,
            UnsupportedOperationException, OperationNotPermittedException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTrailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTrail");

            return clientHandler.execute(new ClientExecutionParams<GetTrailRequest, GetTrailResponse>()
                    .withOperationName("GetTrail").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTrailRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetTrailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a JSON-formatted list of information about the specified trail. Fields include information on delivery
     * errors, Amazon SNS and Amazon S3 errors, and start and stop logging times for each trail. This operation returns
     * trail status from a single region. To return trail status from all regions, you must call the operation on each
     * region.
     * </p>
     *
     * @param getTrailStatusRequest
     *        The name of a trail about which you want the current status.
     * @return Result of the GetTrailStatus operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.GetTrailStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/GetTrailStatus" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTrailStatusResponse getTrailStatus(GetTrailStatusRequest getTrailStatusRequest) throws TrailNotFoundException,
            InvalidTrailNameException, UnsupportedOperationException, OperationNotPermittedException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTrailStatusResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTrailStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTrailStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetTrailStatus");

            return clientHandler.execute(new ClientExecutionParams<GetTrailStatusRequest, GetTrailStatusResponse>()
                    .withOperationName("GetTrailStatus").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getTrailStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetTrailStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all CloudTrail channels.
     * </p>
     *
     * @param listChannelsRequest
     * @return Result of the ListChannels operation returned by the service.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChannelsResponse listChannels(ListChannelsRequest listChannelsRequest) throws InvalidNextTokenException,
            OperationNotPermittedException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChannelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");

            return clientHandler.execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                    .withOperationName("ListChannels").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listChannelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListChannelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all CloudTrail channels.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listChannels(software.amazon.awssdk.services.cloudtrail.model.ListChannelsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListChannelsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListChannelsIterable responses = client.listChannelsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.cloudtrail.model.ListChannelsRequest)} operation.</b>
     * </p>
     *
     * @param listChannelsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListChannels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChannelsIterable listChannelsPaginator(ListChannelsRequest listChannelsRequest) throws InvalidNextTokenException,
            OperationNotPermittedException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            CloudTrailException {
        return new ListChannelsIterable(this, applyPaginatorUserAgent(listChannelsRequest));
    }

    /**
     * <p>
     * Returns information about all event data stores in the account, in the current region.
     * </p>
     *
     * @param listEventDataStoresRequest
     * @return Result of the ListEventDataStores operation returned by the service.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListEventDataStores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListEventDataStores"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEventDataStoresResponse listEventDataStores(ListEventDataStoresRequest listEventDataStoresRequest)
            throws InvalidMaxResultsException, InvalidNextTokenException, OperationNotPermittedException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventDataStoresResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListEventDataStoresResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventDataStoresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEventDataStores");

            return clientHandler.execute(new ClientExecutionParams<ListEventDataStoresRequest, ListEventDataStoresResponse>()
                    .withOperationName("ListEventDataStores").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listEventDataStoresRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEventDataStoresRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about all event data stores in the account, in the current region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEventDataStores(software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListEventDataStoresIterable responses = client.listEventDataStoresPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListEventDataStoresIterable responses = client
     *             .listEventDataStoresPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListEventDataStoresIterable responses = client.listEventDataStoresPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEventDataStores(software.amazon.awssdk.services.cloudtrail.model.ListEventDataStoresRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEventDataStoresRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListEventDataStores
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListEventDataStores"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEventDataStoresIterable listEventDataStoresPaginator(ListEventDataStoresRequest listEventDataStoresRequest)
            throws InvalidMaxResultsException, InvalidNextTokenException, OperationNotPermittedException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, CloudTrailException {
        return new ListEventDataStoresIterable(this, applyPaginatorUserAgent(listEventDataStoresRequest));
    }

    /**
     * <p>
     * Returns a list of failures for the specified import.
     * </p>
     *
     * @param listImportFailuresRequest
     * @return Result of the ListImportFailures operation returned by the service.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListImportFailures
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListImportFailures" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListImportFailuresResponse listImportFailures(ListImportFailuresRequest listImportFailuresRequest)
            throws InvalidNextTokenException, OperationNotPermittedException, UnsupportedOperationException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListImportFailuresResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListImportFailuresResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImportFailuresRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImportFailures");

            return clientHandler.execute(new ClientExecutionParams<ListImportFailuresRequest, ListImportFailuresResponse>()
                    .withOperationName("ListImportFailures").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listImportFailuresRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListImportFailuresRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of failures for the specified import.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listImportFailures(software.amazon.awssdk.services.cloudtrail.model.ListImportFailuresRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListImportFailuresIterable responses = client.listImportFailuresPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListImportFailuresIterable responses = client
     *             .listImportFailuresPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListImportFailuresResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListImportFailuresIterable responses = client.listImportFailuresPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImportFailures(software.amazon.awssdk.services.cloudtrail.model.ListImportFailuresRequest)}
     * operation.</b>
     * </p>
     *
     * @param listImportFailuresRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListImportFailures
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListImportFailures" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListImportFailuresIterable listImportFailuresPaginator(ListImportFailuresRequest listImportFailuresRequest)
            throws InvalidNextTokenException, OperationNotPermittedException, UnsupportedOperationException, AwsServiceException,
            SdkClientException, CloudTrailException {
        return new ListImportFailuresIterable(this, applyPaginatorUserAgent(listImportFailuresRequest));
    }

    /**
     * <p>
     * Returns information on all imports, or a select set of imports by <code>ImportStatus</code> or
     * <code>Destination</code>.
     * </p>
     *
     * @param listImportsRequest
     * @return Result of the ListImports operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListImports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListImports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListImportsResponse listImports(ListImportsRequest listImportsRequest) throws EventDataStoreArnInvalidException,
            InvalidNextTokenException, InvalidParameterException, OperationNotPermittedException, UnsupportedOperationException,
            AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListImportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListImportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImports");

            return clientHandler.execute(new ClientExecutionParams<ListImportsRequest, ListImportsResponse>()
                    .withOperationName("ListImports").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listImportsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListImportsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information on all imports, or a select set of imports by <code>ImportStatus</code> or
     * <code>Destination</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listImports(software.amazon.awssdk.services.cloudtrail.model.ListImportsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListImportsIterable responses = client.listImportsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListImportsIterable responses = client.listImportsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListImportsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListImportsIterable responses = client.listImportsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listImports(software.amazon.awssdk.services.cloudtrail.model.ListImportsRequest)} operation.</b>
     * </p>
     *
     * @param listImportsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListImports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListImports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListImportsIterable listImportsPaginator(ListImportsRequest listImportsRequest)
            throws EventDataStoreArnInvalidException, InvalidNextTokenException, InvalidParameterException,
            OperationNotPermittedException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            CloudTrailException {
        return new ListImportsIterable(this, applyPaginatorUserAgent(listImportsRequest));
    }

    /**
     * <p>
     * Returns all public keys whose private keys were used to sign the digest files within the specified time range.
     * The public key is needed to validate digest files that were signed with its corresponding private key.
     * </p>
     * <note>
     * <p>
     * CloudTrail uses different private and public key pairs per region. Each digest file is signed with a private key
     * unique to its region. When you validate a digest file from a specific region, you must look in the same region
     * for its corresponding public key.
     * </p>
     * </note>
     *
     * @param listPublicKeysRequest
     *        Requests the public keys for a specified time range.
     * @return Result of the ListPublicKeys operation returned by the service.
     * @throws InvalidTimeRangeException
     *         Occurs if the timestamp values are not valid. Either the start time occurs after the end time, or the
     *         time range is outside the range of possible values.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws InvalidTokenException
     *         Reserved for future use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListPublicKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListPublicKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPublicKeysResponse listPublicKeys(ListPublicKeysRequest listPublicKeysRequest) throws InvalidTimeRangeException,
            UnsupportedOperationException, OperationNotPermittedException, InvalidTokenException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPublicKeysResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPublicKeysResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPublicKeysRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPublicKeys");

            return clientHandler.execute(new ClientExecutionParams<ListPublicKeysRequest, ListPublicKeysResponse>()
                    .withOperationName("ListPublicKeys").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listPublicKeysRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPublicKeysRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns all public keys whose private keys were used to sign the digest files within the specified time range.
     * The public key is needed to validate digest files that were signed with its corresponding private key.
     * </p>
     * <note>
     * <p>
     * CloudTrail uses different private and public key pairs per region. Each digest file is signed with a private key
     * unique to its region. When you validate a digest file from a specific region, you must look in the same region
     * for its corresponding public key.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listPublicKeys(software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListPublicKeysIterable responses = client.listPublicKeysPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListPublicKeysIterable responses = client
     *             .listPublicKeysPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListPublicKeysIterable responses = client.listPublicKeysPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPublicKeys(software.amazon.awssdk.services.cloudtrail.model.ListPublicKeysRequest)} operation.</b>
     * </p>
     *
     * @param listPublicKeysRequest
     *        Requests the public keys for a specified time range.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidTimeRangeException
     *         Occurs if the timestamp values are not valid. Either the start time occurs after the end time, or the
     *         time range is outside the range of possible values.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws InvalidTokenException
     *         Reserved for future use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListPublicKeys
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListPublicKeys" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPublicKeysIterable listPublicKeysPaginator(ListPublicKeysRequest listPublicKeysRequest)
            throws InvalidTimeRangeException, UnsupportedOperationException, OperationNotPermittedException,
            InvalidTokenException, AwsServiceException, SdkClientException, CloudTrailException {
        return new ListPublicKeysIterable(this, applyPaginatorUserAgent(listPublicKeysRequest));
    }

    /**
     * <p>
     * Returns a list of queries and query statuses for the past seven days. You must specify an ARN value for
     * <code>EventDataStore</code>. Optionally, to shorten the list of results, you can specify a time range, formatted
     * as timestamps, by adding <code>StartTime</code> and <code>EndTime</code> parameters, and a
     * <code>QueryStatus</code> value. Valid values for <code>QueryStatus</code> include <code>QUEUED</code>,
     * <code>RUNNING</code>, <code>FINISHED</code>, <code>FAILED</code>, <code>TIMED_OUT</code>, or
     * <code>CANCELLED</code>.
     * </p>
     *
     * @param listQueriesRequest
     * @return Result of the ListQueries operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidDateRangeException
     *         A date range for the query was specified that is not valid. Be sure that the start time is
     *         chronologically before the end time. For more information about writing a query, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-create-edit-query.html">Create or
     *         edit a query</a> in the <i>CloudTrail User Guide</i>.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws InvalidQueryStatusException
     *         The query status is not valid for the operation.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListQueries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueriesResponse listQueries(ListQueriesRequest listQueriesRequest) throws EventDataStoreArnInvalidException,
            EventDataStoreNotFoundException, InactiveEventDataStoreException, InvalidDateRangeException,
            InvalidMaxResultsException, InvalidNextTokenException, InvalidParameterException, InvalidQueryStatusException,
            OperationNotPermittedException, UnsupportedOperationException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListQueriesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListQueriesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueries");

            return clientHandler.execute(new ClientExecutionParams<ListQueriesRequest, ListQueriesResponse>()
                    .withOperationName("ListQueries").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listQueriesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListQueriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of queries and query statuses for the past seven days. You must specify an ARN value for
     * <code>EventDataStore</code>. Optionally, to shorten the list of results, you can specify a time range, formatted
     * as timestamps, by adding <code>StartTime</code> and <code>EndTime</code> parameters, and a
     * <code>QueryStatus</code> value. Valid values for <code>QueryStatus</code> include <code>QUEUED</code>,
     * <code>RUNNING</code>, <code>FINISHED</code>, <code>FAILED</code>, <code>TIMED_OUT</code>, or
     * <code>CANCELLED</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listQueries(software.amazon.awssdk.services.cloudtrail.model.ListQueriesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListQueriesIterable responses = client.listQueriesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListQueriesIterable responses = client.listQueriesPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListQueriesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListQueriesIterable responses = client.listQueriesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listQueries(software.amazon.awssdk.services.cloudtrail.model.ListQueriesRequest)} operation.</b>
     * </p>
     *
     * @param listQueriesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidDateRangeException
     *         A date range for the query was specified that is not valid. Be sure that the start time is
     *         chronologically before the end time. For more information about writing a query, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-create-edit-query.html">Create or
     *         edit a query</a> in the <i>CloudTrail User Guide</i>.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws InvalidQueryStatusException
     *         The query status is not valid for the operation.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListQueries" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListQueriesIterable listQueriesPaginator(ListQueriesRequest listQueriesRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException, InactiveEventDataStoreException,
            InvalidDateRangeException, InvalidMaxResultsException, InvalidNextTokenException, InvalidParameterException,
            InvalidQueryStatusException, OperationNotPermittedException, UnsupportedOperationException, AwsServiceException,
            SdkClientException, CloudTrailException {
        return new ListQueriesIterable(this, applyPaginatorUserAgent(listQueriesRequest));
    }

    /**
     * <p>
     * Lists the tags for the trail or event data store in the current region.
     * </p>
     *
     * @param listTagsRequest
     *        Specifies a list of tags to return.
     * @return Result of the ListTags operation returned by the service.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the specified resource is not found.
     * @throws CloudTrailArnInvalidException
     *         This exception is thrown when an operation is called with a trail ARN that is not valid. The following is
     *         the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code>
     * @throws ResourceTypeNotSupportedException
     *         This exception is thrown when the specified resource type is not supported by CloudTrail.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws InvalidTokenException
     *         Reserved for future use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsResponse listTags(ListTagsRequest listTagsRequest) throws ResourceNotFoundException,
            CloudTrailArnInvalidException, ResourceTypeNotSupportedException, InvalidTrailNameException,
            InactiveEventDataStoreException, EventDataStoreNotFoundException, UnsupportedOperationException,
            OperationNotPermittedException, InvalidTokenException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");

            return clientHandler.execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>()
                    .withOperationName("ListTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the trail or event data store in the current region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTags(software.amazon.awssdk.services.cloudtrail.model.ListTagsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListTagsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTags(software.amazon.awssdk.services.cloudtrail.model.ListTagsRequest)} operation.</b>
     * </p>
     *
     * @param listTagsRequest
     *        Specifies a list of tags to return.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the specified resource is not found.
     * @throws CloudTrailArnInvalidException
     *         This exception is thrown when an operation is called with a trail ARN that is not valid. The following is
     *         the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code>
     * @throws ResourceTypeNotSupportedException
     *         This exception is thrown when the specified resource type is not supported by CloudTrail.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws InvalidTokenException
     *         Reserved for future use.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsIterable listTagsPaginator(ListTagsRequest listTagsRequest) throws ResourceNotFoundException,
            CloudTrailArnInvalidException, ResourceTypeNotSupportedException, InvalidTrailNameException,
            InactiveEventDataStoreException, EventDataStoreNotFoundException, UnsupportedOperationException,
            OperationNotPermittedException, InvalidTokenException, AwsServiceException, SdkClientException, CloudTrailException {
        return new ListTagsIterable(this, applyPaginatorUserAgent(listTagsRequest));
    }

    /**
     * <p>
     * Lists trails that are in the current account.
     * </p>
     *
     * @param listTrailsRequest
     * @return Result of the ListTrails operation returned by the service.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListTrails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTrails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTrailsResponse listTrails(ListTrailsRequest listTrailsRequest) throws UnsupportedOperationException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTrailsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTrailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTrailsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTrails");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTrailsRequest, ListTrailsResponse>().withOperationName("ListTrails")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTrailsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTrailsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists trails that are in the current account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTrails(software.amazon.awssdk.services.cloudtrail.model.ListTrailsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTrailsIterable responses = client.listTrailsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.ListTrailsIterable responses = client.listTrailsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.ListTrailsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.ListTrailsIterable responses = client.listTrailsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTrails(software.amazon.awssdk.services.cloudtrail.model.ListTrailsRequest)} operation.</b>
     * </p>
     *
     * @param listTrailsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.ListTrails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/ListTrails" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTrailsIterable listTrailsPaginator(ListTrailsRequest listTrailsRequest) throws UnsupportedOperationException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, CloudTrailException {
        return new ListTrailsIterable(this, applyPaginatorUserAgent(listTrailsRequest));
    }

    /**
     * <p>
     * Looks up <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-management-events"
     * >management events</a> or <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-insights-events"
     * >CloudTrail Insights events</a> that are captured by CloudTrail. You can look up events that occurred in a region
     * within the last 90 days. Lookup supports the following attributes for management events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Web Services access key
     * </p>
     * </li>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * <li>
     * <p>
     * Read only
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource name
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource type
     * </p>
     * </li>
     * <li>
     * <p>
     * User name
     * </p>
     * </li>
     * </ul>
     * <p>
     * Lookup supports the following attributes for Insights events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * </ul>
     * <p>
     * All attributes are optional. The default number of results returned is 50, with a maximum of 50 possible. The
     * response includes a token that you can use to get the next page of results.
     * </p>
     * <important>
     * <p>
     * The rate of lookup requests is limited to two per second, per account, per region. If this limit is exceeded, a
     * throttling error occurs.
     * </p>
     * </important>
     *
     * @param lookupEventsRequest
     *        Contains a request for LookupEvents.
     * @return Result of the LookupEvents operation returned by the service.
     * @throws InvalidLookupAttributesException
     *         Occurs when a lookup attribute is specified that is not valid.
     * @throws InvalidTimeRangeException
     *         Occurs if the timestamp values are not valid. Either the start time occurs after the end time, or the
     *         time range is outside the range of possible values.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidEventCategoryException
     *         Occurs if an event category that is not valid is specified as a value of <code>EventCategory</code>.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.LookupEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/LookupEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public LookupEventsResponse lookupEvents(LookupEventsRequest lookupEventsRequest) throws InvalidLookupAttributesException,
            InvalidTimeRangeException, InvalidMaxResultsException, InvalidNextTokenException, InvalidEventCategoryException,
            UnsupportedOperationException, OperationNotPermittedException, AwsServiceException, SdkClientException,
            CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<LookupEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                LookupEventsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, lookupEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "LookupEvents");

            return clientHandler.execute(new ClientExecutionParams<LookupEventsRequest, LookupEventsResponse>()
                    .withOperationName("LookupEvents").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(lookupEventsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new LookupEventsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Looks up <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-management-events"
     * >management events</a> or <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/cloudtrail-concepts.html#cloudtrail-concepts-insights-events"
     * >CloudTrail Insights events</a> that are captured by CloudTrail. You can look up events that occurred in a region
     * within the last 90 days. Lookup supports the following attributes for management events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Web Services access key
     * </p>
     * </li>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * <li>
     * <p>
     * Read only
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource name
     * </p>
     * </li>
     * <li>
     * <p>
     * Resource type
     * </p>
     * </li>
     * <li>
     * <p>
     * User name
     * </p>
     * </li>
     * </ul>
     * <p>
     * Lookup supports the following attributes for Insights events:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Event ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Event name
     * </p>
     * </li>
     * <li>
     * <p>
     * Event source
     * </p>
     * </li>
     * </ul>
     * <p>
     * All attributes are optional. The default number of results returned is 50, with a maximum of 50 possible. The
     * response includes a token that you can use to get the next page of results.
     * </p>
     * <important>
     * <p>
     * The rate of lookup requests is limited to two per second, per account, per region. If this limit is exceeded, a
     * throttling error occurs.
     * </p>
     * </important><br/>
     * <p>
     * This is a variant of {@link #lookupEvents(software.amazon.awssdk.services.cloudtrail.model.LookupEventsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.LookupEventsIterable responses = client.lookupEventsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudtrail.paginators.LookupEventsIterable responses = client.lookupEventsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudtrail.model.LookupEventsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudtrail.paginators.LookupEventsIterable responses = client.lookupEventsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #lookupEvents(software.amazon.awssdk.services.cloudtrail.model.LookupEventsRequest)} operation.</b>
     * </p>
     *
     * @param lookupEventsRequest
     *        Contains a request for LookupEvents.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidLookupAttributesException
     *         Occurs when a lookup attribute is specified that is not valid.
     * @throws InvalidTimeRangeException
     *         Occurs if the timestamp values are not valid. Either the start time occurs after the end time, or the
     *         time range is outside the range of possible values.
     * @throws InvalidMaxResultsException
     *         This exception is thrown if the limit specified is not valid.
     * @throws InvalidNextTokenException
     *         A token that is not valid, or a token that was previously used in a request with different parameters.
     *         This exception is thrown if the token is not valid.
     * @throws InvalidEventCategoryException
     *         Occurs if an event category that is not valid is specified as a value of <code>EventCategory</code>.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.LookupEvents
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/LookupEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public LookupEventsIterable lookupEventsPaginator(LookupEventsRequest lookupEventsRequest)
            throws InvalidLookupAttributesException, InvalidTimeRangeException, InvalidMaxResultsException,
            InvalidNextTokenException, InvalidEventCategoryException, UnsupportedOperationException,
            OperationNotPermittedException, AwsServiceException, SdkClientException, CloudTrailException {
        return new LookupEventsIterable(this, applyPaginatorUserAgent(lookupEventsRequest));
    }

    /**
     * <p>
     * Configures an event selector or advanced event selectors for your trail. Use event selectors or advanced event
     * selectors to specify management and data event settings for your trail. By default, trails created without
     * specific event selectors are configured to log all read and write management events, and no data events.
     * </p>
     * <p>
     * When an event occurs in your account, CloudTrail evaluates the event selectors or advanced event selectors in all
     * trails. For each trail, if the event matches any event selector, the trail processes and logs the event. If the
     * event doesn't match any event selector, the trail doesn't log the event.
     * </p>
     * <p>
     * Example
     * </p>
     * <ol>
     * <li>
     * <p>
     * You create an event selector for a trail and specify that you want write-only events.
     * </p>
     * </li>
     * <li>
     * <p>
     * The EC2 <code>GetConsoleOutput</code> and <code>RunInstances</code> API operations occur in your account.
     * </p>
     * </li>
     * <li>
     * <p>
     * CloudTrail evaluates whether the events match your event selectors.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>RunInstances</code> is a write-only event and it matches your event selector. The trail logs the event.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>GetConsoleOutput</code> is a read-only event that doesn't match your event selector. The trail doesn't
     * log the event.
     * </p>
     * </li>
     * </ol>
     * <p>
     * The <code>PutEventSelectors</code> operation must be called from the region in which the trail was created;
     * otherwise, an <code>InvalidHomeRegionException</code> exception is thrown.
     * </p>
     * <p>
     * You can configure up to five event selectors for each trail. For more information, see <a href=
     * "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-events-with-cloudtrail.html"
     * >Logging management events for trails </a>, <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html"
     * >Logging data events for trails </a>, and <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Quotas in
     * CloudTrail</a> in the <i>CloudTrail User Guide</i>.
     * </p>
     * <p>
     * You can add advanced event selectors, and conditions for your advanced event selectors, up to a maximum of 500
     * values for all conditions and selectors on a trail. You can use either <code>AdvancedEventSelectors</code> or
     * <code>EventSelectors</code>, but not both. If you apply <code>AdvancedEventSelectors</code> to a trail, any
     * existing <code>EventSelectors</code> are overwritten. For more information about advanced event selectors, see <a
     * href
     * ="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html">Logging
     * data events for trails</a> in the <i>CloudTrail User Guide</i>.
     * </p>
     *
     * @param putEventSelectorsRequest
     * @return Result of the PutEventSelectors operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InvalidHomeRegionException
     *         This exception is thrown when an operation is called on a trail from a region other than the region in
     *         which the trail was created.
     * @throws InvalidEventSelectorsException
     *         This exception is thrown when the <code>PutEventSelectors</code> operation is called with a number of
     *         event selectors, advanced event selectors, or data resources that is not valid. The combination of event
     *         selectors or advanced event selectors and data resources is not valid. A trail can have up to 5 event
     *         selectors. If a trail uses advanced event selectors, a maximum of 500 total values for all conditions in
     *         all advanced event selectors is allowed. A trail is limited to 250 data resources. These data resources
     *         can be distributed across event selectors, but the overall total cannot exceed 250.</p>
     *         <p>
     *         You can:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify a valid number of event selectors (1 to 5) for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid number of data resources (1 to 250) for an event selector. The limit of number of
     *         resources on an individual event selector is configurable up to 250. However, this upper limit is allowed
     *         only if the total number of data resources does not exceed 250 across all event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify up to 500 values for all conditions in all advanced event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid value for a parameter. For example, specifying the <code>ReadWriteType</code> parameter
     *         with a value of <code>read-only</code> is not valid.
     *         </p>
     *         </li>
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.PutEventSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/PutEventSelectors" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutEventSelectorsResponse putEventSelectors(PutEventSelectorsRequest putEventSelectorsRequest)
            throws TrailNotFoundException, InvalidTrailNameException, InvalidHomeRegionException, InvalidEventSelectorsException,
            UnsupportedOperationException, OperationNotPermittedException, NotOrganizationMasterAccountException,
            InsufficientDependencyServiceAccessPermissionException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutEventSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutEventSelectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEventSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEventSelectors");

            return clientHandler.execute(new ClientExecutionParams<PutEventSelectorsRequest, PutEventSelectorsResponse>()
                    .withOperationName("PutEventSelectors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putEventSelectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutEventSelectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lets you enable Insights event logging by specifying the Insights selectors that you want to enable on an
     * existing trail. You also use <code>PutInsightSelectors</code> to turn off Insights event logging, by passing an
     * empty list of insight types. The valid Insights event types in this release are <code>ApiErrorRateInsight</code>
     * and <code>ApiCallRateInsight</code>.
     * </p>
     *
     * @param putInsightSelectorsRequest
     * @return Result of the PutInsightSelectors operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InvalidHomeRegionException
     *         This exception is thrown when an operation is called on a trail from a region other than the region in
     *         which the trail was created.
     * @throws InvalidInsightSelectorsException
     *         The formatting or syntax of the <code>InsightSelectors</code> JSON statement in your
     *         <code>PutInsightSelectors</code> or <code>GetInsightSelectors</code> request is not valid, or the
     *         specified insight type in the <code>InsightSelectors</code> statement is not a valid insight type.
     * @throws InsufficientS3BucketPolicyException
     *         This exception is thrown when the policy on the S3 bucket is not sufficient.
     * @throws InsufficientEncryptionPolicyException
     *         This exception is thrown when the policy on the S3 bucket or KMS key is not sufficient.
     * @throws S3BucketDoesNotExistException
     *         This exception is thrown when the specified S3 bucket does not exist.
     * @throws KmsException
     *         This exception is thrown when there is an issue with the specified KMS key and the trail can’t be
     *         updated.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.PutInsightSelectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/PutInsightSelectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutInsightSelectorsResponse putInsightSelectors(PutInsightSelectorsRequest putInsightSelectorsRequest)
            throws TrailNotFoundException, InvalidTrailNameException, InvalidHomeRegionException,
            InvalidInsightSelectorsException, InsufficientS3BucketPolicyException, InsufficientEncryptionPolicyException,
            S3BucketDoesNotExistException, KmsException, UnsupportedOperationException, OperationNotPermittedException,
            NotOrganizationMasterAccountException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutInsightSelectorsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutInsightSelectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putInsightSelectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutInsightSelectors");

            return clientHandler.execute(new ClientExecutionParams<PutInsightSelectorsRequest, PutInsightSelectorsResponse>()
                    .withOperationName("PutInsightSelectors").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putInsightSelectorsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutInsightSelectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tags from a trail or event data store.
     * </p>
     *
     * @param removeTagsRequest
     *        Specifies the tags to remove from a trail or event data store.
     * @return Result of the RemoveTags operation returned by the service.
     * @throws ResourceNotFoundException
     *         This exception is thrown when the specified resource is not found.
     * @throws CloudTrailArnInvalidException
     *         This exception is thrown when an operation is called with a trail ARN that is not valid. The following is
     *         the format of a trail ARN.</p>
     *         <p>
     *         <code>arn:aws:cloudtrail:us-east-2:123456789012:trail/MyTrail</code>
     * @throws ResourceTypeNotSupportedException
     *         This exception is thrown when the specified resource type is not supported by CloudTrail.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InvalidTagParameterException
     *         This exception is thrown when the specified tag key or values are not valid. It can also occur if there
     *         are duplicate tags or too many tags on the resource.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.RemoveTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/RemoveTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RemoveTagsResponse removeTags(RemoveTagsRequest removeTagsRequest) throws ResourceNotFoundException,
            CloudTrailArnInvalidException, ResourceTypeNotSupportedException, InvalidTrailNameException,
            InvalidTagParameterException, InactiveEventDataStoreException, EventDataStoreNotFoundException,
            UnsupportedOperationException, OperationNotPermittedException, NotOrganizationMasterAccountException,
            AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RemoveTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RemoveTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveTags");

            return clientHandler
                    .execute(new ClientExecutionParams<RemoveTagsRequest, RemoveTagsResponse>().withOperationName("RemoveTags")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(removeTagsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RemoveTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Restores a deleted event data store specified by <code>EventDataStore</code>, which accepts an event data store
     * ARN. You can only restore a deleted event data store within the seven-day wait period after deletion. Restoring
     * an event data store can take several minutes, depending on the size of the event data store.
     * </p>
     *
     * @param restoreEventDataStoreRequest
     * @return Result of the RestoreEventDataStore operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws EventDataStoreMaxLimitExceededException
     *         Your account has used the maximum number of event data stores.
     * @throws InvalidEventDataStoreStatusException
     *         The event data store is not in a status that supports the operation.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws CloudTrailAccessNotEnabledException
     *         This exception is thrown when trusted access has not been enabled between CloudTrail and Organizations.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws OrganizationsNotInUseException
     *         This exception is thrown when the request is made from an Amazon Web Services account that is not a
     *         member of an organization. To make this request, sign in using the credentials of an account that belongs
     *         to an organization.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws OrganizationNotInAllFeaturesModeException
     *         This exception is thrown when Organizations is not configured to support all features. All features must
     *         be enabled in Organizations to support creating an organization trail or event data store.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.RestoreEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/RestoreEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RestoreEventDataStoreResponse restoreEventDataStore(RestoreEventDataStoreRequest restoreEventDataStoreRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException, EventDataStoreMaxLimitExceededException,
            InvalidEventDataStoreStatusException, InvalidParameterException, OperationNotPermittedException,
            UnsupportedOperationException, CloudTrailAccessNotEnabledException,
            InsufficientDependencyServiceAccessPermissionException, OrganizationsNotInUseException,
            NotOrganizationMasterAccountException, OrganizationNotInAllFeaturesModeException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RestoreEventDataStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreEventDataStore");

            return clientHandler.execute(new ClientExecutionParams<RestoreEventDataStoreRequest, RestoreEventDataStoreResponse>()
                    .withOperationName("RestoreEventDataStore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(restoreEventDataStoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RestoreEventDataStoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts an import of logged trail events from a source S3 bucket to a destination event data store.
     * </p>
     * <p>
     * When you start a new import, the <code>Destinations</code> and <code>ImportSource</code> parameters are required.
     * Before starting a new import, disable any access control lists (ACLs) attached to the source S3 bucket. For more
     * information about disabling ACLs, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/about-object-ownership.html">Controlling ownership of
     * objects and disabling ACLs for your bucket</a>.
     * </p>
     * <p>
     * When you retry an import, the <code>ImportID</code> parameter is required.
     * </p>
     *
     * @param startImportRequest
     * @return Result of the StartImport operation returned by the service.
     * @throws AccountHasOngoingImportException
     *         This exception is thrown when you start a new import and a previous import is still in progress.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InvalidEventDataStoreStatusException
     *         The event data store is not in a status that supports the operation.
     * @throws InvalidEventDataStoreCategoryException
     *         This exception is thrown when the event data store category is not valid for the import.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidImportSourceException
     *         This exception is thrown when the provided source S3 bucket is not valid for import.
     * @throws ImportNotFoundException
     *         The specified import was not found.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.StartImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StartImport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartImportResponse startImport(StartImportRequest startImportRequest) throws AccountHasOngoingImportException,
            EventDataStoreArnInvalidException, EventDataStoreNotFoundException, InvalidEventDataStoreStatusException,
            InvalidEventDataStoreCategoryException, InactiveEventDataStoreException, InvalidImportSourceException,
            ImportNotFoundException, InvalidParameterException, OperationNotPermittedException, UnsupportedOperationException,
            AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartImportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartImport");

            return clientHandler.execute(new ClientExecutionParams<StartImportRequest, StartImportResponse>()
                    .withOperationName("StartImport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startImportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartImportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the recording of Amazon Web Services API calls and log file delivery for a trail. For a trail that is
     * enabled in all regions, this operation must be called from the region in which the trail was created. This
     * operation cannot be called on the shadow trails (replicated trails in other regions) of a trail that is enabled
     * in all regions.
     * </p>
     *
     * @param startLoggingRequest
     *        The request to CloudTrail to start logging Amazon Web Services API calls for an account.
     * @return Result of the StartLogging operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InvalidHomeRegionException
     *         This exception is thrown when an operation is called on a trail from a region other than the region in
     *         which the trail was created.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.StartLogging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StartLogging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartLoggingResponse startLogging(StartLoggingRequest startLoggingRequest) throws TrailNotFoundException,
            InvalidTrailNameException, InvalidHomeRegionException, UnsupportedOperationException, OperationNotPermittedException,
            NotOrganizationMasterAccountException, InsufficientDependencyServiceAccessPermissionException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartLoggingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartLoggingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startLoggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartLogging");

            return clientHandler.execute(new ClientExecutionParams<StartLoggingRequest, StartLoggingResponse>()
                    .withOperationName("StartLogging").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startLoggingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartLoggingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a CloudTrail Lake query. The required <code>QueryStatement</code> parameter provides your SQL query,
     * enclosed in single quotation marks.
     * </p>
     *
     * @param startQueryRequest
     * @return Result of the StartQuery operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws InvalidQueryStatementException
     *         The query that was submitted has validation errors, or uses incorrect syntax or unsupported keywords. For
     *         more information about writing a query, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-create-edit-query.html">Create or
     *         edit a query</a> in the <i>CloudTrail User Guide</i>.
     * @throws MaxConcurrentQueriesException
     *         You are already running the maximum number of concurrent queries. Wait a minute for some queries to
     *         finish, and then run the query again.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.StartQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StartQuery" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartQueryResponse startQuery(StartQueryRequest startQueryRequest) throws EventDataStoreArnInvalidException,
            EventDataStoreNotFoundException, InactiveEventDataStoreException, InvalidParameterException,
            InvalidQueryStatementException, MaxConcurrentQueriesException, OperationNotPermittedException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartQuery");

            return clientHandler
                    .execute(new ClientExecutionParams<StartQueryRequest, StartQueryResponse>().withOperationName("StartQuery")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(startQueryRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a specified import.
     * </p>
     *
     * @param stopImportRequest
     * @return Result of the StopImport operation returned by the service.
     * @throws ImportNotFoundException
     *         The specified import was not found.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.StopImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StopImport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopImportResponse stopImport(StopImportRequest stopImportRequest) throws ImportNotFoundException,
            InvalidParameterException, OperationNotPermittedException, UnsupportedOperationException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopImportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopImport");

            return clientHandler
                    .execute(new ClientExecutionParams<StopImportRequest, StopImportResponse>().withOperationName("StopImport")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(stopImportRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopImportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Suspends the recording of Amazon Web Services API calls and log file delivery for the specified trail. Under most
     * circumstances, there is no need to use this action. You can update a trail without stopping it first. This action
     * is the only way to stop recording. For a trail enabled in all regions, this operation must be called from the
     * region in which the trail was created, or an <code>InvalidHomeRegionException</code> will occur. This operation
     * cannot be called on the shadow trails (replicated trails in other regions) of a trail enabled in all regions.
     * </p>
     *
     * @param stopLoggingRequest
     *        Passes the request to CloudTrail to stop logging Amazon Web Services API calls for the specified account.
     * @return Result of the StopLogging operation returned by the service.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws InvalidHomeRegionException
     *         This exception is thrown when an operation is called on a trail from a region other than the region in
     *         which the trail was created.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.StopLogging
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/StopLogging" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopLoggingResponse stopLogging(StopLoggingRequest stopLoggingRequest) throws TrailNotFoundException,
            InvalidTrailNameException, InvalidHomeRegionException, UnsupportedOperationException, OperationNotPermittedException,
            NotOrganizationMasterAccountException, InsufficientDependencyServiceAccessPermissionException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopLoggingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopLoggingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopLoggingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopLogging");

            return clientHandler.execute(new ClientExecutionParams<StopLoggingRequest, StopLoggingResponse>()
                    .withOperationName("StopLogging").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopLoggingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopLoggingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an event data store. The required <code>EventDataStore</code> value is an ARN or the ID portion of the
     * ARN. Other parameters are optional, but at least one optional parameter must be specified, or CloudTrail throws
     * an error. <code>RetentionPeriod</code> is in days, and valid values are integers between 90 and 2557. By default,
     * <code>TerminationProtection</code> is enabled. <code>AdvancedEventSelectors</code> includes or excludes
     * management and data events in your event data store; for more information about
     * <code>AdvancedEventSelectors</code>, see <a>PutEventSelectorsRequest&#36AdvancedEventSelectors</a>.
     * </p>
     *
     * @param updateEventDataStoreRequest
     * @return Result of the UpdateEventDataStore operation returned by the service.
     * @throws EventDataStoreArnInvalidException
     *         The specified event data store ARN is not valid or does not map to an event data store in your account.
     * @throws EventDataStoreNotFoundException
     *         The specified event data store was not found.
     * @throws EventDataStoreHasOngoingImportException
     *         This exception is thrown when you try to update or delete an event data store that currently has an
     *         import in progress.
     * @throws InactiveEventDataStoreException
     *         The event data store is inactive.
     * @throws InvalidParameterException
     *         The request includes a parameter that is not valid.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws CloudTrailAccessNotEnabledException
     *         This exception is thrown when trusted access has not been enabled between CloudTrail and Organizations.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws OrganizationsNotInUseException
     *         This exception is thrown when the request is made from an Amazon Web Services account that is not a
     *         member of an organization. To make this request, sign in using the credentials of an account that belongs
     *         to an organization.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws OrganizationNotInAllFeaturesModeException
     *         This exception is thrown when Organizations is not configured to support all features. All features must
     *         be enabled in Organizations to support creating an organization trail or event data store.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.UpdateEventDataStore
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/UpdateEventDataStore"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEventDataStoreResponse updateEventDataStore(UpdateEventDataStoreRequest updateEventDataStoreRequest)
            throws EventDataStoreArnInvalidException, EventDataStoreNotFoundException, EventDataStoreHasOngoingImportException,
            InactiveEventDataStoreException, InvalidParameterException, OperationNotPermittedException,
            UnsupportedOperationException, CloudTrailAccessNotEnabledException,
            InsufficientDependencyServiceAccessPermissionException, OrganizationsNotInUseException,
            NotOrganizationMasterAccountException, OrganizationNotInAllFeaturesModeException, AwsServiceException,
            SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEventDataStoreResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEventDataStoreResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEventDataStoreRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEventDataStore");

            return clientHandler.execute(new ClientExecutionParams<UpdateEventDataStoreRequest, UpdateEventDataStoreResponse>()
                    .withOperationName("UpdateEventDataStore").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateEventDataStoreRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEventDataStoreRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates trail settings that control what events you are logging, and how to handle log files. Changes to a trail
     * do not require stopping the CloudTrail service. Use this action to designate an existing bucket for log delivery.
     * If the existing bucket has previously been a target for CloudTrail log files, an IAM policy exists for the
     * bucket. <code>UpdateTrail</code> must be called from the region in which the trail was created; otherwise, an
     * <code>InvalidHomeRegionException</code> is thrown.
     * </p>
     *
     * @param updateTrailRequest
     *        Specifies settings to update for the trail.
     * @return Result of the UpdateTrail operation returned by the service.
     * @throws S3BucketDoesNotExistException
     *         This exception is thrown when the specified S3 bucket does not exist.
     * @throws InsufficientS3BucketPolicyException
     *         This exception is thrown when the policy on the S3 bucket is not sufficient.
     * @throws InsufficientSnsTopicPolicyException
     *         This exception is thrown when the policy on the Amazon SNS topic is not sufficient.
     * @throws InsufficientEncryptionPolicyException
     *         This exception is thrown when the policy on the S3 bucket or KMS key is not sufficient.
     * @throws TrailNotFoundException
     *         This exception is thrown when the trail with the given name is not found.
     * @throws InvalidS3BucketNameException
     *         This exception is thrown when the provided S3 bucket name is not valid.
     * @throws InvalidS3PrefixException
     *         This exception is thrown when the provided S3 prefix is not valid.
     * @throws InvalidSnsTopicNameException
     *         This exception is thrown when the provided SNS topic name is not valid.
     * @throws InvalidKmsKeyIdException
     *         This exception is thrown when the KMS key ARN is not valid.
     * @throws InvalidTrailNameException
     *         This exception is thrown when the provided trail name is not valid. Trail names must meet the following
     *         requirements:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Contain only ASCII letters (a-z, A-Z), numbers (0-9), periods (.), underscores (_), or dashes (-)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start with a letter or number, and end with a letter or number
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Be between 3 and 128 characters
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Have no adjacent periods, underscores or dashes. Names like <code>my-_namespace</code> and
     *         <code>my--namespace</code> are not valid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Not be in IP address format (for example, 192.168.5.4)
     *         </p>
     *         </li>
     * @throws TrailNotProvidedException
     *         This exception is no longer in use.
     * @throws InvalidEventSelectorsException
     *         This exception is thrown when the <code>PutEventSelectors</code> operation is called with a number of
     *         event selectors, advanced event selectors, or data resources that is not valid. The combination of event
     *         selectors or advanced event selectors and data resources is not valid. A trail can have up to 5 event
     *         selectors. If a trail uses advanced event selectors, a maximum of 500 total values for all conditions in
     *         all advanced event selectors is allowed. A trail is limited to 250 data resources. These data resources
     *         can be distributed across event selectors, but the overall total cannot exceed 250.</p>
     *         <p>
     *         You can:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Specify a valid number of event selectors (1 to 5) for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid number of data resources (1 to 250) for an event selector. The limit of number of
     *         resources on an individual event selector is configurable up to 250. However, this upper limit is allowed
     *         only if the total number of data resources does not exceed 250 across all event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify up to 500 values for all conditions in all advanced event selectors for a trail.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Specify a valid value for a parameter. For example, specifying the <code>ReadWriteType</code> parameter
     *         with a value of <code>read-only</code> is not valid.
     *         </p>
     *         </li>
     * @throws InvalidParameterCombinationException
     *         This exception is thrown when the combination of parameters provided is not valid.
     * @throws InvalidHomeRegionException
     *         This exception is thrown when an operation is called on a trail from a region other than the region in
     *         which the trail was created.
     * @throws KmsKeyNotFoundException
     *         This exception is thrown when the KMS key does not exist, when the S3 bucket and the KMS key are not in
     *         the same region, or when the KMS key associated with the Amazon SNS topic either does not exist or is not
     *         in the same region.
     * @throws KmsException
     *         This exception is thrown when there is an issue with the specified KMS key and the trail can’t be
     *         updated.
     * @throws InvalidCloudWatchLogsLogGroupArnException
     *         This exception is thrown when the provided CloudWatch Logs log group is not valid.
     * @throws InvalidCloudWatchLogsRoleArnException
     *         This exception is thrown when the provided role is not valid.
     * @throws CloudWatchLogsDeliveryUnavailableException
     *         Cannot set a CloudWatch Logs delivery for this region.
     * @throws UnsupportedOperationException
     *         This exception is thrown when the requested operation is not supported.
     * @throws OperationNotPermittedException
     *         This exception is thrown when the requested operation is not permitted.
     * @throws CloudTrailAccessNotEnabledException
     *         This exception is thrown when trusted access has not been enabled between CloudTrail and Organizations.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/organizations/latest/userguide/orgs_integrate_services.html">Enabling
     *         Trusted Access with Other Amazon Web Services Services</a> and <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a>.
     * @throws InsufficientDependencyServiceAccessPermissionException
     *         This exception is thrown when the IAM user or role that is used to create the organization resource lacks
     *         one or more required permissions for creating an organization resource in a required service.
     * @throws OrganizationsNotInUseException
     *         This exception is thrown when the request is made from an Amazon Web Services account that is not a
     *         member of an organization. To make this request, sign in using the credentials of an account that belongs
     *         to an organization.
     * @throws NotOrganizationMasterAccountException
     *         This exception is thrown when the Amazon Web Services account making the request to create or update an
     *         organization trail or event data store is not the management account for an organization in
     *         Organizations. For more information, see <a href=
     *         "https://docs.aws.amazon.com/awscloudtrail/latest/userguide/creating-an-organizational-trail-prepare.html"
     *         >Prepare For Creating a Trail For Your Organization</a> or <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-event-data-store.html">Create an
     *         event data store</a>.
     * @throws OrganizationNotInAllFeaturesModeException
     *         This exception is thrown when Organizations is not configured to support all features. All features must
     *         be enabled in Organizations to support creating an organization trail or event data store.
     * @throws CloudTrailInvalidClientTokenIdException
     *         This exception is thrown when a call results in the <code>InvalidClientTokenId</code> error code. This
     *         can occur when you are creating or updating a trail to send notifications to an Amazon SNS topic that is
     *         in a suspended Amazon Web Services account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudTrailException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudTrailClient.UpdateTrail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudtrail-2013-11-01/UpdateTrail" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTrailResponse updateTrail(UpdateTrailRequest updateTrailRequest) throws S3BucketDoesNotExistException,
            InsufficientS3BucketPolicyException, InsufficientSnsTopicPolicyException, InsufficientEncryptionPolicyException,
            TrailNotFoundException, InvalidS3BucketNameException, InvalidS3PrefixException, InvalidSnsTopicNameException,
            InvalidKmsKeyIdException, InvalidTrailNameException, TrailNotProvidedException, InvalidEventSelectorsException,
            InvalidParameterCombinationException, InvalidHomeRegionException, KmsKeyNotFoundException, KmsException,
            InvalidCloudWatchLogsLogGroupArnException, InvalidCloudWatchLogsRoleArnException,
            CloudWatchLogsDeliveryUnavailableException, UnsupportedOperationException, OperationNotPermittedException,
            CloudTrailAccessNotEnabledException, InsufficientDependencyServiceAccessPermissionException,
            OrganizationsNotInUseException, NotOrganizationMasterAccountException, OrganizationNotInAllFeaturesModeException,
            CloudTrailInvalidClientTokenIdException, AwsServiceException, SdkClientException, CloudTrailException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTrailResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTrailResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTrailRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudTrail");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTrail");

            return clientHandler.execute(new ClientExecutionParams<UpdateTrailRequest, UpdateTrailResponse>()
                    .withOperationName("UpdateTrail").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateTrailRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTrailRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudTrailException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientDependencyServiceAccessPermissionException")
                                .exceptionBuilderSupplier(InsufficientDependencyServiceAccessPermissionException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreMaxLimitExceededException")
                                .exceptionBuilderSupplier(EventDataStoreMaxLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreTerminationProtectedException")
                                .exceptionBuilderSupplier(EventDataStoreTerminationProtectedException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidKmsKeyIdException")
                                .exceptionBuilderSupplier(InvalidKmsKeyIdException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ChannelARNInvalidException")
                                .exceptionBuilderSupplier(ChannelArnInvalidException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("QueryIdNotFoundException")
                                .exceptionBuilderSupplier(QueryIdNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InactiveEventDataStoreException")
                                .exceptionBuilderSupplier(InactiveEventDataStoreException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TrailNotProvidedException")
                                .exceptionBuilderSupplier(TrailNotProvidedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidQueryStatusException")
                                .exceptionBuilderSupplier(InvalidQueryStatusException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCloudWatchLogsLogGroupArnException")
                                .exceptionBuilderSupplier(InvalidCloudWatchLogsLogGroupArnException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventDataStoreCategoryException")
                                .exceptionBuilderSupplier(InvalidEventDataStoreCategoryException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreAlreadyExistsException")
                                .exceptionBuilderSupplier(EventDataStoreAlreadyExistsException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventCategoryException")
                                .exceptionBuilderSupplier(InvalidEventCategoryException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KmsException").exceptionBuilderSupplier(KmsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OrganizationNotInAllFeaturesModeException")
                                .exceptionBuilderSupplier(OrganizationNotInAllFeaturesModeException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudTrailARNInvalidException")
                                .exceptionBuilderSupplier(CloudTrailArnInvalidException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("S3BucketDoesNotExistException")
                                .exceptionBuilderSupplier(S3BucketDoesNotExistException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagsLimitExceededException")
                                .exceptionBuilderSupplier(TagsLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSnsTopicNameException")
                                .exceptionBuilderSupplier(InvalidSnsTopicNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotOrganizationMasterAccountException")
                                .exceptionBuilderSupplier(NotOrganizationMasterAccountException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("KmsKeyNotFoundException")
                                .exceptionBuilderSupplier(KmsKeyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OperationNotPermittedException")
                                .exceptionBuilderSupplier(OperationNotPermittedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidLookupAttributesException")
                                .exceptionBuilderSupplier(InvalidLookupAttributesException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccountHasOngoingImportException")
                                .exceptionBuilderSupplier(AccountHasOngoingImportException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudTrailInvalidClientTokenIdException")
                                .exceptionBuilderSupplier(CloudTrailInvalidClientTokenIdException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OrganizationsNotInUseException")
                                .exceptionBuilderSupplier(OrganizationsNotInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceTypeNotSupportedException")
                                .exceptionBuilderSupplier(ResourceTypeNotSupportedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTokenException")
                                .exceptionBuilderSupplier(InvalidTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCloudWatchLogsRoleArnException")
                                .exceptionBuilderSupplier(InvalidCloudWatchLogsRoleArnException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidQueryStatementException")
                                .exceptionBuilderSupplier(InvalidQueryStatementException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTimeRangeException")
                                .exceptionBuilderSupplier(InvalidTimeRangeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventSelectorsException")
                                .exceptionBuilderSupplier(InvalidEventSelectorsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedOperationException")
                                .exceptionBuilderSupplier(UnsupportedOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEventDataStoreStatusException")
                                .exceptionBuilderSupplier(InvalidEventDataStoreStatusException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImportNotFoundException")
                                .exceptionBuilderSupplier(ImportNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterCombinationException")
                                .exceptionBuilderSupplier(InvalidParameterCombinationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidMaxResultsException")
                                .exceptionBuilderSupplier(InvalidMaxResultsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudTrailAccessNotEnabledException")
                                .exceptionBuilderSupplier(CloudTrailAccessNotEnabledException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3BucketNameException")
                                .exceptionBuilderSupplier(InvalidS3BucketNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudWatchLogsDeliveryUnavailableException")
                                .exceptionBuilderSupplier(CloudWatchLogsDeliveryUnavailableException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InactiveQueryException")
                                .exceptionBuilderSupplier(InactiveQueryException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTrailNameException")
                                .exceptionBuilderSupplier(InvalidTrailNameException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidImportSourceException")
                                .exceptionBuilderSupplier(InvalidImportSourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreNotFoundException")
                                .exceptionBuilderSupplier(EventDataStoreNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientEncryptionPolicyException")
                                .exceptionBuilderSupplier(InsufficientEncryptionPolicyException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagParameterException")
                                .exceptionBuilderSupplier(InvalidTagParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TrailNotFoundException")
                                .exceptionBuilderSupplier(TrailNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreARNInvalidException")
                                .exceptionBuilderSupplier(EventDataStoreArnInvalidException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaximumNumberOfTrailsExceededException")
                                .exceptionBuilderSupplier(MaximumNumberOfTrailsExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EventDataStoreHasOngoingImportException")
                                .exceptionBuilderSupplier(EventDataStoreHasOngoingImportException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidS3PrefixException")
                                .exceptionBuilderSupplier(InvalidS3PrefixException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidInsightSelectorsException")
                                .exceptionBuilderSupplier(InvalidInsightSelectorsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidDateRangeException")
                                .exceptionBuilderSupplier(InvalidDateRangeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientSnsTopicPolicyException")
                                .exceptionBuilderSupplier(InsufficientSnsTopicPolicyException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ChannelNotFoundException")
                                .exceptionBuilderSupplier(ChannelNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MaxConcurrentQueriesException")
                                .exceptionBuilderSupplier(MaxConcurrentQueriesException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsufficientS3BucketPolicyException")
                                .exceptionBuilderSupplier(InsufficientS3BucketPolicyException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidHomeRegionException")
                                .exceptionBuilderSupplier(InvalidHomeRegionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TrailAlreadyExistsException")
                                .exceptionBuilderSupplier(TrailAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InsightNotEnabledException")
                                .exceptionBuilderSupplier(InsightNotEnabledException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CloudTrailRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
