/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Use event selectors to further specify the management and data event settings for your trail. By default, trails
 * created without specific event selectors will be configured to log all read and write management events, and no data
 * events. When an event occurs in your account, CloudTrail evaluates the event selector for all trails. For each trail,
 * if the event matches any event selector, the trail processes and logs the event. If the event doesn't match any event
 * selector, the trail doesn't log the event.
 * </p>
 * <p>
 * You can configure up to five event selectors for a trail.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventSelector implements SdkPojo, Serializable, ToCopyableBuilder<EventSelector.Builder, EventSelector> {
    private static final SdkField<String> READ_WRITE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EventSelector::readWriteTypeAsString)).setter(setter(Builder::readWriteType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadWriteType").build()).build();

    private static final SdkField<Boolean> INCLUDE_MANAGEMENT_EVENTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(EventSelector::includeManagementEvents)).setter(setter(Builder::includeManagementEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeManagementEvents").build())
            .build();

    private static final SdkField<List<DataResource>> DATA_RESOURCES_FIELD = SdkField
            .<List<DataResource>> builder(MarshallingType.LIST)
            .getter(getter(EventSelector::dataResources))
            .setter(setter(Builder::dataResources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataResources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DataResource> builder(MarshallingType.SDK_POJO)
                                            .constructor(DataResource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(READ_WRITE_TYPE_FIELD,
            INCLUDE_MANAGEMENT_EVENTS_FIELD, DATA_RESOURCES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String readWriteType;

    private final Boolean includeManagementEvents;

    private final List<DataResource> dataResources;

    private EventSelector(BuilderImpl builder) {
        this.readWriteType = builder.readWriteType;
        this.includeManagementEvents = builder.includeManagementEvents;
        this.dataResources = builder.dataResources;
    }

    /**
     * <p>
     * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
     * operation.
     * </p>
     * <p>
     * By default, the value is <code>All</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #readWriteType}
     * will return {@link ReadWriteType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #readWriteTypeAsString}.
     * </p>
     * 
     * @return Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     *         <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only
     *         API operation.</p>
     *         <p>
     *         By default, the value is <code>All</code>.
     * @see ReadWriteType
     */
    public ReadWriteType readWriteType() {
        return ReadWriteType.fromValue(readWriteType);
    }

    /**
     * <p>
     * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
     * operation.
     * </p>
     * <p>
     * By default, the value is <code>All</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #readWriteType}
     * will return {@link ReadWriteType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #readWriteTypeAsString}.
     * </p>
     * 
     * @return Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
     *         <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only
     *         API operation.</p>
     *         <p>
     *         By default, the value is <code>All</code>.
     * @see ReadWriteType
     */
    public String readWriteTypeAsString() {
        return readWriteType;
    }

    /**
     * <p>
     * Specify if you want your event selector to include management events for your trail.
     * </p>
     * <p>
     * For more information, see <a href=
     * "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-management-events"
     * >Management Events</a> in the <i>AWS CloudTrail User Guide</i>.
     * </p>
     * <p>
     * By default, the value is <code>true</code>.
     * </p>
     * 
     * @return Specify if you want your event selector to include management events for your trail.</p>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-management-events"
     *         >Management Events</a> in the <i>AWS CloudTrail User Guide</i>.
     *         </p>
     *         <p>
     *         By default, the value is <code>true</code>.
     */
    public Boolean includeManagementEvents() {
        return includeManagementEvents;
    }

    /**
     * <p>
     * CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions. You can specify up to 250
     * resources for an individual event selector, but the total number of data resources cannot exceed 250 across all
     * event selectors in a trail. This limit does not apply if you configure resource logging for all data events.
     * </p>
     * <p>
     * For more information, see <a href=
     * "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-data-events"
     * >Data Events</a> and <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in AWS
     * CloudTrail</a> in the <i>AWS CloudTrail User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions. You can specify up
     *         to 250 resources for an individual event selector, but the total number of data resources cannot exceed
     *         250 across all event selectors in a trail. This limit does not apply if you configure resource logging
     *         for all data events. </p>
     *         <p>
     *         For more information, see <a href=
     *         "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-data-events"
     *         >Data Events</a> and <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in
     *         AWS CloudTrail</a> in the <i>AWS CloudTrail User Guide</i>.
     */
    public List<DataResource> dataResources() {
        return dataResources;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(readWriteTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(includeManagementEvents());
        hashCode = 31 * hashCode + Objects.hashCode(dataResources());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventSelector)) {
            return false;
        }
        EventSelector other = (EventSelector) obj;
        return Objects.equals(readWriteTypeAsString(), other.readWriteTypeAsString())
                && Objects.equals(includeManagementEvents(), other.includeManagementEvents())
                && Objects.equals(dataResources(), other.dataResources());
    }

    @Override
    public String toString() {
        return ToString.builder("EventSelector").add("ReadWriteType", readWriteTypeAsString())
                .add("IncludeManagementEvents", includeManagementEvents()).add("DataResources", dataResources()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReadWriteType":
            return Optional.ofNullable(clazz.cast(readWriteTypeAsString()));
        case "IncludeManagementEvents":
            return Optional.ofNullable(clazz.cast(includeManagementEvents()));
        case "DataResources":
            return Optional.ofNullable(clazz.cast(dataResources()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventSelector, T> g) {
        return obj -> g.apply((EventSelector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventSelector> {
        /**
         * <p>
         * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
         * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
         * operation.
         * </p>
         * <p>
         * By default, the value is <code>All</code>.
         * </p>
         * 
         * @param readWriteType
         *        Specify if you want your trail to log read-only events, write-only events, or all. For example, the
         *        EC2 <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a
         *        write-only API operation.</p>
         *        <p>
         *        By default, the value is <code>All</code>.
         * @see ReadWriteType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReadWriteType
         */
        Builder readWriteType(String readWriteType);

        /**
         * <p>
         * Specify if you want your trail to log read-only events, write-only events, or all. For example, the EC2
         * <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a write-only API
         * operation.
         * </p>
         * <p>
         * By default, the value is <code>All</code>.
         * </p>
         * 
         * @param readWriteType
         *        Specify if you want your trail to log read-only events, write-only events, or all. For example, the
         *        EC2 <code>GetConsoleOutput</code> is a read-only API operation and <code>RunInstances</code> is a
         *        write-only API operation.</p>
         *        <p>
         *        By default, the value is <code>All</code>.
         * @see ReadWriteType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReadWriteType
         */
        Builder readWriteType(ReadWriteType readWriteType);

        /**
         * <p>
         * Specify if you want your event selector to include management events for your trail.
         * </p>
         * <p>
         * For more information, see <a href=
         * "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-management-events"
         * >Management Events</a> in the <i>AWS CloudTrail User Guide</i>.
         * </p>
         * <p>
         * By default, the value is <code>true</code>.
         * </p>
         * 
         * @param includeManagementEvents
         *        Specify if you want your event selector to include management events for your trail.</p>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-management-events"
         *        >Management Events</a> in the <i>AWS CloudTrail User Guide</i>.
         *        </p>
         *        <p>
         *        By default, the value is <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeManagementEvents(Boolean includeManagementEvents);

        /**
         * <p>
         * CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions. You can specify up to
         * 250 resources for an individual event selector, but the total number of data resources cannot exceed 250
         * across all event selectors in a trail. This limit does not apply if you configure resource logging for all
         * data events.
         * </p>
         * <p>
         * For more information, see <a href=
         * "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-data-events"
         * >Data Events</a> and <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in AWS
         * CloudTrail</a> in the <i>AWS CloudTrail User Guide</i>.
         * </p>
         * 
         * @param dataResources
         *        CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions. You can specify
         *        up to 250 resources for an individual event selector, but the total number of data resources cannot
         *        exceed 250 across all event selectors in a trail. This limit does not apply if you configure resource
         *        logging for all data events. </p>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-data-events"
         *        >Data Events</a> and <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits
         *        in AWS CloudTrail</a> in the <i>AWS CloudTrail User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataResources(Collection<DataResource> dataResources);

        /**
         * <p>
         * CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions. You can specify up to
         * 250 resources for an individual event selector, but the total number of data resources cannot exceed 250
         * across all event selectors in a trail. This limit does not apply if you configure resource logging for all
         * data events.
         * </p>
         * <p>
         * For more information, see <a href=
         * "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-data-events"
         * >Data Events</a> and <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in AWS
         * CloudTrail</a> in the <i>AWS CloudTrail User Guide</i>.
         * </p>
         * 
         * @param dataResources
         *        CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions. You can specify
         *        up to 250 resources for an individual event selector, but the total number of data resources cannot
         *        exceed 250 across all event selectors in a trail. This limit does not apply if you configure resource
         *        logging for all data events. </p>
         *        <p>
         *        For more information, see <a href=
         *        "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-data-events"
         *        >Data Events</a> and <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits
         *        in AWS CloudTrail</a> in the <i>AWS CloudTrail User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataResources(DataResource... dataResources);

        /**
         * <p>
         * CloudTrail supports data event logging for Amazon S3 objects and AWS Lambda functions. You can specify up to
         * 250 resources for an individual event selector, but the total number of data resources cannot exceed 250
         * across all event selectors in a trail. This limit does not apply if you configure resource logging for all
         * data events.
         * </p>
         * <p>
         * For more information, see <a href=
         * "http://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-management-and-data-events-with-cloudtrail.html#logging-data-events"
         * >Data Events</a> and <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/WhatIsCloudTrail-Limits.html">Limits in AWS
         * CloudTrail</a> in the <i>AWS CloudTrail User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<DataResource>.Builder} avoiding the need to
         * create one manually via {@link List<DataResource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<DataResource>.Builder#build()} is called immediately and its
         * result is passed to {@link #dataResources(List<DataResource>)}.
         * 
         * @param dataResources
         *        a consumer that will call methods on {@link List<DataResource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataResources(List<DataResource>)
         */
        Builder dataResources(Consumer<DataResource.Builder>... dataResources);
    }

    static final class BuilderImpl implements Builder {
        private String readWriteType;

        private Boolean includeManagementEvents;

        private List<DataResource> dataResources = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EventSelector model) {
            readWriteType(model.readWriteType);
            includeManagementEvents(model.includeManagementEvents);
            dataResources(model.dataResources);
        }

        public final String getReadWriteTypeAsString() {
            return readWriteType;
        }

        @Override
        public final Builder readWriteType(String readWriteType) {
            this.readWriteType = readWriteType;
            return this;
        }

        @Override
        public final Builder readWriteType(ReadWriteType readWriteType) {
            this.readWriteType(readWriteType.toString());
            return this;
        }

        public final void setReadWriteType(String readWriteType) {
            this.readWriteType = readWriteType;
        }

        public final Boolean getIncludeManagementEvents() {
            return includeManagementEvents;
        }

        @Override
        public final Builder includeManagementEvents(Boolean includeManagementEvents) {
            this.includeManagementEvents = includeManagementEvents;
            return this;
        }

        public final void setIncludeManagementEvents(Boolean includeManagementEvents) {
            this.includeManagementEvents = includeManagementEvents;
        }

        public final Collection<DataResource.Builder> getDataResources() {
            return dataResources != null ? dataResources.stream().map(DataResource::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder dataResources(Collection<DataResource> dataResources) {
            this.dataResources = DataResourcesCopier.copy(dataResources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataResources(DataResource... dataResources) {
            dataResources(Arrays.asList(dataResources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dataResources(Consumer<DataResource.Builder>... dataResources) {
            dataResources(Stream.of(dataResources).map(c -> DataResource.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDataResources(Collection<DataResource.BuilderImpl> dataResources) {
            this.dataResources = DataResourcesCopier.copyFromBuilder(dataResources);
        }

        @Override
        public EventSelector build() {
            return new EventSelector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
