/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudtrail.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about an import failure.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImportFailureListItem implements SdkPojo, Serializable,
        ToCopyableBuilder<ImportFailureListItem.Builder, ImportFailureListItem> {
    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Location").getter(getter(ImportFailureListItem::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Location").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ImportFailureListItem::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> ERROR_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorType").getter(getter(ImportFailureListItem::errorType)).setter(setter(Builder::errorType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorType").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(ImportFailureListItem::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(ImportFailureListItem::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCATION_FIELD, STATUS_FIELD,
            ERROR_TYPE_FIELD, ERROR_MESSAGE_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String location;

    private final String status;

    private final String errorType;

    private final String errorMessage;

    private final Instant lastUpdatedTime;

    private ImportFailureListItem(BuilderImpl builder) {
        this.location = builder.location;
        this.status = builder.status;
        this.errorType = builder.errorType;
        this.errorMessage = builder.errorMessage;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * The location of the failure in the S3 bucket.
     * </p>
     * 
     * @return The location of the failure in the S3 bucket.
     */
    public final String location() {
        return location;
    }

    /**
     * <p>
     * The status of the import.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ImportFailureStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the import.
     * @see ImportFailureStatus
     */
    public final ImportFailureStatus status() {
        return ImportFailureStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the import.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ImportFailureStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the import.
     * @see ImportFailureStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The type of import error.
     * </p>
     * 
     * @return The type of import error.
     */
    public final String errorType() {
        return errorType;
    }

    /**
     * <p>
     * Provides the reason the import failed.
     * </p>
     * 
     * @return Provides the reason the import failed.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    /**
     * <p>
     * When the import was last updated.
     * </p>
     * 
     * @return When the import was last updated.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(errorType());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportFailureListItem)) {
            return false;
        }
        ImportFailureListItem other = (ImportFailureListItem) obj;
        return Objects.equals(location(), other.location()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(errorType(), other.errorType()) && Objects.equals(errorMessage(), other.errorMessage())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImportFailureListItem").add("Location", location()).add("Status", statusAsString())
                .add("ErrorType", errorType()).add("ErrorMessage", errorMessage()).add("LastUpdatedTime", lastUpdatedTime())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Location":
            return Optional.ofNullable(clazz.cast(location()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ErrorType":
            return Optional.ofNullable(clazz.cast(errorType()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Location", LOCATION_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("ErrorType", ERROR_TYPE_FIELD);
        map.put("ErrorMessage", ERROR_MESSAGE_FIELD);
        map.put("LastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ImportFailureListItem, T> g) {
        return obj -> g.apply((ImportFailureListItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImportFailureListItem> {
        /**
         * <p>
         * The location of the failure in the S3 bucket.
         * </p>
         * 
         * @param location
         *        The location of the failure in the S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * The status of the import.
         * </p>
         * 
         * @param status
         *        The status of the import.
         * @see ImportFailureStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImportFailureStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the import.
         * </p>
         * 
         * @param status
         *        The status of the import.
         * @see ImportFailureStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImportFailureStatus
         */
        Builder status(ImportFailureStatus status);

        /**
         * <p>
         * The type of import error.
         * </p>
         * 
         * @param errorType
         *        The type of import error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorType(String errorType);

        /**
         * <p>
         * Provides the reason the import failed.
         * </p>
         * 
         * @param errorMessage
         *        Provides the reason the import failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);

        /**
         * <p>
         * When the import was last updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        When the import was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String location;

        private String status;

        private String errorType;

        private String errorMessage;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportFailureListItem model) {
            location(model.location);
            status(model.status);
            errorType(model.errorType);
            errorMessage(model.errorMessage);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ImportFailureStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getErrorType() {
            return errorType;
        }

        public final void setErrorType(String errorType) {
            this.errorType = errorType;
        }

        @Override
        public final Builder errorType(String errorType) {
            this.errorType = errorType;
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        @Override
        public ImportFailureListItem build() {
            return new ImportFailureListItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
