/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.time.Instant;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Encapsulates the statistical data that CloudWatch computes from metric data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Datapoint implements ToCopyableBuilder<Datapoint.Builder, Datapoint> {
    private final Instant timestamp;

    private final Double sampleCount;

    private final Double average;

    private final Double sum;

    private final Double minimum;

    private final Double maximum;

    private final String unit;

    private final Map<String, Double> extendedStatistics;

    private Datapoint(BuilderImpl builder) {
        this.timestamp = builder.timestamp;
        this.sampleCount = builder.sampleCount;
        this.average = builder.average;
        this.sum = builder.sum;
        this.minimum = builder.minimum;
        this.maximum = builder.maximum;
        this.unit = builder.unit;
        this.extendedStatistics = builder.extendedStatistics;
    }

    /**
     * <p>
     * The time stamp used for the data point.
     * </p>
     * 
     * @return The time stamp used for the data point.
     */
    public Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The number of metric values that contributed to the aggregate value of this data point.
     * </p>
     * 
     * @return The number of metric values that contributed to the aggregate value of this data point.
     */
    public Double sampleCount() {
        return sampleCount;
    }

    /**
     * <p>
     * The average of the metric values that correspond to the data point.
     * </p>
     * 
     * @return The average of the metric values that correspond to the data point.
     */
    public Double average() {
        return average;
    }

    /**
     * <p>
     * The sum of the metric values for the data point.
     * </p>
     * 
     * @return The sum of the metric values for the data point.
     */
    public Double sum() {
        return sum;
    }

    /**
     * <p>
     * The minimum metric value for the data point.
     * </p>
     * 
     * @return The minimum metric value for the data point.
     */
    public Double minimum() {
        return minimum;
    }

    /**
     * <p>
     * The maximum metric value for the data point.
     * </p>
     * 
     * @return The maximum metric value for the data point.
     */
    public Double maximum() {
        return maximum;
    }

    /**
     * <p>
     * The standard unit for the data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The standard unit for the data point.
     * @see StandardUnit
     */
    public StandardUnit unit() {
        return StandardUnit.fromValue(unit);
    }

    /**
     * <p>
     * The standard unit for the data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The standard unit for the data point.
     * @see StandardUnit
     */
    public String unitAsString() {
        return unit;
    }

    /**
     * <p>
     * The percentile statistic for the data point.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The percentile statistic for the data point.
     */
    public Map<String, Double> extendedStatistics() {
        return extendedStatistics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(sampleCount());
        hashCode = 31 * hashCode + Objects.hashCode(average());
        hashCode = 31 * hashCode + Objects.hashCode(sum());
        hashCode = 31 * hashCode + Objects.hashCode(minimum());
        hashCode = 31 * hashCode + Objects.hashCode(maximum());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(extendedStatistics());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Datapoint)) {
            return false;
        }
        Datapoint other = (Datapoint) obj;
        return Objects.equals(timestamp(), other.timestamp()) && Objects.equals(sampleCount(), other.sampleCount())
                && Objects.equals(average(), other.average()) && Objects.equals(sum(), other.sum())
                && Objects.equals(minimum(), other.minimum()) && Objects.equals(maximum(), other.maximum())
                && Objects.equals(unitAsString(), other.unitAsString())
                && Objects.equals(extendedStatistics(), other.extendedStatistics());
    }

    @Override
    public String toString() {
        return ToString.builder("Datapoint").add("Timestamp", timestamp()).add("SampleCount", sampleCount())
                .add("Average", average()).add("Sum", sum()).add("Minimum", minimum()).add("Maximum", maximum())
                .add("Unit", unitAsString()).add("ExtendedStatistics", extendedStatistics()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamp":
            return Optional.of(clazz.cast(timestamp()));
        case "SampleCount":
            return Optional.of(clazz.cast(sampleCount()));
        case "Average":
            return Optional.of(clazz.cast(average()));
        case "Sum":
            return Optional.of(clazz.cast(sum()));
        case "Minimum":
            return Optional.of(clazz.cast(minimum()));
        case "Maximum":
            return Optional.of(clazz.cast(maximum()));
        case "Unit":
            return Optional.of(clazz.cast(unitAsString()));
        case "ExtendedStatistics":
            return Optional.of(clazz.cast(extendedStatistics()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, Datapoint> {
        /**
         * <p>
         * The time stamp used for the data point.
         * </p>
         * 
         * @param timestamp
         *        The time stamp used for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The number of metric values that contributed to the aggregate value of this data point.
         * </p>
         * 
         * @param sampleCount
         *        The number of metric values that contributed to the aggregate value of this data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleCount(Double sampleCount);

        /**
         * <p>
         * The average of the metric values that correspond to the data point.
         * </p>
         * 
         * @param average
         *        The average of the metric values that correspond to the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder average(Double average);

        /**
         * <p>
         * The sum of the metric values for the data point.
         * </p>
         * 
         * @param sum
         *        The sum of the metric values for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sum(Double sum);

        /**
         * <p>
         * The minimum metric value for the data point.
         * </p>
         * 
         * @param minimum
         *        The minimum metric value for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimum(Double minimum);

        /**
         * <p>
         * The maximum metric value for the data point.
         * </p>
         * 
         * @param maximum
         *        The maximum metric value for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximum(Double maximum);

        /**
         * <p>
         * The standard unit for the data point.
         * </p>
         * 
         * @param unit
         *        The standard unit for the data point.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The standard unit for the data point.
         * </p>
         * 
         * @param unit
         *        The standard unit for the data point.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(StandardUnit unit);

        /**
         * <p>
         * The percentile statistic for the data point.
         * </p>
         * 
         * @param extendedStatistics
         *        The percentile statistic for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extendedStatistics(Map<String, Double> extendedStatistics);
    }

    static final class BuilderImpl implements Builder {
        private Instant timestamp;

        private Double sampleCount;

        private Double average;

        private Double sum;

        private Double minimum;

        private Double maximum;

        private String unit;

        private Map<String, Double> extendedStatistics;

        private BuilderImpl() {
        }

        private BuilderImpl(Datapoint model) {
            timestamp(model.timestamp);
            sampleCount(model.sampleCount);
            average(model.average);
            sum(model.sum);
            minimum(model.minimum);
            maximum(model.maximum);
            unit(model.unit);
            extendedStatistics(model.extendedStatistics);
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        public final Double getSampleCount() {
            return sampleCount;
        }

        @Override
        public final Builder sampleCount(Double sampleCount) {
            this.sampleCount = sampleCount;
            return this;
        }

        public final void setSampleCount(Double sampleCount) {
            this.sampleCount = sampleCount;
        }

        public final Double getAverage() {
            return average;
        }

        @Override
        public final Builder average(Double average) {
            this.average = average;
            return this;
        }

        public final void setAverage(Double average) {
            this.average = average;
        }

        public final Double getSum() {
            return sum;
        }

        @Override
        public final Builder sum(Double sum) {
            this.sum = sum;
            return this;
        }

        public final void setSum(Double sum) {
            this.sum = sum;
        }

        public final Double getMinimum() {
            return minimum;
        }

        @Override
        public final Builder minimum(Double minimum) {
            this.minimum = minimum;
            return this;
        }

        public final void setMinimum(Double minimum) {
            this.minimum = minimum;
        }

        public final Double getMaximum() {
            return maximum;
        }

        @Override
        public final Builder maximum(Double maximum) {
            this.maximum = maximum;
            return this;
        }

        public final void setMaximum(Double maximum) {
            this.maximum = maximum;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(StandardUnit unit) {
            this.unit(unit.toString());
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        public final Map<String, Double> getExtendedStatistics() {
            return extendedStatistics;
        }

        @Override
        public final Builder extendedStatistics(Map<String, Double> extendedStatistics) {
            this.extendedStatistics = DatapointValueMapCopier.copy(extendedStatistics);
            return this;
        }

        public final void setExtendedStatistics(Map<String, Double> extendedStatistics) {
            this.extendedStatistics = DatapointValueMapCopier.copy(extendedStatistics);
        }

        @Override
        public Datapoint build() {
            return new Datapoint(this);
        }
    }
}
