/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Encapsulates the information sent to either create a metric or add new values to be aggregated into an existing
 * metric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricDatum implements SdkPojo, Serializable, ToCopyableBuilder<MetricDatum.Builder, MetricDatum> {
    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MetricDatum::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<List<Dimension>> DIMENSIONS_FIELD = SdkField
            .<List<Dimension>> builder(MarshallingType.LIST)
            .getter(getter(MetricDatum::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Dimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(Dimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(MetricDatum::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<Double> VALUE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(MetricDatum::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<StatisticSet> STATISTIC_VALUES_FIELD = SdkField
            .<StatisticSet> builder(MarshallingType.SDK_POJO).getter(getter(MetricDatum::statisticValues))
            .setter(setter(Builder::statisticValues)).constructor(StatisticSet::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatisticValues").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(MetricDatum::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final SdkField<Integer> STORAGE_RESOLUTION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(MetricDatum::storageResolution)).setter(setter(Builder::storageResolution))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageResolution").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_NAME_FIELD,
            DIMENSIONS_FIELD, TIMESTAMP_FIELD, VALUE_FIELD, STATISTIC_VALUES_FIELD, UNIT_FIELD, STORAGE_RESOLUTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metricName;

    private final List<Dimension> dimensions;

    private final Instant timestamp;

    private final Double value;

    private final StatisticSet statisticValues;

    private final String unit;

    private final Integer storageResolution;

    private MetricDatum(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.dimensions = builder.dimensions;
        this.timestamp = builder.timestamp;
        this.value = builder.value;
        this.statisticValues = builder.statisticValues;
        this.unit = builder.unit;
        this.storageResolution = builder.storageResolution;
    }

    /**
     * <p>
     * The name of the metric.
     * </p>
     * 
     * @return The name of the metric.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The dimensions associated with the metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The dimensions associated with the metric.
     */
    public List<Dimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970 00:00:00 UTC.
     * </p>
     * 
     * @return The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970 00:00:00
     *         UTC.
     */
    public Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The value for the metric.
     * </p>
     * <p>
     * Although the parameter accepts numbers of type Double, CloudWatch rejects values that are either too small or too
     * large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10) or 2e-360 to 2e360 (Base 2). In
     * addition, special values (for example, NaN, +Infinity, -Infinity) are not supported.
     * </p>
     * 
     * @return The value for the metric.</p>
     *         <p>
     *         Although the parameter accepts numbers of type Double, CloudWatch rejects values that are either too
     *         small or too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10) or 2e-360 to
     *         2e360 (Base 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are not supported.
     */
    public Double value() {
        return value;
    }

    /**
     * <p>
     * The statistical values for the metric.
     * </p>
     * 
     * @return The statistical values for the metric.
     */
    public StatisticSet statisticValues() {
        return statisticValues;
    }

    /**
     * <p>
     * The unit of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of the metric.
     * @see StandardUnit
     */
    public StandardUnit unit() {
        return StandardUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of the metric.
     * @see StandardUnit
     */
    public String unitAsString() {
        return unit;
    }

    /**
     * <p>
     * Valid values are 1 and 60. Setting this to 1 specifies this metric as a high-resolution metric, so that
     * CloudWatch stores the metric with sub-minute resolution down to one second. Setting this to 60 specifies this
     * metric as a regular-resolution metric, which CloudWatch stores at 1-minute resolution. Currently, high resolution
     * is available only for custom metrics. For more information about high-resolution metrics, see <a href=
     * "http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html#high-resolution-metrics"
     * >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * This field is optional, if you do not specify it the default of 60 is used.
     * </p>
     * 
     * @return Valid values are 1 and 60. Setting this to 1 specifies this metric as a high-resolution metric, so that
     *         CloudWatch stores the metric with sub-minute resolution down to one second. Setting this to 60 specifies
     *         this metric as a regular-resolution metric, which CloudWatch stores at 1-minute resolution. Currently,
     *         high resolution is available only for custom metrics. For more information about high-resolution metrics,
     *         see <a href=
     *         "http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html#high-resolution-metrics"
     *         >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>. </p>
     *         <p>
     *         This field is optional, if you do not specify it the default of 60 is used.
     */
    public Integer storageResolution() {
        return storageResolution;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(dimensions());
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(statisticValues());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(storageResolution());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricDatum)) {
            return false;
        }
        MetricDatum other = (MetricDatum) obj;
        return Objects.equals(metricName(), other.metricName()) && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(timestamp(), other.timestamp()) && Objects.equals(value(), other.value())
                && Objects.equals(statisticValues(), other.statisticValues())
                && Objects.equals(unitAsString(), other.unitAsString())
                && Objects.equals(storageResolution(), other.storageResolution());
    }

    @Override
    public String toString() {
        return ToString.builder("MetricDatum").add("MetricName", metricName()).add("Dimensions", dimensions())
                .add("Timestamp", timestamp()).add("Value", value()).add("StatisticValues", statisticValues())
                .add("Unit", unitAsString()).add("StorageResolution", storageResolution()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "StatisticValues":
            return Optional.ofNullable(clazz.cast(statisticValues()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        case "StorageResolution":
            return Optional.ofNullable(clazz.cast(storageResolution()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricDatum, T> g) {
        return obj -> g.apply((MetricDatum) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricDatum> {
        /**
         * <p>
         * The name of the metric.
         * </p>
         * 
         * @param metricName
         *        The name of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The dimensions associated with the metric.
         * </p>
         * 
         * @param dimensions
         *        The dimensions associated with the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * The dimensions associated with the metric.
         * </p>
         * 
         * @param dimensions
         *        The dimensions associated with the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * The dimensions associated with the metric.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Dimension>.Builder} avoiding the need to
         * create one manually via {@link List<Dimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Dimension>.Builder#build()} is called immediately and its
         * result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<Dimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);

        /**
         * <p>
         * The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970 00:00:00
         * UTC.
         * </p>
         * 
         * @param timestamp
         *        The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970
         *        00:00:00 UTC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The value for the metric.
         * </p>
         * <p>
         * Although the parameter accepts numbers of type Double, CloudWatch rejects values that are either too small or
         * too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10) or 2e-360 to 2e360 (Base
         * 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are not supported.
         * </p>
         * 
         * @param value
         *        The value for the metric.</p>
         *        <p>
         *        Although the parameter accepts numbers of type Double, CloudWatch rejects values that are either too
         *        small or too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10) or 2e-360
         *        to 2e360 (Base 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are not
         *        supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(Double value);

        /**
         * <p>
         * The statistical values for the metric.
         * </p>
         * 
         * @param statisticValues
         *        The statistical values for the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticValues(StatisticSet statisticValues);

        /**
         * <p>
         * The statistical values for the metric.
         * </p>
         * This is a convenience that creates an instance of the {@link StatisticSet.Builder} avoiding the need to
         * create one manually via {@link StatisticSet#builder()}.
         *
         * When the {@link Consumer} completes, {@link StatisticSet.Builder#build()} is called immediately and its
         * result is passed to {@link #statisticValues(StatisticSet)}.
         * 
         * @param statisticValues
         *        a consumer that will call methods on {@link StatisticSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statisticValues(StatisticSet)
         */
        default Builder statisticValues(Consumer<StatisticSet.Builder> statisticValues) {
            return statisticValues(StatisticSet.builder().applyMutation(statisticValues).build());
        }

        /**
         * <p>
         * The unit of the metric.
         * </p>
         * 
         * @param unit
         *        The unit of the metric.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit of the metric.
         * </p>
         * 
         * @param unit
         *        The unit of the metric.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(StandardUnit unit);

        /**
         * <p>
         * Valid values are 1 and 60. Setting this to 1 specifies this metric as a high-resolution metric, so that
         * CloudWatch stores the metric with sub-minute resolution down to one second. Setting this to 60 specifies this
         * metric as a regular-resolution metric, which CloudWatch stores at 1-minute resolution. Currently, high
         * resolution is available only for custom metrics. For more information about high-resolution metrics, see <a
         * href=
         * "http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html#high-resolution-metrics"
         * >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * <p>
         * This field is optional, if you do not specify it the default of 60 is used.
         * </p>
         * 
         * @param storageResolution
         *        Valid values are 1 and 60. Setting this to 1 specifies this metric as a high-resolution metric, so
         *        that CloudWatch stores the metric with sub-minute resolution down to one second. Setting this to 60
         *        specifies this metric as a regular-resolution metric, which CloudWatch stores at 1-minute resolution.
         *        Currently, high resolution is available only for custom metrics. For more information about
         *        high-resolution metrics, see <a href=
         *        "http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html#high-resolution-metrics"
         *        >High-Resolution Metrics</a> in the <i>Amazon CloudWatch User Guide</i>. </p>
         *        <p>
         *        This field is optional, if you do not specify it the default of 60 is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageResolution(Integer storageResolution);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private List<Dimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private Instant timestamp;

        private Double value;

        private StatisticSet statisticValues;

        private String unit;

        private Integer storageResolution;

        private BuilderImpl() {
        }

        private BuilderImpl(MetricDatum model) {
            metricName(model.metricName);
            dimensions(model.dimensions);
            timestamp(model.timestamp);
            value(model.value);
            statisticValues(model.statisticValues);
            unit(model.unit);
            storageResolution(model.storageResolution);
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final Collection<Dimension.Builder> getDimensions() {
            return dimensions != null ? dimensions.stream().map(Dimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        public final Double getValue() {
            return value;
        }

        @Override
        public final Builder value(Double value) {
            this.value = value;
            return this;
        }

        public final void setValue(Double value) {
            this.value = value;
        }

        public final StatisticSet.Builder getStatisticValues() {
            return statisticValues != null ? statisticValues.toBuilder() : null;
        }

        @Override
        public final Builder statisticValues(StatisticSet statisticValues) {
            this.statisticValues = statisticValues;
            return this;
        }

        public final void setStatisticValues(StatisticSet.BuilderImpl statisticValues) {
            this.statisticValues = statisticValues != null ? statisticValues.build() : null;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(StandardUnit unit) {
            this.unit(unit.toString());
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        public final Integer getStorageResolution() {
            return storageResolution;
        }

        @Override
        public final Builder storageResolution(Integer storageResolution) {
            this.storageResolution = storageResolution;
            return this;
        }

        public final void setStorageResolution(Integer storageResolution) {
            this.storageResolution = storageResolution;
        }

        @Override
        public MetricDatum build() {
            return new MetricDatum(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
