/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Encapsulates the information sent to either create a metric or add new values to be aggregated into an existing
 * metric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class MetricDatum implements ToCopyableBuilder<MetricDatum.Builder, MetricDatum> {
    private final String metricName;

    private final List<Dimension> dimensions;

    private final Instant timestamp;

    private final Double value;

    private final StatisticSet statisticValues;

    private final String unit;

    private MetricDatum(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.dimensions = builder.dimensions;
        this.timestamp = builder.timestamp;
        this.value = builder.value;
        this.statisticValues = builder.statisticValues;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The name of the metric.
     * </p>
     * 
     * @return The name of the metric.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The dimensions associated with the metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The dimensions associated with the metric.
     */
    public List<Dimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970 00:00:00 UTC.
     * </p>
     * 
     * @return The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970 00:00:00
     *         UTC.
     */
    public Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The value for the metric.
     * </p>
     * <p>
     * Although the parameter accepts numbers of type Double, Amazon CloudWatch rejects values that are either too small
     * or too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10) or 2e-360 to 2e360 (Base
     * 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are not supported.
     * </p>
     * 
     * @return The value for the metric.</p>
     *         <p>
     *         Although the parameter accepts numbers of type Double, Amazon CloudWatch rejects values that are either
     *         too small or too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10) or 2e-360
     *         to 2e360 (Base 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are not
     *         supported.
     */
    public Double value() {
        return value;
    }

    /**
     * <p>
     * The statistical values for the metric.
     * </p>
     * 
     * @return The statistical values for the metric.
     */
    public StatisticSet statisticValues() {
        return statisticValues;
    }

    /**
     * <p>
     * The unit of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitString}.
     * </p>
     * 
     * @return The unit of the metric.
     * @see StandardUnit
     */
    public StandardUnit unit() {
        return StandardUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit of the metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitString}.
     * </p>
     * 
     * @return The unit of the metric.
     * @see StandardUnit
     */
    public String unitString() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((metricName() == null) ? 0 : metricName().hashCode());
        hashCode = 31 * hashCode + ((dimensions() == null) ? 0 : dimensions().hashCode());
        hashCode = 31 * hashCode + ((timestamp() == null) ? 0 : timestamp().hashCode());
        hashCode = 31 * hashCode + ((value() == null) ? 0 : value().hashCode());
        hashCode = 31 * hashCode + ((statisticValues() == null) ? 0 : statisticValues().hashCode());
        hashCode = 31 * hashCode + ((unitString() == null) ? 0 : unitString().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricDatum)) {
            return false;
        }
        MetricDatum other = (MetricDatum) obj;
        if (other.metricName() == null ^ this.metricName() == null) {
            return false;
        }
        if (other.metricName() != null && !other.metricName().equals(this.metricName())) {
            return false;
        }
        if (other.dimensions() == null ^ this.dimensions() == null) {
            return false;
        }
        if (other.dimensions() != null && !other.dimensions().equals(this.dimensions())) {
            return false;
        }
        if (other.timestamp() == null ^ this.timestamp() == null) {
            return false;
        }
        if (other.timestamp() != null && !other.timestamp().equals(this.timestamp())) {
            return false;
        }
        if (other.value() == null ^ this.value() == null) {
            return false;
        }
        if (other.value() != null && !other.value().equals(this.value())) {
            return false;
        }
        if (other.statisticValues() == null ^ this.statisticValues() == null) {
            return false;
        }
        if (other.statisticValues() != null && !other.statisticValues().equals(this.statisticValues())) {
            return false;
        }
        if (other.unitString() == null ^ this.unitString() == null) {
            return false;
        }
        if (other.unitString() != null && !other.unitString().equals(this.unitString())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (metricName() != null) {
            sb.append("MetricName: ").append(metricName()).append(",");
        }
        if (dimensions() != null) {
            sb.append("Dimensions: ").append(dimensions()).append(",");
        }
        if (timestamp() != null) {
            sb.append("Timestamp: ").append(timestamp()).append(",");
        }
        if (value() != null) {
            sb.append("Value: ").append(value()).append(",");
        }
        if (statisticValues() != null) {
            sb.append("StatisticValues: ").append(statisticValues()).append(",");
        }
        if (unitString() != null) {
            sb.append("Unit: ").append(unitString()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.of(clazz.cast(metricName()));
        case "Dimensions":
            return Optional.of(clazz.cast(dimensions()));
        case "Timestamp":
            return Optional.of(clazz.cast(timestamp()));
        case "Value":
            return Optional.of(clazz.cast(value()));
        case "StatisticValues":
            return Optional.of(clazz.cast(statisticValues()));
        case "Unit":
            return Optional.of(clazz.cast(unitString()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, MetricDatum> {
        /**
         * <p>
         * The name of the metric.
         * </p>
         * 
         * @param metricName
         *        The name of the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The dimensions associated with the metric.
         * </p>
         * 
         * @param dimensions
         *        The dimensions associated with the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * The dimensions associated with the metric.
         * </p>
         * 
         * @param dimensions
         *        The dimensions associated with the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970 00:00:00
         * UTC.
         * </p>
         * 
         * @param timestamp
         *        The time the metric data was received, expressed as the number of milliseconds since Jan 1, 1970
         *        00:00:00 UTC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The value for the metric.
         * </p>
         * <p>
         * Although the parameter accepts numbers of type Double, Amazon CloudWatch rejects values that are either too
         * small or too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10) or 2e-360 to
         * 2e360 (Base 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are not supported.
         * </p>
         * 
         * @param value
         *        The value for the metric.</p>
         *        <p>
         *        Although the parameter accepts numbers of type Double, Amazon CloudWatch rejects values that are
         *        either too small or too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10)
         *        or 2e-360 to 2e360 (Base 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are
         *        not supported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(Double value);

        /**
         * <p>
         * The statistical values for the metric.
         * </p>
         * 
         * @param statisticValues
         *        The statistical values for the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticValues(StatisticSet statisticValues);

        /**
         * <p>
         * The unit of the metric.
         * </p>
         * 
         * @param unit
         *        The unit of the metric.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit of the metric.
         * </p>
         * 
         * @param unit
         *        The unit of the metric.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(StandardUnit unit);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private List<Dimension> dimensions;

        private Instant timestamp;

        private Double value;

        private StatisticSet statisticValues;

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(MetricDatum model) {
            metricName(model.metricName);
            dimensions(model.dimensions);
            timestamp(model.timestamp);
            value(model.value);
            statisticValues(model.statisticValues);
            unit(model.unit);
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final Collection<Dimension.Builder> getDimensions() {
            return dimensions != null ? dimensions.stream().map(Dimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        @Override
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        public final Double getValue() {
            return value;
        }

        @Override
        public final Builder value(Double value) {
            this.value = value;
            return this;
        }

        public final void setValue(Double value) {
            this.value = value;
        }

        public final StatisticSet.Builder getStatisticValues() {
            return statisticValues != null ? statisticValues.toBuilder() : null;
        }

        @Override
        public final Builder statisticValues(StatisticSet statisticValues) {
            this.statisticValues = statisticValues;
            return this;
        }

        public final void setStatisticValues(StatisticSet.BuilderImpl statisticValues) {
            this.statisticValues = statisticValues != null ? statisticValues.build() : null;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(StandardUnit unit) {
            this.unit(unit.toString());
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public MetricDatum build() {
            return new MetricDatum(this);
        }
    }
}
