/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An anomaly detection model associated with a particular CloudWatch metric and statistic. You can use the model to
 * display a band of expected normal values when the metric is graphed.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AnomalyDetector implements SdkPojo, Serializable, ToCopyableBuilder<AnomalyDetector.Builder, AnomalyDetector> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(AnomalyDetector::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(AnomalyDetector::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<List<Dimension>> DIMENSIONS_FIELD = SdkField
            .<List<Dimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(AnomalyDetector::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Dimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(Dimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Stat")
            .getter(getter(AnomalyDetector::stat)).setter(setter(Builder::stat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Stat").build()).build();

    private static final SdkField<AnomalyDetectorConfiguration> CONFIGURATION_FIELD = SdkField
            .<AnomalyDetectorConfiguration> builder(MarshallingType.SDK_POJO).memberName("Configuration")
            .getter(getter(AnomalyDetector::configuration)).setter(setter(Builder::configuration))
            .constructor(AnomalyDetectorConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build()).build();

    private static final SdkField<String> STATE_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateValue").getter(getter(AnomalyDetector::stateValueAsString)).setter(setter(Builder::stateValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD,
            METRIC_NAME_FIELD, DIMENSIONS_FIELD, STAT_FIELD, CONFIGURATION_FIELD, STATE_VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final String metricName;

    private final List<Dimension> dimensions;

    private final String stat;

    private final AnomalyDetectorConfiguration configuration;

    private final String stateValue;

    private AnomalyDetector(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.metricName = builder.metricName;
        this.dimensions = builder.dimensions;
        this.stat = builder.stat;
        this.configuration = builder.configuration;
        this.stateValue = builder.stateValue;
    }

    /**
     * <p>
     * The namespace of the metric associated with the anomaly detection model.
     * </p>
     * 
     * @return The namespace of the metric associated with the anomaly detection model.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name of the metric associated with the anomaly detection model.
     * </p>
     * 
     * @return The name of the metric associated with the anomaly detection model.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * Returns true if the Dimensions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metric dimensions associated with the anomaly detection model.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDimensions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The metric dimensions associated with the anomaly detection model.
     */
    public final List<Dimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The statistic associated with the anomaly detection model.
     * </p>
     * 
     * @return The statistic associated with the anomaly detection model.
     */
    public final String stat() {
        return stat;
    }

    /**
     * <p>
     * The configuration specifies details about how the anomaly detection model is to be trained, including time ranges
     * to exclude from use for training the model, and the time zone to use for the metric.
     * </p>
     * 
     * @return The configuration specifies details about how the anomaly detection model is to be trained, including
     *         time ranges to exclude from use for training the model, and the time zone to use for the metric.
     */
    public final AnomalyDetectorConfiguration configuration() {
        return configuration;
    }

    /**
     * <p>
     * The current status of the anomaly detector's training. The possible values are
     * <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateValue} will
     * return {@link AnomalyDetectorStateValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateValueAsString}.
     * </p>
     * 
     * @return The current status of the anomaly detector's training. The possible values are
     *         <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
     * @see AnomalyDetectorStateValue
     */
    public final AnomalyDetectorStateValue stateValue() {
        return AnomalyDetectorStateValue.fromValue(stateValue);
    }

    /**
     * <p>
     * The current status of the anomaly detector's training. The possible values are
     * <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #stateValue} will
     * return {@link AnomalyDetectorStateValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateValueAsString}.
     * </p>
     * 
     * @return The current status of the anomaly detector's training. The possible values are
     *         <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
     * @see AnomalyDetectorStateValue
     */
    public final String stateValueAsString() {
        return stateValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stat());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(stateValueAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AnomalyDetector)) {
            return false;
        }
        AnomalyDetector other = (AnomalyDetector) obj;
        return Objects.equals(namespace(), other.namespace()) && Objects.equals(metricName(), other.metricName())
                && hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(stat(), other.stat()) && Objects.equals(configuration(), other.configuration())
                && Objects.equals(stateValueAsString(), other.stateValueAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AnomalyDetector").add("Namespace", namespace()).add("MetricName", metricName())
                .add("Dimensions", hasDimensions() ? dimensions() : null).add("Stat", stat())
                .add("Configuration", configuration()).add("StateValue", stateValueAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "Stat":
            return Optional.ofNullable(clazz.cast(stat()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "StateValue":
            return Optional.ofNullable(clazz.cast(stateValueAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AnomalyDetector, T> g) {
        return obj -> g.apply((AnomalyDetector) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AnomalyDetector> {
        /**
         * <p>
         * The namespace of the metric associated with the anomaly detection model.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric associated with the anomaly detection model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The name of the metric associated with the anomaly detection model.
         * </p>
         * 
         * @param metricName
         *        The name of the metric associated with the anomaly detection model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The metric dimensions associated with the anomaly detection model.
         * </p>
         * 
         * @param dimensions
         *        The metric dimensions associated with the anomaly detection model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * The metric dimensions associated with the anomaly detection model.
         * </p>
         * 
         * @param dimensions
         *        The metric dimensions associated with the anomaly detection model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * The metric dimensions associated with the anomaly detection model.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Dimension>.Builder} avoiding the need to
         * create one manually via {@link List<Dimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Dimension>.Builder#build()} is called immediately and its
         * result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<Dimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);

        /**
         * <p>
         * The statistic associated with the anomaly detection model.
         * </p>
         * 
         * @param stat
         *        The statistic associated with the anomaly detection model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stat(String stat);

        /**
         * <p>
         * The configuration specifies details about how the anomaly detection model is to be trained, including time
         * ranges to exclude from use for training the model, and the time zone to use for the metric.
         * </p>
         * 
         * @param configuration
         *        The configuration specifies details about how the anomaly detection model is to be trained, including
         *        time ranges to exclude from use for training the model, and the time zone to use for the metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(AnomalyDetectorConfiguration configuration);

        /**
         * <p>
         * The configuration specifies details about how the anomaly detection model is to be trained, including time
         * ranges to exclude from use for training the model, and the time zone to use for the metric.
         * </p>
         * This is a convenience that creates an instance of the {@link AnomalyDetectorConfiguration.Builder} avoiding
         * the need to create one manually via {@link AnomalyDetectorConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link AnomalyDetectorConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #configuration(AnomalyDetectorConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link AnomalyDetectorConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(AnomalyDetectorConfiguration)
         */
        default Builder configuration(Consumer<AnomalyDetectorConfiguration.Builder> configuration) {
            return configuration(AnomalyDetectorConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * The current status of the anomaly detector's training. The possible values are
         * <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
         * </p>
         * 
         * @param stateValue
         *        The current status of the anomaly detector's training. The possible values are
         *        <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
         * @see AnomalyDetectorStateValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyDetectorStateValue
         */
        Builder stateValue(String stateValue);

        /**
         * <p>
         * The current status of the anomaly detector's training. The possible values are
         * <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
         * </p>
         * 
         * @param stateValue
         *        The current status of the anomaly detector's training. The possible values are
         *        <code>TRAINED | PENDING_TRAINING | TRAINED_INSUFFICIENT_DATA</code>
         * @see AnomalyDetectorStateValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnomalyDetectorStateValue
         */
        Builder stateValue(AnomalyDetectorStateValue stateValue);
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private String metricName;

        private List<Dimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private String stat;

        private AnomalyDetectorConfiguration configuration;

        private String stateValue;

        private BuilderImpl() {
        }

        private BuilderImpl(AnomalyDetector model) {
            namespace(model.namespace);
            metricName(model.metricName);
            dimensions(model.dimensions);
            stat(model.stat);
            configuration(model.configuration);
            stateValue(model.stateValue);
        }

        public final String getNamespace() {
            return namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final Collection<Dimension.Builder> getDimensions() {
            if (dimensions instanceof SdkAutoConstructList) {
                return null;
            }
            return dimensions != null ? dimensions.stream().map(Dimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        public final String getStat() {
            return stat;
        }

        @Override
        public final Builder stat(String stat) {
            this.stat = stat;
            return this;
        }

        public final void setStat(String stat) {
            this.stat = stat;
        }

        public final AnomalyDetectorConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        @Override
        public final Builder configuration(AnomalyDetectorConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(AnomalyDetectorConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        public final String getStateValue() {
            return stateValue;
        }

        @Override
        public final Builder stateValue(String stateValue) {
            this.stateValue = stateValue;
            return this;
        }

        @Override
        public final Builder stateValue(AnomalyDetectorStateValue stateValue) {
            this.stateValue(stateValue == null ? null : stateValue.toString());
            return this;
        }

        public final void setStateValue(String stateValue) {
            this.stateValue = stateValue;
        }

        @Override
        public AnomalyDetector build() {
            return new AnomalyDetector(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
