/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the information that's required to enable a managed Contributor Insights rule for an Amazon Web Services
 * resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedRule implements SdkPojo, Serializable, ToCopyableBuilder<ManagedRule.Builder, ManagedRule> {
    private static final SdkField<String> TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemplateName").getter(getter(ManagedRule::templateName)).setter(setter(Builder::templateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemplateName").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(ManagedRule::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(ManagedRule::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TEMPLATE_NAME_FIELD,
            RESOURCE_ARN_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String templateName;

    private final String resourceARN;

    private final List<Tag> tags;

    private ManagedRule(BuilderImpl builder) {
        this.templateName = builder.templateName;
        this.resourceARN = builder.resourceARN;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The template name for the managed Contributor Insights rule, as returned by <code>ListManagedInsightRules</code>.
     * </p>
     * 
     * @return The template name for the managed Contributor Insights rule, as returned by
     *         <code>ListManagedInsightRules</code>.
     */
    public final String templateName() {
        return templateName;
    }

    /**
     * <p>
     * The ARN of an Amazon Web Services resource that has managed Contributor Insights rules.
     * </p>
     * 
     * @return The ARN of an Amazon Web Services resource that has managed Contributor Insights rules.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of key-value pairs that you can associate with a managed Contributor Insights rule. You can associate as
     * many as 50 tags with a rule. Tags can help you organize and categorize your resources. You also can use them to
     * scope user permissions by granting a user permission to access or change only the resources that have certain tag
     * values. To associate tags with a rule, you must have the <code>cloudwatch:TagResource</code> permission in
     * addition to the <code>cloudwatch:PutInsightRule</code> permission. If you are using this operation to update an
     * existing Contributor Insights rule, any tags that you specify in this parameter are ignored. To change the tags
     * of an existing rule, use <code>TagResource</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A list of key-value pairs that you can associate with a managed Contributor Insights rule. You can
     *         associate as many as 50 tags with a rule. Tags can help you organize and categorize your resources. You
     *         also can use them to scope user permissions by granting a user permission to access or change only the
     *         resources that have certain tag values. To associate tags with a rule, you must have the
     *         <code>cloudwatch:TagResource</code> permission in addition to the <code>cloudwatch:PutInsightRule</code>
     *         permission. If you are using this operation to update an existing Contributor Insights rule, any tags
     *         that you specify in this parameter are ignored. To change the tags of an existing rule, use
     *         <code>TagResource</code>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(templateName());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedRule)) {
            return false;
        }
        ManagedRule other = (ManagedRule) obj;
        return Objects.equals(templateName(), other.templateName()) && Objects.equals(resourceARN(), other.resourceARN())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ManagedRule").add("TemplateName", templateName()).add("ResourceARN", resourceARN())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TemplateName":
            return Optional.ofNullable(clazz.cast(templateName()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ManagedRule, T> g) {
        return obj -> g.apply((ManagedRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedRule> {
        /**
         * <p>
         * The template name for the managed Contributor Insights rule, as returned by
         * <code>ListManagedInsightRules</code>.
         * </p>
         * 
         * @param templateName
         *        The template name for the managed Contributor Insights rule, as returned by
         *        <code>ListManagedInsightRules</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateName(String templateName);

        /**
         * <p>
         * The ARN of an Amazon Web Services resource that has managed Contributor Insights rules.
         * </p>
         * 
         * @param resourceARN
         *        The ARN of an Amazon Web Services resource that has managed Contributor Insights rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * <p>
         * A list of key-value pairs that you can associate with a managed Contributor Insights rule. You can associate
         * as many as 50 tags with a rule. Tags can help you organize and categorize your resources. You also can use
         * them to scope user permissions by granting a user permission to access or change only the resources that have
         * certain tag values. To associate tags with a rule, you must have the <code>cloudwatch:TagResource</code>
         * permission in addition to the <code>cloudwatch:PutInsightRule</code> permission. If you are using this
         * operation to update an existing Contributor Insights rule, any tags that you specify in this parameter are
         * ignored. To change the tags of an existing rule, use <code>TagResource</code>.
         * </p>
         * 
         * @param tags
         *        A list of key-value pairs that you can associate with a managed Contributor Insights rule. You can
         *        associate as many as 50 tags with a rule. Tags can help you organize and categorize your resources.
         *        You also can use them to scope user permissions by granting a user permission to access or change only
         *        the resources that have certain tag values. To associate tags with a rule, you must have the
         *        <code>cloudwatch:TagResource</code> permission in addition to the
         *        <code>cloudwatch:PutInsightRule</code> permission. If you are using this operation to update an
         *        existing Contributor Insights rule, any tags that you specify in this parameter are ignored. To change
         *        the tags of an existing rule, use <code>TagResource</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A list of key-value pairs that you can associate with a managed Contributor Insights rule. You can associate
         * as many as 50 tags with a rule. Tags can help you organize and categorize your resources. You also can use
         * them to scope user permissions by granting a user permission to access or change only the resources that have
         * certain tag values. To associate tags with a rule, you must have the <code>cloudwatch:TagResource</code>
         * permission in addition to the <code>cloudwatch:PutInsightRule</code> permission. If you are using this
         * operation to update an existing Contributor Insights rule, any tags that you specify in this parameter are
         * ignored. To change the tags of an existing rule, use <code>TagResource</code>.
         * </p>
         * 
         * @param tags
         *        A list of key-value pairs that you can associate with a managed Contributor Insights rule. You can
         *        associate as many as 50 tags with a rule. Tags can help you organize and categorize your resources.
         *        You also can use them to scope user permissions by granting a user permission to access or change only
         *        the resources that have certain tag values. To associate tags with a rule, you must have the
         *        <code>cloudwatch:TagResource</code> permission in addition to the
         *        <code>cloudwatch:PutInsightRule</code> permission. If you are using this operation to update an
         *        existing Contributor Insights rule, any tags that you specify in this parameter are ignored. To change
         *        the tags of an existing rule, use <code>TagResource</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A list of key-value pairs that you can associate with a managed Contributor Insights rule. You can associate
         * as many as 50 tags with a rule. Tags can help you organize and categorize your resources. You also can use
         * them to scope user permissions by granting a user permission to access or change only the resources that have
         * certain tag values. To associate tags with a rule, you must have the <code>cloudwatch:TagResource</code>
         * permission in addition to the <code>cloudwatch:PutInsightRule</code> permission. If you are using this
         * operation to update an existing Contributor Insights rule, any tags that you specify in this parameter are
         * ignored. To change the tags of an existing rule, use <code>TagResource</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Tag>.Builder} avoiding the need to
         * create one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String templateName;

        private String resourceARN;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedRule model) {
            templateName(model.templateName);
            resourceARN(model.resourceARN);
            tags(model.tags);
        }

        public final String getTemplateName() {
            return templateName;
        }

        public final void setTemplateName(String templateName) {
            this.templateName = templateName;
        }

        @Override
        public final Builder templateName(String templateName) {
            this.templateName = templateName;
            return this;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ManagedRule build() {
            return new ManagedRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
