/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Encapsulates the statistical data that CloudWatch computes from metric data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Datapoint implements SdkPojo, Serializable, ToCopyableBuilder<Datapoint.Builder, Datapoint> {
    private static final SdkField<Instant> TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("Timestamp").getter(getter(Datapoint::timestamp)).setter(setter(Builder::timestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamp").build()).build();

    private static final SdkField<Double> SAMPLE_COUNT_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("SampleCount").getter(getter(Datapoint::sampleCount)).setter(setter(Builder::sampleCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleCount").build()).build();

    private static final SdkField<Double> AVERAGE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Average")
            .getter(getter(Datapoint::average)).setter(setter(Builder::average))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Average").build()).build();

    private static final SdkField<Double> SUM_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Sum")
            .getter(getter(Datapoint::sum)).setter(setter(Builder::sum))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sum").build()).build();

    private static final SdkField<Double> MINIMUM_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Minimum")
            .getter(getter(Datapoint::minimum)).setter(setter(Builder::minimum))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Minimum").build()).build();

    private static final SdkField<Double> MAXIMUM_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("Maximum")
            .getter(getter(Datapoint::maximum)).setter(setter(Builder::maximum))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Maximum").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Unit")
            .getter(getter(Datapoint::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final SdkField<Map<String, Double>> EXTENDED_STATISTICS_FIELD = SdkField
            .<Map<String, Double>> builder(MarshallingType.MAP)
            .memberName("ExtendedStatistics")
            .getter(getter(Datapoint::extendedStatistics))
            .setter(setter(Builder::extendedStatistics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExtendedStatistics").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIMESTAMP_FIELD,
            SAMPLE_COUNT_FIELD, AVERAGE_FIELD, SUM_FIELD, MINIMUM_FIELD, MAXIMUM_FIELD, UNIT_FIELD, EXTENDED_STATISTICS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant timestamp;

    private final Double sampleCount;

    private final Double average;

    private final Double sum;

    private final Double minimum;

    private final Double maximum;

    private final String unit;

    private final Map<String, Double> extendedStatistics;

    private Datapoint(BuilderImpl builder) {
        this.timestamp = builder.timestamp;
        this.sampleCount = builder.sampleCount;
        this.average = builder.average;
        this.sum = builder.sum;
        this.minimum = builder.minimum;
        this.maximum = builder.maximum;
        this.unit = builder.unit;
        this.extendedStatistics = builder.extendedStatistics;
    }

    /**
     * <p>
     * The time stamp used for the data point.
     * </p>
     * 
     * @return The time stamp used for the data point.
     */
    public final Instant timestamp() {
        return timestamp;
    }

    /**
     * <p>
     * The number of metric values that contributed to the aggregate value of this data point.
     * </p>
     * 
     * @return The number of metric values that contributed to the aggregate value of this data point.
     */
    public final Double sampleCount() {
        return sampleCount;
    }

    /**
     * <p>
     * The average of the metric values that correspond to the data point.
     * </p>
     * 
     * @return The average of the metric values that correspond to the data point.
     */
    public final Double average() {
        return average;
    }

    /**
     * <p>
     * The sum of the metric values for the data point.
     * </p>
     * 
     * @return The sum of the metric values for the data point.
     */
    public final Double sum() {
        return sum;
    }

    /**
     * <p>
     * The minimum metric value for the data point.
     * </p>
     * 
     * @return The minimum metric value for the data point.
     */
    public final Double minimum() {
        return minimum;
    }

    /**
     * <p>
     * The maximum metric value for the data point.
     * </p>
     * 
     * @return The maximum metric value for the data point.
     */
    public final Double maximum() {
        return maximum;
    }

    /**
     * <p>
     * The standard unit for the data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The standard unit for the data point.
     * @see StandardUnit
     */
    public final StandardUnit unit() {
        return StandardUnit.fromValue(unit);
    }

    /**
     * <p>
     * The standard unit for the data point.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The standard unit for the data point.
     * @see StandardUnit
     */
    public final String unitAsString() {
        return unit;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExtendedStatistics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExtendedStatistics() {
        return extendedStatistics != null && !(extendedStatistics instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The percentile statistic for the data point.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExtendedStatistics} method.
     * </p>
     * 
     * @return The percentile statistic for the data point.
     */
    public final Map<String, Double> extendedStatistics() {
        return extendedStatistics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timestamp());
        hashCode = 31 * hashCode + Objects.hashCode(sampleCount());
        hashCode = 31 * hashCode + Objects.hashCode(average());
        hashCode = 31 * hashCode + Objects.hashCode(sum());
        hashCode = 31 * hashCode + Objects.hashCode(minimum());
        hashCode = 31 * hashCode + Objects.hashCode(maximum());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasExtendedStatistics() ? extendedStatistics() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Datapoint)) {
            return false;
        }
        Datapoint other = (Datapoint) obj;
        return Objects.equals(timestamp(), other.timestamp()) && Objects.equals(sampleCount(), other.sampleCount())
                && Objects.equals(average(), other.average()) && Objects.equals(sum(), other.sum())
                && Objects.equals(minimum(), other.minimum()) && Objects.equals(maximum(), other.maximum())
                && Objects.equals(unitAsString(), other.unitAsString())
                && hasExtendedStatistics() == other.hasExtendedStatistics()
                && Objects.equals(extendedStatistics(), other.extendedStatistics());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Datapoint").add("Timestamp", timestamp()).add("SampleCount", sampleCount())
                .add("Average", average()).add("Sum", sum()).add("Minimum", minimum()).add("Maximum", maximum())
                .add("Unit", unitAsString()).add("ExtendedStatistics", hasExtendedStatistics() ? extendedStatistics() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamp":
            return Optional.ofNullable(clazz.cast(timestamp()));
        case "SampleCount":
            return Optional.ofNullable(clazz.cast(sampleCount()));
        case "Average":
            return Optional.ofNullable(clazz.cast(average()));
        case "Sum":
            return Optional.ofNullable(clazz.cast(sum()));
        case "Minimum":
            return Optional.ofNullable(clazz.cast(minimum()));
        case "Maximum":
            return Optional.ofNullable(clazz.cast(maximum()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        case "ExtendedStatistics":
            return Optional.ofNullable(clazz.cast(extendedStatistics()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Datapoint, T> g) {
        return obj -> g.apply((Datapoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Datapoint> {
        /**
         * <p>
         * The time stamp used for the data point.
         * </p>
         * 
         * @param timestamp
         *        The time stamp used for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamp(Instant timestamp);

        /**
         * <p>
         * The number of metric values that contributed to the aggregate value of this data point.
         * </p>
         * 
         * @param sampleCount
         *        The number of metric values that contributed to the aggregate value of this data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleCount(Double sampleCount);

        /**
         * <p>
         * The average of the metric values that correspond to the data point.
         * </p>
         * 
         * @param average
         *        The average of the metric values that correspond to the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder average(Double average);

        /**
         * <p>
         * The sum of the metric values for the data point.
         * </p>
         * 
         * @param sum
         *        The sum of the metric values for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sum(Double sum);

        /**
         * <p>
         * The minimum metric value for the data point.
         * </p>
         * 
         * @param minimum
         *        The minimum metric value for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimum(Double minimum);

        /**
         * <p>
         * The maximum metric value for the data point.
         * </p>
         * 
         * @param maximum
         *        The maximum metric value for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximum(Double maximum);

        /**
         * <p>
         * The standard unit for the data point.
         * </p>
         * 
         * @param unit
         *        The standard unit for the data point.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The standard unit for the data point.
         * </p>
         * 
         * @param unit
         *        The standard unit for the data point.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(StandardUnit unit);

        /**
         * <p>
         * The percentile statistic for the data point.
         * </p>
         * 
         * @param extendedStatistics
         *        The percentile statistic for the data point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extendedStatistics(Map<String, Double> extendedStatistics);
    }

    static final class BuilderImpl implements Builder {
        private Instant timestamp;

        private Double sampleCount;

        private Double average;

        private Double sum;

        private Double minimum;

        private Double maximum;

        private String unit;

        private Map<String, Double> extendedStatistics = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Datapoint model) {
            timestamp(model.timestamp);
            sampleCount(model.sampleCount);
            average(model.average);
            sum(model.sum);
            minimum(model.minimum);
            maximum(model.maximum);
            unit(model.unit);
            extendedStatistics(model.extendedStatistics);
        }

        public final Instant getTimestamp() {
            return timestamp;
        }

        public final void setTimestamp(Instant timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        @Transient
        public final Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public final Double getSampleCount() {
            return sampleCount;
        }

        public final void setSampleCount(Double sampleCount) {
            this.sampleCount = sampleCount;
        }

        @Override
        @Transient
        public final Builder sampleCount(Double sampleCount) {
            this.sampleCount = sampleCount;
            return this;
        }

        public final Double getAverage() {
            return average;
        }

        public final void setAverage(Double average) {
            this.average = average;
        }

        @Override
        @Transient
        public final Builder average(Double average) {
            this.average = average;
            return this;
        }

        public final Double getSum() {
            return sum;
        }

        public final void setSum(Double sum) {
            this.sum = sum;
        }

        @Override
        @Transient
        public final Builder sum(Double sum) {
            this.sum = sum;
            return this;
        }

        public final Double getMinimum() {
            return minimum;
        }

        public final void setMinimum(Double minimum) {
            this.minimum = minimum;
        }

        @Override
        @Transient
        public final Builder minimum(Double minimum) {
            this.minimum = minimum;
            return this;
        }

        public final Double getMaximum() {
            return maximum;
        }

        public final void setMaximum(Double maximum) {
            this.maximum = maximum;
        }

        @Override
        @Transient
        public final Builder maximum(Double maximum) {
            this.maximum = maximum;
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        @Transient
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        @Transient
        public final Builder unit(StandardUnit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        public final Map<String, Double> getExtendedStatistics() {
            if (extendedStatistics instanceof SdkAutoConstructMap) {
                return null;
            }
            return extendedStatistics;
        }

        public final void setExtendedStatistics(Map<String, Double> extendedStatistics) {
            this.extendedStatistics = DatapointValueMapCopier.copy(extendedStatistics);
        }

        @Override
        @Transient
        public final Builder extendedStatistics(Map<String, Double> extendedStatistics) {
            this.extendedStatistics = DatapointValueMapCopier.copy(extendedStatistics);
            return this;
        }

        @Override
        public Datapoint build() {
            return new Datapoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
