/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.cloudwatch.model.CloudWatchException;
import software.amazon.awssdk.services.cloudwatch.model.CloudWatchRequest;
import software.amazon.awssdk.services.cloudwatch.model.DashboardInvalidInputErrorException;
import software.amazon.awssdk.services.cloudwatch.model.DashboardNotFoundErrorException;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAlarmsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAlarmsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteDashboardsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteDashboardsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsForMetricRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsForMetricResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DisableAlarmActionsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DisableAlarmActionsResponse;
import software.amazon.awssdk.services.cloudwatch.model.EnableAlarmActionsRequest;
import software.amazon.awssdk.services.cloudwatch.model.EnableAlarmActionsResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetDashboardRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetDashboardResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricDataRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricDataResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStatisticsRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStatisticsResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricWidgetImageRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricWidgetImageResponse;
import software.amazon.awssdk.services.cloudwatch.model.InternalServiceException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidFormatException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidNextTokenException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidParameterCombinationException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidParameterValueException;
import software.amazon.awssdk.services.cloudwatch.model.LimitExceededException;
import software.amazon.awssdk.services.cloudwatch.model.ListDashboardsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListDashboardsResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricsResponse;
import software.amazon.awssdk.services.cloudwatch.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.cloudwatch.model.PutDashboardRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutDashboardResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricAlarmRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricAlarmResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricDataRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricDataResponse;
import software.amazon.awssdk.services.cloudwatch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudwatch.model.SetAlarmStateRequest;
import software.amazon.awssdk.services.cloudwatch.model.SetAlarmStateResponse;
import software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteAlarmsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteDashboardsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmHistoryRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmsForMetricRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DisableAlarmActionsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.EnableAlarmActionsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetDashboardRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricDataRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricStatisticsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricWidgetImageRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListDashboardsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListMetricsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutDashboardRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricAlarmRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricDataRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.SetAlarmStateRequestMarshaller;

/**
 * Internal implementation of {@link CloudWatchClient}.
 *
 * @see CloudWatchClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudWatchClient implements CloudWatchClient {
    private final SyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudWatchClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified alarms. In the event of an error, no alarms are deleted.
     * </p>
     *
     * @param deleteAlarmsRequest
     * @return Result of the DeleteAlarms operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteAlarms
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAlarmsResponse deleteAlarms(DeleteAlarmsRequest deleteAlarmsRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteAlarmsResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteAlarmsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteAlarmsRequest, DeleteAlarmsResponse>()
                .withOperationName("DeleteAlarms").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteAlarmsRequest)
                .withMarshaller(new DeleteAlarmsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes all dashboards that you specify. You may specify up to 100 dashboards to delete. If there is an error
     * during this call, no dashboards are deleted.
     * </p>
     *
     * @param deleteDashboardsRequest
     * @return Result of the DeleteDashboards operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws DashboardNotFoundErrorException
     *         The specified dashboard does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteDashboards
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDashboardsResponse deleteDashboards(DeleteDashboardsRequest deleteDashboardsRequest)
            throws InvalidParameterValueException, DashboardNotFoundErrorException, InternalServiceException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteDashboardsResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteDashboardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DeleteDashboardsRequest, DeleteDashboardsResponse>()
                .withOperationName("DeleteDashboards").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteDashboardsRequest)
                .withMarshaller(new DeleteDashboardsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the history for the specified alarm. You can filter the results by date range or item type. If an alarm
     * name is not specified, the histories for all alarms are returned.
     * </p>
     * <p>
     * CloudWatch retains the history of an alarm even if you delete the alarm.
     * </p>
     *
     * @param describeAlarmHistoryRequest
     * @return Result of the DescribeAlarmHistory operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmHistoryResponse describeAlarmHistory(DescribeAlarmHistoryRequest describeAlarmHistoryRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmHistoryResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmHistoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeAlarmHistoryRequest, DescribeAlarmHistoryResponse>()
                .withOperationName("DescribeAlarmHistory").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAlarmHistoryRequest)
                .withMarshaller(new DescribeAlarmHistoryRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the history for the specified alarm. You can filter the results by date range or item type. If an alarm
     * name is not specified, the histories for all alarms are returned.
     * </p>
     * <p>
     * CloudWatch retains the history of an alarm even if you delete the alarm.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAlarmHistory(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable responses = client.describeAlarmHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable responses = client
     *             .describeAlarmHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable responses = client.describeAlarmHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAlarmHistory(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAlarmHistoryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmHistory
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmHistoryIterable describeAlarmHistoryPaginator(DescribeAlarmHistoryRequest describeAlarmHistoryRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {
        return new DescribeAlarmHistoryIterable(this, applyPaginatorUserAgent(describeAlarmHistoryRequest));
    }

    /**
     * <p>
     * Retrieves the specified alarms. If no alarms are specified, all alarms are returned. Alarms can be retrieved by
     * using only a prefix for the alarm name, the alarm state, or a prefix for any action.
     * </p>
     *
     * @param describeAlarmsRequest
     * @return Result of the DescribeAlarms operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarms
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAlarmsResponse describeAlarms(DescribeAlarmsRequest describeAlarmsRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeAlarmsRequest, DescribeAlarmsResponse>()
                .withOperationName("DescribeAlarms").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAlarmsRequest)
                .withMarshaller(new DescribeAlarmsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the specified alarms. If no alarms are specified, all alarms are returned. Alarms can be retrieved by
     * using only a prefix for the alarm name, the alarm state, or a prefix for any action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAlarms(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable responses = client.describeAlarmsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable responses = client
     *             .describeAlarmsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable responses = client.describeAlarmsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAlarms(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest)} operation.</b>
     * </p>
     *
     * @param describeAlarmsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarms
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAlarmsIterable describeAlarmsPaginator(DescribeAlarmsRequest describeAlarmsRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {
        return new DescribeAlarmsIterable(this, applyPaginatorUserAgent(describeAlarmsRequest));
    }

    /**
     * <p>
     * Retrieves the alarms for the specified metric. To filter the results, specify a statistic, period, or unit.
     * </p>
     *
     * @param describeAlarmsForMetricRequest
     * @return Result of the DescribeAlarmsForMetric operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmsForMetric
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmsForMetric"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmsForMetricResponse describeAlarmsForMetric(DescribeAlarmsForMetricRequest describeAlarmsForMetricRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmsForMetricResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmsForMetricResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DescribeAlarmsForMetricRequest, DescribeAlarmsForMetricResponse>()
                .withOperationName("DescribeAlarmsForMetric").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(describeAlarmsForMetricRequest)
                .withMarshaller(new DescribeAlarmsForMetricRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Disables the actions for the specified alarms. When an alarm's actions are disabled, the alarm actions do not
     * execute when the alarm state changes.
     * </p>
     *
     * @param disableAlarmActionsRequest
     * @return Result of the DisableAlarmActions operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DisableAlarmActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DisableAlarmActions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DisableAlarmActionsResponse disableAlarmActions(DisableAlarmActionsRequest disableAlarmActionsRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DisableAlarmActionsResponse> responseHandler = protocolFactory
                .createResponseHandler(DisableAlarmActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<DisableAlarmActionsRequest, DisableAlarmActionsResponse>()
                .withOperationName("DisableAlarmActions").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(disableAlarmActionsRequest)
                .withMarshaller(new DisableAlarmActionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Enables the actions for the specified alarms.
     * </p>
     *
     * @param enableAlarmActionsRequest
     * @return Result of the EnableAlarmActions operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.EnableAlarmActions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/EnableAlarmActions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableAlarmActionsResponse enableAlarmActions(EnableAlarmActionsRequest enableAlarmActionsRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<EnableAlarmActionsResponse> responseHandler = protocolFactory
                .createResponseHandler(EnableAlarmActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<EnableAlarmActionsRequest, EnableAlarmActionsResponse>()
                .withOperationName("EnableAlarmActions").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(enableAlarmActionsRequest)
                .withMarshaller(new EnableAlarmActionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Displays the details of the dashboard that you specify.
     * </p>
     * <p>
     * To copy an existing dashboard, use <code>GetDashboard</code>, and then use the data returned within
     * <code>DashboardBody</code> as the template for the new dashboard when you call <code>PutDashboard</code> to
     * create the copy.
     * </p>
     *
     * @param getDashboardRequest
     * @return Result of the GetDashboard operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws DashboardNotFoundErrorException
     *         The specified dashboard does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetDashboard
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetDashboard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDashboardResponse getDashboard(GetDashboardRequest getDashboardRequest) throws InvalidParameterValueException,
            DashboardNotFoundErrorException, InternalServiceException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<GetDashboardResponse> responseHandler = protocolFactory
                .createResponseHandler(GetDashboardResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetDashboardRequest, GetDashboardResponse>()
                .withOperationName("GetDashboard").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDashboardRequest)
                .withMarshaller(new GetDashboardRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * You can use the <code>GetMetricData</code> API to retrieve as many as 100 different metrics in a single request,
     * with a total of as many as 100,800 datapoints. You can also optionally perform math expressions on the values of
     * the returned statistics, to create new time series that represent new insights into your data. For example, using
     * Lambda metrics, you could divide the Errors metric by the Invocations metric to get an error rate time series.
     * For more information about metric math expressions, see <a
     * href="http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax"
     * >Metric Math Syntax and Functions</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * Calls to the <code>GetMetricData</code> API have a different pricing structure than calls to
     * <code>GetMetricStatistics</code>. For more information about pricing, see <a
     * href="https://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * <p>
     * Amazon CloudWatch retains metric data as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Data points with a period of less than 60 seconds are available for 3 hours. These data points are
     * high-resolution metrics and are available only for custom metrics that have been defined with a
     * <code>StorageResolution</code> of 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 60 seconds (1-minute) are available for 15 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 300 seconds (5-minute) are available for 63 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 3600 seconds (1 hour) are available for 455 days (15 months).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data points that are initially published with a shorter period are aggregated together for long-term storage. For
     * example, if you collect data using a period of 1 minute, the data remains available for 15 days with 1-minute
     * resolution. After 15 days, this data is still available, but is aggregated and retrievable only with a resolution
     * of 5 minutes. After 63 days, the data is further aggregated and is available with a resolution of 1 hour.
     * </p>
     *
     * @param getMetricDataRequest
     * @return Result of the GetMetricData operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataResponse getMetricData(GetMetricDataRequest getMetricDataRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricDataResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetMetricDataRequest, GetMetricDataResponse>()
                .withOperationName("GetMetricData").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMetricDataRequest)
                .withMarshaller(new GetMetricDataRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets statistics for the specified metric.
     * </p>
     * <p>
     * The maximum number of data points returned from a single call is 1,440. If you request more than 1,440 data
     * points, CloudWatch returns an error. To reduce the number of data points, you can narrow the specified time range
     * and make multiple requests across adjacent time ranges, or you can increase the specified period. Data points are
     * not returned in chronological order.
     * </p>
     * <p>
     * CloudWatch aggregates data points based on the length of the period that you specify. For example, if you request
     * statistics with a one-hour period, CloudWatch aggregates all data points with time stamps that fall within each
     * one-hour period. Therefore, the number of values aggregated by CloudWatch is larger than the number of data
     * points returned.
     * </p>
     * <p>
     * CloudWatch needs raw data points to calculate percentile statistics. If you publish data using a statistic set
     * instead, you can only retrieve percentile statistics for this data if one of the following conditions is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The SampleCount value of the statistic set is 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Min and the Max values of the statistic set are equal.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Percentile statistics are not available for metrics when any of the metric values are negative numbers.
     * </p>
     * <p>
     * Amazon CloudWatch retains metric data as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Data points with a period of less than 60 seconds are available for 3 hours. These data points are
     * high-resolution metrics and are available only for custom metrics that have been defined with a
     * <code>StorageResolution</code> of 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 60 seconds (1-minute) are available for 15 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 300 seconds (5-minute) are available for 63 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 3600 seconds (1 hour) are available for 455 days (15 months).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data points that are initially published with a shorter period are aggregated together for long-term storage. For
     * example, if you collect data using a period of 1 minute, the data remains available for 15 days with 1-minute
     * resolution. After 15 days, this data is still available, but is aggregated and retrievable only with a resolution
     * of 5 minutes. After 63 days, the data is further aggregated and is available with a resolution of 1 hour.
     * </p>
     * <p>
     * CloudWatch started retaining 5-minute and 1-hour metric data as of July 9, 2016.
     * </p>
     * <p>
     * For information about metrics and dimensions supported by AWS services, see the <a
     * href="http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CW_Support_For_AWS.html">Amazon CloudWatch
     * Metrics and Dimensions Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     *
     * @param getMetricStatisticsRequest
     * @return Result of the GetMetricStatistics operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricStatistics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricStatistics" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMetricStatisticsResponse getMetricStatistics(GetMetricStatisticsRequest getMetricStatisticsRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, InvalidParameterCombinationException,
            InternalServiceException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricStatisticsResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricStatisticsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetMetricStatisticsRequest, GetMetricStatisticsResponse>()
                .withOperationName("GetMetricStatistics").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMetricStatisticsRequest)
                .withMarshaller(new GetMetricStatisticsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * You can use the <code>GetMetricWidgetImage</code> API to retrieve a snapshot graph of one or more Amazon
     * CloudWatch metrics as a bitmap image. You can then embed this image into your services and products, such as wiki
     * pages, reports, and documents. You could also retrieve images regularly, such as every minute, and create your
     * own custom live dashboard.
     * </p>
     * <p>
     * The graph you retrieve can include all CloudWatch metric graph features, including metric math and horizontal and
     * vertical annotations.
     * </p>
     * <p>
     * There is a limit of 20 transactions per second for this API. Each <code>GetMetricWidgetImage</code> action has
     * the following limits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * As many as 100 metrics in the graph.
     * </p>
     * </li>
     * <li>
     * <p>
     * Up to 100 KB uncompressed payload.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMetricWidgetImageRequest
     * @return Result of the GetMetricWidgetImage operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricWidgetImage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricWidgetImage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMetricWidgetImageResponse getMetricWidgetImage(GetMetricWidgetImageRequest getMetricWidgetImageRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricWidgetImageResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricWidgetImageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<GetMetricWidgetImageRequest, GetMetricWidgetImageResponse>()
                .withOperationName("GetMetricWidgetImage").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getMetricWidgetImageRequest)
                .withMarshaller(new GetMetricWidgetImageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the dashboards for your account. If you include <code>DashboardNamePrefix</code>, only those
     * dashboards with names starting with the prefix are listed. Otherwise, all dashboards in your account are listed.
     * </p>
     * <p>
     * <code>ListDashboards</code> returns up to 1000 results on one page. If there are more than 1000 dashboards, you
     * can call <code>ListDashboards</code> again and include the value you received for <code>NextToken</code> in the
     * first call, to receive the next 1000 results.
     * </p>
     *
     * @param listDashboardsRequest
     * @return Result of the ListDashboards operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListDashboards
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListDashboards" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDashboardsResponse listDashboards(ListDashboardsRequest listDashboardsRequest)
            throws InvalidParameterValueException, InternalServiceException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<ListDashboardsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListDashboardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListDashboardsRequest, ListDashboardsResponse>()
                .withOperationName("ListDashboards").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listDashboardsRequest)
                .withMarshaller(new ListDashboardsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the specified metrics. You can use the returned metrics with <a>GetMetricData</a> or
     * <a>GetMetricStatistics</a> to obtain statistical data.
     * </p>
     * <p>
     * Up to 500 results are returned for any one call. To retrieve additional results, use the returned token with
     * subsequent calls.
     * </p>
     * <p>
     * After you create a metric, allow up to fifteen minutes before the metric appears. Statistics about the metric,
     * however, are available sooner using <a>GetMetricData</a> or <a>GetMetricStatistics</a>.
     * </p>
     *
     * @param listMetricsRequest
     * @return Result of the ListMetrics operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetrics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMetricsResponse listMetrics(ListMetricsRequest listMetricsRequest) throws InternalServiceException,
            InvalidParameterValueException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<ListMetricsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<ListMetricsRequest, ListMetricsResponse>()
                .withOperationName("ListMetrics").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listMetricsRequest)
                .withMarshaller(new ListMetricsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * List the specified metrics. You can use the returned metrics with <a>GetMetricData</a> or
     * <a>GetMetricStatistics</a> to obtain statistical data.
     * </p>
     * <p>
     * Up to 500 results are returned for any one call. To retrieve additional results, use the returned token with
     * subsequent calls.
     * </p>
     * <p>
     * After you create a metric, allow up to fifteen minutes before the metric appears. Statistics about the metric,
     * however, are available sooner using <a>GetMetricData</a> or <a>GetMetricStatistics</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMetrics(software.amazon.awssdk.services.cloudwatch.model.ListMetricsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable responses = client.listMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable responses = client.listMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.ListMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable responses = client.listMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMetrics(software.amazon.awssdk.services.cloudwatch.model.ListMetricsRequest)} operation.</b>
     * </p>
     *
     * @param listMetricsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetrics
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMetricsIterable listMetricsPaginator(ListMetricsRequest listMetricsRequest) throws InternalServiceException,
            InvalidParameterValueException, AwsServiceException, SdkClientException, CloudWatchException {
        return new ListMetricsIterable(this, applyPaginatorUserAgent(listMetricsRequest));
    }

    /**
     * <p>
     * Creates a dashboard if it does not already exist, or updates an existing dashboard. If you update a dashboard,
     * the entire contents are replaced with what you specify here.
     * </p>
     * <p>
     * There is no limit to the number of dashboards in your account. All dashboards in your account are global, not
     * region-specific.
     * </p>
     * <p>
     * A simple way to create a dashboard using <code>PutDashboard</code> is to copy an existing dashboard. To copy an
     * existing dashboard using the console, you can load the dashboard and then use the View/edit source command in the
     * Actions menu to display the JSON block for that dashboard. Another way to copy a dashboard is to use
     * <code>GetDashboard</code>, and then use the data returned within <code>DashboardBody</code> as the template for
     * the new dashboard when you call <code>PutDashboard</code>.
     * </p>
     * <p>
     * When you create a dashboard with <code>PutDashboard</code>, a good practice is to add a text widget at the top of
     * the dashboard with a message that the dashboard was created by script and should not be changed in the console.
     * This message could also point console users to the location of the <code>DashboardBody</code> script or the
     * CloudFormation template used to create the dashboard.
     * </p>
     *
     * @param putDashboardRequest
     * @return Result of the PutDashboard operation returned by the service.
     * @throws DashboardInvalidInputErrorException
     *         Some part of the dashboard data is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutDashboard
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutDashboard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutDashboardResponse putDashboard(PutDashboardRequest putDashboardRequest) throws DashboardInvalidInputErrorException,
            InternalServiceException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutDashboardResponse> responseHandler = protocolFactory
                .createResponseHandler(PutDashboardResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutDashboardRequest, PutDashboardResponse>()
                .withOperationName("PutDashboard").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putDashboardRequest)
                .withMarshaller(new PutDashboardRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates or updates an alarm and associates it with the specified metric or metric math expression.
     * </p>
     * <p>
     * When this operation creates an alarm, the alarm state is immediately set to <code>INSUFFICIENT_DATA</code>. The
     * alarm is then evaluated and its state is set appropriately. Any actions associated with the new state are then
     * executed.
     * </p>
     * <p>
     * When you update an existing alarm, its state is left unchanged, but the update completely overwrites the previous
     * configuration of the alarm.
     * </p>
     * <p>
     * If you are an IAM user, you must have Amazon EC2 permissions for some alarm operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>iam:CreateServiceLinkedRole</code> for all alarms with EC2 actions
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeInstanceStatus</code> and <code>ec2:DescribeInstances</code> for all alarms on EC2 instance
     * status metrics
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:StopInstances</code> for alarms with stop actions
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:TerminateInstances</code> for alarms with terminate actions
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:DescribeInstanceRecoveryAttribute</code> and <code>ec2:RecoverInstances</code> for alarms with recover
     * actions
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you have read/write permissions for Amazon CloudWatch but not for Amazon EC2, you can still create an alarm,
     * but the stop or terminate actions are not performed. However, if you are later granted the required permissions,
     * the alarm actions that you created earlier are performed.
     * </p>
     * <p>
     * If you are using an IAM role (for example, an EC2 instance profile), you cannot stop or terminate the instance
     * using alarm actions. However, you can still see the alarm state and perform any other actions such as Amazon SNS
     * notifications or Auto Scaling policies.
     * </p>
     * <p>
     * If you are using temporary security credentials granted using AWS STS, you cannot stop or terminate an EC2
     * instance using alarm actions.
     * </p>
     * <p>
     * The first time you create an alarm in the AWS Management Console, the CLI, or by using the PutMetricAlarm API,
     * CloudWatch creates the necessary service-linked role for you. The service-linked role is called
     * <code>AWSServiceRoleForCloudWatchEvents</code>. For more information, see <a href=
     * "http://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role"
     * >AWS service-linked role</a>.
     * </p>
     *
     * @param putMetricAlarmRequest
     * @return Result of the PutMetricAlarm operation returned by the service.
     * @throws LimitExceededException
     *         The quota for alarms for this customer has already been reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricAlarm
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricAlarm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetricAlarmResponse putMetricAlarm(PutMetricAlarmRequest putMetricAlarmRequest) throws LimitExceededException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutMetricAlarmResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricAlarmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutMetricAlarmRequest, PutMetricAlarmResponse>()
                .withOperationName("PutMetricAlarm").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putMetricAlarmRequest)
                .withMarshaller(new PutMetricAlarmRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Publishes metric data points to Amazon CloudWatch. CloudWatch associates the data points with the specified
     * metric. If the specified metric does not exist, CloudWatch creates the metric. When CloudWatch creates a metric,
     * it can take up to fifteen minutes for the metric to appear in calls to <a>ListMetrics</a>.
     * </p>
     * <p>
     * You can publish either individual data points in the <code>Value</code> field, or arrays of values and the number
     * of times each value occurred during the period by using the <code>Values</code> and <code>Counts</code> fields in
     * the <code>MetricDatum</code> structure. Using the <code>Values</code> and <code>Counts</code> method enables you
     * to publish up to 150 values per metric with one <code>PutMetricData</code> request, and supports retrieving
     * percentile statistics on this data.
     * </p>
     * <p>
     * Each <code>PutMetricData</code> request is limited to 40 KB in size for HTTP POST requests. You can send a
     * payload compressed by gzip. Each request is also limited to no more than 20 different metrics.
     * </p>
     * <p>
     * Although the <code>Value</code> parameter accepts numbers of type <code>Double</code>, CloudWatch rejects values
     * that are either too small or too large. Values must be in the range of 8.515920e-109 to 1.174271e+108 (Base 10)
     * or 2e-360 to 2e360 (Base 2). In addition, special values (for example, NaN, +Infinity, -Infinity) are not
     * supported.
     * </p>
     * <p>
     * You can use up to 10 dimensions per metric to further clarify what data the metric collects. For more information
     * about specifying dimensions, see <a
     * href="http://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
     * Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * Data points with time stamps from 24 hours ago or longer can take at least 48 hours to become available for
     * <a>GetMetricData</a> or <a>GetMetricStatistics</a> from the time they are submitted.
     * </p>
     * <p>
     * CloudWatch needs raw data points to calculate percentile statistics. If you publish data using a statistic set
     * instead, you can only retrieve percentile statistics for this data if one of the following conditions is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>SampleCount</code> value of the statistic set is 1 and <code>Min</code>, <code>Max</code>, and
     * <code>Sum</code> are all equal.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>Min</code> and <code>Max</code> are equal, and <code>Sum</code> is equal to <code>Min</code> multiplied
     * by <code>SampleCount</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param putMetricDataRequest
     * @return Result of the PutMetricData operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricData
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetricDataResponse putMetricData(PutMetricDataRequest putMetricDataRequest) throws InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, InternalServiceException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutMetricDataResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<PutMetricDataRequest, PutMetricDataResponse>()
                .withOperationName("PutMetricData").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(putMetricDataRequest)
                .withMarshaller(new PutMetricDataRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Temporarily sets the state of an alarm for testing purposes. When the updated state differs from the previous
     * value, the action configured for the appropriate state is invoked. For example, if your alarm is configured to
     * send an Amazon SNS message when an alarm is triggered, temporarily changing the alarm state to <code>ALARM</code>
     * sends an SNS message. The alarm returns to its actual state (often within seconds). Because the alarm state
     * change happens quickly, it is typically only visible in the alarm's <b>History</b> tab in the Amazon CloudWatch
     * console or through <a>DescribeAlarmHistory</a>.
     * </p>
     *
     * @param setAlarmStateRequest
     * @return Result of the SetAlarmState operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InvalidFormatException
     *         Data was not syntactically valid JSON.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.SetAlarmState
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/SetAlarmState" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetAlarmStateResponse setAlarmState(SetAlarmStateRequest setAlarmStateRequest) throws ResourceNotFoundException,
            InvalidFormatException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<SetAlarmStateResponse> responseHandler = protocolFactory
                .createResponseHandler(SetAlarmStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();

        return clientHandler.execute(new ClientExecutionParams<SetAlarmStateRequest, SetAlarmStateResponse>()
                .withOperationName("SetAlarmState").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(setAlarmStateRequest)
                .withMarshaller(new SetAlarmStateRequestMarshaller(protocolFactory)));
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValue")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterCombination")
                                .exceptionBuilderSupplier(InvalidParameterCombinationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFormat")
                                .exceptionBuilderSupplier(InvalidFormatException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingParameter")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(DashboardNotFoundErrorException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterInput")
                                .exceptionBuilderSupplier(DashboardInvalidInputErrorException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextToken")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(CloudWatchException::builder).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CloudWatchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
