/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A set of metrics that are associated with an entity, such as a specific service or resource. Contains the entity and
 * the list of metric data associated with it.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EntityMetricData implements SdkPojo, Serializable,
        ToCopyableBuilder<EntityMetricData.Builder, EntityMetricData> {
    private static final SdkField<Entity> ENTITY_FIELD = SdkField.<Entity> builder(MarshallingType.SDK_POJO).memberName("Entity")
            .getter(getter(EntityMetricData::entity)).setter(setter(Builder::entity)).constructor(Entity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Entity").build()).build();

    private static final SdkField<List<MetricDatum>> METRIC_DATA_FIELD = SdkField
            .<List<MetricDatum>> builder(MarshallingType.LIST)
            .memberName("MetricData")
            .getter(getter(EntityMetricData::metricData))
            .setter(setter(Builder::metricData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDatum> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDatum::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENTITY_FIELD,
            METRIC_DATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Entity", ENTITY_FIELD);
                    put("MetricData", METRIC_DATA_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final Entity entity;

    private final List<MetricDatum> metricData;

    private EntityMetricData(BuilderImpl builder) {
        this.entity = builder.entity;
        this.metricData = builder.metricData;
    }

    /**
     * <p>
     * The entity associated with the metrics.
     * </p>
     * 
     * @return The entity associated with the metrics.
     */
    public final Entity entity() {
        return entity;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricData property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetricData() {
        return metricData != null && !(metricData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metric data.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricData} method.
     * </p>
     * 
     * @return The metric data.
     */
    public final List<MetricDatum> metricData() {
        return metricData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(entity());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricData() ? metricData() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EntityMetricData)) {
            return false;
        }
        EntityMetricData other = (EntityMetricData) obj;
        return Objects.equals(entity(), other.entity()) && hasMetricData() == other.hasMetricData()
                && Objects.equals(metricData(), other.metricData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EntityMetricData").add("Entity", entity())
                .add("MetricData", hasMetricData() ? metricData() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Entity":
            return Optional.ofNullable(clazz.cast(entity()));
        case "MetricData":
            return Optional.ofNullable(clazz.cast(metricData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<EntityMetricData, T> g) {
        return obj -> g.apply((EntityMetricData) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EntityMetricData> {
        /**
         * <p>
         * The entity associated with the metrics.
         * </p>
         * 
         * @param entity
         *        The entity associated with the metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entity(Entity entity);

        /**
         * <p>
         * The entity associated with the metrics.
         * </p>
         * This is a convenience method that creates an instance of the {@link Entity.Builder} avoiding the need to
         * create one manually via {@link Entity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Entity.Builder#build()} is called immediately and its result is
         * passed to {@link #entity(Entity)}.
         * 
         * @param entity
         *        a consumer that will call methods on {@link Entity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entity(Entity)
         */
        default Builder entity(Consumer<Entity.Builder> entity) {
            return entity(Entity.builder().applyMutation(entity).build());
        }

        /**
         * <p>
         * The metric data.
         * </p>
         * 
         * @param metricData
         *        The metric data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricData(Collection<MetricDatum> metricData);

        /**
         * <p>
         * The metric data.
         * </p>
         * 
         * @param metricData
         *        The metric data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricData(MetricDatum... metricData);

        /**
         * <p>
         * The metric data.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudwatch.model.MetricDatum.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.cloudwatch.model.MetricDatum#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudwatch.model.MetricDatum.Builder#build()} is called immediately
         * and its result is passed to {@link #metricData(List<MetricDatum>)}.
         * 
         * @param metricData
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudwatch.model.MetricDatum.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricData(java.util.Collection<MetricDatum>)
         */
        Builder metricData(Consumer<MetricDatum.Builder>... metricData);
    }

    static final class BuilderImpl implements Builder {
        private Entity entity;

        private List<MetricDatum> metricData = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EntityMetricData model) {
            entity(model.entity);
            metricData(model.metricData);
        }

        public final Entity.Builder getEntity() {
            return entity != null ? entity.toBuilder() : null;
        }

        public final void setEntity(Entity.BuilderImpl entity) {
            this.entity = entity != null ? entity.build() : null;
        }

        @Override
        public final Builder entity(Entity entity) {
            this.entity = entity;
            return this;
        }

        public final List<MetricDatum.Builder> getMetricData() {
            List<MetricDatum.Builder> result = MetricDataCopier.copyToBuilder(this.metricData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetricData(Collection<MetricDatum.BuilderImpl> metricData) {
            this.metricData = MetricDataCopier.copyFromBuilder(metricData);
        }

        @Override
        public final Builder metricData(Collection<MetricDatum> metricData) {
            this.metricData = MetricDataCopier.copy(metricData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricData(MetricDatum... metricData) {
            metricData(Arrays.asList(metricData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricData(Consumer<MetricDatum.Builder>... metricData) {
            metricData(Stream.of(metricData).map(c -> MetricDatum.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public EntityMetricData build() {
            return new EntityMetricData(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
